#! /usr/bin/perl
#
# system_info.pl:    Queries BIOS information and writes results to a file
#
# chkconfig:         345 96 6 
#
# description:       This script does two things, upon first boot (and only on first boot) \
#                    it sets the Avaya Asset Tag value in the BIOS. Another function of \
#                    this script is to query BIOS information like serial number, \
#                    model name of the server and Avaya Asset Tag information and \
#                    writes the information to /opt/Avaya/system_info/system_info.txt, \
#                    the query functionality is performed at each bootup.

use Switch;

# variables used for processing
my $fragment, $vendor, $asset_tag, $len, $last_char, $tmp_file, $dest_file, $digit;

# variables used for storing the results
my $material_code_value, $model_info_value, $sys_serial_num_value;

# first we clean up, let's remove the temporary file used for parsing it
# should have been removed at the end of this script cleanup but we just
# want to make sure it's not there
$tmp_file = '/tmp/decode.txt';
$dest_file = '/opt/Avaya/system_info/system_info.txt';

# setting default values
$material_code_value = 'N/A';
$model_info_string = 'MODEL_INFO=';
$model_info_value = $model_info_string;
$sys_serial_num_value = 'SYS_SERIAL_NUM=';

# if temporary file exists then delete it
if (-e $tmp_file)
{
	unlink($tmp_file);
}

# start - this portion is only executed once at first bootup
# if /tmp/bios directory exists then this is first time boot
if (-e "/tmp/bios")
{
	system("logger system_info: first boot - setting asset tag value");	
    
	# set assettag flag machine
	system("touch /opt/Avaya/assettag");

	# execute script that sets the asset tag value in BIOS
	system("/tmp/bios/set_system_info.pl");

	# remove all bios-tag rpm package contents, this also
	# removes the /tmp/bios directory
	system("rpm -e bios-tag");
}
# end - this portion ends the first bootup section

# here we run dmidecode to get the BIOS info, we only want the relevant
# portion which is between "System Information" and "SKU" keywords, we get
# this info and put it in the temporary file so we can work on it
system("dmidecode | sed -n \"/System Information/,/SKU/p\" > " . $tmp_file);

open (FILE, $tmp_file);

# let's parse the file
while (<FILE>) 
{
	chomp;
	@words = split;

	# here we retrieve the vendor name "Dell" or "HP" and store it
	# in the $vendor variable
	if ($words[0] eq "Manufacturer:")
	{
		$vendor = $words[1];		
	}

	# here we retrieve the product name (the name of the server) and
	# prepend the $vendor variable so it product name starts with
	# vendor name, lastly we store it in the $model_info_value variable
	if ($words[0] eq "Product")
	{
		# get the substring of this whole line, after the "Name:" which
		# will be the server name
		$fragment =  substr $_, index($_, 'Name:') + 6;
		$model_info_value = $model_info_value . $vendor . " " . $fragment . "\n";
	}
	
	# here we retreive the serial number that vendor uses for service requests
	# and store it in the $sys_serial_num_value veriable
	if ($words[0] eq "Serial")
	{
		$sys_serial_num_value = $sys_serial_num_value . $words[2] . "\n";
	}
}

close (FILE);

# the Asset Tag information is at a different section of the dmidecode information,
# so we repeat process of extracting information between "Chassis Information" and 
# "Thermal" tags, and pipe the results to our temporary file
system("dmidecode | sed -n \"/Chassis Information/,/Thermal/p\" > " . $tmp_file);

open (FILE, $tmp_file);

# parsing for Asset Tag value
while (<FILE>)
{
	chomp;

        @words = split;

	  # we found the Asset Tag line
        if ($words[0] eq "Asset")
        {
		$asset_tag = $words[2];
		
		# getting the length of the asset tag
		$len = length $asset_tag;

		# Avaya valid asset tags are 6-9 characters all digits, this value 
		# (with sufix "Y") is set during staging by our third party in 
		# the server BIOS. There is nothing preventing a business
		# partner from setting asset tag values on the servers they
		# provide. So to make it less likely that a business partner
		# potential asset tag value is not mistaken for our, Avaya 
		# adds a sufix of "Y" to the Avaya asset tag. This means that 
		# we increase the number of valid length from 7-10. Lastly the
		# material codes are obtained from the factory.conf file during
		# staging, in that file the material codes must be padded with
		# zeros (on the left) to always have the length of 9. So 9 plus
		# the "Y" character makes the Asset Tag value always 10 digits long.
		if ($len != 10)
		{
			goto EXIT;
		}	

		# we have an Asset Tag value that might be valid lets
		# check to see if it ends with the "Y" character since
		# Avaya valid Asset Tags have to end with "Y" character

		# here we are getting the last character
		$last_char = substr $asset_tag,-1,1;

		# if the Asset Tag ends with "Y" then we're in business
		if ($last_char eq "Y")
		{
			# subtracting len by 1 since we will be stipping off the "Y"
			$len = $len - 1;

			# getting a substring of the Asset Tag without the last char
			$asset_tag = substr ($asset_tag, 0, $len);
			
			if ($asset_tag =~ /^[+-]?\d+$/)
			{
				# this is numeric value, lets strip off any leading zeros and
				# check that the nuber is 6-9 digits long, since that is a valid
				# Avaya material code
				$digit = $asset_tag + 0;

				$len = length $digit;
				
				# if the asset tag value is less then 6 digits in legth or more
				# then 9 digits then we know this is not a valid Avaya Asset Tag
				if (($len < 6) || ($len > 9))
				{
					goto EXIT;
				}

				# we have a valid Avaya material code value
				$material_code_value = "MATERIAL_CODE=" . $asset_tag . "\n";
			}
			else # there are non-numeric characters present, this is a NON Avaya system
			{
				goto EXIT;
			}
		}
		else # if the character is not "Y" then this is a NON Avaya system
		{
			goto EXIT;
		}		
        }
}

EXIT:
close (FILE);

# clean up by deleting the decode.txt file
if (-e $tmp_file)
{
        unlink($tmp_file);
}

# we got all the data, all that is left is to
# write the information to the output file
open (SYS_INFO_FILE, ">" . $dest_file);

# write the results to the destination file
# write material code only if there was a valid Avaya
# material code present
if ($material_code_value ne 'N/A')
{
	print SYS_INFO_FILE $material_code_value;

	# remove both leading and trailing whitespace including end of line
	$trimmed_model_info = $model_info_value;
	$trimmed_model_info =~ s/^\s+|\s+$//g;

	# on Dell servers if the model info is not available, then
	# key on the material code to set the model info explicitly
	if ($trimmed_model_info eq $model_info_string)
	{
		switch($asset_tag)
		{
			case '000302786'      { $model_info_value = $model_info_string . "Dell PowerEdge R210 II" . "\n"; }
			case '000302787'      { $model_info_value = $model_info_string . "Dell PowerEdge R210 II" . "\n"; }
			case '000302788'      { $model_info_value = $model_info_string . "Dell PowerEdge R210 II" . "\n"; }
			case '000306961'      { $model_info_value = $model_info_string . "Dell PowerEdge R620" . "\n"; }
			case '000380223'      { $model_info_value = $model_info_string . "Dell PowerEdge R220 II" . "\n"; }
			case '000380224'      { $model_info_value = $model_info_string . "Dell PowerEdge R220 II" . "\n"; }
			case '000380225'      { $model_info_value = $model_info_string . "Dell PowerEdge R220 II" . "\n"; }
			case '000380226'      { $model_info_value = $model_info_string . "Dell PowerEdge R220 II" . "\n"; }
			case '000383542'      { $model_info_value = $model_info_string . "Dell PowerEdge R630" . "\n"; }
			case '000390081'      { $model_info_value = $model_info_string . "Dell PowerEdge R230 XL" . "\n"; }
			case '000390082'      { $model_info_value = $model_info_string . "Dell PowerEdge R230 XL" . "\n"; }
			else                  { ; }
		}
    }
}


print SYS_INFO_FILE $model_info_value;
print SYS_INFO_FILE $sys_serial_num_value;

close (SYS_INFO_FILE);

# used for debuging and troubleshooting only
##print $material_code_value;
##print $model_info_value;
##print $sys_serial_num_value;

exit;
