#!/bin/bash

# Prevent shell globbing for patterns containing '*'
set -f
echo $$ > /sys/fs/cgroup/systemd/tasks
CONFIG_DESCR="#MidMarket Avaya Configuration"
CONFIG_EMERG_DESCR="#MidMarket Avaya Emergency Configuration"
CONFIG_MODULES="\n#Modules\n\n\$ModLoad imuxsock\n\$ModLoad imklog\n\$ModLoad immark\n \$WorkDirectory /var/log\n\n"
CONFIG_FORMAT="\$template messageformat,\"<%PRI%> %\$YEAR% %TIMESTAMP% %HOSTNAME% %syslogtag% %msg%\\\n\"\n\$ActionFileDefaultTemplate messageformat\n"
CONFIG_UDP_SERVER="\$UDPServerRun 514\n"
CONFIG_DEFAULT_PERM="\n\$EscapeControlCharactersOnReceive off\n\$MaxMessageSize 8k\n\$SystemLogRateLimitInterval 0 # turn off rate limit\n"
CONFIG_SERVER_REC_TEMPLATE="\n\$template SGLogs,\"/var/log/rsyslog/%fromhost-ip%/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%.log\"\n*.* -?SGLogs"
CONFIG_SERVER_STOP_REMOTE_PROCESSING="if \$fromhost-ip != '127.0.0.1' and \$fromhost-ip != '' then stop"
CONFIG_LOG_INFO_AND_HIGHER="*.info;mail.none;authpriv.none;cron.none\t/var/log/messages"
CONFIG_LOG_MAIL="mail.*\t\t\t\t\t\t/var/log/maillog"
CONFIG_LOG_CRON="cron.*\t\t\t\t\\t\t/var/log/cron"
CONFIG_LOG_EMERGENCY="*.emerg\t\t\t\t\t\t/var/log/emerg"
CONFIG_LOG_BOOT="local7.*\t\t\t\t\t/var/log/local7.log"
CONFIG_LOCAL_DESCR="\n#Local configuration\n"
CONFIG_REMOTE_DESCR="\n#Remote configuration"

TEMPCONFFILE="`mktemp -t rsysconfig_tmp.XXXXXXXX`" #configuration /etc/rsyslog.conf
TEMPSERVERCONF="`mktemp -t networkservercfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/network_server.conf
TEMPJADECONF="`mktemp -t jadeclientcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/jade_network_client.conf
TEMPIPOFFICECONF="`mktemp -t ipofficecfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoffice_network_client.conf
TEMPWEBCONTROLCONF="`mktemp -t webcontrolcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/webcontrol_network_client.conf
TEMPWATCHDOGCONF="`mktemp -t watchdogcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoffice_network_client.conf
TEMPVMPROCONF="`mktemp -t vmprocfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoffice_network_client.conf
TEMPMEDIAMANAGERCONF="`mktemp -t mediamanagercfg_tmp.XXXXXXXX`"
TEMPONEXCONF="`mktemp -t ipofficecfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoffice_network_client.conf
TEMPSYSTEMCONF="`mktemp -t systemcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/system_network_client.conf
TEMPSENDERSCONF="`mktemp -t allowedsendcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/allowed_senders.conf
TEMPWEBMANAGERCONF="`mktemp -t webmanagercfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipowebmanager_network_client.conf
TEMPICRCONF="`mktemp -t icrcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoicr_network_client.conf
TEMPWEBRTCGWCONF="`mktemp -t webrtcgwcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipowebrtcgw_network_client.conf
TEMPAMCONF="`mktemp -t amcfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoam_network_client.conf
TEMPAMSECURITYCONF="`mktemp -t amsecuritycfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/ipoamsecurity_network_client.conf
TEMPSYSMONCONF="`mktemp -t sysmoncfg_tmp.XXXXXXXX`" #configuration /etc/rsyslog.d/filter_sysmon.conf
OUTPUTLOG=/var/log/configure_rsyslog_output.log
CONF_FILE=/opt/Avaya/scripts/config_files/syslog_config
LIST_NUMBERFWD=0
CRON_TIMEOUT=10
maxlogsize="30000000"
maxsysmonsize="179000000"
port_tls=""
port_tcp="514"
port_udp=""
send_protocol=""
otherapp1_ip=""
otherapp1_port=""
otherapp1_protocol=""
otherapp2_ip=""
otherapp2_port=""
otherapp2_protocol=""
server1_ip=""
server1_port=""
server1_protocol=""
server2_ip=""
server2_port=""
server2_protocol=""
sources=""
server_type=""

rm -f $OUTPUTLOG
# Function to log in file
# Args: $1 - Line number
#       $2 - Error 
function log_to_file(){
    echo `date +"%d.%m.%Y %T"`" | Line $1: $2" >> $OUTPUTLOG
}

log_to_file "$LINENO" "Rsyslog configuration script started"

if [ -f /etc/sysconfig/ipoffice ]; then
    . /etc/sysconfig/ipoffice
else
    echo "Could not find ipoffice configuration file"
    log_to_file "$LINENO" "Could not find ipoffice configuration file"
    exit
fi

if [ -z "${IPOFFICE_BE_MODE}" ]; then
    echo "IPOffice is not configured yet"
     log_to_file "$LINENO" "IPOffice is not configured yet"
    exit
fi

if [ -z "${IPOFFICE_LAN1}" ]; then
    echo "Could not find ipoffice running interface"
    log_to_file "$LINENO" "Could not find ipoffice running interface"
    exit
fi


# Function to get values from syslog_conf file
function configuration_values(){
if [ ! -f $CONF_FILE ]; then
    log_to_file "$LINENO" "Syslog configuration file($CONF_FILE) not found."
    exit 1
fi
while read line
do
name=$(echo ${line%=*})
value=$(echo ${line#*=})
    if [ "$name" == "max_size" ]; then
        maxlogsize="$value"
    fi
    if [ "$name" == "max_sysmon_size" ]; then
        maxsysmonsize="$value"
    fi
    if [ "$name" == "port_tcp" ]; then
        port_tcp="$value"
    fi
    if [ "$name" == "port_udp" ]; then
        port_udp="$value"
    fi
    if [ "$name" == "port_tls" ]; then
        port_tls="$value"
    fi

    if [ "$name" == "otherapp1_ip" ]; then
        otherapp1_ip="$value"
    fi
    if [ "$name" == "otherapp1_port" ]; then
        otherapp1_port="$value"
    fi
    if [ "$name" == "otherapp1_protocol" ]; then
        otherapp1_protocol="$value"
    fi
    if [ "$name" == "otherapp2_ip" ]; then
        otherapp2_ip="$value"
    fi
    if [ "$name" == "otherapp2_port" ]; then
        otherapp2_port="$value"
    fi
    if [ "$name" == "otherapp2_protocol" ]; then
        otherapp2_protocol="$value"
    fi
    if [ "$name" == "server1_ip" ]; then
        server1_ip="$value"
    fi
    if [ "$name" == "server1_port" ]; then
        server1_port="$value"
    fi
    if [ "$name" == "server1_protocol" ]; then
        server1_protocol="$value"
    fi
    if [ "$name" == "server2_ip" ]; then
        server2_ip="$value"
    fi
    if [ "$name" == "server2_port" ]; then
        server2_port="$value"
    fi
    if [ "$name" == "server2_protocol" ]; then
        server2_protocol="$value"
    fi
    if [ "$name" == "sources" ]; then
        sources="$value"
    fi

done < $CONF_FILE
    if [ "$port_tls" == "" ]; then
        if [ "$port_tcp" == "" ]; then
            if [ "$port_udp" == "" ]; then
                send_protocol=""
            else
                send_protocol="UDP"
            fi
        else
            send_protocol="TCP"
        fi
    else
        send_protocol="TLS"
    fi
}

# get latest configuration
configuration_values

# Function to add output channel for an application
# Args: $1 - Application name (coresponds with $programname from rsyslog config)
#	$2 - Path to log file
function add_logrotation(){
    app_name=$1
    path=$2
    echo "\n\$outchannel ${app_name}log_rotation,${path},$maxlogsize,/opt/Avaya/scripts/./Log_rotation_script ${app_name}"
    if [ "$app_name" == "secure" ]; then
        echo "\nif \$syslogfacility-text == 'authpriv' and not ( ( \$msg contains 'TTY=unknown ; PWD=/opt/webcontrol ; USER=root ; COMMAND=/usr/bin/curl -k -A WebControlPanel' ) or \$msg contains 'configure_rsyslog.sh' )  then :omfile:\$${app_name}log_rotation\n"
    elif [ "$app_name" == "spooler" ]; then
        echo "\nif (\$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit')) then :omfile:\$${app_name}log_rotation\n"
    else
        echo "\nif \$programname == '${app_name}' then :omfile:\$${app_name}log_rotation\n"
    fi
}

# Function to discard logs from an application
# Args: $1 - Application name (coresponds with $programname from rsyslog config)
function discard_logs(){
    app_name=$1
    if [ "$app_name" == "secure" ]; then
        echo "\nif \$syslogfacility-text == 'authpriv' then stop\n"
    elif [ "$app_name" == "spooler" ]; then
        echo "\nif (\$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit')) then stop\n"
    else
        echo "\nif \$programname == '${app_name}' then stop\n"
    fi
}

# Function to include a configuration file on rsyslog
# Args: $1 - path to configuration file
function add_IncludeConfig(){
    path_to_config=$1
    echo "\n\$IncludeConfig ${path_to_config}"
}

# Function to specify hosts from which messages are being accepted
# Args: $1 - protocol ( TCP or UDP)
#	$2 - host ip
function add_senders(){
    protocol=$1
    sender=$2
    echo "\$AllowedSender ${protocol},${sender}"
}


# Function to create queue and send to forwarding addresses
# creates a new queue for every forward server
# maximum 4 forward servers( 2 set on Manager(usualy Secondary|Primary) and 2 on webcontrol )
# log 
# Args: $1 - conditions
#	$2 - forward address format: [@@|@]A.B.C.D:PORT
function create_list_and_forward()
{
    if [ "$#" != "2" ]; then
        log_to_file "$LINENO" "create_list_and_forward - function needs 2 arguments"
        exit 1
    fi
    if [[ $2 == @@* ]]; then
       echo "\$ActionQueueType LinkedList\n"
       echo "\$ActionQueueFileName srvrfwd$LIST_NUMBERFWD\n"
       echo "\$ActionResumeRetryCount -1\n"
       echo "\$ActionQueueSaveOnShutdown on\n"
       
       ((LIST_NUMBERFWD++))
       if [ -z "${2##*@@(o)*}" ]; then
           echo "\$ActionSendStreamDriver gtls\n"
           echo "\$ActionSendStreamDriverMode 1\n"
           echo "\$ActionSendStreamDriverAuthMode anon\n"
       fi
   fi    
   echo "\nif $1 then $2\n"
}

# Function to send local messages to server1 and server2
# this are set via Manager
# Args: $1 - program name
#       $2 -"system" or null for other apps
function send_logs(){
     if [ "$#" != "2" ]; then
         echo "send_logs - function needs 2 arguments"
         exit 1
     fi
     # initialize conditions 
     tempstr="\$programname == '$1'"
     syst=""
     if [ "$2" == "system" ]; then
         log_to_file "$LINENO" "Forward $2 logs"
         tempstr="$1"
     else
         log_to_file "$LINENO" "Forward $1 logs"
     fi
     fwd=""
     
     # forward to servers defined in Manager
     # these ip-addreses and ports usually are the primary and the secondary servers
     if [ ! -z "$server1_ip" -a ! -z "$server1_port" -a ! -z "$server1_protocol" ]; then
        protocol="@@" #TCP
        if [ "$server1_protocol" == "UDP" ]; then
            protocol="@"
        elif [ "$server1_protocol" == "TLS" ]; then
            protocol="@@(o)"
        fi
        fwd="${protocol}${server1_ip}:${server1_port}"

        # if server is secondary forward only logs received from localhost
        log_to_file "$LINENO" "Forward logs to $fwd"
        if [ "$server_type" == "Secondary" ]; then
	    create_list_and_forward "${tempstr} and ( \$fromhost-ip == '127.0.0.1' or \$fromhost-ip == '' )" "$fwd"
        else
            create_list_and_forward "${tempstr}" "$fwd"
        fi
    fi
    if [ ! -z "$server2_ip" -a ! -z "$server2_port" -a ! -z "$server2_protocol" ]; then
        protocol="@@" #TCP
        if [ "$server2_protocol" == "UDP" ]; then
            protocol="@"
        elif [ "$server2_protocol" == "TLS" ]; then
            protocol="@@(o)"
        fi
        fwd="${protocol}${server2_ip}:${server2_port}"
        log_to_file "$LINENO" "Forward logs to $fwd"
        if [ "$server_type" == "Secondary" ]; then
            create_list_and_forward "${tempstr} and ( \$fromhost-ip == '127.0.0.1' or \$fromhost-ip == '' )" "$fwd"
        else
            create_list_and_forward "${tempstr}" "$fwd"
        fi
    fi

    # forward to 3rd party applications( set from webcontrol ) if the server is not Expansion 
    if [ "$server_type" != "Expansion" ]; then
        if [ ! -z "$otherapp1_ip" -a ! -z "$otherapp1_port" -a ! -z "$otherapp1_protocol" ]; then
            protocol="@@" #TCP
            
            if [ "$otherapp1_protocol" == "UDP" ]; then
                protocol="@"
            elif [ "$otherapp1_protocol" == "TLS" ]; then
                protocol="@@(o)"
            fi
            fwd="${protocol}${otherapp1_ip}:${otherapp1_port}"
            log_to_file "$LINENO" "Forward logs to $fwd"
            create_list_and_forward "${tempstr}" "$fwd"
        fi
        if [ ! -z "$otherapp2_ip" -a ! -z "$otherapp2_port" -a ! -z "$otherapp2_protocol" ]; then
            protocol="@@" #TCP
            if [ "$otherapp2_protocol" == "UDP" ]; then
                protocol="@"
            elif [ "$otherapp2_protocol" == "TLS" ]; then
                protocol="@@(o)"
            fi
            fwd="${protocol}${otherapp2_ip}:${otherapp2_port}"
            log_to_file "$LINENO" "Forward logs to $fwd"
            create_list_and_forward "${tempstr}" "$fwd"
        fi
    fi
}

# Function to create imfile
# Input module for text files - this will get the logs from a file
# every line appendded to the file will mean a new syslog message
# Args: $1 - filepath
#	$2 - filetag
#	$3 - severity
#	$4 - facility
function create_imfile(){
    echo "\n#Read rsyslog logs from file: $1"
    echo "\n\$InputFileName $1"
    echo "\n\$InputFileTag $2:"
    echo "\n\$InputFileStateFile $2"
    echo "\n\$InputFileSeverity $3"
    echo "\n\$InputFileFacility $4"
    echo "\n\$InputRunFileMonitor\n"
    echo "\n\$InputFilePollInterval 10\n"
}

# Function to check if the forwarding ip-adresses set in Manager 
# are the loopback or current server address
# this will avoid infinite loop of the messages
# Args: $1 - current ip address
function check_and_replace_fwd(){
    if [ "$#" != "1" ]; then
        echo "check_and_replace_fwd - function needs 1 arguments"
        exit 1
    fi
    if [ "$server2_ip" == "127.0.0.1" -o "$server2_ip" == "$1" ]; then
        server2_ip=""
        server2_port=""
        server2_protocol=""
    fi
    if [ "$server1_ip" == "127.0.0.1" -o "$server1_ip" == "$1" ]; then
        server1_ip="$server2_ip"
        server1_port="$server2_port"
        server1_protocol="$server2_protocol"
        server2_ip=""
        server2_port=""
        server2_protocol=""
    fi
}

function check_and_replace_fwd_appserver(){
    ifaces=`hostname -I`

    if [ "$server2_ip" == "127.0.0.1" ]; then
        server2_ip=""
        server2_port=""
        server2_protocol=""
    fi

    if [ "$server1_ip" == "127.0.0.1" ]; then
        server1_ip="$server2_ip"
        server1_port="$server2_port"
        server1_protocol="$server2_protocol"
        server2_ip=""
        server2_port=""
        server2_protocol=""
    fi
   
    for iface in ${ifaces[@]}; do
        if [ "$server2_ip" == "${iface}" ]; then
            server2_ip=""
            server2_port=""
            server2_protocol=""
            break
        fi
    done
    
    for iface in ${ifaces[@]}; do
        if [ "$server1_ip" == "${iface}" ]; then
            server1_ip="$server2_ip"
            server1_port="$server2_port"
            server1_protocol="$server2_protocol"
            server2_ip=""
            server2_port=""
            server2_protocol=""
            break
        fi
    done
}
# Function to configure rsyslog server
# Args: $1 - server_type
function configure_network_server(){
    server_type=$1
    echo "\$ModLoad imtcp"
    echo "\n\$ModLoad imptcp"
    echo "\n\$ModLoad imudp"
    echo "\n\$ModLoad imfile\n"
    # if expansion the server will not listen to ports
    if [ "${server_type}" != "Expansion" ]; then
         if [ ! -z "$port_udp" ]; then
             echo "\n\$UDPServerRun ${port_udp}\n"
         fi
         if [ ! -z "$port_tcp" ]; then 
             echo "\n\$InputPTCPServerSupportOctetCountedFraming on\n"
             echo "\n\$InputPTCPServerRun ${port_tcp}\n"
             semanage port -a -t syslogd_port_t -p tcp ${port_tcp} 
         fi

         if [ ! -z "$port_tls" ]; then
            echo "\n\$DefaultNetstreamDriver gtls\n"

            echo "\n\$DefaultNetstreamDriverCAFile ${IPOFFICE_CERTIFICATES_PATH}server.pem\n"
            echo "\n\$DefaultNetstreamDriverCertFile ${IPOFFICE_CERTIFICATES_PATH}cert.pem\n"
            echo "\n\$DefaultNetstreamDriverKeyFile ${IPOFFICE_CERTIFICATES_PATH}key.pem\n"

            echo "\n\$InputTCPServerStreamDriverMode 1\n"
            echo "\n\$InputTCPServerStreamDriverAuthMode anon\n"
            echo "\n\$InputTCPServerSupportOctetCountedFraming on\n"
            echo "\n\$InputTCPServerRun ${port_tls}\n"
            semanage port -a -t syslogd_port_t -p tcp ${port_tls} 
        fi
        add_IncludeConfig "/etc/rsyslog.d/allowed_senders.conf"
   else
       echo "\n\$DefaultNetstreamDriver gtls\n"
       echo "\n\$DefaultNetstreamDriverCAFile ${IPOFFICE_CERTIFICATES_PATH}server.pem\n"
   fi
    # filter,send and discard the logs received from sysmon
    add_IncludeConfig "/etc/rsyslog.d/filter_sysmon.conf"
    # create templates
    echo "\n\n#Templates"
    echo "\n#Template for fowarded messages"
    # special template for the imfiles forwarding because PRI is null
    echo "\n\n\$template fwdimfile,\"<6>%TIMESTAMP% %HOSTNAME% %SYSLOGTAG% %msg%\""
    # default template for forward, if no template is set at forward(@@ip:port;template) this template will be used
    echo "\n\n\$template msgtemplate,\"<%PRI%>%TIMESTAMP% %HOSTNAME% %SYSLOGTAG% %msg%\""
    echo "\n\$ActionForwardDefaultTemplate msgtemplate"
    
    # templates for the location of the local logging files
    echo "\n\n#Templates for local applications "
    echo "\n\n\$template SGLogs,\"/var/log/rsyslog/%fromhost-ip%/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%_%syslogfacility-text%.log\""
    echo "\n\$template SGLogs2,\"/var/log/rsyslog/127.0.0.1/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%_%syslogfacility-text%.log\""
    echo "\ntemplate(name=\"HostFolder\" type=\"string\" string=\"%HOSTNAME%\")"
    echo "\nset \$!host_folder = exec_template(\"HostFolder\");"
    echo "\ntemplate ( name=\"SGLogs3\" type=\"string\" string=\"/var/log/rsyslog/%\$!host_folder%/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%_%syslogfacility-text%.log\")"

    # special templates for operating system logs and imfiles
    echo "\n\n\$template secure_template1,\"/var/log/rsyslog/%fromhost-ip%/%\$YEAR%-%\$MONTH%-%\$DAY%/secure\""
    echo "\n\$template secure_template2,\"/var/log/rsyslog/127.0.0.1/%\$YEAR%-%\$MONTH%-%\$DAY%/secure\""
    echo "\n\$template secure_template3,\"/var/log/rsyslog/%\$!host_folder%/%\$YEAR%-%\$MONTH%-%\$DAY%/secure\""

    echo "\n\n\$template spooler_template1,\"/var/log/rsyslog/%fromhost-ip%/%\$YEAR%-%\$MONTH%-%\$DAY%/spooler\""
    echo "\n\$template spooler_template2,\"/var/log/rsyslog/127.0.0.1/%\$YEAR%-%\$MONTH%-%\$DAY%/spooler\""
    echo "\n\$template spooler_template3,\"/var/log/rsyslog/%\$!host_folder%/%\$YEAR%-%\$MONTH%-%\$DAY%/spooler\""

    echo "\n\n\$template imfiles_template1,\"/var/log/rsyslog/%fromhost-ip%/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%.log\""
    echo "\n\$template imfiles_template2,\"/var/log/rsyslog/127.0.0.1/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%.log\""
    echo "\n\$template imfiles_template3,\"/var/log/rsyslog/%\$!host_folder%/%\$YEAR%-%\$MONTH%-%\$DAY%/%programname%.log\""

    #configure imfile
    create_imfile "/var/log/audit/audit.log" "audit" "6" "13"

    # for avaya applications set default template 
    echo "\n\n#Apply templates"
    echo "\nif (\$programname == 'ipoicr' or \$programname == 'ipoam' or \$programname == 'ipowebrtcgw' or \$programname == 'ipoucm' or \$programname == 'ipoappserver' or \$programname == 'ipoffice' or \$programname == 'ipomediaserver' or \$programname == 'ipowebcontrol' or \$programname == 'ipowebmanager' or \$programname == 'ipowatchdog' or \$programname == 'ipovmpro' or \$programname == 'ipoonexportal' or \$programname == 'ipomediamanager' ) then {"
    echo "\n\tif \$fromhost-ip == '' or \$fromhost-ip == '127.0.0.1' then {"
    echo "\n\t\t?SGLogs2;messageformat"
    echo "\n\t} else {"
    echo "\n\t\tif re_match(\$!host_folder , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
    echo "\n\t\t\t?SGLogs3;messageformat"
    echo "\n\t\t} else {"
    echo "\n\t\t\t?SGLogs;messageformat"
    echo "\n\t\t}"
    echo "\n\t}"
    echo "\n}"

    echo "\n\$template secformat,\"<%PRI%> %\$YEAR% %TIMESTAMP% %HOSTNAME% %syslogtag:R,ERE,1,FIELD:([a-zA-Z\/]+)_security*--end%%syslogtag:R,ERE,2,BLANK:([a-zA-Z\/]+)_security(\[[0-9]{1,5}\]+)--end%: %msg%\\\n\"\n"

    echo "\nif (\$programname == 'ipoam_security') and \$fromhost-ip == '' then ?SGLogs2;secformat"
    echo "\nif (\$programname == 'ipoam_security') and \$fromhost-ip != '' then {"
    echo "\n\tif re_match(\$!host_folder , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
    echo "\n\t\t?SGLogs3;secformat"
    echo "\n\t\t} else {"
    echo "\n\t\t\t?SGLogs;secformat"
    echo "\n\t}"
    echo "\n}"

    # operating system logs
    if [[ "$sources" =~ "secure" ]]; then
        echo "\nif \$syslogfacility-text == 'authpriv' and \$fromhost-ip == '' and not ( ( \$msg contains 'TTY=unknown ; PWD=/opt/webcontrol ; USER=root ; COMMAND=/usr/bin/curl -k -A WebControlPanel' ) or \$msg contains 'configure_rsyslog.sh' )  then ?secure_template2;messageformat"
        echo "\nif \$syslogfacility-text == 'authpriv' and \$fromhost-ip != '' and not ( ( \$msg contains 'TTY=unknown ; PWD=/opt/webcontrol ; USER=root ; COMMAND=/usr/bin/curl -k -A WebControlPanel' ) or \$msg contains 'configure_rsyslog.sh' ) then {"
        echo "\n\tif re_match(\$!host_folder , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
        echo "\n\t\t?secure_template3;messageformat"
        echo "\n\t\t} else {"
        echo "\n\t\t?secure_template1;messageformat"
        echo "\n\t}"
        echo "\n}"
    else
        echo "\nif \$syslogfacility-text == 'authpriv' then stop"
    fi
    if [[ "$sources" =~ "spooler" ]]; then
        echo "\nif (\$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit')) and \$fromhost-ip == '' then ?spooler_template2;messageformat"
        echo "\nif (\$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit')) and \$fromhost-ip != '' then {"
        echo "\n\tif re_match(\$!host_folder , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
        echo "\n\t\t?spooler_template3;messageformat"
        echo "\n\t\t} else {"
        echo "\n\t\t?spooler_template1;messageformat"
        echo "\n\t}"
        echo "\n}"
    else
        echo "\nif (\$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit')) then stop"
    fi
    if [[ "$sources" =~ "audit.log" ]]; then
        echo "\nif ( \$programname == 'audit') and \$fromhost-ip == '' then ?imfiles_template2;messageformat"
        echo "\nif ( \$programname == 'audit') and \$fromhost-ip != '' then {"
        echo "\n\tif re_match(\$!host_folder , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
        echo "\n\t\t?imfiles_template3;messageformat"
        echo "\n\t\t} else {"
        echo "\n\t\t?imfiles_template1;messageformat"
        echo "\n\t}"
        echo "\n}"
    else
        echo "\nif ( \$programname == 'audit') then stop"
    fi
    if [[ "$sources" =~ "httpd" ]]; then
        echo "\nif ( \$programname == 'httpd') and \$fromhost-ip == '' then ?imfiles_template2;messageformat"
        echo "\nif ( \$programname == 'httpd') and \$fromhost-ip != '' then {"
        echo "\n\tif re_match(\$!host_folder , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
        echo "\n\t\t?imfiles_template3;messageformat"
        echo "\n\t\t} else {"
        echo "\n\t\t?imfiles_template1;messageformat"
        echo "\n\t}"
        echo "\n}"
    else
        echo "\nif ( \$programname == 'httpd') then stop"
    fi
}

# Use a webservice to retrive nodes data
function get_current_node_info() {
    lan1_ip=`/sbin/ip addr show $IPOFFICE_LAN1 primary | grep -o ' [0-9\.]\+/' | gawk -F / '{print $1}'`
    lan1_ip=${lan1_ip/ /}
    if [ ! -z "${lan1_ip}" ]; then
        # wait for webservices to start properly
        sleep 30
        counter=0
        while [ $counter -lt 5 -a -z "${nodes_info}" ]; do
            output=`curl -k -A "Avaya-WebAdmin" -u $1:$2 https://${lan1_ip}:8443/ws/system/general 2>> $OUTPUTLOG`
            nodes_info=`echo "$output" | awk 'BEGIN { FS = "[<|>]" } { if ($2 == "NodeAddress") { print "ip-"$3 }; if ($2 == "NodeType") { print "type-"$3 } }'`
            nodes_public=`echo "$output" | awk 'BEGIN { FS = "[<|>]" } { if ($2 == "publicipAddressLAN1" || $2 == "publicipAddressLAN2" || $2 == "NodePublicIPAddress" ) { print $3" " }; }'`
            if [ -z "${nodes_info}" ]; then
                # parse error code
                err=`echo "$output" | awk 'BEGIN { FS = "[<|>]" } { if ($2 == "error_code") { print "Error Code:"$3 }; if ($2 == "error_desc") { print " Error description: "$3 } }'`
                log_to_file "$LINENO" "Failed to retrive SCN nodes.Retry counter: $counter error:$err" 
                # wait a little bit longer
                sleep 20
            fi
            ((counter++))
        done
    fi
    if [ -z "${nodes_info}" ]; then
        log_to_file "$LINENO" "Failed to retrive SCN nodes.Accept only local logging "
    fi
}

#Function to generate the basic configuration
# defines templates
# creates logrotation and outchannels for the applications
function generate_basic_rsyslog_config() {
    echo -e $CONFIG_DESCR > $TEMPCONFFILE
    echo -e "#Last Update : $(date) " >> $TEMPCONFFILE
    echo -e $CONFIG_MODULES >> $TEMPCONFFILE
    echo -e $CONFIG_FORMAT >> $TEMPCONFFILE
    echo -e $CONFIG_DEFAULT_PERM >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/network_server.conf") >> $TEMPCONFFILE
    echo -e "\n#Logrotate all logfiles for a size" >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoffice" "/var/log/ipoffice.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoappserver" "/var/log/ipoappserver.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoucm" "/var/log/ipoucm.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipomediaserver" "/var/log/ipomediaserver.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipowatchdog" "/var/log/ipowatchdog.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipowebcontrol" "/var/log/ipowebcontrol.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipowebmanager" "/var/log/ipowebmanager.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoicr" "/var/log/ipoicr.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipowebrtcgw" "/var/log/ipowebrtcgw.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoam" "/var/log/ipoam.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoam_security" "/var/log/ipoam_security.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipoonexportal" "/var/log/ipoonexportal.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipovmpro" "/var/log/ipovmpro.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "ipomediamanager" "/var/log/ipomediamanager.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "httpd" "/var/log/httpd.log") >> $TEMPCONFFILE
    echo -e $(add_logrotation "spooler" "/var/log/spooler") >> $TEMPCONFFILE
    echo -e $(add_logrotation "secure" "/var/log/secure") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipomediaserver_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipoffice_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/system_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipoonexportal_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipovmpro_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipomediamanager_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipowatchdog_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipowebcontrol_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipowebmanager_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipoicr_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipowebrtcgw_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipoam_network_client.conf") >> $TEMPCONFFILE
    echo -e $(add_IncludeConfig "/etc/rsyslog.d/ipoam_security_network_client.conf") >> $TEMPCONFFILE
    echo -e $CONFIG_SERVER_STOP_REMOTE_PROCESSING >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoffice") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoappserver") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoucm") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipomediaserver") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipowatchdog") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipowebcontrol") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoicr") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipowebmanager") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipowebrtcgw") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoam") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoam_security") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipoonexportal") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipovmpro") >> $TEMPCONFFILE
    echo -e $(discard_logs "ipomediamanager") >> $TEMPCONFFILE
    echo -e $(discard_logs "audit") >> $TEMPCONFFILE
    echo -e $(discard_logs "httpd") >> $TEMPCONFFILE
    echo -e $(discard_logs "spooler") >> $TEMPCONFFILE
    echo -e $(discard_logs "secure") >> $TEMPCONFFILE
    echo -e $CONFIG_LOCAL_DESCR >> $TEMPCONFFILE
    echo -e $CONFIG_LOG_INFO_AND_HIGHER >> $TEMPCONFFILE
    echo -e $CONFIG_LOG_MAIL >> $TEMPCONFFILE
    echo -e $CONFIG_LOG_CRON >> $TEMPCONFFILE
    echo -e $CONFIG_LOG_EMERGENCY >> $TEMPCONFFILE
    echo -e $CONFIG_LOG_BOOT >> $TEMPCONFFILE
}

# Functions to update the configuration and restart te rsyslog service
function update_config() {
   # update configs
   i=0
   systemctl stop rsyslog > /dev/null
   # if rsyslog is not stopped try again to stop
   while systemctl status rsyslog | grep running && (($i < 5)); do
       ((i++))
       log_to_file "$LINENO" "Stop rsyslog service, count=$i"
       sleep 10
       systemctl stop rsyslog > /dev/null
   done
   rsyslog_pid=`pidof rsyslogd`
   # if the rsyslog pid exists kill the process
   if [ ! -z "$rsyslog_pid" ]; then
       log_to_file "$LINENO" "Rsyslog process killed..PID: |$rsyslog_pid|"
       kill -SIGKILL $rsyslog_pid
   fi
   if [ -f /var/log/audit/audit.log ];then
       mv -f /var/log/audit/audit.log /var/log/audit/audit.log.old
   fi
   mv $TEMPCONFFILE /etc/rsyslog.conf
   mv $TEMPSYSMONCONF /etc/rsyslog.d/filter_sysmon.conf
   mv $TEMPJADECONF /etc/rsyslog.d/ipomediaserver_network_client.conf
   mv $TEMPIPOFFICECONF /etc/rsyslog.d/ipoffice_network_client.conf
   mv $TEMPWATCHDOGCONF /etc/rsyslog.d/ipowatchdog_network_client.conf
   mv $TEMPWEBCONTROLCONF /etc/rsyslog.d/ipowebcontrol_network_client.conf
   mv $TEMPWEBMANAGERCONF /etc/rsyslog.d/ipowebmanager_network_client.conf
   mv $TEMPICRCONF /etc/rsyslog.d/ipoicr_network_client.conf
   mv $TEMPWEBRTCGWCONF /etc/rsyslog.d/ipowebrtcgw_network_client.conf
   mv $TEMPAMCONF /etc/rsyslog.d/ipoam_network_client.conf
   mv $TEMPAMSECURITYCONF /etc/rsyslog.d/ipoam_security_network_client.conf
   mv $TEMPVMPROCONF /etc/rsyslog.d/ipovmpro_network_client.conf
   mv $TEMPMEDIAMANAGERCONF /etc/rsyslog.d/ipomediamanager_network_client.conf
   mv $TEMPONEXCONF /etc/rsyslog.d/ipoonexportal_network_client.conf
   mv $TEMPSYSTEMCONF /etc/rsyslog.d/system_network_client.conf
   mv $TEMPSERVERCONF /etc/rsyslog.d/network_server.conf
   mv $TEMPSENDERSCONF /etc/rsyslog.d/allowed_senders.conf
   systemctl start rsyslog > /dev/null
   log_to_file "$LINENO" "Rsyslog updated."
   log_to_file "$LINENO" "Rsyslog status.`systemctl status rsyslog`"
   log_to_file "$LINENO" "Rsyslogd PID:`pidof rsyslogd`"
   # create cron used to manage the size of the /var/log/rsyslog folder
   exist_cron=`crontab -l -u root | grep "cron_rsyslog.sh"`
   echo $exist_cron
   if [ -z "$exist_cron" ]; then
       CRON_JOB="*/$CRON_TIMEOUT * * * * /opt/Avaya/scripts/cron_rsyslog.sh"
       ( crontab -l; echo "${CRON_JOB}" ) | crontab
   fi
}

PRIMARY=""
SECONDARY=""
EXPANSIONS=""
PUBLIC_IPLIST=""

function get_SCN_nodes() {
    get_current_node_info $1 $2
    if [ ! -z "${nodes_info}" ]; then
        nodes_array=(`echo $nodes_info | sed -e 's/ /\n/g'`)
        node_type=( )
        node_address=( )
        for x in "${nodes_array[@]}"
        do
            info=(`echo $x | sed -e 's/-/\n/g'`)
            if [ -z `echo $x | grep "ip-"` ]; then
                node_type=( "${node_type[@]}" "${info[1]}" )
            else
                node_address=( "${node_address[@]}" "${info[1]}" )
            fi
        done
        index=0
        for x in "${node_type[@]}"
        do
            if [ "$x" == "PrimaryServer" ]; then
                # get primary ip
                if [ "${node_address[$index]}" != "0.0.0.0" ]; then
                    PRIMARY=${node_address[$index]}
                fi
            fi
            if [ "$x" == "Expansion" ]; then
                # add expansions to allowed senders for secondary and primary
                if [ "${node_address[$index]}" != "0.0.0.0" ]; then
                    EXPANSIONS+="${node_address[$index]} "

                fi
            fi
            if [ "$x" == "SecondaryServer" ]; then
                if [ "${node_address[$index]}" != "0.0.0.0" ]; then
                    SECONDARY="${node_address[$index]}"
                fi
            fi
        ((index++))
        done
    fi
    index=0
    for x in $nodes_public
    do
        if [ "$x" != "0.0.0.0" ]; then
            PUBLIC_IPLIST+="$x "
        fi
    ((index++))
    done
    log_to_file "$LINENO" "SCN nodes:Primary:|$PRIMARY|,Secondary: |$SECONDARY|,Expansions: |$EXPANSIONS|"
}

# Generate files for sysmon messages and filter the messages
# $1 - server type
function generate_filter_sysmon() {
    log_to_file "$LINENO" "Forwarding Sysmon logs"
    echo "\n\n\$template fwdsysmon,\"<%PRI%>%msg%\""
    echo "\n\$template sysmonformat,\"%msg%\\\n\"\n"
    echo "\ntemplate(name=\"HostFolderSysmon\" type=\"string\" string=\"%HOSTNAME%\")"
    echo "\nset \$!host_folder_sysmon = exec_template(\"HostFolderSysmon\");"
    echo "\n\ntemplate ( name=\"Sysmontemplate1\" type=\"string\" string=\"/var/log/sysmon/%fromhost-ip%/sysmon_%\$YEAR%%\$MONTH%%\$DAY%_%\$HOUR%.log.zip\")"
    echo "\n\ntemplate ( name=\"Sysmontemplate2\" type=\"string\" string=\"/var/log/sysmon/%\$!host_folder_sysmon%/sysmon_%\$YEAR%%\$MONTH%%\$DAY%_%\$HOUR%.log.zip\")"
    echo "\n\$outchannel sysmonlog_rotation,/var/log/sysmon.log,$maxsysmonsize,/opt/Avaya/scripts/./Log_rotation_script sysmon"
    echo "\n\nif \$syslogseverity == 7 and \$syslogfacility == 0 and ( \$msg contains '#^#' ) then :omfile:\$sysmonlog_rotation\n"
    echo "\n\nif \$syslogseverity == 7 and \$syslogfacility == 0 and ( \$msg contains '#^#' ) then {"
    echo "\n\t\tif re_match(\$!host_folder_sysmon , '^([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\\\\\.([0-9]{1,3})\$') then {"
    echo "\n\t\tAction(type=\"omfile\"  DynaFile=\"Sysmontemplate2\" template=\"sysmonformat\" DirCreateMode=\"0755\" FileCreateMode=\"0644\" zipLevel=\"1\")\n"
    echo "\n\t\t} else {"
    echo "\n\t\tAction(type=\"omfile\"  DynaFile=\"Sysmontemplate1\" template=\"sysmonformat\" DirCreateMode=\"0755\" FileCreateMode=\"0644\" zipLevel=\"1\")\n"
    echo "\n\t\t}"
    echo "\n\t}"

    if [ "$1" == "Primary" ]; then
        current_ip="$PRIMARY"
    fi
    if [ "$1" == "Secondary" ]; then
        current_ip="$SECONDARY" 
    fi

    if [ "$1" == "Application" ]; then
        check_and_replace_fwd_appserver
    else
        check_and_replace_fwd "$current_ip"
    fi 
    # use special queue for the 2 possible servers defined for the sysmon logs 
    if [ ! -z "$server1_ip" ]; then
        prot="@@"
        if [ "$server1_protocol" == "UDP" ]; then
            prot="@"
        fi
        if [ "$server1_protocol" == "TLS" ]; then
            prot="@@(o)"
        fi
        log_to_file "$LINENO" "Forward sysmon logs to $prot$server1_ip:$server1_port;fwdsysmon"
        if [ "$server_type" == "Secondary" ]; then
            create_list_and_forward "\$syslogseverity == 7 and \$syslogfacility == 0 and ( \$fromhost-ip == '127.0.0.1' or \$fromhost-ip == '' )" "$prot$server1_ip:$server1_port;fwdsysmon\n"
        else
            create_list_and_forward "\$syslogseverity == 7 and \$syslogfacility == 0" "$prot$server1_ip:$server1_port;fwdsysmon\n"
        fi
    fi
    if [ ! -z "$server2_ip" ]; then
        prot="@@"
        if [ "$server2_protocol" == "UDP" ]; then
            prot="@"
        fi
        if [ "$server2_protocol" == "TLS" ]; then
            prot="@@(o)"
        fi
        log_to_file "$LINENO" "Forward sysmon logs to $prot$server1_ip:$server1_port;fwdsysmon"
        if [ "$server_type" == "Secondary" ]; then
            create_list_and_forward "\$syslogseverity == 7 and \$syslogfacility == 0 and ( \$fromhost-ip == '127.0.0.1' or \$fromhost-ip == '' )" "$prot$server2_ip:$server2_port;fwdsysmon\n"
        else
            create_list_and_forward "\$syslogseverity == 7 and \$syslogfacility == 0" "$prot$server2_ip:$server2_port;fwdsysmon\n"
        fi
    fi
    echo "\n\nif \$syslogseverity == 7 and \$syslogfacility == 0 and ( \$msg contains '#^#' ) then stop\n"
    #restore format to default
    echo "$CONFIG_FORMAT"    
}

function configure_server_rsyslog() {
    server_type="$1"
    # forward the local logs to the primary server
    if [ "$1" != "Application" ]; then
        get_SCN_nodes $2 $3
    fi
    generate_basic_rsyslog_config
    echo -e $(generate_filter_sysmon $1) >> $TEMPSYSMONCONF
    # configuration /etc/rsyslog.d/network_server.conf
    echo -e $(configure_network_server "$1" ) >> $TEMPSERVERCONF #args: server_type udp_port tcp_port
    tempstr="(\$programname == 'audit' or \$programname == 'httpd' or \$syslogfacility-text == 'authpriv' or \$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit') ) and not ( ( \$msg contains 'TTY=unknown ; PWD=/opt/webcontrol ; USER=root ; COMMAND=/usr/bin/curl -k -A WebControlPanel' ) or \$msg contains 'configure_rsyslog.sh' ) "
    if [ "$1" == "Application" ]; then
        check_and_replace_fwd_appserver
        echo -e $(send_logs "ipoappserver" "") >> $TEMPIPOFFICECONF
        echo -e "\n" >> $TEMPIPOFFICECONF
        echo -e $(send_logs "ipoucm" "") >> $TEMPIPOFFICECONF
        echo -e $(send_logs "ipomediaserver" "") >> $TEMPJADECONF
        echo -e $(send_logs "ipowatchdog" "") >> $TEMPWATCHDOGCONF
        echo -e $(send_logs "ipowebcontrol" "") >> $TEMPWEBCONTROLCONF
        echo -e $(send_logs "ipowebmanager" "") >> $TEMPWEBMANAGERCONF
        echo -e $(send_logs "ipoicr" "") >> $TEMPICRCONF
        echo -e $(send_logs "ipowebrtcgw" "") >> $TEMPWEBRTCGWCONF
        echo -e $(send_logs "ipoam" "") >> $TEMPAMCONF
        echo -e $(send_logs "ipoam_security" "") >> $TEMPAMSECURITYCONF
        echo -e $(send_logs "ipovmpro" "") >> $TEMPVMPROCONF
        echo -e $(send_logs "ipomediamanager" "") >> $TEMPMEDIAMANAGERCONF
        echo -e $(send_logs "ipoonexportal" "") >> $TEMPONEXCONF
        echo -e $(send_logs "$tempstr" "system") >> $TEMPSYSTEMCONF
    fi

    if [ "$1" == "Secondary" ]; then
       check_and_replace_fwd "$SECONDARY"
       # send logs to primary (send_logs programname [null|system])
       echo -e $(send_logs "ipoffice" "") >> $TEMPIPOFFICECONF
       echo -e $(send_logs "ipomediaserver" "") >> $TEMPJADECONF
       echo -e $(send_logs "ipowatchdog" "") >> $TEMPWATCHDOGCONF
       echo -e $(send_logs "ipowebcontrol" "") >> $TEMPWEBCONTROLCONF
       echo -e $(send_logs "ipowebmanager" "") >> $TEMPWEBMANAGERCONF
       echo -e $(send_logs "ipoicr" "") >> $TEMPICRCONF
       echo -e $(send_logs "ipowebrtcgw" "") >> $TEMPWEBRTCGWCONF
       echo -e $(send_logs "ipoam" "") >> $TEMPAMCONF
       echo -e $(send_logs "ipoam_security" "") >> $TEMPAMSECURITYCONF
       echo -e $(send_logs "ipovmpro" "") >> $TEMPVMPROCONF
       echo -e $(send_logs "ipoonexportal" "") >> $TEMPONEXCONF
       echo -e $(send_logs "$tempstr" "system") >> $TEMPSYSTEMCONF
   fi
   if [ "$1" == "Primary" ]; then
       check_and_replace_fwd "$PRIMARY"
       # send logs to primary ((send_logs programname [null|system])
       echo -e $(send_logs "ipoffice" "") >> $TEMPIPOFFICECONF
       echo -e $(send_logs "ipomediaserver" "") >> $TEMPJADECONF
       echo -e $(send_logs "ipowatchdog" "") >> $TEMPWATCHDOGCONF
       echo -e $(send_logs "ipowebcontrol" "") >> $TEMPWEBCONTROLCONF
       echo -e $(send_logs "ipowebmanager" "") >> $TEMPWEBMANAGERCONF
       echo -e $(send_logs "ipoicr" "") >> $TEMPICRCONF
       echo -e $(send_logs "ipowebrtcgw" "") >> $TEMPWEBRTCGWCONF
       echo -e $(send_logs "ipoam" "") >> $TEMPAMCONF
       echo -e $(send_logs "ipoam_security" "") >> $TEMPAMSECURITYCONF
       echo -e $(send_logs "ipovmpro" "") >> $TEMPVMPROCONF
       echo -e $(send_logs "ipomediamanager" "") >> $TEMPMEDIAMANAGERCONF
       echo -e $(send_logs "ipoonexportal" "") >> $TEMPONEXCONF
       echo -e $(send_logs "$tempstr" "system") >> $TEMPSYSTEMCONF
   fi
   echo -e $(add_senders "TCP" "127.0.0.1") >> $TEMPSENDERSCONF
   echo -e $(add_senders "UDP" "127.0.0.1") >> $TEMPSENDERSCONF 
   text="\nif ( "
   if [ "$1" == "Primary" -a ! -z "$SECONDARY" ]; then
       echo -e $(add_senders "TCP" "$SECONDARY") >> $TEMPSENDERSCONF
       echo -e $(add_senders "UDP" "$SECONDARY") >> $TEMPSENDERSCONF
       text="$text \$fromhost-ip != '$SECONDARY' and"
   fi
   for s in $EXPANSIONS; do
        echo -e $(add_senders "TCP" "$s") >> $TEMPSENDERSCONF
        echo -e $(add_senders "UDP" "$s") >> $TEMPSENDERSCONF
	text="$text \$fromhost-ip != '$s' and"
   done
   for s in $PUBLIC_IPLIST; do
        echo -e $(add_senders "TCP" "$s") >> $TEMPSENDERSCONF
        echo -e $(add_senders "UDP" "$s") >> $TEMPSENDERSCONF
        text="$text \$fromhost-ip != '$s' and"
   done

   update_config
}

function configure_expansion_rsyslog() {
    server_type="$1"
    generate_basic_rsyslog_config
    server_apps=("ipoffice" "ipomediaserver" "ipowatchdog" "ipowebcontrol")
    echo -e $(generate_filter_sysmon $1) >> $TEMPSYSMONCONF
    echo -e $(configure_network_server "$1" "$port_udp" "$port_tcp" "$port_tls") > $TEMPSERVERCONF
    tempstr="(\$programname == 'httpd' or \$programname == 'audit' or \$syslogfacility-text == 'authpriv' or \$syslogfacility-text == 'uucp' or (\$syslogfacility-text == 'news' and \$syslogseverity-text == 'crit') and not ( ( \$msg contains 'TTY=unknown ; PWD=/opt/webcontrol ; USER=root ; COMMAND=/usr/bin/curl -k -A WebControlPanel' ) or \$msg contains 'configure_rsyslog.sh' )  )"
    echo -e $(send_logs "ipoffice"  "") >> $TEMPIPOFFICECONF
    echo -e $(send_logs "ipomediaserver"  "") >> $TEMPJADECONF
    echo -e $(send_logs "ipowatchdog" "" ) >> $TEMPWATCHDOGCONF
    echo -e $(send_logs "ipowebcontrol"  "" ) >> $TEMPWEBCONTROLCONF
    echo -e $(send_logs "ipowebmanager"  "" ) >> $TEMPWEBMANAGERCONF
    echo -e $(send_logs "ipowebrtcgw"  "" ) >> $TEMPWEBRTCGWCONF
    echo -e $(send_logs "ipoam" "") >> $TEMPAMCONF
    echo -e $(send_logs "ipoam_security" "") >> $TEMPAMSECURITYCONF
    echo -e $(send_logs "$tempstr" "system") >> $TEMPSYSTEMCONF
    update_config
}

log_to_file "$LINENO" "Ipoffice Mode: $IPOFFICE_BE_MODE"
if [ "$IPOFFICE_BE_MODE" == "Primary" -o "$IPOFFICE_BE_MODE" == "Secondary" -o "$IPOFFICE_BE_MODE" == "Application server on Linux PC" -o "$IPOFFICE_BE_MODE" == "Application server on UCM" ]; then
    configure_server_rsyslog $IPOFFICE_BE_MODE $1 $2
elif [ "$IPOFFICE_BE_MODE" == "Expansion" ]; then
    configure_expansion_rsyslog $IPOFFICE_BE_MODE $1 $2
else
    echo "Unknown machine type"
fi
