#!/bin/bash
# This script provides basic functions for handling Avaya IPOL Certificates
# 
# ACTIONS:
#    --generate-ca-cert           Generate the Root CA Certificate
#    --generate-server-cert       Generate the Server Certificate and Private Key
#    --export-ca <file>           Export the CA Certificate in file <file>
#    --import-ca <file>           Import the CA Certificate from file <file>
#    --distribute-server-cert     Distribute the Server Certificate to all Avaya Apps
#    --print-default-ca-vars      Print the default ca certificate variables
#    --print-default-server-vars  Print the default server certificate variables
# 
# OPTIONS:
#    -h|--help                    Show this message
#    --rsa1024                    Use the RSA-1024 instead of RSA-2048
#    --sha1                       Use SHA-1 instead of SHA-256
#    --valid-to <days>            Specify the validity of the certificate
#    --id <ID>                    Specify the ID used in the generation of IssuedTo field.
#                                 If not specified, the default ID is the hostname for CA
#                                 Certificate and the eth0 MAC for Server Certificate.
#    --subject-alt-name <SAN>     Specify the SubjectAltName
#                                 Example:
#                                     --subject-alt-name "IP:1.2.3.4,URI:server.avaya.com"
#    --pass <password>            Password for imported/exported CA Certificate
#    --server-ip <IP>             When generating a server certificate, the IP of the server can
#                                 be specified, so that the certificate is named like: server_<IP>.pem
#    --hostname                   Specify a new hostname
#    --regenerate                 Option to regenerate the CA
#
# RETURN VALUES:
#    ESUCCESS=0          # Success
#    EFAIL=1             # General Error
#    EINVAL=2            # Invalid argument
#    EACCESS=3           # Access error
#    EGENRSA=4           # Cannot generate RSA Key
#    EGENCERT=5          # Cannot cenerate Certificate/Certificate request
#    ESIGNCERT=6         # Cannot Sign the certificate
#    ECERTNOTFOUND=7     # Cannot find or load the certificate
#    EKEYNOTFOUND=8      # Cannot find or load the RSA Key
#    ENOTMATCH=9         # Non Matching Certificate-Private Key
# 
# Examples:
# 
# 1. Generate CA Certificate:
# ./gen_certs.sh --generate-ca-cert
# 
# 2. Generate a Server Certificate signed by the CA (using RSA-1024 and SHA-1 with a validity of one year):
# ./gen_certs.sh --generate-server-cert --rsa1024 -sha1 --valid-to 365 --id 00:01:02:03:04:05
# 
# 3. Distribute the Server Certificate to all Avaya Applications:
# ./gen_certs.sh --distribute-server-cert

echo $$ > /sys/fs/cgroup/systemd/tasks

if [ -f /opt/Avaya/scripts/config_files/apps_logging_config ]; then
    . /opt/Avaya/scripts/config_files/apps_logging_config
fi

# DEFAULT PARAMETERS

# Default CA Certificate Validity
DEFAULT_CA_CERT_VALIDITY=$((365*10)) # 10 years
# Default Server Certificate Validity
DEFAULT_SERVER_CERT_VALIDITY=$((824)) # 824 days - maximum allowed for certificates lifetime minus 1 day.
# Default RSA Key Size
DEFAULT_RSA_KEY_LEN=2048
# Default SHA Version
DEFAULT_SHA_VER=-sha256

# Is primary server
IS_PRIMARY=false

# Enable CA Chain functionality depending on WCP settings
ENABLE_CA_CHAIN=false
CA_CHAIN=
GENERATE_CA_CHAIN=false
CHAIN_NO=1
CA_REGENERATE=false
ENABLE_NO_RESTART_MODE=false
ENABLE_FORCE_RESTART_MODE=false
CERT_IS_SELF_SIGNED=false
CERT_IS_PKCS12_FORMAT=false
VERIFY_CSR=false

# ACTIONS

# --import-ca action
IMPORT_CA=false
# --export-ca
EXPORT_CA=false
# --generate-ca-cert
GENERATE_CA_CERT=false
# --generate-server-cert
GENERATE_SRV_CERT=false
# --generate-csr
GENERATE_CSR=false
# --distribute-server-cert
DISTRIBUTE_SRV_CERT=false
# --print-default-ca-vars
PRINT_DEFAULT_CA_VARS=false
# --print-default-server-vars
PRINT_DEFAULT_SERVER_VARS=false
# --print-ca-info
PRINT_CA_INFO=false
# --decrypt-p12certn
INSTALL_P12CERTIFICATE=false
# --convert-to-der
CONVERT_TODER_FORMAT=false
# --addca-to-ipotcs
IMPORTCA_TO_IPOTCS=false

# These applications could be absent from the build
ONEXPORTAL_PRESENT=false
VMPRO_PRESENT=false

# All the temporary files should be added to this variable
# They will be all removed in the cleanup function, when the script exits
TMP_FILES=

SERVER_IP=

if [ ! -z $GEN_CERTS_LOG_DIR_NAME ]; then
    LOG_DIR="${GEN_CERTS_LOG_DIR_NAME}"
else
    LOG_DIR="/opt/Avaya/scripts/logs"
fi
mkdir -p $LOG_DIR
if [ ! -z $GEN_CERTS_LOG_FILE_NAME ]; then
    LOG_FILE="$LOG_DIR""/""${GEN_CERTS_LOG_FILE_NAME}"
else
    LOG_FILE=$LOG_DIR/gen_certs.log
fi
touch $LOG_FILE

LOCKFILE="/var/lock/gencert.LOCK"
SCRIPT_RUNNING=`ps -ef|grep gen_certs|grep -v "grep"|grep -E "generate-server-cert|install-p12cert|convert-to-der|distribute-server-cert"`
if [ -z "$SCRIPT_RUNNING" ]; then
    if [ -f $LOCKFILE ]; then
        rm -f $LOCKFILE
    fi
fi

DISTRIBUTION_INPROGRESS_FLAG="/opt/Avaya/certs/.distrib_inprogress"
DISTRIBUTION_DONE_FLAG="/opt/Avaya/certs/.distrib_complete"

# Help function
usage()
{
cat << EOF
usage: $0 <action> <options>

This script provides the basic utilities for handling Avaya IPOL Certificates

ACTIONS:
   --generate-ca-cert           Generate the Root CA Certificate
   --generate-server-cert       Generate the Server Certificate and Private Key
   --generate-csr               Generate a Certificate sign request
   --export-ca <file>           Export the CA Certificate in file <file>
   --import-ca <file>           Import the CA Certificate from file <file>
   --distribute-server-cert     Distribute the Server Certificate to all Avaya Apps
   --print-default-ca-vars      Prints the default ca certificate variables
   --print-default-server-vars  Prints the default server certificate variables
   --print-ca-info              Prints the Certificate Authority Information
   --generate-ca-chain <X>      Generate the X certificate in the chain (debug purpose).
   --install-p12cert            Installs a p12 certificate into system
   --convert-to-der <file>      Converts to der the file in pem format
   --addca-to-ipotcs            Add the current CA to IPOffice TCS

OPTIONS:
   -h|--help                    Show this message
   --rsa1024                    Use the RSA-1024 instead of RSA-2048
   --sha1                       Use SHA-1 instead of SHA-256
   --valid-to <days>            Specify the validity of the certificate
   --id <ID>                    Specify the ID used in the generation of IssuedTo field.
                                If not specified, the default ID is the hostname for CA
                                Certificate and the eth0 MAC for Server Certificate.
   --issued-to <subject>        Specify a new Subject
   --subject-alt-name <SAN>     Specify the SubjectAltName
                                It needs to be in the following format:
                                    "type1:value1,type2:value2,..."
                                Example:
                                    --subject-alt-name "IP:1.2.3.4,URI:server.avaya.com"
   --pass <password>            Password for imported/exported CA Certificate
   --primary                    If primary server, need to copy the server certificate
                                to cert_to_import.pem
                                This option needs to be enabled when generating a certificate
                                for the primary server.
   --self-signed                User for generating a self-signed server certificate
                                This certificate will be copied automatically to cert_to_import.pem
   --pkcs12                     This will generate the server certificate in pkcs12 format.
                                It works in conjunction with --pass switch
   --enable-ca-chain            Enable Certificate Chain support
   --no-restart                 Don't restart the services when updating the certificates.
                                (For WebControl Ignition)
   --force-restart              Force restart applications when distributing certificates
   --server-ip <IP>             When generating a server certificate, the IP of the server can
                                be specified, so that the certificate is named like: server_<IP>.pem
   --hostname <name>            Specify a new hostname
   --verify-csr-signature       Used when importing a P12 cert so to import only certs
                                with CSRs generated on the same machine

RETURN VALUES:
   ESUCCESS=0                   Success
   EFAIL=1                      General Error
   EINVAL=2                     Invalid argument
   EACCESS=3                    Access error
   EGENRSA=4                    Cannot generate RSA Key
   EGENCERT=5                   Cannot cenerate Certificate/Certificate request
   ESIGNCERT=6                  Cannot Sign the certificate
   ECERTNOTFOUND=7              Cannot find or load the certificate
   EKEYNOTFOUND=8               Cannot find or load the RSA Key
   ENOTMATCH=9                  Non Matching Certificate-Private Key

Examples:

1. Generate CA Certificate:
$0 --generate-ca-cert

2. Generate a Server Certificate signed by the CA (using RSA-1024 and SHA-1 with a validity of one year):
$0 --generate-server-cert --rsa1024 -sha1 --valid-to 365 --id 00:01:02:03:04:05

3. Generate a Self Signed Server Certificate
$0 --generate-server-cert --self-signed --hostname test --rsa1024 -sha1 --valid-to 365 --id 00:01:02:03:04:05

4. Generate a pkcs12 Signed Certificate

$0 --generate-server-cert --pkcs12 --pass test --server-ip <ip> --hostname secondary

5. Installing a pkcs12 Signed Certificate on a machine

$0 --install-p12cert --pass test --server-ip <ip>

6. Distribute the Server Certificate to all Avaya Applications:
$0 --distribute-server-cert

EOF
}

# Import IPOffice variables
import_isa_data() {
    if [ -f /etc/sysconfig/ipoffice ]; then
        source /etc/sysconfig/ipoffice
    fi
    if [ -f /etc/sysconfig/webcontrol ]; then
        source /etc/sysconfig/webcontrol
    fi
    if [ -f /etc/sysconfig/onexportal ]; then
        source /etc/sysconfig/onexportal
        ONEXPORTAL_PRESENT=true
    fi
    if [ -f /etc/sysconfig/vmpro ]; then
        source /etc/sysconfig/vmpro
        VMPRO_PRESENT=true
    fi

    # Avaya Certificate Directory
    CERT_DIR=$IPOFFICE_CERTIFICATES_PATH
    # CA Certificate Directory
    CA_CERT_DIR=$CERT_DIR/ca
    # CA Private key file
    CA_KEY=$CA_CERT_DIR/root-ca.key
    # CA Certificate in PEM format
    CA_CERT=$CA_CERT_DIR/root-ca.pem

    if [ ! -d $CERT_DIR ]
    then
        mkdir -p "$CERT_DIR"
    fi
    chmod 700 $CERT_DIR

    if [ ! -d $CA_CERT_DIR ]
    then
        mkdir -p "$CA_CERT_DIR"
    fi
    chmod 700 $CA_CERT_DIR

    if [ -f $CA_CERT_DIR/chain.pem ]
    then
        ENABLE_CA_CHAIN=true
    fi

    CHAIN_SEARCH=$ENABLE_CA_CHAIN
    if ! $CHAIN_SEARCH; then
        if [ ! -f $CA_CERT ]; then
            CHAIN_SEARCH=true
        fi
    fi
    if $CHAIN_SEARCH
    then
        CA_CHAIN_CERTS=`find $CA_CERT_DIR -maxdepth 1 -name "ca_*.pem" | sort`
        CA_CHAIN="$CA_CERT_DIR/chain.pem"
        if [ ! -f $CA_CHAIN ]; then
            log "Chain file is missing, we will not offer it"
            ENABLE_CA_CHAIN=false
        fi
    fi
}

# Process the script arguments
process_args() {
    local ACTIONS_CNT=0

    # Check the options
    OPTS=`getopt -o h -l generate-ca-cert -l regenerate -l generate-server-cert -l generate-csr -l export-ca: -l convert-to-der: \
          -l distribute-server-cert -l addca-to-ipotcs -l rsa1024 -l sha1 -l valid-to: -l import-ca: -l install-p12cert \
          -l id: -l issued-to: -l subject-alt-name: -l pass: -l help -l primary -l pkcs12 -l print-ca-info \
          -l print-default-ca-vars -l print-default-server-vars -l enable-ca-chain -l self-signed \
          -l generate-ca-chain: -l hostname: -l no-restart -l verify-csr-signature -l force-restart -l server-ip: -- "$@"`

    eval set -- "$OPTS"

    # Parse arguments
    while true 
    do
        case "$1" in
            -h|--help)
                usage
                exit;;
            # Actions
            --generate-ca-cert)
                GENERATE_CA_CERT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --generate-server-cert)
                GENERATE_SRV_CERT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --generate-csr)
                GENERATE_CSR=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --import-ca)
                IMPORT_CA=true
                PKCS12_FILE=$2
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift 2;;
            --export-ca)
                EXPORT_CA=true
                PKCS12_FILE=$2
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift 2;;
            --distribute-server-cert)
                DISTRIBUTE_SRV_CERT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --print-default-ca-vars)
                PRINT_DEFAULT_CA_VARS=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --print-default-server-vars)
                PRINT_DEFAULT_SERVER_VARS=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --print-ca-info)
                PRINT_CA_INFO=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --generate-ca-chain)
                GENERATE_CA_CHAIN=true
                CHAIN_NO=$2
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift 2;;
            --install-p12cert)
                INSTALL_P12CERTIFICATE=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --convert-to-der)
                CONVERT_TODER_FORMAT=true
                FILE_TO_CONVERT_TODER=$2
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift 2;;
            --addca-to-ipotcs)
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                IMPORTCA_TO_IPOTCS=true
                shift;;
            # Options
            --rsa1024)
                RSA_KEY_LEN=1024
                shift;;
            --sha1)
                SHA_VER=-sha1
                shift;;
            --valid-to)
                CERT_VALIDITY="$2"
                shift 2;;
            --id)
                ID="$2"
                shift 2;;
            --issued-to)
                ISSUED_TO="$2"
                shift 2;;
            --subject-alt-name)
                SUBJECT_ALT_NAME="$2"
                shift 2;;
            --pass)
                PKCS12_PASS="$2"
                shift 2;;
            --primary)
                IS_PRIMARY=true
                shift;;
            --enable-ca-chain)
                ENABLE_CA_CHAIN=true
                shift;;
            --self-signed)
                CERT_IS_SELF_SIGNED=true
                shift;;
            --pkcs12)
                CERT_IS_PKCS12_FORMAT=true
                shift;;
            --hostname)
                HOST_NAME="$2"
                shift 2;;
            --no-restart)
                ENABLE_NO_RESTART_MODE=true
                shift;;
            --force-restart)
                ENABLE_FORCE_RESTART_MODE=true
                shift;;
            --server-ip)
                SERVER_IP="$2"
                shift 2;;
            --regenerate)
                CA_REGENERATE=true
                shift;;
            --verify-csr-signature)
                VERIFY_CSR=true
                shift;;
            --)
                shift; break;;
            *)
                usage
                err_exit $EINVAL "Invalid option $1";;
        esac
    done

    # No further arguments are required
    if [ $# -ne 0 ]
    then
        usage
        err_exit $EINVAL "Invalid argument: $@"
    fi

    # Only one action at a time
    if [ "$ACTIONS_CNT" -ne "1" ]
    then
        usage
        err_exit $EINVAL "Multiple or no actions specified"
    fi

    # Need to provide password when exporting and importing a PKCS12 Certificate
    if $EXPORT_CA || $IMPORT_CA
    then
        if [ -z "$PKCS12_PASS" ]
        then
            usage
            err_exit $EINVAL "Please provide password [--pass password]"
        fi
    fi

}

# Generate an OpenSSL config file so that certain parameters can be filled
# (like keyUsage, SubjectAltName, CA flag...)
generate_config_file() {
    local EXT_FILE="$1"
    local CA=$2
    local KEY_USAGE=$3
    local SUBJECT_ALT_NAME=$4
    local EXTENDED_KEY_USAGE=$5
    local EXTENDED_CSR=$6

    if $EXTENDED_CSR; then
        echo "[ req ]"                                         > $EXT_FILE
        echo "distinguished_name = req_distinguished_name"    >> $EXT_FILE
        echo "req_extensions = v3_req"                        >> $EXT_FILE
        echo "prompt = no"                                    >> $EXT_FILE
        echo "[ req_distinguished_name ]"                     >> $EXT_FILE
        echo "C = US"                                         >> $EXT_FILE
        echo "ST = New Jersey"                                >> $EXT_FILE
        echo "L = Basking Ridge"                              >> $EXT_FILE
        echo "O = Avaya Inc"                                  >> $EXT_FILE
        echo "OU = GCS"                                       >> $EXT_FILE
        
        CN="$ISSUED_TO/emailAddress=support@avaya.com"
        if [[ ${#CN} -gt 64 ]]; then
            echo "CN = iposerver.avaya.com/emailAddress=support@avaya.com" >> $EXT_FILE
        else
            echo "CN = $CN" >> $EXT_FILE
        fi

        echo "[ v3_req ]"                                     >> $EXT_FILE
        echo "basicConstraints = $CA"                         >> $EXT_FILE
        echo "keyUsage = $KEY_USAGE"                          >> $EXT_FILE
        echo "subjectAltName = $SUBJECT_ALT_NAME"             >> $EXT_FILE
        echo "subjectKeyIdentifier   = hash"                  >> $EXT_FILE
        echo "extendedKeyUsage   = serverAuth,clientAuth"     >> $EXT_FILE
    else
        # [ v3_req ]
        echo "[ v3_req ]"                           > $EXT_FILE
        echo "basicConstraints = $CA"              >> $EXT_FILE
        echo "keyUsage = $KEY_USAGE"               >> $EXT_FILE
        echo "subjectAltName = $SUBJECT_ALT_NAME"  >> $EXT_FILE
        echo "authorityKeyIdentifier = keyid"      >> $EXT_FILE
        echo "subjectKeyIdentifier   = hash"       >> $EXT_FILE
        if $EXTENDED_KEY_USAGE
        then
            echo "extendedKeyUsage   = serverAuth,clientAuth" >> $EXT_FILE
        fi
    fi
}

init_ca_variables() {
    # Use the default ID if none specified
    if [ -z "$ID" ]
    then
        ID=`hostname`
    fi

    # Use the default CN if none specified
    if [ -z "$ISSUED_TO" ]
    then
        ISSUED_TO="ipoffice-root-$ID.avaya.com"
    fi

    # Use the default SubjectAltName if none specified
    if [ -z "$SUBJECT_ALT_NAME" ]
    then
        SUBJECT_ALT_NAME="URI.1:ipoffice-root-$ID.avaya.com"
    fi

    # Default CA Certificate Validity
    if [ -z "$CERT_VALIDITY" ]
    then
        CERT_VALIDITY=$DEFAULT_CA_CERT_VALIDITY
    fi

    # Default RSA Key Size
    if [ -z "$RSA_KEY_LEN" ]
    then
        RSA_KEY_LEN=$DEFAULT_RSA_KEY_LEN
    fi

    # Default RSA Key Size
    if [ -z "$SHA_VER" ]
    then
        SHA_VER=$DEFAULT_SHA_VER
    fi
}

# Generate Root CA's Certificate and Private Key
generate_ca_cert() {
    local EXT_FILE=`mktemp`
    local KEY_USAGE="digitalSignature, keyCertSign, cRLSign"
    local CA="CA:TRUE"
    local CERT_SUBJECT="/C=US/ST=New Jersey/L=Basking Ridge/O=Avaya Inc/OU=GCS/CN=$ISSUED_TO/emailAddress=support@avaya.com"
    local CA_CERT_REQ=$CA_CERT_DIR/root-ca.csr
    local CA_KEY_PASS=$CA_CERT_DIR/root-ca_pass.key
    local DEFA_PASS=1234

    TMP_FILES="$TMP_FILES $EXT_FILE $CA_KEY_PASS $CA_CERT_REQ"

    # Generate OPENSSL config file for CA certificate
    generate_config_file "$EXT_FILE" "$CA" "$KEY_USAGE" "$SUBJECT_ALT_NAME" false

    if $CA_REGENERATE
    then
       if [ -f $CA_KEY ]; then
           REGENERATING_CA=true
           log "Regenerating the CA"
       else
           err_exit $EGENCERT "Cannot find the RSA Key"
       fi
    else
        # Generate RSA key
        log "Generate RSA key"
        openssl genrsa -des3 -out $CA_KEY_PASS -passout pass:$DEFA_PASS $RSA_KEY_LEN \
            || err_exit $EGENKEY "Cannot generate RSA Key"
        # Remove passphrase
        openssl rsa -passin pass:$DEFA_PASS -in $CA_KEY_PASS -out $CA_KEY

        # Request CA Certificate
    fi
    log "Request CA Certificate"
    openssl req -new -batch $SHA_VER -out $CA_CERT_REQ -key $CA_KEY -subj "$CERT_SUBJECT" \
        || err_exit $EGENCERT "Cannot generate Certificate Request"

    # Self Sign the CA Certificate
    log "Self Sign the CA Certificate"

    # the certificate is generated to be valid since 24 hours ago.
    # openssl calculates the validiy (-days param.) since now not since -startdate.
    local CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY-1))

    local DateMin=`date --date="01/01/2014" +"%s"`
    local DateMax=`date --date="01/01/2029" +"%s"`
    local DateDefault=`date --date="24 hours ago" +"%s"`

    #set default date to 24 h ago
    local StartDate=$DateDefault

    #if resulting date is invalid change it to default value
    if [ $DateDefault -le $DateMin ]; then
        DateDiff=$((DateMin-StartDate))
        ((DateDiffDays=${DateDiff}/(60*60*24)))
        CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY+DateDiffDays))
        StartDate=$DateMin
    elif [ $DateDefault -gt $DateMax ]; then
        DateDiff=$((StartDate-DateMin))
        ((DateDiffDays=${DateDiff}/(60*60*24)))
        CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY-DateDiffDays))
        StartDate=$DateMin
    fi


    #convert date from "seconds from epoch" to YYMMDDHHMMSSZ (UTC format)
    local StartDateFormated=`date -d @$StartDate +%g%m%d%H%M%S`"Z"
    #sign certificate request with openssl ca
    if [ ! -f /etc/pki/CA/index.txt ]; then
        touch /etc/pki/CA/index.txt
    fi

    if [ ! -f /etc/pki/CA/index.txt.attr ]; then
        echo "unique_subject = no">> /etc/pki/CA/index.txt.attr
    fi

    local CERT_SERIAL
    #get a random string
    CERT_SERIAL="`printf "%06x" $RANDOM`"
    #concatenate date as hex
    CERT_SERIAL=$CERT_SERIAL"$(echo "obase=16; "$(date +%s)""|bc)"

    if [ ! -f /etc/pki/CA/serial ]; then
        echo $CERT_SERIAL >> /etc/pki/CA/serial
    fi

    local DigestType=${SHA_VER#?}

    openssl ca -policy policy_anything -notext -batch -selfsign -md $DigestType -startdate $StartDateFormated -days $CERT_VALIDITY_EXTENDED -in $CA_CERT_REQ -keyfile $CA_KEY -out $CA_CERT -extfile $EXT_FILE -extensions v3_req \
    ||err_exit $ESIGNCERT "Cannot Sign the Certificate"

    convert_to_der
    
    # cleaup potential other previous CAs
    rm -f $CA_CERT_DIR/ca_*.pem
    rm -f $CA_CERT_DIR/ca_*.key

    add_ca_to_ipo_tcs
}

init_server_variables() {
    if [ ! -z "`/sbin/ifconfig | grep $IPOFFICE_LAN1`" ];then
        ETH0_MAC=`cat /sys/class/net/$IPOFFICE_LAN1/address`
        local SRV_ETH0_IP=`ip addr show dev $IPOFFICE_LAN1 | grep "inet " | cut -d " " -f 6 | cut -d "/" -f 1`
    fi
    if [ ! -z "$IPOFFICE_LAN2" ]; then
        if [ ! -z "`/sbin/ifconfig | grep $IPOFFICE_LAN2`" ];then
            local SRV_ETH1_IP=`ip addr show dev $IPOFFICE_LAN2 | grep "inet " | cut -d " " -f 6 | cut -d "/" -f 1`            
        fi
    fi

    # Get the hostname
    hName=$HOST_NAME
    if [ -z "$hName" ]; then
        hName=`hostnamectl --static`
    fi
    # Check if the ip of lan1 is resolvable
    if [ -z "$hName" -o "$hName" == "localhost" -o "$hName" == "localhost.localdomain" -o "$hName" == "$ETH0_MAC" ]; then
        hName=""
        if [ -n "$SRV_ETH0_IP" ]; then
            dNSName=`/usr/bin/host $SRV_ETH0_IP | cut -d' ' -f5`
        fi
        if [[ -z "$dNSName" || "$dNSName" =~ "NXDOMAIN" || "$dNSName" =~ "no" ]]; then
            if [ -n "$SRV_ETH1_IP" ]; then
                dNSName="`/usr/bin/host $SRV_ETH1_IP | cut -d' ' -f5`"
            fi
        fi
        if [[ -z "$dNSName" || "$dNSName" =~ "NXDOMAIN" || "$dNSName" =~ "no" ]]; then
            dNSName=""
        else
            dNSName=${dNSName%"."}
        fi
    fi

    # Use the defaut ID if none specified
    if [ -z "$ID" ]
    then
        # Default Server ID
        if [ ! -z "$hName" ]; then
            ID="$hName"
        elif [ ! -z "$dNSName" ]; then
            ID="$dNSName"
        else
            ID="$ETH0_MAC"
        fi
    fi

    # Use the defaut IssuedTo if none specified
    if [ -z "$ISSUED_TO" ]
    then
        if [ ! -z "$hName" -o ! -z "$dNSName" ]; then
            ISSUED_TO="$ID"
        else
            ISSUED_TO="iposerver-$ID.avaya.com"
        fi
    fi

    # Use the default SubjectAltName if none specified
    if [ -z "$SUBJECT_ALT_NAME" ]
    then
        if [ ! -z "$hName" -o ! -z "$dNSName" ]; then
            SUBJECT_ALT_NAME="DNS:$ID"
        else
            SUBJECT_ALT_NAME="DNS:iposerver-$ID.avaya.com"
        fi
        if [ -n "$SRV_ETH0_IP" ]
        then
            SUBJECT_ALT_NAME="$SUBJECT_ALT_NAME, IP:$SRV_ETH0_IP"
        fi
        if [ -n "$SRV_ETH1_IP" ]
        then
            SUBJECT_ALT_NAME="$SUBJECT_ALT_NAME, IP:$SRV_ETH1_IP"
        fi
        if [ "$1" == "stamp" ]; then
            # uniquely identify the machine
            SUBJECT_ALT_NAME="$SUBJECT_ALT_NAME, URI.0:ipoffice-$ETH0_MAC.avaya.com"
        fi
    fi
 
    # Use the default SubjectAltName if none specified
    if [ -z "$SERVER_IP" ]
    then
       [ -n "$SRV_ETH0_IP" ] && SERVER_IP="$SRV_ETH0_IP" || SERVER_IP="$SRV_ETH1_IP"
    fi
    [ -z "$SERVER_IP" ] && SERVER_IP="no_IP"

    # Default Server Certificate Validity
    if [ -z "$CERT_VALIDITY" ]
    then
        CERT_VALIDITY=$DEFAULT_SERVER_CERT_VALIDITY
    fi

    # Default RSA Key Size
    if [ -z "$RSA_KEY_LEN" ]
    then
        RSA_KEY_LEN=$DEFAULT_RSA_KEY_LEN
    fi

    # Default SHA Version
    if [ -z "$SHA_VER" ]
    then
        SHA_VER=$DEFAULT_SHA_VER
    fi

    # Server Certificate in PEM format
    if $IS_PRIMARY
    then
        SRV_CERT=$CERT_DIR/cert_to_import.pem
    else
        if $CERT_IS_SELF_SIGNED
        then
            SRV_CERT=$CERT_DIR/cert_to_import.pem
        else
            if $CERT_IS_PKCS12_FORMAT || $INSTALL_P12CERTIFICATE
            then
                SRV_CERT=$CERT_DIR/server_$SERVER_IP.p12
            else
                SRV_CERT=$CERT_DIR/server_$SERVER_IP.pem
            fi
        fi
    fi

    # if the CA Chain is enabled, find a suitable CA certficate to sign the
    # ID certificate
    SEARCH_CA=$ENABLE_CA_CHAIN
    if ! $SEARCH_CA; then
        if [ ! -f $CA_CERT ]; then
            SEARCH_CA=true
        fi
    fi
    if $SEARCH_CA
    then
        find_ca_cert
    fi
}

find_ca_cert() {
    local INT_CA=
    for INT_CA in $CA_CHAIN_CERTS
    do
        local CERT_VALID=false
        local KEY_VALID=false
        openssl x509 -text -in $INT_CA > /dev/null 2>&1 && CERT_VALID=true
        openssl rsa -text -in ${INT_CA%.pem}.key > /dev/null 2>&1 && KEY_VALID=true
        if $CERT_VALID && $KEY_VALID
        then
            CA_CERT=$INT_CA
            CA_KEY=${INT_CA%.pem}.key
        fi
    done
}

# Generate Server's Certificate and Private Key
generate_server_cert() {
    local KEY_USAGE="digitalSignature, nonRepudiation, keyAgreement, dataEncipherment, keyEncipherment"
    local CA="CA:FALSE"
    local CERT_SUBJECT="/C=US/ST=New Jersey/L=Basking Ridge/O=Avaya Inc/OU=GCS/CN=$ISSUED_TO/emailAddress=support@avaya.com"
    local EXT_FILE=`mktemp`
    local DEFA_PASS=1234
    local SCRIPT_LOCKED=false
    local ERROR_DETECTED=false
    # Encrypted Server Private Key
    local SRV_KEY_PASS=$CERT_DIR/server_$ID_pass.key
    # Server Certificate Request
    local SRV_CERT_REQ=$CERT_DIR/server_$ID.csr
    # Server Certificate
    local TMP_SRV_CERT=$CERT_DIR/server_$ID.crt
    # Server Private Key file
    local SRV_KEY=$CERT_DIR/server_$ID.key
    TMP_FILES="$TMP_FILES $EXT_FILE $SRV_CERT_REQ $TMP_SRV_CERT $SRV_KEY $SRV_KEY_PASS"
    # openssl calculates the validity -days since now not since startdate.
    local CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY-1))
(
    local DateMin=`date --date="01/01/2014" +"%s"`
    local DateMax=`date --date="01/01/2029" +"%s"`
    local DateDefault=`date --date="24 hours ago" +"%s"`

    # set default date to 24 h ago
    local StartDate=$DateDefault

    # if resulting date is invalid change it to default value
    if [ $DateDefault -le $DateMin ]; then
        DateDiff=$((DateMin-StartDate))
        ((DateDiffDays=${DateDiff}/(60*60*24)))
        CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY+DateDiffDays))
        StartDate=$DateMin
    elif [ $DateDefault -gt $DateMax ]; then
        DateDiff=$((StartDate-DateMin))
        ((DateDiffDays=${DateDiff}/(60*60*24)))
        CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY-DateDiffDays))
        StartDate=$DateMin
    fi


    # Due to security standards, since 1 March 2018 validity of a certificates shall not be over 825 days.
    # openssl calculates the validity -days since now not since startdate. IPOL certificates are generated since 24 hours ago.
    if [ $CERT_VALIDITY_EXTENDED -gt $((DEFAULT_SERVER_CERT_VALIDITY-1)) ]; then
        CERT_VALIDITY_EXTENDED=$((DEFAULT_SERVER_CERT_VALIDITY-1))
    fi

    #convert date from "seconds from epoch" to YYMMDDHHMMSSZ (UTC format)
    local StartDateFormated=`date -d @$StartDate +%g%m%d%H%M%S`"Z"

    local DigestType=${SHA_VER#?}
    local CERT_SERIAL
    #get random number
    CERT_SERIAL="`printf "%06x" $RANDOM`"
    #concatenate date as hex
    CERT_SERIAL=$CERT_SERIAL"$(echo "obase=16; "$(date +%s)""|bc)"

    local PROCEED="no"
    if $CERT_IS_PKCS12_FORMAT; then
        PROCEED="yes"
    elif flock -x -w 15 200; then
        PROCEED="yes"
        SCRIPT_LOCKED=true
        log "Locking gen_certs script"
    fi
    if [ "$PROCEED" == "yes" ]; then
        if $CERT_IS_PKCS12_FORMAT
        then
            if [ -z "$PKCS12_PASS" ]; then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $ENOPASS "You must specify a password for the pkcs12 format"
            fi
            if $CERT_IS_SELF_SIGNED
            then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $EINVAL "Cannot generate a pkcs12 self-signed certificate"
            fi
        fi
        # Generate OPENSSL config file for Server certificate
        generate_config_file "$EXT_FILE" "$CA" "$KEY_USAGE" "$SUBJECT_ALT_NAME" true

        if $CERT_IS_SELF_SIGNED
        then
            log "Generating a self signed certificate"
            # Generate RSA key
            log "Generate RSA key"
            openssl genrsa -des3 -out $SRV_KEY_PASS -passout pass:$DEFA_PASS $RSA_KEY_LEN > /dev/null 2>&1 \
                || ERROR_DETECTED=true
            if $ERROR_DETECTED; then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $EGENKEY "Cannot generate RSA Key"
            fi
            # Remove passphrase
            openssl rsa -passin pass:$DEFA_PASS -in $SRV_KEY_PASS -out $SRV_KEY > /dev/null 2>&1

            # Generate a certificate request
            log "Generating a certificate request"
            openssl req -new -batch $SHA_VER -key $SRV_KEY -out $SRV_CERT_REQ -subj "$CERT_SUBJECT" > /dev/null 2>&1 \
                || ERROR_DETECTED=true
            if $ERROR_DETECTED; then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $EGENCERT "Cannot generate Certificate Request"
            fi
 
            # Self Sign the certificate request
            log "Self Sign the generated certificate"
            
            if [ ! -f /etc/pki/CA/index.txt ]; then
                touch /etc/pki/CA/index.txt
            fi

            if [ ! -f /etc/pki/CA/index.txt.attr ]; then
                echo "unique_subject = no">> /etc/pki/CA/index.txt.attr
            fi

            if [ ! -f /etc/pki/CA/serial ]; then
                echo $CERT_SERIAL >> /etc/pki/CA/serial
            fi

            openssl ca -policy policy_anything -notext -batch -selfsign -md $DigestType -startdate $StartDateFormated -days $CERT_VALIDITY_EXTENDED -in $SRV_CERT_REQ -keyfile $SRV_KEY -out $TMP_SRV_CERT> /dev/null 2>&1 \
                || ERROR_DETECTED=true
            if $ERROR_DETECTED; then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $ESIGNCERT "Cannot Sign the Certificate"
            fi
        else
            log "Generate the Server Key and Request"
            openssl req -newkey rsa:$RSA_KEY_LEN -passout pass:$DEFA_PASS -keyout $SRV_KEY_PASS -batch $SHA_VER -out $SRV_CERT_REQ -subj "$CERT_SUBJECT" > /dev/null 2>&1 \
                || ERROR_DETECTED=true
            if $ERROR_DETECTED; then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $EGENKEY "Cannot Generate Key"
            fi
            openssl rsa -passin pass:$DEFA_PASS -in $SRV_KEY_PASS -out $SRV_KEY > /dev/null 2>&1

            log "Sign the Server Certificate"
           
            if [ ! -f /etc/pki/CA/index.txt ]; then
                touch /etc/pki/CA/index.txt
            fi
            
            if [ ! -f /etc/pki/CA/index.txt.attr ]; then
                echo "unique_subject = no">> /etc/pki/CA/index.txt.attr
            fi
            
            if [ ! -f /etc/pki/CA/serial ]; then
               echo $CERT_SERIAL >> /etc/pki/CA/serial
            fi
           
            openssl ca -policy policy_anything -notext -batch -cert $CA_CERT -md $DigestType -startdate $StartDateFormated -days $CERT_VALIDITY_EXTENDED -in $SRV_CERT_REQ -keyfile $CA_KEY -out $TMP_SRV_CERT -extfile $EXT_FILE -extensions v3_req > /dev/null 2>&1\
 || ERROR_DETECTED=true
            if $ERROR_DETECTED; then
                if $SCRIPT_LOCKED; then
                    log "Unlocking gen_certs script"
                    flock -u 200
                fi
                err_exit $ESIGNCERT "Cannot Sign the Certificate"
            fi
            if $CERT_IS_PKCS12_FORMAT
            then
                openssl pkcs12 -export -in $TMP_SRV_CERT -inkey $SRV_KEY -out $SRV_CERT -certfile $CA_CERT -caname root -passout pass:$PKCS12_PASS > /dev/null 2>&1 \
                    || ERROR_DETECTED=true
                if $ERROR_DETECTED; then
                    if $SCRIPT_LOCKED; then
                        log "Unlocking gen_certs script"
                        flock -u 200
                    fi
                    err_exit $EPKCS12CERT "Cannot generate pkcs12 certificate"
                fi
            fi
        fi

        if $CERT_IS_PKCS12_FORMAT
        then
            CERT_NAME=${SRV_CERT%%.p12*}
            RAND_CERT="$CERT_NAME""_""$RANDOM""$RANDOM"".p12"
            mv $SRV_CERT $RAND_CERT
            echo "PATH=$RAND_CERT"
            chmod 400 $RAND_CERT
        else
            cat $SRV_KEY $TMP_SRV_CERT > $SRV_CERT
            chmod 400 $SRV_CERT
        fi
        if $SCRIPT_LOCKED; then
            log "Unlocking gen_certs script"
            flock -u 200
        fi
    else
        err_exit $ELOCK "Error - cannot aquire lock"
    fi
) 200>$LOCKFILE
}

generate_csr() {
    local KEY_USAGE="digitalSignature, nonRepudiation, keyAgreement, dataEncipherment, keyEncipherment"
    local CA="CA:FALSE"
    local CERT_SUBJECT="/C=US/ST=New Jersey/L=Basking Ridge/O=Avaya Inc/OU=GCS/CN=$ISSUED_TO/emailAddress=support@avaya.com"
    local EXT_FILE=`mktemp`
    local DEFA_PASS=1234
    local SCRIPT_LOCKED=false
    local ERROR_DETECTED=false
    local ID_pass=$(cat /dev/urandom | tr -dc 'a-zA-Z0-9' | fold -w 8 | head -n 1)
    # Encrypted Server Private Key
    local SRV_KEY_PASS=$CERT_DIR/server_$ID_pass.key
    # Server Certificate Request
    local SRV_CERT_REQ=$CERT_DIR/server_$ID_pass.csr
    local SRV_KEY=$CERT_DIR/server_$ID.key
    
    TMP_FILES="$TMP_FILES $SRV_CERT_REQ $EXT_FILE $SRV_KEY_PASS $SRV_KEY"

    generate_config_file "$EXT_FILE" "$CA" "$KEY_USAGE" "$SUBJECT_ALT_NAME" true true

    log "Generating RSA key length $RSA_KEY_LEN"

    openssl genrsa -des3 -out $SRV_KEY_PASS -passout pass:$DEFA_PASS $RSA_KEY_LEN > /dev/null 2>&1 || ERROR_DETECTED=true

    if $ERROR_DETECTED; then
        log "Exiting..."
        err_exit $EGENKEY "Cannot generate RSA Key"
    fi

    openssl rsa -passin pass:$DEFA_PASS -in $SRV_KEY_PASS -out $SRV_KEY > /dev/null 2>&1

    openssl req -new -batch $SHA_VER -key $SRV_KEY -out $SRV_CERT_REQ -subj "$CERT_SUBJECT" -reqexts v3_req -config $EXT_FILE > /dev/null 2>&1 || ERROR_DETECTED=true

    if $ERROR_DETECTED; then
        err_exit $EGENCERT "Cannot generate Certificate Request"
    fi

    cat $SRV_CERT_REQ
}

install_p12_certificate() {
    local TMP_CERT=`mktemp`
    local DECRYPTED_CERT=`mktemp`
    local CA_DECRYPTED_CERT=`mktemp`
    local TMP_KEY=`mktemp`
    local OUT_CERT=$CERT_DIR/cert_to_import.pem

    TMP_FILES="$TMP_FILES $DECRYPTED_CERT $CA_DECRYPTED_CERT $TMP_CERT $TMP_KEY"

    if [ -z "$PKCS12_PASS" ]; then
        err_exit $ENOPASS "No password specified"
    fi

    if [ ! -f $SRV_CERT ]; then
        err_exit $ENOCERT "No pkcs12 certificate found"
    else
        if [  -z "`openssl pkcs12 -in $SRV_CERT -nodes -passin pass:$PKCS12_PASS`" ]; then
            err_exit $EP12DECRYPT "Error decrypting pkcs12 certificate"
        fi
    fi

(
    local PROCEED="no"
    if flock -x -w 15 200; then
        PROCEED="yes"
        log "Locking gen_certs script"
    fi    
    if [ "$PROCEED" == "yes" ]; then
        openssl pkcs12 -in $SRV_CERT -nokeys -nodes -passin pass:$PKCS12_PASS | \
            sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' > $TMP_CERT
        COUNTER=0
        while read line
        do
            if [ "$line" == "-----BEGIN CERTIFICATE-----" ]; then
                if [[ $COUNTER -eq 0 ]]; then
                    OUTPUT_FILE=$DECRYPTED_CERT
                else
                    OUTPUT_FILE=$CA_DECRYPTED_CERT
                fi
                let COUNTER=COUNTER+1 
            fi
            echo $line >> $OUTPUT_FILE
        done < $TMP_CERT
        if $VMPRO_PRESENT
        then
            if [ ! -f $CA_CERT ]; then
                cp $CA_DECRYPTED_CERT $CA_CERT
                chmod 400 $CA_CERT
            fi
        fi 
        openssl pkcs12 -in $SRV_CERT -nocerts -nodes -passin pass:$PKCS12_PASS | \
            sed -n '/-----BEGIN PRIVATE KEY-----/,/-----END PRIVATE KEY-----/p' > $TMP_KEY

        SAN="`openssl x509 -in $TMP_CERT -text|grep URI`"
        SAN=${SAN##*URI:}
        DMAC="`echo $SAN|cut -d"-" -f2 | cut -d"." -f1`"

        openssl rsa -in $TMP_KEY > $TMP_CERT
        cat $DECRYPTED_CERT >> $TMP_CERT

        if $VERIFY_CSR; then
            if [ "$DMAC" == "$ETH0_MAC" ]; then
                cp $TMP_CERT $OUT_CERT
                rm -f $SRV_CERT
                log "Unlocking gen_certs script"
                flock -u 200
            else
                rm -f $SRV_CERT
                log "Unlocking gen_certs script"
                flock -u 200
                err_exit $ENOTMATCH "CSR does not match"
            fi
        else
            cp $TMP_CERT $OUT_CERT
            rm -f $SRV_CERT
            log "Unlocking gen_certs script"
            flock -u 200
        fi
    else
        err_exit $ELOCK "Error aquiring lock"
    fi
) 200>$LOCKFILE
}

convert_to_der() {
(
    log "Requesting lock to convert certificate to der format"
    if flock -x -w 600 200; then
        log "Locking gen_certs script"
        if [ -z "$FILE_TO_CONVERT_TODER" ]; then
            # if no file specified then probably we need to convert the
            # root ca to der
            if [ ! -f $CA_CERT ]; then
                flock -u 200
                log "Unlocking gen_certs script"
                err_exit $ENOCERT "No certificate found"
            else
                CERT_DER_NAME="${CA_CERT%%.pem*}"".crt"
                openssl x509 -outform der -in $CA_CERT -out $CERT_DER_NAME
            fi
        else
            CERT_TO_CONVERT="$CERT_DIR""/""$FILE_TO_CONVERT_TODER"
            if [ ! -f $CERT_TO_CONVERT ]; then
                flock -u 200
                log "Unlocking gen_certs script"
                err_exit $ENOCERT "No certificate found"
            else
                CERT_DER_NAME="${CERT_TO_CONVERT%%.pem*}"".crt"
                openssl x509 -outform der -in $CERT_TO_CONVERT -out $CERT_DER_NAME
            fi
        fi
        log "Unlocking gen_certs script"
        flock -u 200
    else
        log "Error: could not acquire lock"
    fi
) 200>$LOCKFILE
}

add_ca_to_ipo_tcs() {
    local IPOFFICE_TCS_ADD="/opt/ipoffice/system/primary/certificates/tcs/add/"
    local IPOFFICE_TCS_DEL="/opt/ipoffice/system/primary/certificates/tcs/delete/"
 
    if [ "$1" == "restored" ]; then
        REVOKED_CERTS=`ls $IPOFFICE_TCS_DEL`
        for rc in $REVOKED_CERTS
        do
            if [ "$rc" != "readme.txt" ]; then
                rm -f $IPOFFICE_TCS_DEL/$rc
            fi
        done
    fi
    #get a random string
    RAND_SERIAL="`printf "%06x" $RANDOM`"
    #concatenate date as hex
    RAND_SERIAL=$RAND_SERIAL"$(echo "obase=16; "$(date +%s)""|bc)"
    OLD_CERT=`ls $IPOFFICE_TCS_ADD`
    for oc in $OLD_CERT
    do
        if [ "$oc" != "readme.txt" ]; then
            if [ "$1" == "restored" ]; then
                rm -f $IPOFFICE_TCS_ADD/$oc
            else
                OLD_CERT_EXT=${oc##*.}
                OLD_CERT_NAME=${oc%.*}
                REVOKED_CERT="$OLD_CERT_NAME""_""$RAND_SERIAL"".""$OLD_CERT_EXT"
                mv "$IPOFFICE_TCS_ADD/$oc" "$IPOFFICE_TCS_DEL/$REVOKED_CERT"
            fi
        fi
    done
    if [ -f $CA_CERT_DIR/ca_1.pem ]; then
        cp $CA_CERT_DIR/ca_*.pem $IPOFFICE_TCS_ADD
        if [ -f $CA_CERT_DIR/chain.pem ]; then
            cp $CA_CERT_DIR/chain.pem $IPOFFICE_TCS_ADD
        fi
    else
        cp $CA_CERT_DIR/root-*.pem $IPOFFICE_TCS_ADD
    fi
}

init_chain_variables() {
    # Use the defaut IssuedTo if none specified
    if [ -z "$ISSUED_TO" ]
    then
        ISSUED_TO="ipoffice-int$CHAIN_NO.avaya.com"
    fi

    # Use the default SubjectAltName if none specified
    if [ -z "$SUBJECT_ALT_NAME" ]
    then
        local SRV_ETH0_IP=`ip addr show dev eth0 | grep "inet " | cut -d " " -f 6 | cut -d "/" -f 1`
        local SRV_ETH1_IP=`ip addr show dev eth1 | grep "inet " | cut -d " " -f 6 | cut -d "/" -f 1`

        SUBJECT_ALT_NAME="URI.1:ipoffice-int$CHAIN_NO.avaya.com"
        if [ -n "$SRV_ETH0_IP" ]
        then
            SUBJECT_ALT_NAME="$SUBJECT_ALT_NAME, IP.1:$SRV_ETH0_IP"
        fi
        if [ -n "$SRV_ETH1_IP" ]
        then
            SUBJECT_ALT_NAME="$SUBJECT_ALT_NAME, IP.2:$SRV_ETH1_IP"
        fi
    fi

    # Default Server Certificate Validity
    if [ -z "$CERT_VALIDITY" ]
    then
        CERT_VALIDITY=$DEFAULT_SERVER_CERT_VALIDITY
    fi

    # Default RSA Key Size
    if [ -z "$RSA_KEY_LEN" ]
    then
        RSA_KEY_LEN=$DEFAULT_RSA_KEY_LEN
    fi

    # Default SHA Version
    if [ -z "$SHA_VER" ]
    then
        SHA_VER=$DEFAULT_SHA_VER
    fi

    # Server Certificate in PEM format
    CHAIN_CERT=$CERT_DIR/ca/ca_$CHAIN_NO.pem

    if [ "$CHAIN_NO" -ne "1" ]
    then
        local PARENT_NO=$((CHAIN_NO-1))
        CA_CERT="$CA_CERT_DIR/ca_$PARENT_NO.pem"
        CA_KEY="$CA_CERT_DIR/ca_$PARENT_NO.pem"
    fi
}

# Generate CA Chain
generate_ca_chain() {
    local KEY_USAGE="digitalSignature, keyCertSign, cRLSign"
    local CA="CA:TRUE"
    local CERT_SUBJECT="/C=US/ST=New Jersey/L=Basking Ridge/O=Avaya Inc/OU=GCS/CN=$ISSUED_TO/emailAddress=support@avaya.com"
    local EXT_FILE=`mktemp`
    local DEFA_PASS=1234
    # Encrypted Server Private Key
    local CHAIN_KEY_PASS=$CERT_DIR/server_$ID.key
    # Server Certificate Request
    local CHAIN_CERT_REQ=$CERT_DIR/server_$ID.csr
    # Server Certificate
    local TMP_CHAIN_CERT=$CERT_DIR/server_$ID.crt
    # Server Private Key file
    local CHAIN_KEY=$CERT_DIR/server_$ID.key

    TMP_FILES="$TMP_FILES $EXT_FILE $CHAIN_CERT_REQ $TMP_CHAIN_CERT $CHAIN_KEY"

    # Generate OPENSSL config file for Server certificate
    generate_config_file "$EXT_FILE" "$CA" "$KEY_USAGE" "$SUBJECT_ALT_NAME" false

    log "Generate the Server Key and Request"
    openssl req -newkey rsa:$RSA_KEY_LEN -passout pass:$DEFA_PASS -keyout $CHAIN_KEY_PASS -batch $SHA_VER -out $CHAIN_CERT_REQ -subj "$CERT_SUBJECT" \
        || err_exit $EGENKEY "Cannot Generate Key"
    openssl rsa -passin pass:$DEFA_PASS -in $CHAIN_KEY_PASS -out $CHAIN_KEY

    log "Sign the Server Certificate"

    
    local DateMin=`date --date="01/01/2014" +"%s"`
    local DateMax=`date --date="01/01/2029" +"%s"`
    local DateDefault=`date --date="24 hours ago" +"%s"`

    #set default date to 24 h ago
    local StartDate=$DateDefault

    #if resulting date is invalid change it to default value
    if [ $DateDefault -le $DateMin ] || [ $DateDefault -gt $DateMax ]; then

      StartDate=$DateMin

    fi

    #convert date from "seconds from epoch" to YYMMDDHHMMSSZ (UTC format)
    local StartDateFormated=`date -d @$StartDate +%g%m%d%H%M%S`"Z"

    #sign certificate request with openssl ca
    local CERT_VALIDITY_EXTENDED=$((CERT_VALIDITY+1))
    if [ ! -f /etc/pki/CA/index.txt ]; then
      touch /etc/pki/CA/index.txt
    fi

    if [ ! -f /etc/pki/CA/index.txt.attr ]; then
      echo "unique_subject = no">> /etc/pki/CA/index.txt.attr
    fi

    local CERT_SERIAL
    #get random number
    CERT_SERIAL="`printf "%06x" $RANDOM`"
    #concatenate date as hex
    CERT_SERIAL=$CERT_SERIAL"$(echo "obase=16; "$(date +%s)""|bc)"

    if [ ! -f /etc/pki/CA/serial ]; then
        echo $CERT_SERIAL >> /etc/pki/CA/serial
    fi

    local DigestType=${SHA_VER#?}

    openssl ca -policy policy_anything -notext -batch -cert $CA_CERT -md $DigestType -startdate $StartDateFormated -days $CERT_VALIDITY_EXTENDED -in $CHAIN_CERT_REQ -keyfile $CA_KEY -out $TMP_CHAIN_CERT -extfile $EXT_FILE -extensions v3_req \
 ||err_exit $ESIGNCERT "Cannot Sign the Certificate"

    cat $TMP_CHAIN_CERT $CHAIN_KEY > $CHAIN_CERT

    rm -f $CA_CERT_DIR/chain.pem

    for chain_file in `find $CA_CERT_DIR -maxdepth 1 -name "ca_*.pem" | sort`
    do
        openssl x509 -in $chain_file >> $CA_CERT_DIR/chain.pem
    done
}

# Export the CA's Certificate and Private Key in PKCS-12 format
export_ca() {
    if [ ! -f $CA_CERT ]
    then
        err_exit $ECERTNOTFOUND "Please generate the CA Certificate first."
    fi

    if [ ! -f $CA_KEY ]
    then
        err_exit $EKEYNOTFOUND "Please generate the CA Key first."
    fi

    # Export the PKCS-12 certificate
    openssl pkcs12 -export -out $PKCS12_FILE -in $CA_CERT -inkey $CA_KEY -password pass:"$PKCS12_PASS" \
        || err_exit $EFAIL "Cannot Export CA Certificate"
}

# Import the CA's Certificate and Private Key in PKCS-12 format
import_ca() {
    if [ ! -f $PKCS12_FILE ]
    then
        err_exit $ECERTNOTFOUND "PKCS-12 file $PKCS12_FILE not found."
    fi

    local TMP_KEY=`mktemp`
    local TMP_CERT=`mktemp`
    local ROOT_CA=false

    TMP_FILES="$TMP_FILES $TMP_KEY $TMP_CERT $PKCS12_FILE"

    # Extract CA Certificate
    log "Extracting CA Certificate"
    openssl pkcs12 -in "$PKCS12_FILE" -nokeys -out "$TMP_CERT" -nodes -passin pass:"$PKCS12_PASS" \
        || err_exit $ECERTNOTFOUND "Cannot Load Certificate from $PKCS12_FILE"

    # Extract CA RSA Key
    log "Extracting CA Key"
    openssl pkcs12 -in "$PKCS12_FILE" -nocerts -out "$TMP_KEY" -nodes -passin pass:"$PKCS12_PASS" \
        || err_exit $EKEYNOTFOUND "Cannot Load Private Key from $PKCS12_FILE"
    openssl rsa -in  $TMP_KEY -out $TMP_KEY

    # Check if the public key from the extracted certificate matches the public
    # key from the key file
    local certificate_key_modulus=`openssl x509 -noout -modulus -in "$TMP_CERT"`
    local private_key_modulus=`openssl rsa -noout -modulus -in "$TMP_KEY"`
    if [ "$certificate_key_modulus" != "$private_key_modulus" ]
    then
        err_exit $ENOTMATCH "Non-Matching Certificate and Private Key from $PKCS12_FILE"
    fi

    # Check if the CA Certificate is self signed
    CA_SUBJECT=`openssl x509 -in $TMP_CERT -inform PEM -noout -subject`
    CA_SUBJECT=${CA_SUBJECT##*subject= }
    CA_ISSUER=`openssl x509 -in $TMP_CERT -inform PEM -noout -issuer`
    CA_ISSUER=${CA_ISSUER##*issuer= }
    if [ "$CA_SUBJECT" == "$CA_ISSUER" ]; then
        ROOT_CA=true
    fi
    if ! $ROOT_CA
    then
        log "Found intermediate CA"
        rm -f $CA_CERT_DIR/ca_*.pem
        CA_CERT=$CA_CERT_DIR/ca_1.pem
        CA_KEY=$CA_CERT_DIR/ca_1.key
        CA_CHAIN=$CA_CERT_DIR/chain.pem
        cp $TMP_CERT $CA_CHAIN
        rm -f $CA_CERT_DIR/root-ca*
    fi

    # Import the Certificate and Private Key files
    cp $TMP_CERT $CA_CERT
    cp $TMP_KEY $CA_KEY

    convert_to_der

    add_ca_to_ipo_tcs
}

need_restart() {
    local SERVICE=$1
    local NEED_RESTART=1

    if $ENABLE_NO_RESTART_MODE
    then
        if [ "$ENABLE_FORCE_RESTART_MODE" == "false" ]; then
            log "Skip restarting $SERVICE because of --no-restart"
            return $NEED_RESTART
        fi
    fi

    # Need to restart services only if the auto mode is enabled
    if [ -f /opt/Avaya/certs/.auto ]; then
        if $ENABLE_FORCE_RESTART_MODE
        then
            if [ "$SERVICE" != "webcontrol" ]; then
                # We will force start all services that are marked
                # to start at boot
                RLEVEL=`runlevel | cut -d" " -f2`
                RUNAPP=`systemctl list-unit-files | grep "^$SERVICE.service" | grep "enabled"`
                if [ ! -z "$RUNAPP" ]; then
                    NEED_RESTART=0
                fi
            fi
        else
            if [ ! -f /opt/Avaya/certs/.wcp_no_restart ]; then
                if [ "$SERVICE" == "onexportal" ]; then
                    /etc/init.d/$SERVICE status 2>/dev/null | grep -E "^.*(running|starting)" && NEED_RESTART=0
                else
                    /etc/init.d/$SERVICE status 2>/dev/null | grep running && NEED_RESTART=0
                fi
            else
                if [ "$SERVICE" != "webcontrol" ]; then
                    if [ "$SERVICE" == "onexportal" ]; then
                        /etc/init.d/$SERVICE status 2>/dev/null | grep -E "^.*(running|starting)" && NEED_RESTART=0
                    else
                        /etc/init.d/$SERVICE status 2>/dev/null | grep running && NEED_RESTART=0
                    fi
                fi
            fi
        fi
    fi

    if [ $NEED_RESTART -eq 0 ]; then
        log "Restart $SERVICE"
    fi

    return $NEED_RESTART
}

distribute_server_cert_to_webcontrol() {
    log "Starting webcontrol distribution process"

    local WEBCONTROL_CERT_DIR=$WEBCONTROL_HOME/certificate
    local NEED_UPDATE=0

    if [ ! -d $WEBCONTROL_CERT_DIR ]
    then
        log "Warning: Could not find webcontrol directory $WEBCONTROL_CERT_DIR"
        return 1
    fi

    local WEBCONTROL_CERT=$WEBCONTROL_CERT_DIR/server.pem
    local CERT=$1

    local SAME_CERT=false
    local SAME_CHAIN=false

    # Check if the certificate needs to be updated
    if [ -f $WEBCONTROL_CERT ] && diff $CERT $WEBCONTROL_CERT > /dev/null
    then
        SAME_CERT=true
    fi

    local WEBCONTROL_CHAIN=$WEBCONTROL_CERT_DIR/chain.pem
    if $ENABLE_CA_CHAIN
    then
        local TMP_WEBCONTROL_CHAIN=`mktemp`
        TMP_FILES="$TMP_FILES $TMP_WEBCONTROL_CHAIN"

        cat $CERT $CA_CHAIN > $TMP_WEBCONTROL_CHAIN

        if [ -f $WEBCONTROL_CHAIN ] && diff $TMP_WEBCONTROL_CHAIN $WEBCONTROL_CHAIN > /dev/null
        then
            SAME_CHAIN=true
        fi

        cp $TMP_WEBCONTROL_CHAIN $WEBCONTROL_CHAIN
        # Need to set rights so that webcontrol can use the certificate
        chown wcp:wcp $WEBCONTROL_CHAIN
        chmod 400 $WEBCONTROL_CHAIN
    else
        if [ -f $WEBCONTROL_CHAIN ]; then
            rm -f $WEBCONTROL_CHAIN
        fi
        SAME_CHAIN=true
    fi

    if $SAME_CERT && $SAME_CHAIN
    then
        log "The webcontrol Server Certificate is already updated."
        return 1
    fi

    log "Copy Certificate to webcontrol"
    cp -f $CERT $WEBCONTROL_CERT
    # Need to set rights so that webcontrol can use the certificate
    chmod 400 $WEBCONTROL_CERT

    # Restart WebControl Service

    need_restart webcontrol && systemctl restart webcontrol && NEED_UPDATE=1


    log "Webcontrol distribution process completed"

    return $NEED_UPDATE
}

distribute_server_cert_to_onexportal() {
    log "Starting onexportal distribution process"

    local ONEX_CERT_DIR=$ONEX_PATH/apache-tomcat/conf
    local ONEX_CERT=$ONEX_CERT_DIR/server.pem
    local CERT=$1
    local need_update=false

    local OLD_CERT=`mktemp`
    local TMP_KEYSTORE=`mktemp`
    local TMP_PRIVATE_KEY=`mktemp`
    local TMP_NEW_CERT=`mktemp`
    local STORE_PASS="admin@12345"
    local ONEX_STORE="$ONEX_PATH/apache-tomcat/webapps/pa.bin"
    local ALIAS="onex"
    local FOUND_ONEX_CERT=true

    TMP_FILES="$TMP_FILES $OLD_CERT $TMP_KEYSTORE $TMP_PRIVATE_KEY $TMP_NEW_CERT"

    if [ ! -d $ONEX_PATH/apache-tomcat/webapps ]
    then
        log "Warning: Could not find onex directory $ONEX_PATH/apache-tomcat/webapps"
        return 1
    fi
    
    if [ ! -d $ONEX_CERT_DIR ]
    then
        log "Warning: Could not find onex directory $ONEX_CERT_DIR"
        return 1
    fi

    cat $CERT | sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' > $TMP_NEW_CERT
    if $ENABLE_CA_CHAIN
    then
        if [ -f $CA_CHAIN ]
        then
            cat $CA_CHAIN | sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' >> $TMP_NEW_CERT
        fi
    fi
    cat $CERT | sed -n '/-----BEGIN RSA PRIVATE KEY-----/,/-----END RSA PRIVATE KEY-----/p' >> $TMP_NEW_CERT

    # Export the onex store
    rm -f $TMP_KEYSTORE
    keytool -importkeystore -srckeystore $ONEX_STORE -destkeystore $TMP_KEYSTORE -deststoretype PKCS12 \
        -alias $ALIAS -srcstorepass $STORE_PASS -deststorepass $STORE_PASS -srckeypass $STORE_PASS -noprompt \
        || FOUND_ONEX_CERT=false

    if $FOUND_ONEX_CERT
    then
        # Export the current certificate
        openssl pkcs12 -in $TMP_KEYSTORE -nokeys -nodes -passin pass:$STORE_PASS | \
            sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' > $OLD_CERT

        # Export the current private key
        openssl pkcs12 -in $TMP_KEYSTORE -nocerts -nodes -passin pass:$STORE_PASS | \
            sed -n '/-----BEGIN PRIVATE KEY-----/,/-----END PRIVATE KEY-----/p' > $TMP_PRIVATE_KEY
        openssl rsa -in $TMP_PRIVATE_KEY >> $OLD_CERT

        if diff $OLD_CERT $TMP_NEW_CERT > /dev/null
        then
            log "The OneX Binary Server Certificate is already updated."
        fi
    fi

    distribute_server_cert_to_openfire

    local SAME_CERT=false
    local SAME_CHAIN=false
    # Check if the certificate needs to be updated
    if [ -f $ONEX_CERT ] && diff $CERT $ONEX_CERT > /dev/null
    then
        SAME_CERT=true
    else
        log "Copy PEM Certificate to onexportal"
        # Distribute the server certificate
        cp -f $CERT $ONEX_CERT
        chown onexp:onexp $ONEX_CERT
        chmod 400 $ONEX_CERT

        if $ENABLE_CA_CHAIN
        then
            rm -f $ONEX_CERT_DIR/chain.pem
            if [ "$CA_CHAIN" ]
            then
                cp $CA_CHAIN $ONEX_CERT_DIR
                chown onexp:onexp "$ONEX_CERT_DIR/chain.pem"
                chmod 400 "$ONEX_CERT_DIR/chain.pem"
            fi
        else
            if [ -f "$ONEX_CERT_DIR/chain.pem" ]; then
                rm -f $ONEX_CERT_DIR/chain.pem
            fi
            SAME_CHAIN=true
        fi
    fi

    if ! $SAME_CERT || ! $SAME_CHAIN
    then
        need_update=true
    else
        log "The OneXportal PEM Server Certificate is already updated."
    fi

    if ! $need_update
    then
        return 1
    fi

    need_restart onexportal && systemctl restart onexportal 

    log "Onexportal distribution process completed"

    return 0
}

distribute_server_cert_to_openfire() {
    log "Starting openfire distribution process"   

    # Distribute the server certificate
    bash $SCRIPTS_PATH/keystore.sh

    log "Openfire distribution process completed" 

    return 0
}

distribute_server_cert_to_vmpro() {
    log "Starting vmpro distribution process"

    local VMPRO_CERT_DIR=$VMPRO_PATH/Certificate
    if [ ! -d $VMPRO_CERT_DIR ]
    then
        log "Warning: Could not find vmpro directory $VMPRO_CERT_DIR"
    fi

    local VMPRO_CERT=$VMPRO_CERT_DIR/server.pem
    local CERT=$1
    local TMP_VMPRO_CERT=`mktemp`

    TMP_FILES="$TMP_FILES $TMP_VMPRO_CERT"

    cp -f $CERT $TMP_VMPRO_CERT
    if $ENABLE_CA_CHAIN
    then
        cat $CA_CHAIN >> $TMP_VMPRO_CERT
    fi

    # Check if the certificate needs to be updated
    if [ -f $VMPRO_CERT ] && diff $TMP_VMPRO_CERT $VMPRO_CERT > /dev/null
    then
        log "The vmpro Server Certificate is already updated."
        return 1
    fi

    log "Copy Certificate to vmpro"
    # Distribute the server certificate
    cp -f $TMP_VMPRO_CERT $VMPRO_CERT
    cp -f $CA_CERT $VMPRO_CERT_DIR

    need_restart vmpro && systemctl restart vmpro &

    log "Vmpro distribution process continuing in the background"

    return 0
}

distribute_server_cert_to_weblm() {
    log "Starting WebLm distribution process"

    local WEBLM_CERT_DIR=/opt/Avaya/apache-tomcat/webapps/WebLM/WEB-INF
    if [ ! -d $WEBLM_CERT_DIR ]
    then
        log "Warning: Could not find weblm directory $WEBLM_CERT_DIR"
    fi

    local WEBLM_CERT=$WEBLM_CERT_DIR/server.pem
    local CERT=$1
    local SAME_CERT=false
    local SAME_CHAIN=false
    local need_update=false

    # Check if the certificate needs to be updated
    if [ -f $WEBLM_CERT ] && diff $CERT $WEBLM_CERT > /dev/null
    then
        SAME_CERT=true
    fi

    if $ENABLE_CA_CHAIN
    then
        if [ -f $WEBLM_CERT_DIR/chain.pem ] && diff $WEBLM_CERT_DIR/chain.pem $CA_CHAIN > /dev/null
        then
            SAME_CHAIN=true
        fi
        rm -f $WEBLM_CERT_DIR/chain.pem
        if [ -f "$CA_CHAIN" ]
        then
            cp $CA_CHAIN $WEBLM_CERT_DIR
        fi
    else
        if [ -f "$WEBLM_CERT_DIR/chain.pem" ]; then
            rm -f $WEBLM_CERT_DIR/chain.pem
        fi
        SAME_CHAIN=true
    fi

    if ! $SAME_CERT || ! $SAME_CHAIN
    then
        need_update=true
    else
        log "The weblm Server Certificate is already updated."
    fi

    log "Copy Certificate to weblm"
    # Distribute the server certificate
    cp -f $CERT $WEBLM_CERT

    if ! $need_update
    then
        return 1
    fi

    need_restart WebLM && systemctl restart WebLM

    log "WebLm distribution process completed"

    return 0
}

distribute_server_cert_to_authmodule() {
    # TODO: Distribute certificate to Authentication Module if necessary.
    # For now, just start/restart it when necessary (ignition, network changes).
    log "Starting AuthModule distribution process"

    need_restart AuthModule && systemctl restart AuthModule&

    log "AuthModule process restarting continuing in the background"

    return 0
}

distribute_server_cert_to_callanalytics() {
    # For now, just start/restart it when necessary (ignition, network changes).
    log "Starting CallAnalytics distribution process"

    need_restart CallAnalytics && systemctl restart CallAnalytics &

    log "CallAnalytics process restarting continuing in the background"

    return 0
}

distribute_server_cert_to_webrtc() {
    # For now, just start/restart it when necessary (ignition, network changes).
    log "Starting WebRTC distribution process"

    need_restart webrtcgw && systemctl restart webrtcgw  &

    log "WebRTCGateway process restarting continuing in the background"

    return 0
}


distribute_server_cert_to_ipoffice() {
    # No need to copy the Certificate to IPOffice. Just restart the service.

    # Restart the ISA only when not having the auto behavior.
    # In case of auto, the server certificate is already in ipoffice's TCS
    # when this script is called
    if [ ! -f /opt/Avaya/certs/.auto ]
    then
        # Restart IpOffice Service
        need_restart isa && systemctl restart isa
    fi
    return 0
}

# Distribute the server certificate to Contact Recorder
distribute_server_cert_to_csipo() {
    log "Starting Csipo distribution process"

    local CERT=$1
    local OLD_CERT=`mktemp`
    local TMP_KEYSTORE=`mktemp`
    local TMP_PRIVATE_KEY=`mktemp`
    local STORE_PASS="Contact5tor3"
    local CSIPO_STORE="/opt/witness/keystore/keystore.jks"
    local ALIAS="tomcat"

    TMP_FILES="$TMP_FILES $OLD_CERT $TMP_KEYSTORE $TMP_PRIVATE_KEY"

    if [ ! -d /opt/witness/keystore/ ]
    then
        log "Warning: Could not find csipo directory /opt/witness/keystore"
        return 1
    fi

    if [ -f $CSIPO_STORE ]
    then
        # Export the csipo store
        rm -f $TMP_KEYSTORE
        keytool -importkeystore -srckeystore $CSIPO_STORE -destkeystore $TMP_KEYSTORE -deststoretype PKCS12 \
            -alias $ALIAS -srcstorepass $STORE_PASS -deststorepass $STORE_PASS -srckeypass $STORE_PASS -noprompt \
            || err_exit $ECERTNOTFOUND "Cannot Load CSIPO Keystore from $CSIPO_STORE"

        # TODO: Verify Chain
        # Export the current certificate
        openssl pkcs12 -in $TMP_KEYSTORE -nokeys -nodes -passin pass:$STORE_PASS | \
            sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' > $OLD_CERT

        # Export the current private key
        openssl pkcs12 -in $TMP_KEYSTORE -nocerts -nodes -passin pass:$STORE_PASS | \
            sed -n '/-----BEGIN PRIVATE KEY-----/,/-----END PRIVATE KEY-----/p' > $TMP_PRIVATE_KEY
        openssl rsa -in $TMP_PRIVATE_KEY >> $OLD_CERT

        if diff $OLD_CERT $CERT > /dev/null 
        then
            log "The csipo Server Certificate is already updated."
            return 1
        fi
    fi

    log "Copy Certificate to csipo"
    bash $SCRIPTS_PATH/keystore_csipo.sh
    chown witness:witness $CSIPO_STORE

    need_restart csipo && systemctl restart csipo

    log "Csipo distribution process completed"

    return 0
}

# Distribute the server certificate to Web Manager
distribute_server_cert_to_webmanager() {
    log "Starting WebManager distribution process"

    local CERT=$1
    local OLD_CERT=`mktemp`
    local TMP_NEW_CERT=`mktemp`
    local TMP_KEYSTORE=`mktemp`
    local TMP_PRIVATE_KEY=`mktemp`
    local TMP_FULL_CA_CHAIN=`mktemp`
    local STORE_PASS="changeit"
    local SMA_STORE="/opt/Avaya/apache-tomcat/conf/.keystore"
    local ALIAS="tomcat"

    TMP_FILES="$TMP_FILES $OLD_CERT $TMP_NEW_CERT $TMP_KEYSTORE $TMP_PRIVATE_KEY $TMP_FULL_CA_CHAIN"

    if [ ! -d /opt/Avaya/apache-tomcat/conf/ ]
    then
        log "Warning: Could not find web manager keystore directory /opt/apache-tomcat/conf"
        return 1
    fi

    cat $CERT | sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' > $TMP_NEW_CERT
    if $ENABLE_CA_CHAIN
    then
        if [ -f $CA_CHAIN ]
        then
            cat $CA_CHAIN | sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' >> $TMP_FULL_CA_CHAIN
        fi
    fi
    if [ -f $CA_CERT ]
    then
        cat $CA_CERT | sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' >> $TMP_FULL_CA_CHAIN
    fi
    cat $TMP_FULL_CA_CHAIN >> $TMP_NEW_CERT
    cat $CERT | sed -n '/-----BEGIN RSA PRIVATE KEY-----/,/-----END RSA PRIVATE KEY-----/p' >> $TMP_NEW_CERT

    if [ -f $SMA_STORE ]
    then
        # Export the current certificate
        openssl pkcs12 -in $SMA_STORE -nokeys -nodes -passin pass:$STORE_PASS | \
            sed -n '/-----BEGIN CERTIFICATE-----/,/-----END CERTIFICATE-----/p' > $OLD_CERT

        # Export the current private key
        openssl pkcs12 -in $SMA_STORE -nocerts -nodes -passin pass:$STORE_PASS | \
            sed -n '/-----BEGIN PRIVATE KEY-----/,/-----END PRIVATE KEY-----/p' > $TMP_PRIVATE_KEY
        openssl rsa -in $TMP_PRIVATE_KEY >> $OLD_CERT

        if diff $OLD_CERT $TMP_NEW_CERT > /dev/null 
        then
            log "The web manager Server Certificate is already updated."
            return 1
        fi
    fi

    log "Copy Certificate to web manager"
    # IPOFFICE-141844 - Web Manager restarts after applying a patch for one of the Server Edition applications
    if [ -f $CA_CERT ]
    then
        openssl pkcs12 -export -in $CERT -inkey $CERT -out $SMA_STORE -name tomcat \
        -certfile $TMP_FULL_CA_CHAIN -caname root -passout pass:$STORE_PASS
    else
        openssl pkcs12 -export -in $CERT -inkey $CERT -out $SMA_STORE -name tomcat \
        -caname root -passout pass:$STORE_PASS
    fi

    need_restart apache-tomcat && systemctl restart apache-tomcat
    
    need_restart WebManager && systemctl restart WebManager

    log "WebManager distribution process completed"
    
    return 0
}

# Distribute the server certificate to all Avaya Apps
distribute_server_cert() {
(
    log "Requesting lock for distributing the certificates"
    if flock -x -w 1200 200; then
        log "Locking gen_certs script"
        sudo rm -f $DISTRIBUTION_DONE_FLAG
        sudo touch $DISTRIBUTION_INPROGRESS_FLAG
        sudo chmod 777 $DISTRIBUTION_INPROGRESS_FLAG
        local MARK_DISTRIB_COMPLETE=false
        local CERT=$CERT_DIR/server.pem
        local CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        local STOP_DISTRIBUTING=false
        if [ ! -s $CERT ]
        then
            flock -u 200
            sudo rm -f $DISTRIBUTION_INPROGRESS_FLAG
            err_exit $ECERTNOTFOUND "The server certificate $CERT contains no valid data"
        fi

        # Make sure the certificate is correct
        ERROR_DETECTED=false
        openssl x509 -text -noout -in $CERT > /dev/null \
            || ERROR_DETECTED=true
        if $ERROR_DETECTED
        then
            flock -u 200
            sudo rm -f $DISTRIBUTION_INPROGRESS_FLAG
            err_exit $ECERTNOTFOUND "Invalid Server Certificate $CERT"
        fi

        openssl rsa -text -noout -in $CERT > /dev/null \
            || ERROR_DETECTED=true
        if $ERROR_DETECTED
        then
            flock -u 200
            sudo rm -f $DISTRIBUTION_INPROGRESS_FLAG
            err_exit $EKEYNOTFOUND "Invalid Server Certificate $CERT"
        fi

        if $ENABLE_CA_CHAIN
        then
            if [ -f $CA_CHAIN ]
            then
                sudo chmod 444 $CA_CHAIN
            fi
        fi

        local need_update=false
        distribute_server_cert_to_webcontrol $CERT && need_update=true
        if $need_update || $ENABLE_FORCE_RESTART_MODE
        then
            MARK_DISTRIB_COMPLETE=true
        fi
        C_CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        if [ "$CERT_MD5SUM" != "$C_CERT_MD5SUM" ]; then
            STOP_DISTRIBUTING=true
            log "Certificate changed, will stop distributing"
        fi
        if $ENABLE_CA_CHAIN; then
            if [ ! -f $CA_CHAIN ]; then
                STOP_DISTRIBUTING=true
                log "Chain file missing, will stop distributing"
            fi
        fi
        if $ONEXPORTAL_PRESENT
        then
            if ! $STOP_DISTRIBUTING; then
                distribute_server_cert_to_onexportal $CERT && need_update=true
            fi
        fi
        C_CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        if [ "$CERT_MD5SUM" != "$C_CERT_MD5SUM" ]; then
            STOP_DISTRIBUTING=true
            log "Certificate changed, will stop distributing"
        fi
        if $ENABLE_CA_CHAIN; then
            if [ ! -f $CA_CHAIN ]; then
                STOP_DISTRIBUTING=true
                log "Chain file missing, will stop distributing"
            fi
        fi
        if $VMPRO_PRESENT
        then
            if ! $STOP_DISTRIBUTING; then
                distribute_server_cert_to_vmpro $CERT && need_update=true
            fi
        fi
        C_CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        if [ "$CERT_MD5SUM" != "$C_CERT_MD5SUM" ]; then
            STOP_DISTRIBUTING=true
            log "Certificate changed, will stop distributing"
        fi
        if $ENABLE_CA_CHAIN; then
            if [ ! -f $CA_CHAIN ]; then
                STOP_DISTRIBUTING=true
                log "Chain file missing, will stop distributing"
            fi
        fi
        if ! $STOP_DISTRIBUTING; then
            distribute_server_cert_to_weblm $CERT && need_update=true
        fi
        C_CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        if [ "$CERT_MD5SUM" != "$C_CERT_MD5SUM" ]; then
            STOP_DISTRIBUTING=true
            log "Certificate changed, will stop distributing"
        fi
        if $ENABLE_CA_CHAIN; then
            if [ ! -f $CA_CHAIN ]; then
                STOP_DISTRIBUTING=true
                log "Chain file missing, will stop distributing"
            fi
        fi
        if ! $STOP_DISTRIBUTING; then
            distribute_server_cert_to_csipo $CERT && need_update=true
        fi
        C_CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        if [ "$CERT_MD5SUM" != "$C_CERT_MD5SUM" ]; then
            STOP_DISTRIBUTING=true
            log "Certificate changed, will stop distributing"
        fi
        if $ENABLE_CA_CHAIN; then
            if [ ! -f $CA_CHAIN ]; then
                STOP_DISTRIBUTING=true
                log "Chain file missing, will stop distributing"
            fi
        fi
        if ! $STOP_DISTRIBUTING; then
            distribute_server_cert_to_webmanager $CERT && need_update=true
        fi
        C_CERT_MD5SUM=`md5sum $CERT | cut -d" " -f1`
        if [ "$CERT_MD5SUM" != "$C_CERT_MD5SUM" ]; then
            STOP_DISTRIBUTING=true
            log "Certificate changed, will stop distributing"
        fi
        if $ENABLE_CA_CHAIN; then
            if [ ! -f $CA_CHAIN ]; then
                STOP_DISTRIBUTING=true
                log "Chain file missing, will stop distributing"
            fi
        fi
        if ! $STOP_DISTRIBUTING; then
            distribute_server_cert_to_authmodule $CERT && need_update=true
        fi
        if ! $STOP_DISTRIBUTING; then
            distribute_server_cert_to_callanalytics $CERT && need_update=true
        fi
        if ! $STOP_DISTRIBUTING; then
            distribute_server_cert_to_webrtc $CERT && need_update=true
        fi
        # distribute_server_cert_to_ipoffice $CERT && need_update=true

        if ! $need_update
        then
            log "The Server Certificate is already updated."
            # the httpd service is restarted from ipoffice
            # it is configured to use directly the certificate file
            # that ipoffice exports
        fi

        sudo rm -f $DISTRIBUTION_INPROGRESS_FLAG

        if $MARK_DISTRIB_COMPLETE
        then
            sudo touch $DISTRIBUTION_DONE_FLAG
            sudo chmod 777 $DISTRIBUTION_DONE_FLAG
        fi
        
        rm -f $TMP_FILES
        rm -f /opt/Avaya/certs/.upgrade
        
        log "Unlocking gen_certs script"

        flock -u 200
    else
        log "Error: could not acquire lock for distributing the certificates"
    fi
) 200>$LOCKFILE
}

print_variables() {
    echo "Subject: $ISSUED_TO"
    echo "Subject-alt-name: $SUBJECT_ALT_NAME"
    echo "Validity: $CERT_VALIDITY"
    echo "RSA Key Length: $RSA_KEY_LEN"
    echo "SHA Version: $SHA_VER"
}

print_ca_info() {
    if [ -f "$CA_CERT" ]; then
        OPENSSL_CA_INFO=`openssl x509 -in $CA_CERT -text -noout`
        ISSUEDTO_INFO=`echo $OPENSSL_CA_INFO| grep "Issuer:"`
        ISSUE_INFO=${ISSUEDTO_INFO##*CN=}
        CN=`echo $ISSUE_INFO|cut -d"/" -f1`
        ORG=${OPENSSL_CA_INFO##*O=}
        ORG=`echo $ORG|cut -d"," -f1`
        ORGUNIT=${OPENSSL_CA_INFO##*OU=}
        ORGUNIT=`echo $ORGUNIT|cut -d"," -f1`
        ROOTCA_INFO=(`echo $OPENSSL_CA_INFO`)
        ISSUED_TO="IssuedTo:CN=$CN,O=$ORG,OU=$ORGUNIT,SerialNumber=${ROOTCA_INFO[7]}"
        #echo "$ISSUED_TO"
        ISSUEDBY=`echo $OPENSSL_CA_INFO|grep "Subject:"`
        ISSUER=${ISSUEDBY##*CN=}
        CN=`echo $ISSUER|cut -d"/" -f1`
        ORG=${OPENSSL_CA_INFO##*O=}
        ORG=`echo $ORG|cut -d"," -f1`
        ORGUNIT=${OPENSSL_CA_INFO##*OU=}
        ORGUNIT=`echo $ORGUNIT|cut -d"," -f1`
        ISSUED_BY="IssuedBy:CN=$CN,O=$ORG,OU=$ORGUNIT"
        echo -e "$ISSUED_TO\n$ISSUED_BY"
    fi
}

log() {
    if [ -n $LOG_FILE ]
    then
        echo "[$(date)] $1" >> $LOG_FILE
    fi
    # echo "[$(date)] $1"
}

init_error_codes() {
    ESUCCESS=0          # Success
    EFAIL=1             # General Error
    EINVAL=2            # Invalid argument
    EACCESS=3           # Access error
    EGENRSA=4           # Cannot generate RSA Key
    EGENCERT=5          # Cannot cenerate Certificate/Certificate request
    ESIGNCERT=6         # Cannot Sign the certificate
    ECERTNOTFOUND=7     # Cannot find or load the certificate
    EKEYNOTFOUND=8      # Cannot find or load the RSA Key
    ENOTMATCH=9         # Non Matching Certificate-Private Key
    ENOPASS=10          # No password specified
    EPKCS12CERT=11      # Cannot generate pkcs12 cert
    ENOCERT=12          # No certificate found
    EP12DECRYPT=13      # Error decrypting pkcs12 certificate
    ELOCK=14            # Error aquiring lock
}

err_exit() {
    log "$2"
    echo $2
    exit $1
}

cleanup() {
    rc=$?

    # Cleanup temp files
    sudo rm -f $DISTRIBUTION_INPROGRESS_FLAG
    sudo rm -f $TMP_FILES
    exit $rc
}

# Main

init_error_codes

# Execute the script as root
if [ "$UID" -ne 0 ]
then
    err_exit $EACCESS "Please run as root"
fi

# Change umask so that the Certificates and Private Keys cannot be read by
# other users.
umask 077

# Toggle -e to exit the script when an error occurs
set -e

# Redirect stderr to stdout so that webcontrol can analyze all the output
exec 2>&1

# Set the cleanup function
trap cleanup EXIT
trap "err_exit $EFAIL \"Script Interrupted\"" SIGHUP SIGINT SIGQUIT SIGTERM

process_args "$@"

import_isa_data

if $GENERATE_CA_CERT
then
    log "Generate CA Certificate"
    init_ca_variables
    generate_ca_cert
elif $GENERATE_SRV_CERT
then
    log "Generate SRV Certificate"
    init_server_variables
    generate_server_cert
elif $GENERATE_CSR
then
    log "Generate CSR"
    init_server_variables stamp
    generate_csr
elif $EXPORT_CA
then
    log "Export CA PKCS12"
    export_ca
elif $IMPORT_CA
then
    log "Import CA PKCS12"
    import_ca
elif $DISTRIBUTE_SRV_CERT
then
    log "Distribute Server Certificate"
    init_server_variables
    distribute_server_cert
elif $PRINT_DEFAULT_CA_VARS
then
    init_ca_variables
    print_variables
elif $PRINT_DEFAULT_SERVER_VARS
then
    init_server_variables
    print_variables
elif $PRINT_CA_INFO
then
    init_server_variables
    print_ca_info
elif $GENERATE_CA_CHAIN
then
    init_chain_variables
    generate_ca_chain
elif $INSTALL_P12CERTIFICATE
then
    log "Installing p12 Certificate"
    init_server_variables
    install_p12_certificate
elif $CONVERT_TODER_FORMAT
then
    log "Converting to der format"
    init_server_variables
    convert_to_der
elif $IMPORTCA_TO_IPOTCS
then
    log "Importing CA to IPOffice TCS"
    add_ca_to_ipo_tcs restored
else
    usage
    err_exit $EINVAL
fi

