#!/bin/bash

# Restore node specific configuraton and application databases from either archive or directory
# tree under local folder or archive on Backup Server. A system reboot will be performed upon
# successful restore in order to make sure that all modified system settings will be applied.
#
# When restoring from local folder, having a directory tree rather than an archive at
# the specified location is controlled by providing the "-d" command line option

# Usage:
# ipol_restore.sh [-d] <backup URL>

#set -x

BASEDIR=$(dirname $0)
. $BASEDIR/ipol_common.sh

echo $$ > /sys/fs/cgroup/systemd/tasks

while getopts :d:s opt; do
  case $opt in
    d)
      wantDirectoryTree=1
      ;;
    s)
      #1 stop isa service
      #2 unpack backup archive
      #3 cp ipoffice backed files to primary
      #4 reboot the system
      systemRestore=1
      archiveName="Linux_Backup.tar.gz"
      ;;
    \?)
      echo "Invalid option: -$OPTARG" >&2
      ;;
  esac
done

# Done with command line options, discarding them
shift $((OPTIND-1))

#Copy archive file from remote server using scp
#arg1 - Backup server username account
#arg2 - Backup server password account
#arg3 - Backup server IP Address
#arg4 - Backup server absolut path
function copy_tarballFrom () {
        
    if [ -f "$br_path/$md5sumfile" ]; then
        rm -rf $br_path/$md5sumfile
    fi
    if [ -f "$br_path/$rStatusFile" ]; then
        rm -rf $br_path/$rStatusFile
    fi
    
    here="."
    
    if [ "$urlPROTO" == "sftp" -o "$urlPROTO" == "scp" ]; then
    	#get files from backup server
        python $BASEDIR/ipol_copy.py  $1 $2 $3 $4/$archiveName $here $operation &>$br_path/$rmessageRcv
        if (($? != 0)); then
                eventrcvd=`head -n 1 $br_path/$rmessageRcv`
                br_LogEvent "$brinfoLog" "Error-Restore: Failed due to copying file from server $3 {$eventrcvd}." "$br_path/$restoreLog"
                br_LogEvent "$brinfoStatus" "Restore failed. Failed due to copying file from server $3 {$eventrcvd}." "$br_path/$rStatusFile"
                if [ -f $br_path/$rmessageRcv ]; then
                      rm -rf $br_path/$rmessageRcv
                fi
                remove_lock $operation
                exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
        else
                check_KeyVerificationFailed $br_path/$rmessageRcv $br_path/$restoreLog
        fi

        python $BASEDIR/ipol_copy.py  $1 $2 $3 $4/$md5sumfile $here $operation &>$br_path/$rmessageRcv
        if (($? != 0)); then
                eventrcvd=`head -n 1 $br_path/$rmessageRcv`
                br_LogEvent "$brinfoLog" "Error-Restore: Failed due to copying file from server $3 {$eventrcvd}." "$br_path/$restoreLog"
                br_LogEvent "$brinfoStatus" "Restore failed. Failed due to copying file from server $3 {$eventrcvd}." "$br_path/$rStatusFile"
                if [ -f $br_path/$rmessageRcv ]; then
                      rm -rf $br_path/$rmessageRcv
                fi
                remove_lock $operation
                exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
        else
                check_KeyVerificationFailed $br_path/$rmessageRcv $br_path/$restoreLog
        fi
    elif [ "$urlPROTO" == "http" -o "$urlPROTO" == "https" ]; then
        if [ -z "$urlPORT" ]; then
                #default port on primary-secondary ipol
                uport=":8000"
        else
                uport=":$urlPORT"
        fi
        #get files from backup server
        wget $urlPROTO://$urlIPADDRESS$uport/$urlPATH/{$md5sumfile,$archiveName}
        if (($? != 0)); then
                br_LogEvent "$brinfoLog" "Error-Restore: Failed to get files from Server $3." "$br_path/$restoreLog"
                br_LogEvent "$brinfoStatus" "Restore failed. Failed to get files from Server $3." "$br_path/$rStatusFile"
                remove_lock $operation
                exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
	fi
    else
        br_LogEvent "$brinfoLog" "Error-Restore: Protocol $urlPROTO not supported." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Protocol $urlPROTO not supported." "$br_path/$rStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_URL_PROTOCOL_NOTSUPPORTED
    fi

    if [ ! -f $md5sumfile -o ! -f $archiveName ]; then
        br_LogEvent "$brinfoLog" "Error-Restore: Get files from Server $3 failed." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Get files from Server $3 failed." "$br_path/$rStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
    fi
    
    #verify that md5 matches
    cmd5="`md5sum $archiveName | awk '{ print $1}'`"
    md5file="`cat $md5sumfile`"
    if [ "$cmd5" == "$md5file" ]; then
        mv -f $md5sumfile $br_path/$md5sumfile
        mv -f $archiveName $br_path/$archiveName
    else
        br_LogEvent "$brinfoLog" "Error-Restore: md5 $archiveName mismatched." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. md5 $archiveName mismatched." "$br_path/$rStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_MD5SUM_MISMATCH
    fi
}

# Merge environment variables from input to output file
# No commented texts merged with these variables
# arg1 input file
# arg2 output file
function mergeIO () {

    if [ -n "$1" -a -n "$2" ]; then
         cat "$1" | grep -E "export .*=.*" > $tmp_backuprestore_folder/in_ipoffice_sysconfig
         while read line
         do
             if [ -n "$line" ]
             then
                 varenv="${line%=*}"
                 # search variable into output file. If does not exist, it will be added
                 env_searched=`cat "$2" | grep -E "$varenv"`
                 if [ -z "$env_searched" ]; then
                      echo "" >> "$2"
                      echo "# Added due to restore operation" >> "$2"
                      echo "$line" >> "$2"
                      echo "" >> "$2" 
                 fi
             fi
         done < $tmp_backuprestore_folder/in_ipoffice_sysconfig
    fi
}

#Main execution
operation="Restore"

#Create lock file for restorep operation
#Only one instance at a time is permited
if [ ! -f "$br_path/ipol_backup_restore.lock" ]; then
        touch $br_path/ipol_backup_restore.lock
else
        br_LogEvent "$brinfoLog" "Error-Restore: Only one backup/restore instance is allowed.Exit." "$br_path/$restoreLog"
        exit $BR_ERRORCODE_LOCK_TAKEN
fi

if [ -f "$br_path/$rStatusFile" ]; then
        rm -rf $br_path/$rStatusFile
fi

if [ $systemRestore == 1 ]; then
        rURL="file:///$br_systemLocation"
else
        rURL=$1
        if [ -z "$rURL" ]; then
            br_LogEvent "$brinfoLog" "Error-Restore: Empty url found." "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore failed. Empty url found." "$br_path/$rStatusFile"
            remove_lock $operation
            exit $BR_ERRORCODE_URL_EMPTY
        fi

        USERNAME=$2
        USERPASSWORD=$3
fi
#get url protocol|url server|url absolut path
parse_url  $operation $rURL

if [ -f $br_path/parse.ini ]; then
	. $br_path/parse.ini
	rm -f $br_path/parse.ini
else
        br_LogEvent "$brinfoLog" "Error-Restore: Parse url failed." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Parse url failed." "$br_path/$rStatusFile"
	remove_lock $operation
	exit $BR_ERRORCODE_URL_PARSE
fi

if [ -z "$urlPROTO" ]; then
	remove_lock $operation
	exit $BR_ERRORCODE_URL_PROTOCOL_EMPTY
fi

#change url protocol to lowercase 
urlPROTO=`echo $urlPROTO | tr A-Z a-z`
case "$urlPROTO" in
    "scp" | "sftp")
    if [ -z "$USERNAME" -o -z "$USERPASSWORD" ]; then
        br_LogEvent "$brinfoLog" "Error-Restore: No credentials found in arguments." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. No credentials found in arguments." "$br_path/$rStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_CREDENTIALS_MISMATCH
    fi

    if [ ! -z "$urlIPADDRESS" ]; then
        #verify conectivity with backup server
        ping -c 1 $urlIPADDRESS
        if (($? != 0)); then
            br_LogEvent "$brinfoLog" "WARNING-Restore: Conectivity with {$urlIPADDRESS} server failed!" "$br_path/$restoreLog"
        fi

        #get backup archive from remote server
        copy_tarballFrom $USERNAME $USERPASSWORD $urlIPADDRESS $urlPATH
        if (($? == 0)); then
            br_LogEvent "$brinfoLog" "Info-Restore: Got files from {$urlIPADDRESS}." "$br_path/$restoreLog"
        fi

        #unpack backup archive
        tar -z $tarExtractOptions $br_path/$archiveName
        if (($? != 0)); then
            br_LogEvent "$brinfoLog" "Error-Restore: Unpack failed!" "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore failed. Unpack failed!" "$br_path/$rStatusFile"
        else
            #execute vmpro restore utility
            if [ -f $rvmproUtility -a -f $bvmproScript ]; then
                br_vmproLocationDepthOne=`ls -l $br_vmproLocation | grep '^d' | awk '{print $9}'`
                if [ -n $br_vmproLocationDepthOne -a -d $br_vmproLocation/$br_vmproLocationDepthOne ]; then
                    $rvmproUtility -Restore:$br_vmproLocation/$br_vmproLocationDepthOne
                    br_LogEvent "$brinfoLog" "Info-VMPro_Restore: Check VMPro logs to find out the vmpro restore state." "$br_path/$restoreLog"
                else
                    br_LogEvent "$brinfoLog" "Info-VMPro_Restore: VMPro may failed." "$br_path/$restoreLog"
                fi
            else
                br_LogEvent "$brinfoLog" "Info-VMPro_Restore: VMPro may failed." "$br_path/$restoreLog"
            fi
            br_LogEvent "$brinfoLog" "Info-Restore: Finished operation.Succeeded." "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore succeeded. Finished operation." "$br_path/$rStatusFile"
            restartRequired=1
        fi
    else
        br_LogEvent "$brinfoLog" "Error-Restore: Empty urlIPADDRESS found." "$br_path/$restorLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Empty urlIPADDRESS found." "$br_path/$rStatusFile"
    fi
    ;;
    "file")
    if [ ! -d $urlPATH ]; then
        br_LogEvent "$brinfoLog" "Error-Restore: Folder {$urlPATH} does not exist!" "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Folder {$urlPATH} does not exist!" "$br_path/$rStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_FOLDER_DOESNOTEXIST
    fi

    if [ $wantDirectoryTree == 0 -o $systemRestore == 1 ]; then
        # Verify that archive file is present in $urlPATH
        if [ ! -f $urlPATH/$archiveName ]; then
            br_LogEvent "$brinfoLog" "Error-Restore: No archive file at {$urlPATH} location." "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore failed. No archive file at {$urlPATH} location." "$br_path/$rStatusFile"
            remove_lock $operation
            exit $BR_ERRORCODE_FILE_DOESNOTEXIST
        fi

        if [ $systemRestore == 1 ]; then
            mkdir -p $ipoffice_backupLocation
            if [ ! -d "$ipoffice_backupLocation" ]; then
                 br_LogEvent "$brinfoLog" "Error-Restore: Folder $ipoffice_backupLocation does not exist" "$br_path/$restoreLog"
                 br_LogEvent "$brinfoStatus" "Restore failed. Folder "$ipoffice_backupLocation" does not exist." "$br_path/$rStatusFile"
                 remove_lock $operation
                 exit $BR_ERRORCODE_FOLDER_DOESNOTEXIST
            elif [ ! -d "$ipoffice_primaryLocation" ]; then
                 br_LogEvent "$brinfoLog" "Error-Restore: Folder $ipoffice_primaryLocation does not exist" "$br_path/$restoreLog"
                 br_LogEvent "$brinfoStatus" "Restore failed. Folder $ipoffice_primaryLocation does not exist." "$br_path/$rStatusFile"
                 remove_lock $operation
                 exit $BR_ERRORCODE_FOLDER_DOESNOTEXIST
            fi
            /usr/lib/ipoffice/ipoffice_reset.sh stop maxnoresponsetime
        fi

        # Unpack backup archive
        temp_unpacked_location=`mktemp -d`
        if [ ! -d "$temp_unpacked_location" ]; then
             br_LogEvent "$brinfoLog" "Error-Restore: Cannot create temp folder." "$br_path/$restoreLog"
             br_LogEvent "$brinfoStatus" "Restore failed. Cannot create temp folder." "$br_path/$rStatusFile"
             remove_lock $operation
             exit $BR_ERRORCODE_FOLDER_FAILEDTOCREATE
        fi
        additionalExtractOptionsWithPatch="-C $temp_unpacked_location -xvf"
        tar -z $tarExtractOptionsNoPathDefined $additionalExtractOptionsWithPatch $urlPATH/$archiveName
        if (($? != 0)); then
            br_LogEvent "$brinfoLog" "Error-Restore: Unpack failed." "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore failed. Unpack failed." "$br_path/$rStatusFile"
            remove_lock $operation
            exit $BR_ERRORCODE_ARCHIVE_FAILEDTOUNPACK
        else
            br_LogEvent "$brinfoLog" "Restore: Unpack succeeded." "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore: Unpack succeeded." "$br_path/$rStatusFile"
            if [ $systemRestore == 1 ]; then
                 #before applying backed archive perform some verifications in sysconfig ipoffice
                 # 1) Server type should be the same
                 # 2) At least one LAN interface should have a valide value(not empty value)
                 env_current=`cat /etc/sysconfig/ipoffice | grep "export IPOFFICE_BE_MODE" | cut -d'=' -f2`
                 env_backed=`cat $temp_unpacked_location/etc/sysconfig/ipoffice | grep "export IPOFFICE_BE_MODE" | cut -d'=' -f2`
                 if [ "${env_current//\"}" != "${env_backed//\"}" ]; then
                      br_LogEvent "$brinfoLog" "Error-Restore: Could not apply this archive. Different server type found." "$br_path/$restoreLog"
                      br_LogEvent "$brinfoStatus" "Restore failed. Could not apply this archive. Different server type found." "$br_path/$rStatusFile"
                      rm -rf $temp_unpacked_location
                      remove_lock $operation
                      exit $BR_ERRORCODE_ARCHIVE_CANNOTAPPLY
                 fi

                 env_current=`cat $temp_unpacked_location/etc/sysconfig/ipoffice | grep "export IPOFFICE_LAN1=" | cut -d'=' -f2`
                 env_backed=`cat $temp_unpacked_location/etc/sysconfig/ipoffice | grep "export IPOFFICE_LAN2=" | cut -d'=' -f2`
                 if [ -z "$env_current" -a -z "$env_backed" ]; then
                      br_LogEvent "$brinfoLog" "Error-Restore: Could not apply this archive. No valid interface found." "$br_path/$restoreLog"
                      br_LogEvent "$brinfoStatus" "Restore failed. Could not apply this archive. No valid interface found." "$br_path/$rStatusFile"
                      rm -rf $temp_unpacked_location
                      remove_lock $operation
                      exit $BR_ERRORCODE_ARCHIVE_CANNOTAPPLY
                 fi
                 
                 # Create backup file of current sysconfig ipoffice file(in case the backup archive contains a different one)
                 if [ ! -d "$tmp_backuprestore_folder" ]; then
                       mkdir -p "$tmp_backuprestore_folder"
                 fi

                 env_current=`cat /etc/sysconfig/ipoffice | grep "export IPOFFICE_FULL_VERSION"`
                 env_backed=`cat $temp_unpacked_location/etc/sysconfig/ipoffice | grep "export IPOFFICE_FULL_VERSION"`
                 if [ "$env_current" != "$env_backed" ]; then
                      # 1. update backed version with the current one (avoid upgrade issues)
                        #    search end replace escape characters
                        env_current="${env_current//\\/\\\\}"
                        env_current="${env_current//\'/\\\'}"
                        env_current="${env_current//\"/\\\"}"
                        env_current="${env_current//\//\/}"
                        envReplaced="${env_current%=*}"
                        # replace backed value with the current one
                        perl -pi -e "s/${envReplaced}.*/$env_current/" $temp_unpacked_location/etc/sysconfig/ipoffice
                      # 2. merge that fields from current version to backed one that are not present in backed file (ipoffice service is runnig with the current one)
                        mergeIO "/etc/sysconfig/ipoffice" "$temp_unpacked_location/etc/sysconfig/ipoffice"
                 fi

                 time_stamp=`date +"%a%d%b_%H%M%S_%Y"`
                 cp "/etc/sysconfig/ipoffice" "$tmp_backuprestore_folder/sysconfig_ipoffice_$time_stamp"

                 # restore backed archive
                 cp -Rf $temp_unpacked_location/. $root_path 
                 if (($? != 0)); then
                     br_LogEvent "$brinfoLog" "Error-Restore: Failed to copy configs backed files." "$br_path/$restoreLog"
                     br_LogEvent "$brinfoStatus" "Restore failed. Failed to copy configs backed files." "$br_path/$rStatusFile"
                     remove_lock $operation
                     rm -rf $temp_unpacked_location
                     exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
                 fi

                 cp -rf $ipoffice_backupLocation/. $ipoffice_primaryLocation
                 if (($? != 0)); then
                     br_LogEvent "$brinfoLog" "Error-Restore: Failed to copy ipoffice backed files." "$br_path/$restoreLog"
                     br_LogEvent "$brinfoStatus" "Restore failed. Failed to copy ipoffice backed files." "$br_path/$rStatusFile"
                     remove_lock $operation
                     exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
                 else
                     br_LogEvent "$brinfoLog" "Restore: Copy ipoffice files succeeded." "$br_path/$restoreLog"
                     br_LogEvent "$brinfoStatus" "Restore: Copy ipoffice files succeeded." "$br_path/$rStatusFile"
                     if [ -f $restore_status ]; then
                        rm -rf $restore_status
                     fi
                     echo "0" > $restore_status
                 fi
                 rm -rf $temp_unpacked_location
            fi
        fi
        
    else
        # Verify that $urlPATH specifies a directory
        if [ ! -d $urlPATH ]; then
            br_LogEvent "$brinfoLog" "Error-Restore: {$urlPATH} is not a directory." "$br_path/$restoreLog"
            br_LogEvent "$brinfoStatus" "Restore failed. {$urlPATH} is not a directory." "$br_path/$rStatusFile"
            remove_lock $operation
            exit $BR_ERRORCODE_FOLDER_DOESNOTEXIST
        fi

        # Restore files from specified directory
        tar -C $urlPATH -cf - . | tar $tarExtractOptions -
    fi

    if (($? != 0)); then
        br_LogEvent "$brinfoLog" "Error-Restore: Unpack failed!" "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Unpack failed!" "$br_path/$rStatusFile"
    else
        if [ $systemRestore == 0 ]; then
            #execute vmpro restore utility
            if [ -f $rvmproUtility -a -f $bvmproScript ]; then
                br_vmproLocationDepthOne=`ls -l $br_vmproLocation | grep '^d' | awk '{print $9}'`
                if [ -n "$br_vmproLocationDepthOne" -a -d $br_vmproLocation/$br_vmproLocationDepthOne ]; then
                    $rvmproUtility -Restore:$br_vmproLocation/$br_vmproLocationDepthOne
                    br_LogEvent "$brinfoLog" "Info-VMPro_Restore: Check VMPro logs to find out the vmpro restore state." "$br_path/$restoreLog"
                else
                    br_LogEvent "$brinfoLog" "Info-VMPro_Restore: VMPro may failed." "$br_path/$restoreLog"
                fi
            else
                br_LogEvent "$brinfoLog" "Info-VMPro_Restore: VMPro may failed." "$br_path/$restoreLog"
            fi
        fi
        br_LogEvent "$brinfoLog" "Info-Restore: Finished operation. Succeeded." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore succeeded. Finished operation." "$br_path/$rStatusFile"
        restartRequired=1
    fi
    ;;
    "http" | "https")
    #get archive file from remote server
    copy_tarballFrom $USERNAME $USERPASSWORD $urlIPADDRESS $urlPATH
    #unpack backup archive
    tar -z $tarExtractOptions $br_path/$archiveName
    if (($? != 0)); then
        br_LogEvent "$brinfoLog" "Error-Restore: Unpack failed!" "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore failed. Unpack failed!" "$br_path/$rStatusFile"
    else
        br_LogEvent "$brinfoLog" "Info-Restore: Finished operation. Succedeed." "$br_path/$restoreLog"
        br_LogEvent "$brinfoStatus" "Restore succeeded. Finished operation." "$br_path/$rStatusFile"
        restartRequired=1
    fi
    ;;
    *)
    br_LogEvent "$brinfoLog" "Warning-Restore: Failed. Protocol {$urlPROTO} not supported." "$br_path/$restoreLog"
    br_LogEvent "$brinfoStatus" "Restore failed. Protocol {$urlPROTO} not supported." "$br_path/$rStatusFile"
    ;;
esac
remove_lock $operation

if [ $restartRequired == 1 ]; then
    shutdown -r now "Restarting the system to apply restored configuration data"
fi
