#/bin/bash
# usage: ./system_status_script [<sec>] [<cleanup>]
# purpose: log system information in /var/log/system_status_log at <sec> interval 
#          and keep the logs from the last <cleanup> days

sec=1 # frequency to check system status - default 1 sec
float_scale=2 
aux_file="/tmp/.system_status_log"
path="/var/log/system_status_log/"
block_size=1024
cleanup_period=7 # days

function float_eval()
{
	local stat=0
	local result=0.0
	
	if [[ $# -gt 0 ]]; then
		result=$(echo "scale=$float_scale; $*" | bc -q 2>/dev/null)
		stat=$?
		if [[ $stat -eq 0  &&  -z "$result" ]]; then stat=1; fi
	fi
	
	echo $result
	return $stat
}

function float_cond()
{
    local cond=0
    if [[ $# -gt 0 ]]; then
        cond=$(echo "$*" | bc -q 2>/dev/null)
        if [[ -z "$cond" ]]; then cond=0; fi
        if [[ "$cond" != 0  &&  "$cond" != 1 ]]; then cond=0; fi
    fi
    local stat=$((cond == 0))
    return $stat
}


function bytestohumanreadable(){
    multiplier=0
    number=$1
    
    while float_cond "$number > 1024"  ; do
        multiplier=$(($multiplier+1))
        number=$(float_eval "$number / 1024")
    done

    case "$multiplier" in
        1)
            result="$number Kb"
            ;;
        2)
            result="$number Mb"
        ;;
        3)
            result="$number Gb"
        ;;
        4)
            result="$number Tb"
        ;;
         *)
        result="$1 B"
        ;;
      esac

    echo "$result"
}

function cpu_load()
{
    nr_cores=`lscpu | grep "CPU(s)" | head -1 | awk -F ":" '{print $2}' | tr -s ' '`
    speed=`cat /proc/cpuinfo | grep "model name" | awk -F "@" '{print $2}' | sed -n 1p`

    command=$1
    out_file=$2

    idle=`echo $command | cut -d ' ' -f15`
    iowait=`echo $command | cut -d ' ' -f16`

    echo `date` >> $out_file
    echo "CPU(S): $nr_cores @$speed" >> $out_file

    cpu_load=$(float_eval "100 - $idle")
    echo "cpu: $cpu_load% iowait: $iowait%" >> $out_file
}

function mem_load()
{
    out_file=$1
    eval free > $aux_file
	
    total_mem=`cat $aux_file | sed -n 2p | tr -s ' '| cut -d ' ' -f2`
    used_mem=`cat $aux_file | sed -n 3p | tr -s ' '| cut -d ' ' -f3`

    total_swap=`cat $aux_file | sed -n 4p | tr -s ' ' | cut -d ' ' -f 2`
    used_swap=`cat $aux_file | sed -n 4p | tr -s ' ' | cut -d ' ' -f 3`

    used_mem_percent=$(float_eval "$used_mem*100/$total_mem")
    used_swap_percent=$(float_eval "$used_swap*100/$total_swap")

    total_mem=`bytestohumanreadable $((total_mem * 1024))`
    total_swap=`bytestohumanreadable $((total_swap * 1024))`

    echo "Mem: $total_mem total, $used_mem_percent% used" >> $output_file
    echo "Swap: $total_swap total, $used_swap_percent% used" >> $output_file
}

function dsk_load()
{
    command=$1
    out_file=$2

    reads=`echo $command | cut -d ' ' -f9` # this is in blocks
    writes=`echo $command | cut -d ' ' -f10` # this is in blocks

    reads=`bytestohumanreadable $(($reads * $block_size))`
    writes=`bytestohumanreadable $(($writes * $block_size))`

    echo "Dsk: $reads r/s  $writes w/s" >> $out_file
}

function nic_load()
{
    nr_int="$1" #number of eth interfaces
    oldrx_vec="$2"
    oldtx_vec="$3"
    rx_vec="$4"
    tx_vec="$5"
    out_file=$6
    
    for i in $(seq 0 $(($nr_int - 1)))
    do
        old_rx=`echo $oldrx_vec | cut -d ' ' -f$(($i+1))`
        old_tx=`echo $oldtx_vec | cut -d ' ' -f$(($i+1))`
        
        rx=`echo $rx_vec | cut -d ' ' -f$(($i+1))`
        tx=`echo $tx_vec | cut -d ' ' -f$(($i+1))`

        rx=`bytestohumanreadable $(($rx - $old_rx))` 
        tx=`bytestohumanreadable $(($tx - $old_tx))` 

        echo "NET eth$i $rx RXbytes $tx TXbytes " >> $out_file
    done

}

oldrx_vec=""
oldtx_vec=""
rx_vec=""
tx_vec=""
function prepare_net()
{
    if [ "$rx_vec" ]; then
        oldrx_vec=$rx_vec
        oldtx_vec=$tx_vec
    fi

    nr_int=`ifconfig | grep -v 'eth[0-9][0-9]*\.[0-9][0-9]*' | grep Ethernet | wc -l`
    rx_vec=""
    tx_vec=""
    for i in $(seq 0 $((nr_int - 1))) 
    do
        rx=$(printrxbytes "eth$i")
        tx=$(printtxbytes "eth$i")

        rx_vec="$rx_vec $rx"
        tx_vec="$tx_vec $tx"
    done
}


function printrxbytes()
{
    interface=$1
    rx=`ifconfig $interface | grep "RX packets" | tr -s ' ' | cut -d ' ' -f6`

    echo $rx
    return "$rx"
}

function printtxbytes()
{
    interface=$1
    tx=`ifconfig $interface | grep "TX packets" | tr -s ' ' | cut -d ' ' -f6`

    echo $tx
    return "$tx"
}

function create_log_dir()
{
    if [ ! -d $path ]; then
        mkdir $path
    fi
}

#check if a log file is older than the configured days
function check_cleanup()
{
    if [ $cleanup_period -gt 3 ]; then
        find $path -type f -ctime +$((cleanup_period - 2)) -exec rm {} \;
    fi
}


################################### BEGIN ###########################################
if [[ -n "$1"   && -n "$2" ]] #check for arguments
then
    sec=$1
    cleanup_period=$2
fi
    
while true;
do
    create_log_dir
    check_cleanup
    month=`date | tr -s ' ' | cut -d ' ' -f2`
    day=`date | tr -s ' ' | cut -d ' ' -f3`
    output_file=${path}"log_"${month}"_"${day}

    if [ ! -f $output_file ]; then
        touch $output_file
    fi

    prepare_net

    if [ ! "$oldrx_vec" ]; then
        sleep $sec
        continue
    fi

    command=`vmstat $sec 2 | tail -n 1`

    cpu_load "$command" $output_file
    mem_load  $output_file
    dsk_load  "$command" $output_file
    nic_load "$nr_int" "$oldrx_vec" "$oldtx_vec" "$rx_vec" "$tx_vec" "$output_file"

    echo -e "\n" >> $output_file

done

