#
# Copyright 2012 Avaya Inc. All Rights Reserved.
#

#
# Various utility functions for manipulating IP routes, grouped as a Perl module
#

# Usage:
#
# use RouteUtils;

package RouteUtils;

use strict;
use warnings;
use IO::File;
use File::Copy;
use NetAddr::IP;

# Private variables initialized on startup
my $lockFileHandle;

BEGIN {
  require Exporter;

  # set the version for version checking
  our $VERSION = 1.00;

  # Inherit from Exporter to export functions and variables
  our @ISA = qw(Exporter);

  # Functions and variables which are exported by default
  our @EXPORT = qw(getDeviceFromInstalledRoute lookupPersistentRoute addPersistentRoute delPersistentRoute getPersistentRouteLock releasePersistentRouteLock getNextHopDevice $routeFilePrefix);

  # Functions and variables which can be optionally exported
  our @EXPORT_OK = qw();

  $lockFileHandle = IO::File->new(">/opt/Avaya/persistent_routes.lock");
  unless (defined($lockFileHandle)) {
    die "Could not open persistent routes lock file!";
  }
}

# Exported constants
our $routeFilePrefix = "/etc/sysconfig/network-scripts/route-";

# Exported functions

# Search the routing table for matching entry and extract outgoing device if match is found
#
# Parameters:
# - prefix: The network prefix to look for
# - metric: The route metric to look for (may be undefined)
#
# Return value:
#   Outgoing device for installed route (if matching route found) or undef
sub getDeviceFromInstalledRoute {
  my ($prefix, $metric) = @_;
  my $dev;

  if (defined($metric) && ($metric == 0)) {
    undef $metric;
  }

  open CMDOUT, "/sbin/ip route show |" or die "Unable to retrieve kernel routing table: $!";
  while (<CMDOUT>) {
    if (m/$prefix.*?\s+dev\s+(\S+).*?(?:metric\s+(\d+))?\s*$/) {
      # Metric 0 is the same with no metric
      my $matchedMetric = (defined($2) && ($2 == 0)) ? undef : $2;

      # Want the metric to match, as well
      if ((defined($metric) && defined($matchedMetric) && ($metric == $matchedMetric))
          || (!defined($metric) && !defined($matchedMetric))) {

        $dev=$1;
        last;
      }
    }
  }
  close CMDOUT;

  return $dev;
}

# Select outgoing device by matching next hop address against local networks
#
# Parameters:
# - nextHopAddr: a next hop address (represented as a NetAddr::IP object reference),
#                assumed to be in a locally reachable network
#
# Return value:
#   Outgoing device for matching local network
sub getNextHopDevice {
  my $nextHopAddr = shift;
  my ($dev, $matchingDev);

  open CMDOUT, "/sbin/ip addr show |" or die "Unable to retrieve local IP configuration: $!";
  while (<CMDOUT>) {
    if (m/^\d+\:\s+(.*?)\:/) {
      $dev = $1;
      next;
    }

    if (m/^\s+inet\s+(\S+)\s/) {
      if ($nextHopAddr->within(new NetAddr::IP($1))) {
        # The provided next hop address falls within the local network on current device
        $matchingDev = $dev;
        last;
      }
    }
  }
  close CMDOUT;

  return $matchingDev;
}

# Search a per-interface CentOS persistent route file for a matching entry
#
# Parameters:
# - prefix: The network prefix to look for
# - metric: The route metric to look for (may be undefined)
# - devName: the outgoing network device for the persistent route
#
# Return value:
#   1 if match found, 0 otherwise
sub lookupPersistentRoute {
  my ($prefix, $metric, $devName) = @_;

  my $matchFound = 0;

  if (open(RTFILE,"<${routeFilePrefix}$devName")) {
    while (<RTFILE>) {
      # Skip commented out lines
      if (m/^\s*(\#.*)?$/) {
        next;
      }

      if (m/$prefix.*?(?:metric\s+(\d+))?$/) {
        # Metric 0 is the same with no metric
        my $matchedMetric = (defined($1) && ($1 == 0)) ? undef : $1;

        if ((defined($metric) && defined($matchedMetric) && ($metric == $matchedMetric))
            || (!defined($metric) && !defined($matchedMetric))) {
          $matchFound = 1;
        }
      }
    }
    close RTFILE;
  }

  return $matchFound;
}

# Add route to a per-interface CentOS persistent route file
#
# Parameters:
# - prefix: The network prefix to look for
# - gateway: The gateway address
# - devName: the outgoing network device for the persistent route
# - metric: The route metric to look for (may be undefined)
#
# Return value:
#   none
sub addPersistentRoute {
  my ($prefix, $gateway, $devName, $metric) = @_;

  my $fileName = "${routeFilePrefix}$devName";
  my $line = $prefix;
  if (defined($gateway)) {
    $line .= " via " . $gateway;
  }
  $line .= " dev $devName";
  if (defined($metric)) {
    $line .= " metric $metric";
  }

  # Append route to file
  open(RTFILE,">>$fileName") || die("Could not append to $fileName!");
  print RTFILE "$line\n";
  close RTFILE;
}


# Delete route from a per-interface CentOS persistent route file
#
# Parameters:
# - prefix: The network prefix to look for
# - devName: the outgoing network device for the persistent route
# - metric: The route metric to look for (may be undefined)
#
# Return value:
#   none
sub delPersistentRoute {
  my ($prefix, $devName, $metric) = @_;
  my $fileName = "${routeFilePrefix}$devName";
  my $tmpFileName = "${fileName}.tmp";

  # Open existing file for reading
  unless (open(RTFILE,"<$fileName")) {
    print STDERR "Could not open \"$fileName\" for reading!\n";
    return;
  }

  # Open temporary file for writing
  unless (open(TMPRTFILE, ">$tmpFileName")) {
    print STDERR "Could not open \"$tmpFileName\" for writing!\n";
    close RTFILE;
    return;
  }

  while (<RTFILE>) {
      # Skip commented out lines
      if (m/^\s*(\#.*)?$/) {
        next;
      }

      # Skip the route that we want to delete
      if (m/$prefix.*?(?:metric\s+(\d+))?$/) {
        # Metric 0 is the same with no metric
        my $matchedMetric = (defined($1) && ($1 == 0)) ? undef : $1;

        if ((defined($metric) && defined($matchedMetric) && ($metric == $matchedMetric))
            || (!defined($metric) && !defined($matchedMetric))) {
          next;
        }
      }

      # Write the other lines to the temporary file
      print TMPRTFILE $_;
  }

  close RTFILE;
  close TMPRTFILE;

  # Move the temporary file over the original one
  move($tmpFileName, $fileName);
}

# Get lock for changing persistent route files
#
# Return value:
#   none
sub getPersistentRouteLock {
  flock $lockFileHandle, 2;
}

# Release lock for changing persistent route files
#
# Return value:
#   none
sub releasePersistentRouteLock {
  flock $lockFileHandle, 8;
}

END {
  undef $lockFileHandle;
}

1; # don't forget to return a true value from the file
