#!/usr/bin/perl

#
# Copyright 2012 Avaya Inc. All Rights Reserved.
#

# Usage:        add_persistent_route.pl <dest> <mask> <gateway> <dev> [<metric>]
# Purpose:      Add a persistent route to a CentOS 6.x machine
#
# Description for command line arguments:
# dest    - the address of the destination network or host
# mask    - the netmask for destination
# gateway - the address of the IP gateway device
# dev     - the outgoing network inteface for the route
# metric  - the metric for the toute
#

# Pick custom modules from specific path
use lib '/opt/Avaya/scripts';
use File::Copy;

use strict;

use RouteUtils;

use vars qw($shortName $usage $dst $gw $dev $metric $prefix $fileName);
use vars qw($line $duplicate);

# Main execution thread starts here
$shortName = eval { my @path = (split "/", $0) ; pop @path; };

$usage = <<EOF;
  Usage: $shortName <dest> <mask> <gateway> <dev> [<metric>]
EOF

open(my $fLog, '>>', '/opt/ipoffice/network_changes.log');
my $datestring = localtime();
my $strParam = "";
foreach my $argnum (0 .. $#ARGV) {
  $strParam =  "$strParam $ARGV[$argnum]";
}
print $fLog "$datestring: add_persistent_route-> $strParam\n";
close $fLog;

if ((@ARGV != 5) && (@ARGV != 4) && (@ARGV != 3)) {
  print STDERR "Invalid number of arguments!\n";
  print STDERR $usage;
  exit 1;
}

# Validate destination and mask
$dst = new NetAddr::IP ($ARGV[0], $ARGV[1]);
unless (defined($dst)) {
  print STDERR "Could not parse network from destination \"$ARGV[0]\" and netmask \"$ARGV[1]\"\n";
  exit 2;
}

# Extract the string representation of the network prefix/host address
if ($dst->masklen() == 32) {
  # Only the address is actually needed for host routes
  $prefix=$dst->addr();
} else {
  $prefix=$dst->network();
}

# Validate gateway address, if present
if ($ARGV[2] =~ m/\S/) {
  $gw = new NetAddr::IP ($ARGV[2]);
  unless (defined($gw)) {
    print STDERR "Could not parse gateway address from \"$ARGV[2]\"\n";
    exit 4;
  }
}

if (defined($ARGV[4]) && ($ARGV[4] =~ m/\S/)) {
  unless ($ARGV[4] =~ m/\d+/) {
    print STDERR "Invalid route metric \"$ARGV[4]\"\n";
    exit 5;
  }
  $metric = ($ARGV[4] == 0) ? undef : $ARGV[4];
}

# Find appropriate outgoing network device if none provided, abort if unable to do so
if ($ARGV[3] =~ m/\S/) {
  $dev = $ARGV[3];
} else {
  # Identify the outgoing network device based on the gateway address, if provided
  unless (defined($gw)) {
    print STDERR "Neither gateway address nor outgoing network device were provided for route\n";
    exit 6;
  }

  $dev = getNextHopDevice($gw);

  unless (defined($dev)) {
    print STDERR "Could not find outgoing device for \"$prefix\" and next hop \"" . $gw->addr() . "\"\n";
    exit 7;
  }
}
getPersistentRouteLock();
# Add a default route (gateway)
# In order to be added, a default route must have a valid next hop address
if (($dst eq "0.0.0.0/0" )  && !($gw eq "0.0.0.0/32" )) {
  my $networkfile = "/etc/sysconfig/network";
  my $newnetworkfile = "/etc/sysconfig/network.new";
  my $gateway = "GATEWAY=$ARGV[2]\n";
  my $gatewaydev = "GATEWAYDEV=$dev\n";
  if (open(file1, "<$networkfile") && open(file2, ">$newnetworkfile")) {
    # Exclude the old gateway (if it exists)
    while($line = <file1>) {
      if (($line !~ /^\s*GATEWAY\s*=/) && ($line !~ /^\s*GATEWAYDEV\s*=/) ) {
        print file2 $line;
      }
    }
    
    # Add the new gateway
    print file2 $gateway;
    print file2 $gatewaydev;
    close(file2);
    close(file1);
    move($newnetworkfile, $networkfile);
  }
  chmod 0664, $networkfile;
  system("ip route add default via $ARGV[2] dev $dev");
  releasePersistentRouteLock();
  
  exit 0;
}


# Check for duplicate routes
$duplicate = lookupPersistentRoute($prefix, $metric, $dev);
if ($duplicate) {
  my $metricText = defined($metric) ? "metric $metric":"no metric";
  print STDERR "Persistent route for $prefix and $metricText already exists on $dev\n";
  releasePersistentRouteLock();
  exit 8;
}

addPersistentRoute($prefix, defined($gw) ? $gw->addr() : undef, $dev, $metric);

releasePersistentRouteLock();
