#!/bin/bash
# This script provides basic functions for handling Avaya IPOL Certificates for the CLI interface
# 

if [ -f /opt/Avaya/scripts/config_files/apps_logging_config ]; then
    . /opt/Avaya/scripts/config_files/apps_logging_config
fi

# DEFAULT PARAMETERS

# Default CA Certificate Validity
DEFAULT_CA_CERT_VALIDITY=$((365*10)) # 10 years
# Default Server Certificate Validity
DEFAULT_SERVER_CERT_VALIDITY=$((365*7)) # 7 years
# Default RSA Key Size
DEFAULT_RSA_KEY_LEN=2048
# Default SHA Version
DEFAULT_SHA_VER=-sha256

# ACTIONS

# --set subjectName
SET_SN_PARAMETER=false
# --set subjectAltName
SET_SAN_PARAMETER=false
# --display current csr parameters
DISPLAY_CSR_CNF=false
# --import
IMPORT_CERT=false
# --generate-csr
GENERATE_CSR=false
# --cleanup
CLEANUP_ACTION=false
# --display-auto-flag
DISPLAY_AUTO_FLAG=false
# --set-auto-flag
SET_AUTO_FLAG=false

# All the temporary files should be added to this variable
# They will be all removed in the cleanup function, when the script exits
TMP_FILES=

if [ ! -z $GEN_CERTS_LOG_DIR_NAME ]; then
    LOG_DIR="${GEN_CERTS_LOG_DIR_NAME}"
else
    LOG_DIR="/opt/Avaya/scripts/logs"
fi
mkdir -p $LOG_DIR
if [ ! -z $GEN_CSR_LOG_FILE_NAME ]; then
    LOG_FILE="$LOG_DIR""/""${GEN_CERTS_LOG_FILE_NAME}"
else
    LOG_FILE=$LOG_DIR/cli_csr.log
fi
touch $LOG_FILE

# Help function
usage()
{
cat << EOF
usage: $0 <action> <options>

This script provides the basic utilities for handling Avaya IPOL Certificates

ACTIONS:
   --set-sn                     Sets the subject name
   --set-san                    Sets the subject alternative name
   --set-auto-flag              Sets/Resets auto distribution flag
   --display-csr-cnf            Displays the current csr configuration
   --import-cert                Imports a certificate
   --generate-csr               Generates a certificate request based on the current configuration
   --display-auto-flag          Displays the auto distribution flag
   --cleanup                    Cleans up the temporary configuration file

OPTIONS:
   -h|--help                    Show this message
   --sn-country                 Inputs the subject name country initials
   --sn-state                   Inputs the subject name state
   --sn-locality                Inputs the subject name locality
   --sn-organization            Inputs the subject name organization
   --sn-organizationunit        Inputs the subject name organization unit
   --sn-commonname              Inputs the subject name common name
   --sn-email                   Inputs the subject name email
   --add-san                    Adds a san record to the configuration
   --del-san                    Deletes a san record from the configuration

RETURN VALUES:
   ESUCCESS=0          # Success
   EFAIL=1             # General Error
   EINVAL=2            # Invalid argument
   EGENRSA=3           # Cannot generate RSA Key
   EGENCERT=4          # Cannot generate certificate
   ECERTINVALID=5      # Certificate invalid
   EKEYNOTFOUND=6      # Cannot find or load the RSA Key
   ENOTMATCH=7         # Non Matching Certificate-Private Key

EOF
}

# Import IPOffice variables
import_isa_data() {
    if [ -f /etc/sysconfig/ipoffice ]; then
        source /etc/sysconfig/ipoffice
    fi

    # Avaya Certificate Directory
    CERT_DIR=$IPOFFICE_CERTIFICATES_PATH
    # CA Certificate Directory
    PRIVATE_KEYS_DIR=$CERT_DIR/keys

    if [ ! -d $CERT_DIR ]
    then
        mkdir -p "$CERT_DIR"
    fi
    chmod 700 $CERT_DIR

    if [ ! -d $PRIVATE_KEYS_DIR ]
    then
        mkdir -p "$PRIVATE_KEYS_DIR"
    fi
    chmod 700 $PRIVATE_KEYS_DIR
}

# Process the script arguments
process_args() {
    local ACTIONS_CNT=0

    # Check the options
    OPTS=`getopt -o h -l set-sn -l set-san -l set-auto-flag: -l display-csr-cnf -l import-cert -l generate-csr -l display-auto-flag -l cleanup \
          -l help -l sn-country: -l sn-state: -l sn-locality: -l sn-organization: -l sn-organizationunit: -l sn-commonname: -l sn-email: \
          -l add-san: -l del-san: -- "$@"`

    eval set -- "$OPTS"

    # Parse arguments
    while true
    do
        case "$1" in
            -h|--help)
                usage
                exit;;
            # Actions
            --set-sn)
                SET_SN_PARAMETER=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --set-san)
                SET_SAN_PARAMETER=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --set-auto-flag)
                SET_AUTO_FLAG=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                AUTO_FLAG_VALUE="$2"
                shift 2;;
            --display-csr-cnf)
                DISPLAY_CSR_CNF=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --display-auto-flag)
                DISPLAY_AUTO_FLAG=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --import-cert)
                IMPORT_CERT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --generate-csr)
                GENERATE_CSR=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --cleanup)
                CLEANUP_ACTION=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            # Options
            --sn-country)
                SN_COUNTRY="$2"
                shift 2;;
            --sn-state)
                SN_STATE="$2"
                shift 2;;
            --sn-locality)
                SN_LOCALITY="$2"
                shift 2;;
            --sn-organization)
                SN_ORGANIZATION="$2"
                shift 2;;
            --sn-organizationunit)
                SN_ORGANIZATIONUNIT="$2"
                shift 2;;
            --sn-commonname)
                SN_COMMONNAME="$2"
                shift 2;;
            --sn-email)
                SN_EMAIL="$2"
                shift 2;;
            --add-san)
                SAN_ADD="$2"
                shift 2;;
            --del-san)
                SAN_DEL="$2"
                shift 2;;
            --)
                shift; break;;
            *)
                usage
                err_exit $EINVAL "Invalid option $1";;
        esac
    done

    # No further arguments are required
    if [ $# -ne 0 ]
    then
        usage
        err_exit $EINVAL "Invalid argument: $@"
    fi

    # Only one action at a time
    if [ "$ACTIONS_CNT" -ne "1" ]
    then
        usage
        err_exit $EINVAL "Multiple or no actions specified"
    fi

    if $SET_SN_PARAMETER;
    then
        PCOUNT=0
        if [ ! -z "${SN_COUNTRY}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [ ! -z "${SN_STATE}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [ ! -z "${SN_LOCALITY}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [ ! -z "${SN_ORGANIZATION}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [ ! -z "${SN_ORGANIZATIONUNIT}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [ ! -z "${SN_COMMONNAME}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [ ! -z "${SN_EMAIL}" ]; then
            let PCOUNT=PCOUNT+1
        fi
        if [[ ${PCOUNT} -ne 1 ]]; then
            err_exit $EINVAL "More than one value given"
        fi
    fi

    if $SET_SAN_PARAMETER;
    then
        if [ -z "$SAN_ADD" -a -z "$SAN_DEL" ]; then
            err_exit $EINVAL "No values for SAN to add or delete"
        fi
    fi
}

# Generate an OpenSSL config file so that certain parameters can be filled
# (like keyUsage, SubjectAltName, CA flag...)
generate_config_file() {
    local EXT_FILE="$1"
    local KEY_USAGE=$2
    local PTYPE=$3
    local PVALUE="$4"

    if [ ! -f ${EXT_FILE} ]; then
        echo "[ req ]"                                         > $EXT_FILE
        echo "distinguished_name = req_distinguished_name"    >> $EXT_FILE
        echo "req_extensions = v3_req"                        >> $EXT_FILE
        echo "prompt = no"                                    >> $EXT_FILE
        echo "[ req_distinguished_name ]"                     >> $EXT_FILE
        echo                                                  >> $EXT_FILE

        echo "[ v3_req ]"                                     >> $EXT_FILE
        echo "basicConstraints = CA:FALSE"                    >> $EXT_FILE
        echo "keyUsage = $KEY_USAGE"                          >> $EXT_FILE
        #echo "subjectName = $SUBJECT_NAME"                   >> $EXT_FILE
        echo "subjectAltName = @alt_names"                    >> $EXT_FILE
        echo "subjectKeyIdentifier = hash"                    >> $EXT_FILE
        echo "[ alt_names ]"                                  >> $EXT_FILE
        echo                                                  >> $EXT_FILE
    else
        MODIFY_SN=false
        MODIFY_SAN=false

        EXT_TMP_FILE=`mktemp`
        if [ "${PTYPE}" == "SN_C" -o "${PTYPE}" == "SN_ST" -o \
             "${PTYPE}" == "SN_L" -o "${PTYPE}" == "SN_O" -o \
             "${PTYPE}" == "SN_OU" -o "${PTYPE}" == "SN_CN" -o "${PTYPE}" == "SN_E" ]; then
            MODIFY_SN=true
        elif [ "${PTYPE}" == "SAN_ADD" -o "${PTYPE}" == "SAN_DEL" ]; then
            MODIFY_SAN=true
        fi

        START_INJECTION=false
        END_INJECTION=false
        BUFFER_SAN=false
        
        while read line;
        do
            if $START_INJECTION; then
                if ${MODIFY_SN}; then
                    declare -a snarray
                    if [ "${PTYPE}" == "SN_C" ]; then
                        local SN_COUNTRY="C=${PVALUE}"
                    else
                        local SN_COUNTRY=`cat ${EXT_FILE}|grep "C="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_COUNTRY}")
                    if [ "${PTYPE}" == "SN_ST" ]; then
                        local SN_STATE="ST=${PVALUE}"
                    else
                        local SN_STATE=`cat ${EXT_FILE}|grep "ST="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_STATE}")
                    if [ "${PTYPE}" == "SN_L" ]; then
                        local SN_LOC="L=${PVALUE}"
                    else
                        local SN_LOC=`cat ${EXT_FILE}|grep "L="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_LOC}")
                    if [ "${PTYPE}" == "SN_O" ]; then
                        local SN_ORG="O=${PVALUE}"
                    else
                        local SN_ORG=`cat ${EXT_FILE}|grep "O="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_ORG}")
                    if [ "${PTYPE}" == "SN_OU" ]; then
                        local SN_ORGU="OU=${PVALUE}"
                    else
                        local SN_ORGU=`cat ${EXT_FILE}|grep "OU="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_ORGU}")
                    if [ "${PTYPE}" == "SN_CN" ]; then
                        local SN_COMN="CN=${PVALUE}"
                    else
                        local SN_COMN=`cat ${EXT_FILE}|grep "CN="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_COMN}")
                    if [ "${PTYPE}" == "SN_E" ]; then
                        local SN_EMAIL="emailAddress=${PVALUE}"
                    else
                        local SN_EMAIL=`cat ${EXT_FILE}|grep "emailAddress="|grep -v "URI"`
                    fi
                    snarray=("${snarray[@]}" "${SN_EMAIL}")
                    for sn_o in "${snarray[@]}"
                    do
                        if [ ! -z "${sn_o}" ]; then
                            echo "${sn_o}" >> ${EXT_TMP_FILE}
                        fi
                    done
                    START_INJECTION=false
                    END_INJECTION=true
                elif ${MODIFY_SAN}; then
                    SAN_ARRAY=(${PVALUE//,/ })
                    for san_o in "${SAN_ARRAY[@]}";
                    do
                        echo "${san_o}" >> ${EXT_TMP_FILE}
                    done
                    START_INJECTION=false
                    END_INJECTION=true
                else
                    START_INJECTION=false
                    END_INJECTION=true
                fi
            elif $END_INJECTION; then
                if ${MODIFY_SN}; then
                    if [[ "${line}" == *"[ v3_req ]"* ]]; then
                        echo ${line} >> ${EXT_TMP_FILE}
                        END_INJECTION=false
                    fi
                elif ${MODIFY_SAN}; then
                    START_INJECTION=false
                    END_INJECTION=true # not ending
                else
                    START_INJECTION=false
                    END_INJECTION=false
                fi
            else
                if ${MODIFY_SN}; then
                    if [[ "${line}" == *"[ req_distinguished_name ]"* ]]; then
                        START_INJECTION=true
                    fi
                    if $BUFFER_SAN; then
                        if [ ${#BUFFERED_SAN[@]} -eq 0 ]; then
                            BUFFERED_SAN=("$line")
                        else
                            BUFFERED_SAN=("${BUFFERED_SAN}" "$line")
                        fi
                    else
                       if [[ "${line}" == *"[ alt_names ]"* ]]; then
                            echo $line >> ${EXT_TMP_FILE}
                            BUFFER_SAN=true
                        else
                            echo $line >> ${EXT_TMP_FILE}
                        fi
                    fi
                elif ${MODIFY_SAN}; then
                    if $BUFFER_SAN; then
                        if [ ${#BUFFERED_SAN[@]} -eq 0 ]; then
                            BUFFERED_SAN=("$line")
                        else
                            BUFFERED_SAN=("${BUFFERED_SAN}" "$line")
                        fi
                    else
                        if [[ "${line}" == *"[ alt_names ]"* ]]; then
                            BUFFER_SAN=true
                        fi
                        echo $line >> ${EXT_TMP_FILE}
                    fi
                else
                    echo $line >> ${EXT_TMP_FILE}
                fi
            fi
        done < ${EXT_FILE}

        if $BUFFER_SAN; then
            # check the COMMON_NAME
            if ${MODIFY_SAN}; then
                if [ "${PTYPE}" == "SAN_ADD" ]; then
                    for san_o in "${BUFFERED_SAN[@]}"
                    do
                        if [ ! -z "${san_o}" ]; then
                            echo "${san_o}" >> ${EXT_TMP_FILE}
                        fi
                    done
                    echo "${PVALUE}" >> ${EXT_TMP_FILE}
                elif [ "${PTYPE}" == "SAN_DEL" ]; then
                    SAN_WRITTEN=false
                    for san_o in "${BUFFERED_SAN[@]}"
                    do
                        if [ ! -z "${san_o}" ]; then
                            if [ -z "`echo "${san_o}"|grep "${PVALUE}"`" ]; then
                                SAN_WRITTEN=true
                                echo "${san_o}" >> ${EXT_TMP_FILE}
                            fi
                        fi
                    done
                    if ! $SAN_WRITTEN; then
                        echo >> ${EXT_TMP_FILE}
                    fi
                else
                    SAN_WRITTEN=false
                    for san_o in "${BUFFERED_SAN[@]}"
                    do
                        if [ ! -z "${san_o}" ]; then
                            SAN_WRITTEN=true
                            echo "${san_o}" >> ${EXT_TMP_FILE}
                        fi
                    done
                    if ! $SAN_WRITTEN; then
                        echo >> ${EXT_TMP_FILE}
                    fi
                fi
            else
                SN_COMMON_FILE="`cat ${EXT_TMP_FILE}`"
                readarray -t SN_COMMON_CAT <<< "${SN_COMMON_FILE}"
                for i in "${SN_COMMON_CAT[@]}"
                do
                    if [[ "${i}" == *"CN="* ]]; then
                        SN_COMMON_NAME="${i}"
                        break
                    fi
                done
                SN_CMN_ARRAY=(${SN_COMMON_NAME//,/ })
                if [[ ${#SN_CMN_ARRAY[@]} -gt 1 ]]; then
                    SAN_REC_0="DNS.0="
                    SAN_REC_0="${SAN_REC_0}""${SN_CMN_ARRAY[0]##*=}"
                    echo "${SAN_REC_0}" >> ${EXT_TMP_FILE}
                else
                    SN_CMN_ARRAY=(${SN_COMMON_NAME//;/ })
                    if [[ ${#SN_CMN_ARRAY[@]} -gt 1 ]]; then
                        SAN_REC_0="DNS.0="
                        SAN_REC_0="${SAN_REC_0}""${SN_CMN_ARRAY[0]}"
                        echo "${SAN_REC_0}" >> ${EXT_TMP_FILE}
                    fi
                fi
                SAN_WRITTEN=false
                for san_o in "${BUFFERED_SAN[@]}"
                do
                    if [ ! -z "${san_o}" ]; then
                        if [ ! -z "${SAN_REC_0}" ]; then
                            if [[ ${san_o} != *"DNS.0"* ]]; then
                                SAN_WRITTEN=true
                                echo "${san_o}" >> ${EXT_TMP_FILE}
                            fi
                        else
                            SAN_WRITTEN=true
                            echo "${san_o}" >> ${EXT_TMP_FILE}
                        fi
                    fi
                done
                if ! $SAN_WRITTEN; then
                    echo >> ${EXT_TMP_FILE}
                fi
            fi
        fi

        mv ${EXT_TMP_FILE} ${EXT_FILE}
    fi
}

init_csr_variables() {
    PARENT_SUDO=`ps -ef|grep $PPID|grep -v "ps -ef"|grep "sudo"| tr -s \ |cut -d" " -f3`
    PARENT_CLISH=`ps -ef|grep ${PARENT_SUDO}|grep "sh -c"|tr -s \ |cut -d" " -f3`

    if [ -z "${PARENT_CLISH}" ]; then
        err_exit $EFAIL "This should be invoked through the clish shell"
    fi
    OPENSSL_CNF="/tmp/csrtmp_""$PARENT_CLISH"".cnf"
    KEY_USAGE="digitalSignature, nonRepudiation, keyAgreement, dataEncipherment, keyEncipherment"
    if [ ! -f "$OPENSSL_CNF" ]; then
        if [ ! -z "`/sbin/ifconfig | grep $IPOFFICE_LAN1`" ];then
            ETH0_MAC=`cat /sys/class/net/$IPOFFICE_LAN1/address|tr -d ":"`
            local SRV_ETH0_IP=`ip addr show dev $IPOFFICE_LAN1 | grep "inet " | cut -d " " -f 6 | cut -d "/" -f 1`
        fi
        if [ ! -z "$IPOFFICE_LAN2" ]; then
            if [ ! -z "`/sbin/ifconfig | grep $IPOFFICE_LAN2`" ];then
                local SRV_ETH1_IP=`ip addr show dev $IPOFFICE_LAN2 | grep "inet " | cut -d " " -f 6 | cut -d "/" -f 1`
            fi
        fi

        # Get the hostname
        hName=$HOST_NAME
        if [ -z "$hName" ]; then
            hName=`hostname`
        fi
        # Check if the ip of lan1 is resolvable
        if [ -z "$hName" -o "$hName" == "localhost" -o "$hName" == "localhost.localdomain" -o "$hName" == "$ETH0_MAC" ]; then
            hName=""
            if [ -n "$SRV_ETH0_IP" ]; then
                dNSName=`/usr/bin/host $SRV_ETH0_IP | cut -d' ' -f5`
            fi
            if [[ -z "$dNSName" || "$dNSName" =~ "NXDOMAIN" || "$dNSName" =~ "no" ]]; then
                if [ -n "$SRV_ETH1_IP" ]; then
                    dNSName="`/usr/bin/host $SRV_ETH1_IP | cut -d' ' -f5`"
                fi
            fi
            if [[ -z "$dNSName" || "$dNSName" =~ "NXDOMAIN" || "$dNSName" =~ "no" ]]; then
                dNSName=""
            else
                dNSName=${dNSName%"."}
            fi
        fi

        # Use the defaut ID if none specified
        if [ -z "$ID" ]
        then
            # Default Server ID
            if [ ! -z "$hName" ]; then
                ID="$hName"
            elif [ ! -z "$dNSName" ]; then
                ID="$dNSName"
            else
                ID="$ETH0_MAC"
            fi
        fi

        # Use the defaut IssuedTo if none specified
        if [ -z "$ISSUED_TO" ]
        then
            if [ ! -z "$hName" -o ! -z "$dNSName" ]; then
                ISSUED_TO="$ID"
            else
                ISSUED_TO="iposerver-$ID.avaya.com"
            fi
        fi
        generate_config_file "$OPENSSL_CNF" "$KEY_USAGE"
    fi
}

verify_wildcard() {
    local vv="$1"
    local __resultvar=$2

    if [[ "${vv}" == *"*"* ]]; then
        echo -n "Avaya does not endorse the use of wildcard characters – i.e. wildcard certificates. Any use of wildcard certificates will be entirely at the customer's own risk. Do you wish to continue? [y/N]"
        read -n1 answer
        if [ "${answer}" == "y" -o "${answer}" == "Y" ]; then
            eval $__resultvar=true
        else
            eval $__resultvar=false
        fi
        echo
    else
        eval $__resultvar=true
    fi
}

verify_san_string() {
    local vv="$1"
    local __resultvar=$2
    ST_CHARS=${vv:0:3}
    if [ "${ST_CHARS}" == "DNS" -o "${ST_CHARS}" == "URI" ]; then
        if [[ "${vv}" == *"="* ]]; then
            eval $__resultvar=true
        else
            echo "ERROR: your string does not contain an mandatory equal sign"
            eval $__resultvar=false
        fi
    else
        ST_CHARS=${vv:0:2}
        if [ "${ST_CHARS}" == "IP" ]; then
            if [[ "${vv}" == *"="* ]]; then
                eval $__resultvar=true
            else
                echo "ERROR: your string does not contain an mandatory equal sign"
                eval $__resultvar=false
            fi
        else
            echo "ERROR: your string should start with a mandatory DNS,URI or IP word"
            eval $__resultvar=false
        fi
    fi
}

set_parameter() {
    local PTYPE="$1"
    local PNAME="$2"
    local PVALUE="$3"

    EVAL=false
    verify_wildcard "${PVALUE}" EVAL

    if ${EVAL}; then
        if [ "${PTYPE}" == "subjectName" ]; then
            if [ "${PNAME}" == "country" ]; then
                # check value
                VLENGTH=${#PVALUE}
                if [[ ${VLENGTH} -ne 2 ]]; then
                    err_exit $EINVAL "ERROR - Country is a 2 letter code"
                fi
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_C" "${PVALUE}"
            elif [ "${PNAME}" == "state" ]; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_ST" "${PVALUE}"
            elif [ "${PNAME}" == "locality" ]; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_L" "${PVALUE}"
            elif [ "${PNAME}" == "organization" ]; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_O" "${PVALUE}"
            elif [ "${PNAME}" == "organizationunit" ]; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_OU" "${PVALUE}"
            elif [ "${PNAME}" == "commonname" ]; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_CN" "${PVALUE}"
            elif [ "${PNAME}" == "email" ]; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "SN_E" "${PVALUE}"
            fi
        elif [ "${PTYPE}" == "subjectAltName" ]; then
            if [ "${PNAME}" == "SAN_ADD" ]; then
                EVAL=false
                verify_san_string "${PVALUE}" EVAL
            else
                EVAL=true
            fi
            if ${EVAL}; then
                generate_config_file "$OPENSSL_CNF" "$KEY_USAGE" "${PNAME}" "${PVALUE}"
            fi
        fi
    else
        echo "Wildcard character not accepted"
    fi
}

display_csr_cnf() {
    cat $OPENSSL_CNF
}

cleanup_action() {
    if [ -f $OPENSSL_CNF ]; then
        rm -f $OPENSSL_CNF
    fi
}

find_ca_cert() {
    local INT_CA=
    for INT_CA in $CA_CHAIN_CERTS
    do
        local CERT_VALID=false
        local KEY_VALID=false
        openssl x509 -text -in $INT_CA > /dev/null 2>&1 && CERT_VALID=true
        openssl rsa -text -in ${INT_CA%.pem}.key > /dev/null 2>&1 && KEY_VALID=true
        if $CERT_VALID && $KEY_VALID
        then
            CA_CERT=$INT_CA
            CA_KEY=${INT_CA%.pem}.key
        fi
    done
}

generate_csr() {
    local EXT_FILE=`mktemp`
    local OPENSSL_TMP_CNF=`mktemp`
    local DEFA_PASS=1234
    local ERROR_DETECTED=false
    local ID=$(cat /dev/urandom | tr -dc 'a-zA-Z0-9' | fold -w 8 | head -n 1)
    # Encrypted Server Private Key
    local SRV_KEY_PASS="${CERT_DIR}/keys/server_${ID}_pass.key"
    # Server Certificate Request
    local SRV_CERT_REQ="/tmp/server_${ID}.csr"
    local SRV_KEY="${CERT_DIR}/keys/server_${ID}.key"
    local RSA_KEY_LEN=${DEFAULT_RSA_KEY_LEN}
    local SHA_VER=${DEFAULT_SHA_VER}

    TMP_FILES="$TMP_FILES $SRV_CERT_REQ $EXT_FILE $SRV_KEY_PASS $OPENSSL_TMP_CNF"

    if [ ! -f ${OPENSSL_CNF} ]; then
        err_exit $EFAIL "ERROR: No request configuration found"
    fi

    log "Generating RSA key length $RSA_KEY_LEN"

    openssl genrsa -des3 -out $SRV_KEY_PASS -passout pass:$DEFA_PASS $RSA_KEY_LEN > /dev/null 2>&1 || ERROR_DETECTED=true

    if $ERROR_DETECTED; then
        err_exit $EGENKEY "Cannot generate RSA Key"
    fi

    openssl rsa -passin pass:$DEFA_PASS -in $SRV_KEY_PASS -out $SRV_KEY > /dev/null 2>&1

    # processing OPENSSL_CNF file
    READ_CAT_CNF="`cat ${OPENSSL_CNF}`"
    readarray -t READ_CNF_ARRAY <<< "${READ_CAT_CNF}"
    FOUND_SAN_REC=false
    CHECK_SAN_REC=false
    for cnf_o in "${READ_CNF_ARRAY[@]}"
    do
        if [[ "${cnf_o}" == *"[ alt_names ]"* ]]; then
            CHECK_SAN_REC=true
        elif $CHECK_SAN_REC; then
            if [ ! -z "${cnf_o}" ]; then
                FOUND_SAN_REC=true
                break
            fi
        fi
    done

    if $FOUND_SAN_REC; then
        cp ${OPENSSL_CNF} ${OPENSSL_TMP_CNF}
    else
        PUSH_TO_TMP=true
        for cnf_o in "${READ_CNF_ARRAY[@]}"
        do
            if [[ "${cnf_o}" != *"subjectAltName"* ]]; then
                if $PUSH_TO_TMP; then
                    echo "${cnf_o}" >> ${OPENSSL_TMP_CNF}
                fi
                if [[ "${cnf_o}" == *"[ alt_names ]"* ]]; then
                    PUSH_TO_TMP=false
                fi
            fi
        done
    fi

    openssl req -new -batch $SHA_VER -key $SRV_KEY -out $SRV_CERT_REQ -reqexts v3_req -config ${OPENSSL_TMP_CNF} > ${EXT_FILE} 2>&1 || ERROR_DETECTED=true

    if $ERROR_DETECTED; then
        TMP_FILES="$TMP_FILES $SRV_KEY"
        
        # getting and printing the error
        ERROR_RES="`cat ${EXT_FILE}`"
        readarray -t earray <<< "${ERROR_RES}"
        echo "ERROR while generating certificate request:"
        echo "${earray[0]}"
        IFS=':' read -r -a dearray <<< ${earray[${#earray[@]}-1]}
        DE_SIZE=${#dearray[@]}
        ERR_EXPLAINED=$((DE_SIZE - 4))
        ERR_POINTER=$((DE_SIZE - 1))
        if [[ ${ERR_EXPLAINED} -gt -1 ]]; then
            CSR_ERROR="${dearray[${ERR_EXPLAINED}]} -> "
            CSR_ERROR="$CSR_ERROR""${dearray[${ERR_POINTER}]}"
            echo "${CSR_ERROR}"
        fi
        err_exit $EGENCERT "Cannot generate Certificate Request"
    fi

    openssl req -text -noout -in $SRV_CERT_REQ
    echo

    cat $SRV_CERT_REQ
}

set_auto_flag() {
    local FLAG_VALUE="$1"
    local AUTO_FLAG=`ls -al /opt/Avaya/certs|grep .auto|wc -l`

    if [ "${FLAG_VALUE}" == "on" ]; then
        if [[ ${AUTO_FLAG} -eq 0 ]]; then
            echo "Certificate distribution state change: manual -> automatical"
            echo "This will restart the running applications on the machine"
            touch /opt/Avaya/certs/.auto
            chmod 600 /opt/Avaya/certs/.auto
        else
            echo "Certificate distribution state did not change"
        fi
    else
        if [[ ${AUTO_FLAG} -eq 1 ]]; then
            echo "Certificate distribution state change: automatical -> manual"
            rm /opt/Avaya/certs/.auto
        else
            echo "Certificate distribution state did not change"
        fi
    fi
}

display_auto_flag() {
    local AUTO_FLAG=`ls -al /opt/Avaya/certs|grep .auto|wc -l`
    if [[ ${AUTO_FLAG} -eq 0 ]]; then
        echo "Certificate distribuion state: manual"
    else
        echo "Certificate distribuion state: automatical"
    fi
}

import_certificate() {
    local EXT_FILE=`mktemp`
    local CERT_PBKEY=`mktemp`
    local PVKEY_PBKEY=`mktemp`
    local ERROR_DETECTED=false
    # read the certificate from the command line
    echo "Please paste your certificate here:"
    continue_reading=true
    end_reading=false
    while $continue_reading;
    do
        read cline
        cline_length=${#cline}
        if [ -z "${cert_read}" ]; then
            cert_read="${cline}"
        else
            cert_read="${cert_read}"$'\r'$'\n'"${cline}"
        fi
        if [ ${cline_length} -eq 0 ]; then
            continue_reading=false
        elif $end_reading; then
            continue_reading=false
        elif [ ${cline_length} -lt 64 ]; then
            if [ -z "`echo ${cline}|grep "BEGIN CERTIFICATE"`" ]; then
                end_reading=true
            fi
        fi
        if [ ! -z "`echo $cline|grep "END"`" ]; then
            continue_reading=false
        fi
    done
    echo "$cert_read" > ${EXT_FILE}
    # displaying certificate
    openssl x509 -in ${EXT_FILE} -text -noout > /dev/null 2>&1 || ERROR_DETECTED=true
    if $ERROR_DETECTED; then
        TMP_FILES="$TMP_FILES $EXT_FILE $CERT_PBKEY $PVKEY_PBKEY"
        err_exit $ECERTINVALID "Invalid certificate"
    fi

    openssl x509 -in ${EXT_FILE} -text -noout

    echo
    echo "Importing certificate ...."

    openssl x509 -pubkey -noout -in ${EXT_FILE} > ${CERT_PBKEY}

    MATCHED_PVKEY=""
    for privkey in /opt/Avaya/certs/keys/*
    do
        openssl rsa -in ${privkey} -pubout > ${PVKEY_PBKEY} 2>/dev/null
        MD5SUM_PBKEYCERT="`md5sum ${CERT_PBKEY}|tr -s \ |cut -d" " -f1`"
        MD5SUM_PBKEYPVKEY="`md5sum ${PVKEY_PBKEY}|tr -s \ |cut -d" " -f1`"
 
        if [ "${MD5SUM_PBKEYCERT}" == "${MD5SUM_PBKEYPVKEY}" ]; then
            MATCHED_PVKEY=${privkey}
            break;
        fi
    done

    if [ ! -z "${MATCHED_PVKEY}" ]; then
        cat ${MATCHED_PVKEY} > ${PVKEY_PBKEY}
        cat ${EXT_FILE} >> ${PVKEY_PBKEY}
        mv ${PVKEY_PBKEY} /opt/Avaya/certs/cert_to_import.pem
        rm -f ${MATCHED_PVKEY}
        echo "Operation successful"
        local AUTO_FLAG=`ls -al /opt/Avaya/certs|grep .auto|wc -l`
        if [[ ${AUTO_FLAG} -eq 1 ]]; then
            echo "Certificate auto distribution flag is set, services will be disrupted"
        fi
    else
        echo "Operation failed - no matching private key found"
    fi
 
    TMP_FILES="$TMP_FILES $EXT_FILE $CERT_PBKEY"
}

print_variables() {
    echo "Subject: $ISSUED_TO"
    echo "Subject-alt-name: $SUBJECT_ALT_NAME"
    echo "Validity: $CERT_VALIDITY"
    echo "RSA Key Length: $RSA_KEY_LEN"
    echo "SHA Version: $SHA_VER"
}

log() {
    if [ -n $LOG_FILE ]
    then
        echo "[$(date)] $1" >> $LOG_FILE
    fi
    # echo "[$(date)] $1"
}

init_error_codes() {
    ESUCCESS=0          # Success
    EFAIL=1             # General Error
    EINVAL=2            # Invalid argument
    EGENRSA=3           # Cannot generate RSA Key
    EGENCERT=4          # Cannot generate certificate
    ECERTINVALID=5      # Certificate invalid
    EKEYNOTFOUND=6      # Cannot find or load the RSA Key
    ENOTMATCH=7         # Non Matching Certificate-Private Key
}

err_exit() {
    log "$2"
    echo $2
    exit $1
}

cleanup() {
    rc=$?

    # Cleanup temp files
    sudo rm -f $TMP_FILES
    exit $rc
}

# Main

init_error_codes

# Execute the script as root
if [ "$UID" -ne 0 ]
then
    err_exit $EACCESS "Please run as root"
fi

# Change umask so that the Certificates and Private Keys cannot be read by
# other users.
umask 077

# Toggle -e to exit the script when an error occurs
set -e

# Redirect stderr to stdout so that webcontrol can analyze all the output
exec 2>&1

# Set the cleanup function
trap cleanup EXIT
trap "err_exit $EFAIL \"Script Interrupted\"" SIGHUP SIGINT SIGQUIT SIGTERM

process_args "$@"

import_isa_data

init_csr_variables

if $SET_SN_PARAMETER
then
    if [ ! -z "${SN_COUNTRY}" ]; then
        log "Setting country initials"
        set_parameter subjectName country "${SN_COUNTRY}"
    elif [ ! -z "${SN_STATE}" ]; then
        log "Setting state name"
        set_parameter subjectName state "${SN_STATE}"
    elif [ ! -z "${SN_LOCALITY}" ]; then
        log "Setting locality"
        set_parameter subjectName locality "${SN_LOCALITY}"
    elif [ ! -z "${SN_ORGANIZATION}" ]; then
        log "Setting organization"
        set_parameter subjectName organization "${SN_ORGANIZATION}"
    elif [ ! -z "${SN_ORGANIZATIONUNIT}" ]; then
        log "Setting organization unit"
        set_parameter subjectName organizationunit "${SN_ORGANIZATIONUNIT}"
    elif [ ! -z "${SN_COMMONNAME}" ]; then
        log "Setting common name"
        set_parameter subjectName commonname "${SN_COMMONNAME}"
    elif [ ! -z "${SN_EMAIL}" ]; then
        log "Setting email"
        set_parameter subjectName email "${SN_EMAIL}"
    fi
elif $SET_SAN_PARAMETER
then
    log "Setting subjectAltName"
    if [ ! -z "${SAN_ADD}" ]; then
        set_parameter subjectAltName "SAN_ADD" "${SAN_ADD}"
    elif [ ! -z "${SAN_DEL}" ]; then
        set_parameter subjectAltName "SAN_DEL" "${SAN_DEL}"
    fi
elif $DISPLAY_CSR_CNF
then
    display_csr_cnf
elif $IMPORT_CERT
then
    log "Requesting certificate import"
    import_certificate
elif $GENERATE_CSR
then
    log "Requesting generation of a certificate sign request"
    generate_csr
elif $SET_AUTO_FLAG
then
    log "Setting auto flag"
    set_auto_flag "${AUTO_FLAG_VALUE}"
elif $DISPLAY_AUTO_FLAG
then
    display_auto_flag
elif $CLEANUP_ACTION
then
    cleanup_action
else
    usage
    err_exit $EINVAL
fi
