#!/usr/bin/perl

#
# Copyright 2012 Avaya Inc. All Rights Reserved.
#

# Usage:        del_persistent_route.pl <dest> <mask> <dev> [<metric>]
# Purpose:      Delete a persistent route from a CentOS 6.x machine
#
# Description for command line arguments:
# dest    - the address of the destination network or host
# mask    - the netmask for destination
# dev     - the outgoing network inteface for the route
# metric  - the metric for the toute
#
# NOTEs:
#  - If no network device is provided by the caller, all persistent route files will be processed

# Pick custom modules from specific path
use lib '/opt/Avaya/scripts';
use File::Copy;

use strict;

use RouteUtils;

use vars qw($shortName $usage $dst $dev @devList $metric $prefix);

# Main execution thread starts here
$shortName = eval { my @path = (split "/", $0) ; pop @path; };

$usage = <<EOF;
  Usage: $shortName <dest> <mask> <dev> [<metric>]
EOF

open(my $fLog, '>>', '/opt/ipoffice/network_changes.log');
my $datestring = localtime();
my $strParam = "";
foreach my $argnum (0 .. $#ARGV) {
  $strParam =  "$strParam $ARGV[$argnum]";
}
print $fLog "$datestring: del_persistent_route-> $strParam\n";
close $fLog;

if ((@ARGV != 4) && (@ARGV != 3)) {
  print STDERR "Invalid number of arguments!\n";
  print STDERR $usage;
  exit 1;
}

# Validate destination and mask
$dst = new NetAddr::IP ($ARGV[0], $ARGV[1]);
unless (defined($dst)) {
  print STDERR "Could not parse network from destination \"$ARGV[0]\" and netmask \"$ARGV[1]\"\n";
  exit 2;
}


# Extract the string representation of the network prefix/host address
if ($dst->masklen() == 32) {
  # Only the address is actually needed for host routes
  $prefix=$dst->addr();
} else {
  $prefix=$dst->network();
}

# If a proper network device was provided, use it; otherwise all persistent route
# files will be searched
if ($ARGV[2] =~ m/\S/) {
  push @devList, $ARGV[2];
} else {
  # Build device list, based on the existing files
  @devList = map {s/$routeFilePrefix//;$_} glob "$routeFilePrefix*";
}

# Retrieve route metric, if provided
if (defined($ARGV[3]) && ($ARGV[3] =~ m/\S/)) {
  unless ($ARGV[3] =~ m/\d+/) {
    print STDERR "Invalid route metric \"$ARGV[3]\"\n";
    exit 4;
  }
  $metric = ($ARGV[3] == 0) ? undef : $ARGV[3];
}

getPersistentRouteLock();

# Remove the default route (gateway)
if ($dst eq '0.0.0.0/0' && $metric == 0) {
  my $networkfile = "/etc/sysconfig/network";
  my $newnetworkfile = "/etc/sysconfig/network.new";
  my $line = "";
  if (open(file1, $networkfile) && open(file2, ">$newnetworkfile")) {
    while($line = <file1>) {
      if (($line !~ /^\s*GATEWAY\s*=/) && ($line !~ /^\s*GATEWAYDEV\s*=/) ) {
        print file2 $line;
      }
    }
    close(file2);
    close(file1);
    move($newnetworkfile, $networkfile);
  }
  chmod 0664, $networkfile;
  releasePersistentRouteLock();
  exit 0;
}

foreach $dev (@devList) {
  # Remove route if match found in device specific file
  my $matched = lookupPersistentRoute($prefix, $metric, $dev);
  if ($matched) {
    delPersistentRoute($prefix, $dev, $metric);
  }
}

releasePersistentRouteLock();
