#!/bin/bash

# Backup node specific configuraton and application databases as either archive or directory tree
# under local folder or archive on Backup Server.
#
# When backing up to local folder, creating a directory tree rather than an archive at
# the specified location is controlled by providing the "-d" command line option

# Usage:
# ipol_backup.sh [-d] <backup URL>

#URL format
#scp_URI  = "scp://userinfo@host:port/abs_path"   or scp_URI  = "scp://host:port/abs_path"
#sftp_URI = "sftp://userinfo@host:port/abs_path"  or sftp_URI = "sftp://host:port/abs_path"
#http_URI = "http://host:port/abs_path            or https_URI = "https://host:port/abs_path
#file_URL = "file:///abs_path"

#set -x

BASEDIR=$(dirname $0)
. $BASEDIR/ipol_common.sh

fileList="$BASEDIR/ipol_fileList"

while getopts :d:s opt; do
  case $opt in
    d)
      wantDirectoryTree=1
      ;;
    s)
      systemBackup=1
      ;;
    \?)
      echo "Invalid option: -$OPTARG" >&2
      ;;
  esac
done

# Done with command line options, discarding them
shift $((OPTIND-1))

#Create tar.gz archive 
#dstArchive      - folder where archive will be created
#srcFiles        - file list archive
function build_archive () {

    if [ "$urlPROTO" == "file" ]; then
        if [ $systemBackup == 1 ]; then
            archiveName=Linux_Backup.tar.gz
        else
            archiveName=ABE_Backup.tar.gz
        fi
    else
        archiveName=ABE_Backup_tmp.tar.gz
    fi

    srcFiles=$fileList
    if [ -z "$1" ]; then
        dstArchive=$br_path
    else
        dstArchive=$1
    fi

    if [ $systemBackup == 0 ]; then
        if [ -f "$bvmproScript" -a -d "$br_vmproLocation" ]; then

             rm -rf $br_vmproLocation/*
             $bvmproScript $br_vmproLocation &>> $br_path/$backupLog
             if (($? != 0)); then
                 br_LogEvent "$brinfoLog" "Error-Backup: Vmpro backup failed." "$br_path/$backupLog"
             fi
        fi

        #avoid failure of tar operation
        if [ ! -d $oneXbackupLocation ]; then
             br_LogEvent "$brinfoLog" "Warning-Backup: oneXportal path not found.Create $oneXbackupLocation" "$br_path/$backupLog"
             mkdir -p $oneXbackupLocation
        fi
    fi

    if [ $wantDirectoryTree == 0 ]; then
        IFS='\n' printf '%s\n' $(cat $srcFiles) | grep -v "\*"|tar -czv  -f $dstArchive/$archiveName --files-from -
    else
        IFS='\n' printf '%s\n' $(cat $srcFiles) | grep -v "\*"|tar -cf - --files-from - | tar -C $dstArchive -p --same-owner --atime-preserve -xvf -
    fi

    if (($? != 0)); then
        br_LogEvent "$brinfoLog" "Error-Backup: Create tar archive failed." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Create tar archive failed." "$br_path/$bStatusFile"
        if [ -d "$br_vmproLocation" ]; then
            rm -rf $br_vmproLocation/*
        fi
        if [ -f $dstArchive/$archiveName ]; then
            br_LogEvent "$brinfoLog" "Error-Backup: Archive created does not contain all expected files. It will be deleted!" "$br_path/$backupLog"
            rm -rf $dstArchive/$archiveName
        fi
        remove_lock $operation
        exit $BR_ERRORCODE_ARCHIVE_FAILEDTOCREATE
    fi

    if [ $systemBackup == 1 ]; then
        if [ -f $br_metadata ]; then
             rm -rf $br_metadata
        fi

        rfile=$se_rf
        if [ -f "$apc_rf" ]; then
            rfile=$apc_rf
        elif [ -f $appl_rf -a ! -f $se_rf ]; then
            rfile=$appl_rf
        fi
        
        br_getVersionFromReleaseFile $rfile
        #create metadata file
        touch $br_metadata 
        echo "Linux" > $br_metadata
        echo "$release_version" >> $br_metadata
        echo "You may run the following command in order to check content of archive:" >> $br_metadata
        echo "tar -tvf archivename" >> $br_metadata
    fi
    #remove any vmpro backed files
    if [ $systemBackup == 0 -a -d "$br_vmproLocation" ]; then
        rm -rf $br_vmproLocation/*
    fi
}

#Copy archive file to remote server using scp
#arg1 - MD5 of  archive file
#arg2 - Fullpath filename copy to
#arg3 - Backup server username account
#arg4 - Backup server password account
#arg5 - Backup server IP Address
#arg6 - Backup server storage location
#arg7 - Operation
function copy_tarball2remote () {
       
    #compute md5 for archive file
    md5sumfile="md5sumfile"
    if [ -f "$br_path/$md5sumfile" ]; then
        rm -rf $br_path/$md5sumfile
    fi
    if [ -f "$br_path/$bStatusFile" ]; then
        rm -rf $br_path/$bStatusFile
    fi

    echo "`md5sum $1 | awk '{ print $1}'`" > "$br_path/$md5sumfile"
    #send files to backup server
    python $BASEDIR/ipol_copy.py $br_path/$md5sumfile $1 $2 $3 $4 $5 $operation &>$br_path/$bmessageRcv
    if (($? != 0)); then
        eventrcvd=`head -n 1 $br_path/$bmessageRcv`
        br_LogEvent "$brinfoLog" "Error-Backup: Failed copy file to $operation Server $4 {$eventrcvd}" "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Failed copy file to $operation Server $4 {$eventrcvd}" "$br_path/$bStatusFile"
        if [ -f $br_path/$bmessageRcv ]; then
            rm -rf $br_path/$bmessageRcv
        fi
        remove_lock $operation
        exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
    else
        check_KeyVerificationFailed $br_path/$bmessageRcv $br_path/$backupLog
    fi

    here="."
    #get status transfer from backup server
    python $BASEDIR/ipol_copy.py  $2 $3 $4 $5/$bStatusFile $here $operation &>$br_path/$bmessageRcv
    if (($? != 0)); then
        eventrcvd=`head -n 1 $br_path/$bmessageRcv`
        br_LogEvent "$brinfoLog" "Error-Backup: Files transfer failed {$eventrcvd}" "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Files transfer failed {$eventrcvd}" "$br_path/$bStatusFile"
        if [ -f $br_path/$bmessageRcv ]; then
            rm -rf $br_path/$bmessageRcv
        fi
        remove_lock $operation
        exit $BR_ERRORCODE_FILE_FAILEDTOCOPY
    else
        check_KeyVerificationFailed $br_path/$bmessageRcv $br_path/$backupLog
    fi

    if [ ! -s ./$bStatusFile ]; then
        br_LogEvent "$brinfoLog" "Error-Backup: Get backup status file with size 0." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Get backup status file with size 0." "$br_path/$bStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_FILE_INVALIDSIZE
    else
        message=`cat ./$bStatusFile`
        if [[ "$message" =~ "failed" ]]; then
              br_LogEvent "$brinfoLog" "Error-Backup: Message received from remote server {$message}" "$br_path/$backupLog"
              mv -f ./$bStatusFile $br_path/$bStatusFile
              br_LogEvent "$brinfoStatus" "Backup failed. Message received from remote server {$message}" "$br_path/$bStatusFile"
              remove_lock $operation
              exit $BR_ERRORCODE_FAILURE_MSGRCV
        fi
    fi

    mv -f ./$bStatusFile $br_path/$bStatusFile
    if (($? != 0)); then
        br_LogEvent "$brinfoLog" "Error-Backup: Get backup status file failed." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Get backup status file failed." "$br_path/$bStatusFile"
    else
        br_LogEvent "$brinfoLog" "Info-Backup: $operation successful created on server $4." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup succeeded. $operation successful created on server $4." "$br_path/$bStatusFile"
    fi
}


#Main execution
operation="Backup"

##check if AuthModule files should be included in backup and update fileList accordingly
if [[ -n `rpm -q AuthModule | grep "not installed"` ]]; then
    #remove AuthModule files from fileList
    if [[ -n `cat $fileList | grep $AuthModuleBackupFilesPath` ]]; then
        sed "/$EscapedAuthModuleBackupFilesPath/d" $fileList 
    fi
else
    #append AuthModule files if not present
    if [[ -z `cat $fileList | grep $AuthModuleBackupFilesPath ` ]]; then
        echo "$AuthModuleBackupFilesPath" >> $fileList
    fi
fi

if [ ! -d "$br_path" ]; then
    mkdir "$br_path"
fi

#Create lock file for backup operation
#Only one instance at a time is permited
if [ ! -f "$br_path/ipol_backup_restore.lock" ]; then
        touch $br_path/ipol_backup_restore.lock
else
        br_LogEvent "$brinfoLog" "Error-Backup: Only one backup/restore instance is allowed. Exit." "$br_path/$backupLog"
        exit $BR_ERRORCODE_LOCK_TAKEN
fi

if [ -f "$br_path/$bStatusFile" ]; then
        rm -rf $br_path/$bStatusFile
fi

if [ $systemBackup == 1 ]; then
    #local backup.
    bURL="file:///$br_systemLocation"
    if [ ! -d $br_systemLocation ]; then
         mkdir -p $br_systemLocation
    fi
else
    bURL=$1
    if [ -z "$bURL" ]; then
         br_LogEvent "$brinfoLog" "Error-Backup: Empty url found." "$br_path/$backupLog"
         br_LogEvent "$brinfoStatus" "Backup failed. Empty url found." "$br_path/$bStatusFile"
         remove_lock $operation
         exit $BR_ERRORCODE_URL_EMPTY
    fi

    USERNAME=$2
    USERPASSWORD=$3
fi

#get url protocol|url server|url absolut path 
parse_url  $operation $bURL

if [ -f $br_path/parse.ini ]; then
    . $br_path/parse.ini
    rm -f $br_path/parse.ini
else
    br_LogEvent "$brinfoLog" "Error-BAckup: Parse url failed." "$br_path/$backupLog"
    br_LogEvent "$brinfoStatus" "Backup failed. Parse url failed." "$br_path/$bStatusFile"
    remove_lock $operation
    exit $BR_ERRORCODE_URL_PARSE
fi

if [ -z "$urlPROTO" ]; then
    br_LogEvent "Error-Backup: Empty urlPROTO found." "$br_path/$backupLog"
    br_LogEvent "Backup failed. Empty urlPROTO found." "$br_path/$bStatusFile"
    remove_lock $operation
    exit $BR_ERRORCODE_URL_PROTOCOL_EMPTY
fi

#change url protocol to lowercase        
urlPROTO=`echo $urlPROTO | tr A-Z a-z`
case "$urlPROTO" in
    "scp" | "sftp")

    if [ -z "$USERNAME" -o -z "$USERPASSWORD" ]; then
        br_LogEvent "$brinfoLog" "Error-Backup: No credentials found in arguments." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. No credentials found in arguments." "$br_path/$bStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_CREDENTIALS_MISMATCH
    fi

    if [ ! -z "$urlIPADDRESS" ]; then
        #verify conectivity with backup server
        ping -c 1 $urlIPADDRESS
        if (($? != 0)); then
            br_LogEvent "$brinfoLog" "Warning-Backup: Conectivity with {$urlIPADDRESS} server failed!" "$br_path/$backupLog"
        fi

        #create tarball archive
        build_archive 

        #copy backup archive to remote server
        copy_tarball2remote $br_path/$archiveName $USERNAME $USERPASSWORD $urlIPADDRESS $urlPATH
    else
        br_LogEvent "$brinfoLog" "Error-Backup: Empty urlIPADDRESS found." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Empty urlIPADDRESS found." "$br_path/$bStatusFile"
    fi
    ;;
    "file")
    if [ $wantDirectoryTree == 0 -a $systemBackup == 0 ]; then
        #Local backup is permited only in user's home directory(default user is Administrator)
        if [ -z "$USERNAME" ]; then
            USERNAME="Administrator"
        fi

        permitedpath=`getent passwd $USERNAME | cut -d':' -f6`
        if (($? != 0)); then
            br_LogEvent "$brinfoLog" "Error-Backup: Cannot resolve home user path." "$br_path/$backupLog"
            br_LogEvent "$brinfoStatus" "Backup failed. Cannot resolve home user path." "$br_path/$bStatusFile"
            remove_lock $operation
            exit $BR_ERRORCODE_USERPATH_NOTFOUND
        fi
        if [ -z "$permitedpath" ]; then
            br_LogEvent "$brinfoLog" "Error-Backup: Not valid home user path." "$br_path/$backupLog"
            br_LogEvent "$brinfoStatus" "Backup failed. Not valid home user path." "$br_path/$bStatusFile"
            remove_lock $operation
            exit $BR_ERRORCODE_USERPATH_INVALID
        fi
    else
        # Backing up as a directory was introduced to assist IP Office remote backup,
        # that can only access files under the path below
        permitedpath="/opt/ipoffice/"
    fi

    #configurated path
    if [ -z "$urlPATH" -o "$urlPATH" == "/" -o "$urlPATH" == ":" ]; then
        configuratedpath=$permitedpath
    elif [ ${#urlPATH} -ge 2 -o $systemBackup == 1 ]; then
        configuratedpath=$urlPATH
    else
        configuratedpath=""
    fi
    if [[ ! $configuratedpath =~ ^[\/]*$permitedpath ]]; then
        br_LogEvent "$brinfoLog" "Error-Backup: Local backup is allowed only in user's home directory." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Local backup is allowed only in user's home directory." "$br_path/$bStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_ACCESSDENIED
    fi

    if [ ! -d $configuratedpath ]; then
        br_LogEvent "$brinfoLog" "Error-Backup: Folder {$configuratedpath} does not exist." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Folder {$configuratedpath} does not exist." "$br_path/$bStatusFile"
        remove_lock $operation
        exit $BR_ERRORCODE_FOLDER_DOESNOTEXIST
    fi

    #create and copy backup archive to local server
    build_archive $configuratedpath
        
    if [ $wantDirectoryTree == 0 ]; then
        #restore user rights
        if [ "$USERNAME" != "root" ]; then
            chown $USERNAME:$USERNAME $configuratedpath/$archiveName
            chmod g+rw $configuratedpath/$archiveName
            if (($? != 0)); then
                br_LogEvent "$brinfoLog" "Error-Backup: Set rights failed on {$configuratedpath/$archiveName}." "$br_path/$backupLog"
                br_LogEvent "$brinfoStatus" "Backup failed. Set rights failed." "$br_path/$bStatusFile"
                remove_lock $operation
                exit $BR_ERRORCODE_ACCESSDENIED
            fi
        fi

        if [ ! -f $configuratedpath/$archiveName ]; then
            br_LogEvent "$brinfoLog" "Error-Backup: Backup failed. No archive created to configured path." "$br_path/$backupLog"
            br_LogEvent "$brinfoStatus" "Backup failed. No archive created to configured path." "$br_path/$bStatusFile"
        else
            br_LogEvent "$brinfoLog" "Info-Backup: Successful archive created to local server {$configuratedpath}." "$br_path/$backupLog"
            br_LogEvent "$brinfoStatus" "Backup succeeded. Successful archive created to local server {$configuratedpath}." "$br_path/$bStatusFile"
        fi
    else
            br_LogEvent "$brinfoLog" "Info-Backup: local backup set successfully created at {$configuratedpath}." "$br_path/$backupLog"
            br_LogEvent "$brinfoStatus" "Backup succeeded. Local backup set created at {$configuratedpath}." "$br_path/$bStatusFile"
    fi
    ;;
    "http" | "https")
            br_LogEvent "$brinfoLog" "Error-Backup: Upload files using {$urlPROTO} not supported yet." "$br_path/$backupLog"
            br_LogEvent "$brinfoStatus" "Backup failed. Upload files using {$urlPROTO} not supported yet." "$br_path/$bStatusFile"
            ;;
    *)
        br_LogEvent "$brinfoLog" "Error-Backup: Protocol {$urlPROTO} not supported." "$br_path/$backupLog"
        br_LogEvent "$brinfoStatus" "Backup failed. Protocol {$urlPROTO} not supported." "$br_path/$bStatusFile"
        ;;
esac
remove_lock $operation
