#!/bin/bash

# ACTIONS
#    --backup
#    --restore
#    --check

# OPTIONS
#    --path

# DEFAULT PARAMETERS
BACKUP_PATH=/opt/Avaya/Backup_Restore/SecureBackup

# Actions
BACKUP_ACT=false
RESTORE_ACT=false
CHECK_ACT=false

# Help function
usage() {
cat << EOF
usage: $0 <action> <option>

This script provides secured backup functionalities to IPOL

ACTIONS:
    --backup		Flag specific path to be securely backed up
    --restore		Restores a specific backed up file/directory
    --check		Checks if a specific path needs to be updated

OPTIONS:
    --path		Specific path flagged to be backed up
    --file		File to be restored; this file has to be in the restore directory
    --backupname        The name of the archive

Exaples:
1. ./secure_backup.sh --backup --path=/opt/var/www/test.txt
2. ./secure_backup.sh --check --path=/opt/Avaya/BackDir/ca.tar.gz
3. ./secure_backup.sh --restore --file=ca.tar.gz
EOF
}

err_exit() {
    echo $2
    exit $1
}

init_error_codes() {
    ESUCCESS=0		# Success
    EFAIL=1             # General Error
    EDIFFER=2		# Check differs
    ENOTFOUND=3		# Checked file not found
    ECORRUPTARCH=4	# Corrupt archive?
    EACCESS=5           # Access error
    EINVAL=6            # Invalid argument
}

backup_procedure() {
    BACKUP_DIR="$BACKUP_PATH"/"backup"
    CURRENT_DIR=`pwd`
    if [ ! -d $BACKUP_DIR ]; then
        mkdir $BACKUP_DIR
    fi
    if [ -f $BCKP_PATH ]; then
        FILE_MD5SUM=`md5sum $BCKP_PATH|cut -d" " -f1`
        FILE_NAME=${BCKP_PATH##*/}
        ARCH_NAME=${FILE_NAME%.*}
        FILE_PATH=${BCKP_PATH%/*}
        BCKP_FILE_NAME="$ARCH_NAME"".tar.gz"
        cd $BACKUP_PATH
        if [ -f "$BACKUP_DIR/$BCKP_FILE_NAME" ]; then
            cd $CURRENT_DIR
            err_exit $EACCESS "Backed up file exists"
        fi
        echo "md5sum=$FILE_MD5SUM" > info.txt
        echo "path=$FILE_PATH" >> info.txt
        cd $BACKUP_DIR
        tar -C "$FILE_PATH" -zcvf "$BCKP_FILE_NAME" "$FILE_NAME" "$BACKUP_PATH/info.txt" > /dev/null 2>&1
        rm -f "$BACKUP_PATH/info.txt"
        if [ -f "$BCKP_FILE_NAME" ]; then
            err_exit $ESUCCESS "Backup successfull"
        fi
    elif [ -d $BCKP_PATH ]; then
        # testing for slash to the end of the directory name
        dir_size=${#BCKP_PATH}
        let slash_pos=dir_size-1
        if [ "${BCKP_PATH:${slash_pos}:${dir_size}}" == "/" ]; then
            BCKP_PATH="${BCKP_PATH%?}"
        fi
        DIR_NAME=${BCKP_PATH##*/}
        DIR_MD5SUM=`tar -cPf - "$BCKP_PATH"|md5sum|cut -d" " -f1`
        DIR_PATH=${BCKP_PATH%/*}
        BCKP_FILE_NAME="$DIR_NAME"".tar.gz"
        cd $BACKUP_PATH
        if [ -f "$BACKUP_DIR/$BCKP_FILE_NAME" ]; then
            cd $CURRENT_DIR
            err_exit $EACCESS "Backed up file exists"
        fi
        echo "md5sum=$DIR_MD5SUM" > info.txt
        echo "path=$DIR_PATH" >> info.txt
        cd $BACKUP_DIR
        tar -C "$DIR_PATH" -zcvf "$BCKP_FILE_NAME" "$DIR_NAME" "$BACKUP_PATH/info.txt" > /dev/null 2>&1
        rm -f "$BACKUP_PATH/info.txt"
        if [ -f "$BCKP_FILE_NAME" ]; then
            err_exit $ESUCCESS "Backup successfull"
        fi
    fi
}

check_procedure() {
    RESTORE_PATH="$BACKUP_PATH"/"restore"
    CURRENT_DIR=`pwd`
    if [ ! -d $RESTORE_PATH ]; then
        mkdir $RESTORE_PATH
    fi
    if [ -f $BCKP_PATH ]; then
        DIR_PATH=${BCKP_PATH%/*}
        FILE_NAME=${BCKP_PATH##*/}
        CHK_FILE_PATH="$RESTORE_PATH"/"$FILE_NAME"
        if [ ! -f $CHK_FILE_PATH ]; then
            cp $BCKP_PATH $CHK_FILE_PATH
        fi
        cd $RESTORE_PATH
        FILES_IN_DIR=`ls`
        for fl in $FILES_IN_DIR
        do
            if [ "$fl" != "$FILE_NAME" ]; then
                rm -rf $fl
            fi
        done
        # check if this is an archive
        tar -tf $FILE_NAME > /dev/null 2>&1
        if [ "$?" != "0" ]; then
            rm -f $FILE_NAME
            err_exit $ECORRUPTARCH "Corrupt backup archive, it does not match the known format"
        fi
        INFO=`tar -O -xf $FILE_NAME "opt/Avaya/Backup_Restore/SecureBackup/info.txt"`
        tar --exclude='info.txt' -zxvf $FILE_NAME > /dev/null 2>&1
        rm -f $FILE_NAME
        info_array=(${INFO//\n/ })
        for i in "${info_array[@]}"
        do
            field=`echo $i|cut -d"=" -f1`
            if [ "$field" == "md5sum" ]; then
                MD5SUM=`echo $i|cut -d"=" -f2`
            elif [ "$field" == "path" ]; then
                SEARCHED_PATH=`echo $i|cut -d"=" -f2`
            fi
        done
        if [ ! -z "$MD5SUM" ]; then
            ARCH_FILE_NAME=`ls`
            ARCH_FILES_NUMBER=`ls|wc -l`
            CURRENT_FILE="$SEARCHED_PATH/$ARCH_FILE_NAME"
            if [ "$ARCH_FILES_NUMBER" == "1" ]; then
                if [ -f $ARCH_FILE_NAME ]; then
                    # backed up file is actually a file
                    # testing if the file has changed
                    if [ ! -f $CURRENT_FILE ]; then
                        # the file does not exist
                        # was it erased?
                        rm -rf $ARCH_FILE_NAME
                        err_exit $ENOTFOUND "File $CURRENT_FILE not found"
                    else
                        CURRENT_MD5SUM=`md5sum $CURRENT_FILE|cut -d" " -f1`
                        if [ "$CURRENT_MD5SUM" == "$MD5SUM" ]; then
                            rm -rf $ARCH_FILE_NAME
                            err_exit $ESUCCESS "File $CURRENT_FILE checks ok"
                        else
                            rm -rf $ARCH_FILE_NAME
                            err_exit $EDIFFER "File $CURRENT_FILE has been changed"
                        fi
                    fi
                elif [ -d $ARCH_FILE_NAME ]; then
                    # backed up file is a directory
                    # testing if the directory has changed
                    if [ ! -d $CURRENT_FILE ]; then
                         # the directory does not exist
                         # what happened?
                         rm -rf $ARCH_FILE_NAME
                         err_exit $ENOTFOUND "Directory $CURRENT_FILE not found"
                    else
                        CURRENT_MD5SUM=`tar -cPf - "$CURRENT_FILE"|md5sum|cut -d" " -f1`
                        if [ "$CURRENT_MD5SUM" == "$MD5SUM" ]; then
                            rm -rf $ARCH_FILE_NAME
                            err_exit $ESUCCESS "Directory $CURRENT_FILE checks ok"
                        else
                            rm -rf $ARCH_FILE_NAME
                            err_exit $EDIFFER "Directory $CURRENT_FILE has been changed"
                        fi
                    fi
                else
                    err_exit $EFAIL "File is not a normal file or directory"
                fi
            else
                err_exit $EFAIL "Multiple files in checked archive"
            fi
        else
            err_exit $ECORRUPTARCH "Cannot extract the md5sum from checked archive"
        fi
    fi
}

restore_procedure() {
    RESTORE_PATH="$BACKUP_PATH"/"restore"
    CURRENT_DIR=`pwd`
    if [ ! -d $RESTORE_PATH ]; then
        err_exit $EACCESS "Cannot find the restore directory"
    fi
    cd $RESTORE_PATH
    if [ ! -f $RESTORE_FILE ]; then
        err_exit $EACCESS "Cannot find the restore file"
    fi
    # testing the itegrity of the archive
    tar -tf $RESTORE_FILE > /dev/null 2>&1
    if [ "$?" != "0" ]; then
        err_exit $ECORRUPTARCH "Corrupt backup archive, cannot decrypt"
    fi

    INFO=`tar -O -xf $RESTORE_FILE "opt/Avaya/Backup_Restore/SecureBackup/info.txt"`
    tar --exclude='info.txt' -zxvf $RESTORE_FILE > /dev/null 2>&1
    info_array=(${INFO//\n/ })
    for i in "${info_array[@]}"
    do
        field=`echo $i|cut -d"=" -f1`
        if [ "$field" == "path" ]; then
            BACKEDUP_PATH=`echo $i|cut -d"=" -f2`
        fi
    done
    if [ -z "$BACKEDUP_PATH" ]; then
        err_exit $ECORRUPTARCH "Corrupt backup archive, cannot find path"
    fi
    if [ ! -d "$BACKEDUP_PATH" ]; then
        err_exit $ECORRUPTARCH "Corrupt backup archive, backed up path does not exist"
    fi
    rm -f $RESTORE_FILE
    ARCH_FILE_NAME=`ls`
    ARCH_FILES_NUMBER=`ls|wc -l`
    if [ "$ARCH_FILES_NUMBER" != "1" ]; then
        err_exit $EFAIL "Multiple files in backup archive"
    fi
    if [ -f $ARCH_FILE_NAME ]; then
        rm -f "$BACKEDUP_PATH/$ARCH_FILE_NAME"
        cp "$ARCH_FILE_NAME" "$BACKEDUP_PATH"
        rm -f $ARCH_FILE_NAME
        err_exit $ESUCCESS "Restore completed successfully"
    elif [ -d $ARCH_FILE_NAME ]; then
        CLEAN_PATH="$BACKEDUP_PATH/$ARCH_FILE_NAME""/"
        cd $CLEAN_PATH
        rm -rf *
        cd "$RESTORE_PATH/$ARCH_FILE_NAME"
        cp -r * "$BACKEDUP_PATH/$ARCH_FILE_NAME/"
        cd $RESTORE_PATH
        rm -rf $ARCH_FILE_NAME
        err_exit $ESUCCESS "Restore completed successfully"
    else
        err_exit $EFAIL "Unknown backed up file format"
    fi
}

# Process script arguments
process_args() {
    local ACTIONS_CNT=0

    OPTS=`getopt -o h -l backup -l restore -l check -l path: -l file: -- "$@"`

    eval set -- "$OPTS"

    while true
    do
        case "$1" in
            -h|--help)
                usage
                exit;;
            # Actions
            --backup)
                BACKUP_ACT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --restore)
                RESTORE_ACT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            --check)
                CHECK_ACT=true
                ACTIONS_CNT=$((ACTIONS_CNT + 1))
                shift;;
            # Options
            --path)
                BCKP_PATH="$2"
                shift 2;;
            --file)
                RESTORE_FILE="$2"
                shift 2;;
            --)
                shift;
                break;;
            *)
                usage
                err_exit $EINVAL "Invalid option $1";;
        esac
    done

    # No further arguments are required
    if [ $# -ne 0 ]
    then
        usage
        err_exit $EINVAL "Invalid argument: $@"
    fi

    # Only one action at a time
    if [ "$ACTIONS_CNT" -ne "1" ]
    then
        usage
        err_exit $EINVAL "Multiple or no actions specified"
    fi

    if $BACKUP_ACT || $CHECK_ACT
    then
        if [ -z $BCKP_PATH ]
        then
            usage
            err_exit $EINVAL "Please provide a path to be backed up"
        fi
    fi

    if $RESTORE_ACT
    then
        if [ -z $RESTORE_FILE ]
        then
            usage
            err_exit $EINVAL "Please provide a file to be restored"
        fi
    fi
}

init_error_codes

# Execute the script as root
if [ "$UID" -ne 0 ]
then
    err_exit $EACCESS "Please run as root"
fi

process_args "$@"

if $BACKUP_ACT
then
    backup_procedure
elif $RESTORE_ACT
then
    restore_procedure
elif $CHECK_ACT
then
    check_procedure
fi
