using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
using System.Threading;
using System.Runtime.InteropServices;

using AxisMediaViewerLib;
using AxisMediaParserLib;


// This sample shows how to use the AXIS Media Viewer component, where a thread reads
// media frames from a file (e.g. created in the MediaParser sample) and lets the viewer component
// render them. Please see the documentation for possible values and error codes.
class VideoForm : Form
{
	// The render thread that parses the file and renders each frame
	private Thread renderThread;

	private const string FilePath = "C:\\Axis\\Rec_18_15_9.bin";   // file in current directory to read from
	bool closing = false;


	static void Main()
	{
		Application.EnableVisualStyles();
		Application.SetCompatibleTextRenderingDefault(false);
		Application.Run(new VideoForm());
	}


	public VideoForm()
	{
		InitializeComponent();

	}

	// This event handler is called after the dialog is shown, since we need the
	// win32 window handle as a parameter to the Viewer
	private void VideoForm_Load(object sender, EventArgs e)
	{
		// Create the render thread that will read file and render frames
		renderThread = new Thread(new ParameterizedThreadStart(RenderThread));

		// Set MTA apartment to allow multiple threads (UI and render thread) to access COM object
		renderThread.SetApartmentState(ApartmentState.MTA);

		// Start the render thread and send in the Win32 handle to this form
		renderThread.Start(this.Handle);
	}


	// A thread started in VideoForm() and reads a binary media file and renders the frames in the
	// viewer
	private void RenderThread(object obj)
	{
		// Get the hWnd windows handle provided in VideoForm() method above
		IntPtr hWnd = (IntPtr)obj;

		// Create the AXIS Media Viewer COM object
		AxisMediaViewerClass viewer = new AxisMediaViewerClass();

		try
		{
			// Set some media properties
			viewer.VMR9 = false;
			viewer.ColorSpace = (short)AMV_COLOR_SPACE.AMV_CS_RGB32;

			// Open binary input file to read parsed video frames from
			using (FileStream inFileStream = new FileStream(FilePath, FileMode.Open))
			using (BinaryReader inFile = new BinaryReader(inFileStream, System.Text.Encoding.UTF32))
			{
				// Read media type information from inFile
				int mediaTypeSize = inFile.ReadInt32();
				byte[] mediaTypeBuffer = inFile.ReadBytes(mediaTypeSize);

				// Init the Viewer with media type buffer and a win32 hWnd handle to the window. 
				viewer.Init(0, mediaTypeBuffer, hWnd.ToInt64());

				// Get the video frame size and resize the form to fit the video stream 
				int width;
				int height;
				viewer.GetVideoSize(out width, out height);
				ResizeFormToFitVideoSize(width, height);

				// Start the viewer before rendering the first frame
				viewer.Start();

				// While there are frames in the in file, read each frame and let the viewer render it
				while (inFile.PeekChar() != -1 && !closing)
				{
					// Read frame data
					int sampleType = inFile.ReadInt32();
					int sampleFlags = inFile.ReadInt32();
					ulong startTime = inFile.ReadUInt64();
					ulong stopTime = inFile.ReadUInt64();
					int bufferSize = inFile.ReadInt32();
					byte[] bufferBytes = inFile.ReadBytes(bufferSize);

					// Let the viewer render the video or audio frame
					if (sampleType != (int)AMP_VIDEO_SAMPLE_TYPE.AMP_VST_MPEG4_AUDIO)
					{
						viewer.RenderVideoSample(sampleFlags, startTime, stopTime, bufferBytes);
					}
					else
					{
						viewer.RenderAudioSample(sampleFlags, startTime, stopTime, bufferBytes);
					}
				}

				viewer.Stop();

				// Inform the GC that COM object no longer will be used
				Marshal.FinalReleaseComObject(viewer);
				viewer = null;
			}
		}
		catch (COMException e)
		{
			MessageBox.Show(string.Format("Exception for {0}, {1}", FilePath, e.Message));
		}
	}


	// Function and delegate to resize the form from the background render thread.
	private delegate void ResizeFormToFitVideoSizeDelegate(int width, int height);

	private void ResizeFormToFitVideoSize(int width, int height)
	{
		if (InvokeRequired)
		{
			// If called from a non UI thread, let the UI thread perform the call 
			Invoke(new ResizeFormToFitVideoSizeDelegate(ResizeFormToFitVideoSize), 
				new object[] { width, height });
			return;
		}

		ClientSize = new System.Drawing.Size(width, height);
	}

	private void VideoForm_FormClosing(object sender, FormClosingEventArgs e)
	{
		closing = true;
		renderThread.Join();
	}

	#region Windows Form Designer generated code
	
	// Required designer variable.
	private System.ComponentModel.IContainer components = null;

	// Clean up any resources being used.
	protected override void Dispose(bool disposing)
	{
		if (disposing && (components != null))
		{
			components.Dispose();
		}
		base.Dispose(disposing);
	}


	/// <summary>
	/// Required method for Designer support - do not modify
	/// the contents of this method with the code editor.
	/// </summary>
	private void InitializeComponent()
	{
		this.SuspendLayout();
		// 
		// VideoForm
		// 
		this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 13F);
		this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
		this.ClientSize = new System.Drawing.Size(292, 266);
		this.Name = "VideoForm";
		this.Text = "VideoForm";
		this.FormClosing += new System.Windows.Forms.FormClosingEventHandler(this.VideoForm_FormClosing);
		this.Load += new System.EventHandler(this.VideoForm_Load);
		this.ResumeLayout(false);

	}

	#endregion

}
