/** \MediaParserFileSourceDecodeC++.cpp
* 
* Defines the entry point for the console application.
*
* (C) Copyright 2010, Axis Communications AB, LUND, SWEDEN
****************************************************************************/


/****************** INCLUDE FILES SECTION ***********************************/

#include <Windows.h>
#include <tchar.h>
#include <iostream>
#include <time.h>

#include <Streams.h>
#include <dmo.h>

#include "initguid.h"
#include "../Interfaces/AxH264Dec.h"
#include "../Interfaces/AxMP4Dec.h"

#include "MediaBuffer.h"
#include "AMPFileSource.h"

/****************** DEFINITION SECTION **************************************/

#define MAX_NUMBER_OF_STREAMS 2

// DMO CLSIDs
const CLSID CLSID_G7xxDecoder = { 0xD1F283E1, 0xDF6F, 0x41e4,
                                 { 0xAE, 0xD1, 0xEB, 0x1D, 0x6B, 0x4B, 0x0A, 0xE1 } }; // Audio.dll
const CLSID CLSID_AACDecoder = { 0XBA7A56EB, 0XD1B9, 0X443B,
                               { 0X96, 0XE9, 0X8, 0X65, 0X32, 0XA3, 0X78, 0XF1 } };
const CLSID CLSID_MPEG4Decoder = { 0XC32FE9F1, 0XA857, 0X48B0,
                                 { 0XB7, 0XBF, 0X6, 0X5B, 0X57, 0X92, 0XF2, 0X8D} };
const CLSID CLSID_H264Decoder = { 0x7340F0E4, 0xAEDA, 0x47C6,
                                {0x89, 0x71, 0x9D, 0xB3, 0x14, 0x3, 0xB, 0xD7 } };
const CLSID CLSID_MJPEGDecoder = { 0X29020FE0, 0X3833, 0X4BE2,
                                 { 0XA7, 0XF0, 0X42, 0X84, 0X1D, 0X4B, 0X45, 0XF0} };

// Custom media subtype GUIDs
#define FOURCC_H264 mmioFOURCC('H','2','6','4')
#define FOURCC_DX50 mmioFOURCC('D','X','5','0') 

const GUID GUID_H264 = 
{FOURCC_H264, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
const GUID GUID_DX50 = 
{FOURCC_DX50, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
const GUID GUID_AAC = 
{ 0x000000ff, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
const GUID GUID_G711 = //preferred
{ 0x00000007, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
const GUID GUID_G726_32 = //preferred
{ 0x00000014, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
const GUID GUID_G726_24 = //preferred
{ 0x00000040, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
// Alternative G7xx GUIDs
const GUID GUID_G711_ALT = { 
  0x000001ef, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} };
const GUID GUID_G721 = { //equivalent to G726_32
  0x000002ef, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} }; 
const GUID GUID_G723 = { //equivalent to G726_24
  0x000003ef, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71} }; 

using namespace std;

HRESULT InitializeDecoder(CLSID theDmoClsid, AM_MEDIA_TYPE* theMediaType, IMediaObject** theDecoder, bool useQualitySetting)
{
  HRESULT hr;
  DMO_MEDIA_TYPE anOutputMediaType;

  hr = ::CoCreateInstance(theDmoClsid, NULL, CLSCTX_INPROC,
                          IID_IMediaObject, (LPVOID *)theDecoder);
  if(SUCCEEDED(hr))
  {
    hr = (*theDecoder)->SetInputType(0, (DMO_MEDIA_TYPE*)theMediaType, 0);
    if(SUCCEEDED(hr))
    {
      hr = (*theDecoder)->GetOutputType(0,0, &anOutputMediaType);
      if(SUCCEEDED(hr))
      {
        hr = (*theDecoder)->SetOutputType(0, &anOutputMediaType, 0);
        if(FAILED(hr))
        {
          cout << "Could not set output media type!" << endl;
        }
        MoFreeMediaType(&anOutputMediaType);
      }
      else
      {
        cout << "Could not get output media type!" << endl;
      }
    }
    else
    {
      cout << "Could not set input media type!" << endl;
    }

    if(FAILED(hr))
    {
      (*theDecoder)->Release();
      *theDecoder = NULL;
    }
  }
  else
  {
    cout << "Decoder not installed." << endl;
  }

  if(SUCCEEDED(hr))
  {
    IAxH264Dec2* anAxIH264Dec2 = NULL;
    IAxMP4Dec* anAxMP4Dec = NULL;
    if(SUCCEEDED((*theDecoder)->QueryInterface(IID_IAxH264Dec2, (void**) &anAxIH264Dec2)))
    {
      anAxIH264Dec2->SetQualitySettings(useQualitySetting ? 9 : 0);
      anAxIH264Dec2->Release();
    }
    else if(SUCCEEDED((*theDecoder)->QueryInterface(IID_IAxMP4Dec, (void**)&anAxMP4Dec)))
    {
      anAxMP4Dec->SetDeblock(useQualitySetting ? 1 : 0);
      anAxMP4Dec->Release();
    }
  }

  return hr;
}

HRESULT CreateBuffers(IMediaObject* theDecoder, IMediaBuffer** theInputBuffer, IMediaBuffer** theOutputBuffer)
{
  HRESULT hr;
  DWORD aMinBufferSize, aLookahead, anAlignment;

  // INPUT
  hr = theDecoder->GetInputSizeInfo(0, &aMinBufferSize, &aLookahead, &anAlignment);
  if(SUCCEEDED(hr))
  {
    if(aMinBufferSize > 10000000)
    {
      // Older H.264 decoder doesn't return correct input buffer min size
      // Setting buffer size to a fixed 10 MB
      //
      // A more correct buffer size can be approximated using the image dimension info
      // from the input media type:
      // aBitmapInfoHeader.biWidth * aBitmapInfoHeader.biHeight * 2;
      hr = CMediaBuffer::Create(10000000, theInputBuffer);
    }
    else
    {
      hr = CMediaBuffer::Create(aMinBufferSize, theInputBuffer);
    }
  }
  if(FAILED(hr))
  {
    cout << "Failed to allocate input buffer" << endl;
    return E_FAIL;
  }

  // OUTPUT
  hr = theDecoder->GetOutputSizeInfo(0, &aMinBufferSize, &anAlignment);
  if(SUCCEEDED(hr))
  {
    hr = CMediaBuffer::Create(aMinBufferSize, theOutputBuffer);
  }
  if(FAILED(hr))
  {
    cout << "Failed to allocate output buffer" << endl;
    return E_FAIL;
  }
  return S_OK;
}


int _tmain(int argc, _TCHAR* argv[])
{
  HRESULT hr;
  _TCHAR* aFileName;
  int aNumIterations = 1;
  bool useQualityDec = true; // impact H.264 and MPEG-4 decoding speed
  time_t aStartTime = time(NULL);
  USES_CONVERSION;

  if(argc <= 1)
  {
    cout << "Usage: MediaParserFileSourceDecodeC++ <AMP_file> [num iterations]" << endl;
    cout << "Press enter to quit.";
    cin.get();
    return 1;
  }
  else if(argc == 3)
  {
    aNumIterations = atoi(T2A(argv[2]));
  }
  aFileName = argv[1];

  cout << ctime (&aStartTime) << "Starting sample MediaParserFileSourceDecodeC++";
  cout << " (" << aNumIterations << " iterations)" << endl;

  CoInitialize(NULL);

  int anIterationCount;
  for(anIterationCount = 0; anIterationCount < aNumIterations; anIterationCount++)
  {
    AMPFileSource aSource;
    IMediaObject* someDecoders[MAX_NUMBER_OF_STREAMS];
    IMediaBuffer* someInputBuffers[MAX_NUMBER_OF_STREAMS];
    IMediaBuffer* someOutputBuffers[MAX_NUMBER_OF_STREAMS];
    DWORD someInputSampleCounts[MAX_NUMBER_OF_STREAMS];
    DWORD someOutputSampleCounts[MAX_NUMBER_OF_STREAMS];
    ::ZeroMemory(&someInputSampleCounts, sizeof(someInputSampleCounts));
    ::ZeroMemory(&someOutputSampleCounts, sizeof(someOutputSampleCounts));

    cout << "--------------------------------------" << endl;
    cout << "Source file: " << T2A(aFileName) << endl;
    // Initialize source
    hr = aSource.Load(aFileName);
    if(FAILED(hr))
    {
      cout << "Failed loading source!" << endl;
      break;
    }

    // Get media types 
    // Initiate decoders
    AM_MEDIA_TYPE* aMediaType;
    for (int i = 0; i < MAX_NUMBER_OF_STREAMS; i++)
    {
      if(S_OK == aSource.GetStreamInfo(&aMediaType, i))
      {
        if(GUID_H264 == aMediaType->subtype)
        {
          cout << "Stream#" << i << ": " << "H.264 Video" << endl;
          hr = InitializeDecoder(CLSID_H264Decoder, aMediaType, &someDecoders[i], useQualityDec);       
        }
        else if(GUID_DX50 == aMediaType->subtype)
        {
          cout << "Stream#" << i << ": " << "MPEG-4 Video" << endl;
          hr = InitializeDecoder(CLSID_MPEG4Decoder, aMediaType, &someDecoders[i], useQualityDec);        
        }
        else if(MEDIASUBTYPE_MJPG == aMediaType->subtype)
        {
          cout << "Stream#" << i << ": " << "MJPEG Video" << endl;
          hr = InitializeDecoder(CLSID_MJPEGDecoder, aMediaType, &someDecoders[i],useQualityDec);        
        }
        else if(GUID_AAC == aMediaType->subtype)
        {
          cout << "Stream#" << i << ": " << "AAC Audio" << endl;
          hr = InitializeDecoder(CLSID_AACDecoder, aMediaType, &someDecoders[i],useQualityDec);        
        }
        else if(GUID_G711 == aMediaType->subtype ||
                GUID_G726_24 == aMediaType->subtype ||
                GUID_G726_32 == aMediaType->subtype ||
                GUID_G711_ALT == aMediaType->subtype ||
                GUID_G721 == aMediaType->subtype ||
                GUID_G723 == aMediaType->subtype)
        {
          cout << "Stream#" << i << ": " << "G7.xx Audio" << endl;
          hr = InitializeDecoder(CLSID_G7xxDecoder, aMediaType, &someDecoders[i],useQualityDec);        
        }
        else
        {
          cout << "Stream#" << i << ": " << "The stream type is not supported" << endl;
          someDecoders[i] = NULL;
        }

        if(FAILED(hr))
        {
          cout << "Stream#" << i << ": " << "Failed to initiate decoder, hr = 0x" << hex <<  hr << endl;
          someDecoders[i] = NULL;
        }

        DeleteMediaType(aMediaType);
      }
      else
      {
        someDecoders[i] = NULL;
      }
    }


    // Allocate media buffers
    for (int i = 0; i < MAX_NUMBER_OF_STREAMS; i++)
    {
      hr = E_FAIL;
      if(someDecoders[i] != NULL)
      {
        hr = ::CreateBuffers(someDecoders[i], &someInputBuffers[i], &someOutputBuffers[i]);
      }
      
      if(FAILED(hr))
      {
        someInputBuffers[i] = NULL;
        someOutputBuffers[i] = NULL;
      }
    }


    // Start decoding loop
    DWORD aStreamIndex;
    REFERENCE_TIME aStartTime;
    REFERENCE_TIME aStopTime;
    BOOL isSyncPoint;
    DWORD aSampleSize;
    DWORD someInputFlags;
    DWORD anInputStatus;
    DWORD anOutputStatus;
    BYTE* aBuffer;
    DWORD aBufferMaxLength;
    DWORD aBufferOffset;
    DMO_OUTPUT_DATA_BUFFER aDMOOutputDataBuffer;
    while(true)
    {
      hr = aSource.GetNextSampleProperties(&aStreamIndex, &aSampleSize, &aStartTime,
                                            &aStopTime, &isSyncPoint);
      if(S_OK != hr)
      {
        // end-of-stream or read error
        if(FAILED(hr))
        {
          cout << "Source read error" << endl;
        }
        else
        {
          cout << "End of source" << endl;
        }
        break; //exit
      }

      someInputSampleCounts[aStreamIndex]++;

      if(someDecoders[aStreamIndex] == NULL ||
         someInputBuffers[aStreamIndex] == NULL ||
         someOutputBuffers[aStreamIndex] == NULL)
      {
        //skip buffer
        if(FAILED(aSource.SkipBuffer()))
        {
          cout << "Source read error" << endl;
          break; //exit
        }
        continue; // goto next sample
      }

      // prepare input buffer
      hr = someInputBuffers[aStreamIndex]->GetMaxLength(&aBufferMaxLength);
      if(SUCCEEDED(hr))
      {
        hr = someInputBuffers[aStreamIndex]->GetBufferAndLength(&aBuffer, &aBufferOffset);
      }

      if(FAILED(hr))
      {
        cout << "Stream#" << aStreamIndex << ": " << "Input buffer error" << endl;
        if(FAILED(aSource.SkipBuffer()))
        {
          cout << "Source read error" << endl;
          break; //exit
        }
        continue; // goto next sample
      }
      
      hr = aSource.FillBuffer(aBuffer+aBufferOffset, aBufferMaxLength-aBufferOffset);
      if(FAILED(hr))
      {
        cout << "Source read error" << endl;
        break; //exit
      }

      hr = someInputBuffers[aStreamIndex]->SetLength(aSampleSize);
      if(FAILED(hr))
      {
        cout << "Stream#" << aStreamIndex << ": " << "Failed setting input buffer length" << endl;
        continue; // goto next sample
      }

      bool anInputSampleProcessed = false;
      while(!anInputSampleProcessed)
      {
        hr = someDecoders[aStreamIndex]->GetInputStatus(0,&anInputStatus);
        if(SUCCEEDED(hr) && (anInputStatus & DMO_INPUT_STATUSF_ACCEPT_DATA) > 0)
        {
          someInputFlags = DMO_INPUT_DATA_BUFFERF_TIME | DMO_INPUT_DATA_BUFFERF_TIMELENGTH;
          someInputFlags |= isSyncPoint ? DMO_INPUT_DATA_BUFFERF_SYNCPOINT : 0;
          hr = someDecoders[aStreamIndex]->ProcessInput(0, someInputBuffers[aStreamIndex],
                                             someInputFlags, aStartTime, aStopTime-aStartTime);
          if(FAILED(hr))
          {
            cout << "Stream#" << aStreamIndex << ": " << "Failed processing input sample (#" <<
              (someInputSampleCounts[aStreamIndex]-1) << "), hr = 0x" << hex <<  hr << endl;
          }
          // reset buffer
          (void)someInputBuffers[aStreamIndex]->SetLength(0);
          anInputSampleProcessed = true;
        }

        ZeroMemory(&aDMOOutputDataBuffer, sizeof(aDMOOutputDataBuffer));
        aDMOOutputDataBuffer.pBuffer = someOutputBuffers[aStreamIndex];
        do
        {
          hr = someDecoders[aStreamIndex]->ProcessOutput(0, 1, &aDMOOutputDataBuffer, &anOutputStatus);
        } while(hr == S_OK &&
                (aDMOOutputDataBuffer.dwStatus & DMO_OUTPUT_DATA_BUFFERF_INCOMPLETE) > 0);

        if(hr == S_OK)
        {
          someOutputSampleCounts[aStreamIndex]++;

          // Uncomment for sample info output
          /*BYTE* aBuf;
          DWORD aLength;
          someOutputBuffers[aStreamIndex]->GetBufferAndLength(&aBuf, &aLength);
          cout << "Stream#" << dec << aStreamIndex << ": " << "OUT: ";
          cout << "SIZE:" << aLength << "\t"; 
          cout << "TIME/DURATION:\t" << aDMOOutputDataBuffer.rtTimestamp;
          cout << "/" << aDMOOutputDataBuffer.rtTimelength;
          if((aDMOOutputDataBuffer.dwStatus & DMO_OUTPUT_DATA_BUFFERF_SYNCPOINT) > 0)
          {
            cout << " [KF] ";
          }
          cout << endl;*/

          // Uncomment for data preview output
          /*cout << hex ;
          for(int b = 0; b < min(10,aLength); b++)
          {
            cout << "0x"<< (int)aBuf[min(b,aLength-1)] << " ";
          }
          cout << endl;*/
        }
        else if(FAILED(hr))
        {
          // Note that older H.264 decoder versions will return E_INVALIDARG when 
          // no data i produced when only configuration data (SPS/PPS) was inputted
          cout << "Stream#" << aStreamIndex << ": " << "Failed processing output sample (#" <<
            (someInputSampleCounts[aStreamIndex]-1) << "), hr = 0x" << hex <<  hr << endl;
        }

        // reset buffer
        someOutputBuffers[aStreamIndex]->SetLength(0);
      }
    }

    // Clean up
    for (int i = 0; i < MAX_NUMBER_OF_STREAMS; i++)
    {
      if(someDecoders[i] != NULL)
      {
        someDecoders[i]->Release();
      }
      if(someInputBuffers[i] != NULL)
      {
        someInputBuffers[i]->Release();
      }
      if(someOutputBuffers[i] != NULL)
      {
        someOutputBuffers[i]->Release();
      }
    }


    cout << "Iteration#" << dec << anIterationCount << " ended." << endl;
    for (int i = 0; i < MAX_NUMBER_OF_STREAMS; i++)
    {
      if(someInputSampleCounts[i] > 0)
      {
        cout << "\tStream#" << i << " IN: " << someInputSampleCounts[i] <<
             " OUT: " <<  someOutputSampleCounts[i] << endl;
      }
    }
  }

  CoUninitialize();

  time_t anEndTime = time(NULL);
  cout << endl;
  cout << "######################################" << endl;
  cout << ctime(&anEndTime);
  cout << "Finished " << anIterationCount << " iterations" << endl;
  cout << "Process time: " << difftime(anEndTime, aStartTime) << " sec" << endl;
  cout << "Press enter to quit." << endl;
  cin.get();
	return 0;
}
