//---------------------------------------------------------------------------- 
// Copyright 2010 (c) Axis Communications AB, SWEDEN. All rights reserved.
//----------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using DirectShowLib;
using System.IO;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace MediaParserFileSourceDecodeCSharp
{
  public enum AMP_VIDEO_SAMPLE_TYPE
  {
    AMP_VST_UNKNOWN = 0,
    AMP_VST_MPEG4_VIDEO_CONFIG = 200,
    AMP_VST_MPEG4_VIDEO_IVOP = 201,
    AMP_VST_MPEG4_VIDEO_PVOP = 202,
    AMP_VST_MPEG4_VIDEO_BVOP = 203,
    AMP_VST_MPEG4_VIDEO_FRAGMENT = 204,
    AMP_VST_MJPEG_VIDEO = 205,
    AMP_VST_H264_VIDEO_CONFIG = 206,
    AMP_VST_H264_VIDEO_IDR = 207,
    AMP_VST_H264_VIDEO_NON_IDR = 208,
    AMP_VST_H264_VIDEO_SEI = 209,
    AMP_VST_MPEG4_AUDIO = 300,
    AMP_VST_METADATA_ONVIF = 400,
  }

  public enum AMP_SAMPLE
  {
    AMP_S_SYNCPOINT = 1,
    AMP_S_INCOMPLETE = 4,
    AMP_S_STARTTIMEVALID = 16,
    AMP_S_STOPTIMEVALID = 256,
  }
  /// <summary>
  /// Class for reading media from standard AMP file
  /// </summary>
  class AMPFileSource
  {
    public const int MaxNumberOfStreams = 2;

    private static Guid MediaTypeAxisCommunications = new Guid(0x34f14b, 0x454c, 0x400e, 
      0xa0, 0x75, 0xd8, 0xba, 0x5e, 0x14, 0x92, 0xc8);

    BinaryReader myBinaryReader;

    private uint myNumberOfStreams = 0;
    private AMMediaType[] myStreamMediaTypes = null;

    private bool isLoaded = false;

    private int myVideoStreamIndex = -1;
    private int myAudioStreamIndex = -1;

    private int myNextSampleSize = 0;

    public AMPFileSource()
    {

    }

    /// <summary>
    /// Loads a file created with AXIS Media Parser.
    /// </summary>
    public void Load(string theFile)
    {
      Guid aGuid;
      byte aFileVersion = 0;
      uint aBufferSize;

      if(isLoaded)
      {
        this.Close();
      }

      Debug.Assert(myStreamMediaTypes == null);

      FileStream aFileStream = new FileStream(theFile, FileMode.Open, FileAccess.Read);
      myBinaryReader = new BinaryReader(aFileStream);

      aBufferSize = myBinaryReader.ReadUInt32();

      aGuid = new Guid(myBinaryReader.ReadBytes(16));

      if (aGuid == MediaTypeAxisCommunications)
      {
        aFileVersion = myBinaryReader.ReadByte();
        myNumberOfStreams = myBinaryReader.ReadByte();
        myNumberOfStreams = Math.Min(myNumberOfStreams, MaxNumberOfStreams);
      }
      else
      {
        // single AMP_MEDIA_TYPE
        aFileVersion = 0;
        myNumberOfStreams = 1;
      }

      myStreamMediaTypes = new AMMediaType[myNumberOfStreams];

      for (uint aStreamIndex = 0; aStreamIndex < myNumberOfStreams; aStreamIndex++)
      {
        if (aFileVersion == 0)
        {
          Debug.Assert(myNumberOfStreams == 1);

          AMMediaType aDstMediaType = new AMMediaType();

          aDstMediaType = new AMMediaType();
          aDstMediaType.majorType = aGuid;
          aDstMediaType.subType = new Guid(myBinaryReader.ReadBytes(16));
          aDstMediaType.formatType = new Guid(myBinaryReader.ReadBytes(16)); 
          aDstMediaType.formatSize = myBinaryReader.ReadInt32();
          if (aDstMediaType.formatSize > 0)
          {
            aDstMediaType.formatPtr = Marshal.AllocCoTaskMem(aDstMediaType.formatSize);
            byte[] formatData = myBinaryReader.ReadBytes(aDstMediaType.formatSize);
            Marshal.Copy(formatData, 0, aDstMediaType.formatPtr, aDstMediaType.formatSize);
            myBinaryReader.ReadBytes(3); // padding
          }
          else
          {
            myBinaryReader.ReadBytes(4);
          }
          
          myStreamMediaTypes[aStreamIndex] = aDstMediaType;
        }
        else if (aFileVersion == 1)
        {
          AMMediaType aDstMediaType = new AMMediaType();
          aDstMediaType.majorType = new Guid(myBinaryReader.ReadBytes(16));
          aDstMediaType.subType = new Guid(myBinaryReader.ReadBytes(16));
          aDstMediaType.fixedSizeSamples = myBinaryReader.ReadInt32() > 0;
          aDstMediaType.temporalCompression = myBinaryReader.ReadInt32() > 0;
          aDstMediaType.sampleSize = myBinaryReader.ReadInt32();
          aDstMediaType.formatType = new Guid(myBinaryReader.ReadBytes(16));
          aDstMediaType.unkPtr = IntPtr.Zero;
          aDstMediaType.formatSize = myBinaryReader.ReadInt32();
          if(aDstMediaType.formatSize > 0)
          {
            aDstMediaType.formatPtr = Marshal.AllocCoTaskMem(aDstMediaType.formatSize);
            byte[] formatData = myBinaryReader.ReadBytes(aDstMediaType.formatSize);
            Marshal.Copy(formatData,0,aDstMediaType.formatPtr, aDstMediaType.formatSize);
            myBinaryReader.ReadBytes(3); // padding
          }
          else
          {
            myBinaryReader.ReadBytes(4);
          }

          myStreamMediaTypes[aStreamIndex] = aDstMediaType;
        }
        else
        {
          throw new Exception("Unsupported AMP file version");
        }
      }

      for(int aStreamIndex = 0; aStreamIndex < myNumberOfStreams; aStreamIndex++)
      {
        if(myStreamMediaTypes[aStreamIndex].majorType == MediaType.Video)
        {
          Debug.Assert(myVideoStreamIndex == -1, "AMP format inherently only support one video stream");
          myVideoStreamIndex = aStreamIndex;
        }
        else if (myStreamMediaTypes[aStreamIndex].majorType == MediaType.Audio)
        {
          Debug.Assert(myAudioStreamIndex == -1, "AMP format inherently only support one video stream");
          myAudioStreamIndex = aStreamIndex;
        }
      }

      isLoaded = true;
    }

    /// <summary>
    /// Retrieves media type information for the streams contained in the loaded file.
    /// </summary>
    public bool GetStreamInfo(out AMMediaType theMediaType, int theIndex)
    {
      if (!isLoaded)
      {
        throw new Exception("File not loaded.");
      }

      if (theIndex < myNumberOfStreams)
      {
        theMediaType = myStreamMediaTypes[theIndex];
        return true;
      }
      theMediaType = null;
      return false;
    }

    /// <summary>
    /// Get the stream index, size and other properties of the next samples in the file stream.
    /// </summary>
    public bool GetNextSampleProperties(out int theStreamIndex, out int theSampleSize,
                                        out long theStartTime, out long theStopTime,
                                        out bool isSyncPoint)
    {
      uint aSampleType;
      uint someFlags;

      theStreamIndex = -1;
      theSampleSize = 0;
      theStartTime = 0;
      theStopTime = 0;
      isSyncPoint = false;

      if (!isLoaded)
      {
        throw new Exception("File not loaded.");
      }

      try
      {
        aSampleType = myBinaryReader.ReadUInt32();
        someFlags = myBinaryReader.ReadUInt32();
        theStartTime = myBinaryReader.ReadInt64();
        theStopTime = myBinaryReader.ReadInt64();
        myNextSampleSize = myBinaryReader.ReadInt32();

        if (aSampleType == (uint)AMP_VIDEO_SAMPLE_TYPE.AMP_VST_MPEG4_AUDIO) //audio
        {
          Debug.Assert(myAudioStreamIndex >= 0);
          theStreamIndex = myAudioStreamIndex;
        }
        else if (aSampleType != (uint)AMP_VIDEO_SAMPLE_TYPE.AMP_VST_METADATA_ONVIF) // video
        {
          Debug.Assert(myVideoStreamIndex >= 0);
          theStreamIndex = myVideoStreamIndex;
        }

        isSyncPoint = (someFlags & (uint)AMP_SAMPLE.AMP_S_SYNCPOINT) > 0;
        theSampleSize = myNextSampleSize;
      }
      catch (EndOfStreamException)
      {
        return false; // end of stream
      }

      return true;
    }

    /// <summary>
    /// Copies the sample data of the next sample to the provided buffer.
    /// Call the GetNextSampleProperties before calling this method.
    /// </summary>
    public void FillBuffer(IntPtr theBuffer, int theBufferLength)
    {
      if (!isLoaded)
      {
        throw new Exception("File not loaded.");
      }

      if (theBuffer == IntPtr.Zero)
      {
        throw new NullReferenceException();
      }

      if (myNextSampleSize == 0)
      {
        throw new Exception("Call GetNextSampleProperties before calling this function");
      }

      if (myNextSampleSize > theBufferLength)
      {
        throw new Exception("Insufficient buffer size");
      }

      byte[] aSampleBuffer = myBinaryReader.ReadBytes(myNextSampleSize);
      if (aSampleBuffer.Length != myNextSampleSize)
      {
        myNextSampleSize = 0;
        throw new Exception("Unexpected end of stream");
      }

      Marshal.Copy(aSampleBuffer, 0, theBuffer, aSampleBuffer.Length);

      myNextSampleSize = 0;
    }

    /// <summary>
    /// Skips the next sample data.
    /// Call the GetNextSampleProperties before calling this method.
    /// </summary>
    public void SkipBuffer()
    {
      if (!isLoaded)
      {
        throw new Exception("File not loaded.");
      }

      if (myNextSampleSize == 0)
      {
        throw new Exception("Call GetNextSampleProperties before calling this function");
      }

      long anOldPos = myBinaryReader.BaseStream.Position;
      long aNewPos = myBinaryReader.BaseStream.Seek(myNextSampleSize, SeekOrigin.Current);
      if ((aNewPos-anOldPos) != myNextSampleSize)
      {
        myNextSampleSize = 0;
        throw new Exception("Unexpected end of stream");
      }
      myNextSampleSize = 0;
    }

    private void Close()
    {
      if(myBinaryReader != null)
      {
        myBinaryReader.Close();
        myBinaryReader = null;
      }

      if (myStreamMediaTypes != null)
      {
        for (uint aStreamIndex = 0; aStreamIndex < myNumberOfStreams; aStreamIndex++)
        {
          DsUtils.FreeAMMediaType(myStreamMediaTypes[aStreamIndex]);
          myStreamMediaTypes[aStreamIndex] = null;
        }
        myStreamMediaTypes = null;
      }

      myNumberOfStreams = 0;
      isLoaded = false;
      myVideoStreamIndex = -1;
      myAudioStreamIndex = -1;
      myNextSampleSize = 0;
    }
  }
}
