//---------------------------------------------------------------------------- 
// Copyright 2010 (c) Axis Communications AB, SWEDEN. All rights reserved.
//----------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using DirectShowLib.DMO;
using DirectShowLib;
using System.Runtime.InteropServices;

namespace MediaParserFileSourceDecodeCSharp
{
  class Program
  {
    static void Main(string[] args)
    {
      int hr;
      string aFileName;
      int aNumIterations = 1;
      bool useQualityDec = true; // impact H.264 and MPEG-4 decoding speed
      DateTime aProgramStartTime = DateTime.Now;

      if (args.Length == 0)
      {
        Console.WriteLine("Usage: MediaParserFileSourceDecodeCSharp <AMP_file> [num iterations]");
        Console.WriteLine("Press enter to quit.");
        Console.ReadKey(false);
        return;
      }
      else if (args.Length == 2)
      {
        aNumIterations = Int32.Parse(args[1]);
      }
      aFileName = args[0];

      Console.WriteLine(aProgramStartTime.ToString());
      Console.WriteLine("Starting sample MediaParserFileSourceDecodeC++" +
                        " (" + aNumIterations + " iterations)");

      int anIterationCount;
      for (anIterationCount = 0; anIterationCount < aNumIterations; anIterationCount++)
      {
        AMPFileSource aSource = new AMPFileSource();
        IMediaObject[] someDecoders = new IMediaObject[AMPFileSource.MaxNumberOfStreams];
        IMediaBuffer[] someInputBuffers = new IMediaBuffer[AMPFileSource.MaxNumberOfStreams];
        IMediaBuffer[] someOutputBuffers = new IMediaBuffer[AMPFileSource.MaxNumberOfStreams];
        uint[] someInputSampleCounts = new uint[AMPFileSource.MaxNumberOfStreams];
        uint[] someOutputSampleCounts = new uint[AMPFileSource.MaxNumberOfStreams];
        for(int i = 0; i < AMPFileSource.MaxNumberOfStreams; i++)
        {
          someDecoders[i] = null;
          someInputBuffers[i] = null;
          someOutputBuffers[i] = null;
          someInputSampleCounts[i] = 0;
          someOutputSampleCounts[i] = 0;
        }

        Console.WriteLine("--------------------------------------");
        Console.WriteLine("Source file: " + aFileName);

        try
        {
          // Initialize source
          aSource.Load(aFileName);
        }
        catch(Exception ex)
        {
          Program.ReportError("Failed to load file", ex);
          break;
        }

        // Get media types 
        // Initiate decoders
        AMMediaType aMediaType;
        for (int i = 0; i < AMPFileSource.MaxNumberOfStreams; i++)
        {
          if (aSource.GetStreamInfo(out aMediaType, i))
          {
            try
            {
              if(MediaSubType.H264 == aMediaType.subType)
              {
                Console.WriteLine("Stream#" + i + ": " + "H.264 Video");
                Program.InitializeDecoder(AxisDmoClsid.H264, aMediaType,
                                          out someDecoders[i], useQualityDec);
              }
              else if (CustomMediaSubType.DX50 == aMediaType.subType)
              {
                Console.WriteLine("Stream#" + i + ": " + "MPEG-4 Video");
                Program.InitializeDecoder(AxisDmoClsid.MPEG4, aMediaType,
                                          out someDecoders[i], useQualityDec);
              }
              else if(MediaSubType.MJPG == aMediaType.subType)
              {
                Console.WriteLine("Stream#" + i + ": " + "Motion JPEG Video");
                Program.InitializeDecoder(AxisDmoClsid.MJPEG, aMediaType,
                                          out someDecoders[i], useQualityDec);
              }
              else if(CustomMediaSubType.AAC == aMediaType.subType)
              {
                Console.WriteLine("Stream#" + i + ": " + "AAC Audio");
                Program.InitializeDecoder(AxisDmoClsid.AAC, aMediaType,
                                          out someDecoders[i], useQualityDec);
              }
              else if(CustomMediaSubType.G711 == aMediaType.subType ||
                      CustomMediaSubType.G726_24 == aMediaType.subType ||
                      CustomMediaSubType.G726_32 == aMediaType.subType ||
                      CustomMediaSubType.G711_ALT == aMediaType.subType ||
                      CustomMediaSubType.G721 == aMediaType.subType ||
                      CustomMediaSubType.G723 == aMediaType.subType)
              {                      
                Console.WriteLine("Stream#" + i + ": " + "G.7xx Audio");
                Program.InitializeDecoder(AxisDmoClsid.G7xx, aMediaType,
                                          out someDecoders[i], useQualityDec);
              }
              else
              {
                Console.WriteLine("Stream#" + i + ": " + "The stream type is not supported (" +
                                  DsToString.MediaSubTypeToString(aMediaType.subType) + ")");
                someDecoders[i] = null;
              }
            }
            catch(Exception ex)
            {
              Program.ReportStreamError(i, "Failed to initiate decoder", ex);
              someDecoders[i] = null;
            }
          }
          else
          {
            someDecoders[i] = null;
          }
        }

        // Allocate media buffers
        for (int i = 0; i < AMPFileSource.MaxNumberOfStreams; i++)
        {
          try
          {
            if (someDecoders[i] != null)
            {
              Program.CreateBuffers(someDecoders[i],
                                    out someInputBuffers[i], out someOutputBuffers[i]);
            }
          }
          catch(Exception ex)
          {
            Program.ReportStreamError(i, "Could not initialize I/O buffers", ex);
            someInputBuffers[i] = null;
            someOutputBuffers[i] = null;
          }
        }

        
        // Start decoding loop
        int aStreamIndex;
        long aStartTime;
        long aStopTime;
        bool isSyncPoint;
        int aSampleSize;
        DMOInputDataBuffer someInputFlags;
        DMOInputStatusFlags anInputStatus;
        int anOutputStatus;
        IntPtr aBuffer;
        int aBufferMaxLength;
        int aBufferOffset;
        DMOOutputDataBuffer[] aDMOOutputDataBuffer = new DMOOutputDataBuffer[1];
        while(true)
        {
          try 
	        {	        
		        if(!aSource.GetNextSampleProperties(out aStreamIndex, out aSampleSize, out aStartTime,
                                                      out aStopTime, out isSyncPoint))
            {
              Console.WriteLine("End of source");
              break;
            }
          }
	        catch (Exception ex)
	        {
            Program.ReportError("Source read error", ex);
            break;
	        }

          someInputSampleCounts[aStreamIndex]++;

          if(someDecoders[aStreamIndex] == null ||
             someInputBuffers[aStreamIndex] == null ||
             someOutputBuffers[aStreamIndex] == null)
          {
            //skip buffer
            try
            {
              aSource.SkipBuffer();
            }
            catch(Exception ex)
            {
              Program.ReportError("Source read error", ex);
              break;
            }
            continue; // goto next sample
          }

          // prepare input buffer
          try
          {
            hr = someInputBuffers[aStreamIndex].GetMaxLength(out aBufferMaxLength);
            DsError.ThrowExceptionForHR(hr);
            hr = someInputBuffers[aStreamIndex].GetBufferAndLength(out aBuffer, out aBufferOffset);
            DsError.ThrowExceptionForHR(hr);
          }
          catch(Exception ex)
          {
            Program.ReportStreamError(aStreamIndex, "Input buffer error", ex);
            //skip buffer
            try
            {
              aSource.SkipBuffer();
            }
            catch(Exception ex2)
            {
              Program.ReportError("Source read error", ex2);
              break;
            }
            continue; // goto next sample
          }
          
          try 
	        {
            // ToInt64 is safe on both 32 and 64 bit systems
            aSource.FillBuffer(new IntPtr(aBuffer.ToInt64() + aBufferOffset),
                               aBufferMaxLength - aBufferOffset);
	        }
	        catch (Exception ex)
          {
            Program.ReportError("Source read error", ex);
            break; //exit
	        }

          try
          {
            someInputBuffers[aStreamIndex].SetLength(aSampleSize);
          }
          catch (Exception ex)
          {
            Program.ReportStreamError(aStreamIndex, "Failed setting input buffer length", ex);
            continue; // goto next sample
          }

          bool anInputSampleProcessed = false;
          while(!anInputSampleProcessed)
          {
            try
            {
              hr = someDecoders[aStreamIndex].GetInputStatus(0, out anInputStatus);
              DsError.ThrowExceptionForHR(hr);
              if(anInputStatus == DMOInputStatusFlags.AcceptData)
              {
                someInputFlags = DMOInputDataBuffer.Time | DMOInputDataBuffer.TimeLength;
                someInputFlags |= isSyncPoint ? DMOInputDataBuffer.SyncPoint : 0;
                hr = someDecoders[aStreamIndex].ProcessInput(0, someInputBuffers[aStreamIndex],
                                                             someInputFlags, aStartTime,
                                                             aStopTime-aStartTime);
                DsError.ThrowExceptionForHR(hr);
              }
            }
            catch (Exception ex)
            {
              Program.ReportStreamError(aStreamIndex, "Failed processing input sample (#" +
                                        (someInputSampleCounts[aStreamIndex]-1) + ")", ex);
            }
            finally
            {
              // reset buffer
              someInputBuffers[aStreamIndex].SetLength(0);
              anInputSampleProcessed = true;
            }

            try
            {
              aDMOOutputDataBuffer[0].pBuffer = someOutputBuffers[aStreamIndex];
              do
              {
                hr = someDecoders[aStreamIndex].ProcessOutput(0, 1, aDMOOutputDataBuffer,
                                                              out anOutputStatus);
                DsError.ThrowExceptionForHR(hr);
              } while(hr == 0 &&
                      (aDMOOutputDataBuffer[0].dwStatus & DMOOutputDataBufferFlags.InComplete) > 0);

              if(hr == 0)
              {
                someOutputSampleCounts[aStreamIndex]++;

                // Uncomment to copy to managed buffer
                //IntPtr aBuf;
                //int aLength;
                //someOutputBuffers[aStreamIndex].GetBufferAndLength(out aBuf, out aLength);
                //byte[] aManagedBuf = new byte[aLength];
                //Marshal.Copy(aBuf, aManagedBuf, 0, aLength);

                // Uncomment for output sample info
                //IntPtr aBuf;
                //int aLength;
                //someOutputBuffers[aStreamIndex].GetBufferAndLength(out aBuf, out aLength);
                //Console.Write("Stream#" + aStreamIndex + ": " + "OUT: " +
                //              "SIZE:" + aLength + "\t" +
                //              "TIME/DURATION:\t" + aDMOOutputDataBuffer[0].rtTimestamp + 
                //              "/" + aDMOOutputDataBuffer[0].rtTimelength);
                //if((aDMOOutputDataBuffer[0].dwStatus & DMOOutputDataBufferFlags.SyncPoint) > 0)
                //{
                //  Console.Write(" [KF] ");
                //}
                //Console.WriteLine("");

                // Uncomment for output data preview
                //int aNumBytesToDisplay = Math.Min(10,aLength);
                //byte[] aManagedBuf = new byte[aNumBytesToDisplay];
                //Marshal.Copy(aBuf, aManagedBuf, 0, aNumBytesToDisplay);
                //for(int b = 0; b < aNumBytesToDisplay; b++)
                //{
                //  Console.Write("0x" + aManagedBuf[b].ToString("x") + " ");
                //}
                //Console.WriteLine("");
              }
              else
              {
                  Console.WriteLine("Stream#" + aStreamIndex + ": No output generated for sample #" +
                                    (someInputSampleCounts[aStreamIndex] - 1));
              }
            }
            catch(Exception ex)
            {
              // Note that older H.264 decoder versions will return E_INVALIDARG when 
              // no data i produced when only configuration data (SPS/PPS) was inputted
              Program.ReportStreamError(aStreamIndex, "Failed processing output sample (#" +
                                        (someInputSampleCounts[aStreamIndex]-1) + ")", ex);
            }
            finally
            {
              // reset buffer
              someOutputBuffers[aStreamIndex].SetLength(0);
            }
          }
        }

        // Clean up
        try
        {
          for (int i = 0; i < AMPFileSource.MaxNumberOfStreams; i++)
          {
            if (someDecoders[i] != null)
            {
              Marshal.ReleaseComObject(someDecoders[i]);
            }
            if (someInputBuffers[i] != null)
            {
              (someInputBuffers[i] as IDisposable).Dispose();
            }
            if (someOutputBuffers[i] != null)
            {
              (someOutputBuffers[i] as IDisposable).Dispose();
            }
          }
        }
        catch (Exception ex)
        {
          Program.ReportError("Failed to clean up", ex);
        }

        Console.WriteLine("Iteration#" + anIterationCount + " ended.");
        for (int i = 0; i < AMPFileSource.MaxNumberOfStreams; i++)
        {
          if (someInputSampleCounts[i] > 0)
          {
            Console.WriteLine("\tStream#" + i + " IN: " + someInputSampleCounts[i] +
                              " OUT: " + someOutputSampleCounts[i]);
          }
        }

      }

      Console.WriteLine();
      Console.WriteLine("######################################");
      Console.WriteLine(DateTime.Now.ToString());
      Console.WriteLine("Finished " + anIterationCount + " iterations");
      Console.WriteLine("Process time: " + (DateTime.Now - aProgramStartTime));
      Console.WriteLine("Press enter to quit.");
      Console.ReadKey(false);
    }

    private static void CreateBuffers(IMediaObject theDecoder,
                                      out IMediaBuffer theInputBuffer,
                                      out IMediaBuffer theOutputBuffer)
    {
      int hr;
      int aMinBufferSize, aLookahead, anAlignment;

      // INPUT
      try 
	    {	        
		    hr = theDecoder.GetInputSizeInfo(0, out aMinBufferSize, out aLookahead, out anAlignment);
        DsError.ThrowExceptionForHR(hr);
        if(aMinBufferSize <= 0 || aMinBufferSize > 10000000)
        {
          // Older H.264 decoder doesn't set the input buffer min size
          // Setting buffer size to a fixed 10 MB
          //
          // A more correct buffer size can be approximated using the image dimension info
          // from the input media type:
          // aBitmapInfoHeader.biWidth * aBitmapInfoHeader.biHeight * 2;
          theInputBuffer = new MediaBuffer(10000000);
        }
        else
        {
          theInputBuffer = new MediaBuffer(aMinBufferSize);
        }
	    }
	    catch (Exception ex)
	    {
		    throw new Exception("Failed to allocate input buffer: " + ex.Message);
	    }

      // OUTPUT
      try
      {
        hr = theDecoder.GetOutputSizeInfo(0, out aMinBufferSize, out anAlignment);
        DsError.ThrowExceptionForHR(hr);
        theOutputBuffer = new MediaBuffer(aMinBufferSize);
      }
      catch (Exception ex)
      {
        throw new Exception("Failed to allocate output buffer: " + ex.Message);
      }
    }

    static void InitializeDecoder(Guid theDmoClsid, AMMediaType theMediaType,
                                  out IMediaObject theDecoder, bool useQualitySetting)
    {
      int hr;
      AMMediaType anOutputMediaType = new AMMediaType();
      theDecoder = null;

      Type type = Type.GetTypeFromCLSID(theDmoClsid);
      theDecoder = Activator.CreateInstance(type) as IMediaObject;
      if(theDecoder == null)
      {
        throw new Exception("Decoder not installed");
      }

      try 
	    {	        
		    hr = theDecoder.SetInputType(0, theMediaType, DMOSetType.None);
        DsError.ThrowExceptionForHR(hr);

        hr = theDecoder.GetOutputType(0, 0, anOutputMediaType);
        DsError.ThrowExceptionForHR(hr);

        hr = theDecoder.SetOutputType(0, anOutputMediaType, 0);
        DsError.ThrowExceptionForHR(hr);
	    }
	    catch (Exception ex)
	    {
        if (theDecoder != null)
        {
            Marshal.ReleaseComObject(theDecoder);
            theDecoder = null;
        }
		    throw new Exception("Could not setup decoder DMO. " + ex.Message);
	    }
      finally
      {
        DsUtils.FreeAMMediaType(anOutputMediaType);
      }


#warning "Setting decoding quality has not been implemented. Must define .Net interfaces for IAxH264Dec2 and IAxMP4Dec"
      //IAxH264Dec2 anAxIH264Dec2 = theDecoder as IAxH264Dec2;
      //if (anAxIH264Dec2 != null)
      //{
      //  anAxIH264Dec2.SetQualitySettings(useQualitySetting ? 9 : 0);
      //  Marshal.Release(Marshal.GetIUnknownForObject((object)anAxIH264Dec2));
      //}
      //else
      //{
      //  IAxMP4Dec anAxMP4Dec = theDecoder as IAxMP4Dec;
      //  if (anAxMP4Dec != null)
      //  {
      //    anAxMP4Dec.SetDeblock(useQualitySetting ? 1 : 0);
      //    Marshal.Release(Marshal.GetIUnknownForObject((object)anAxMP4Dec));
      //  }
      //}
    }


    static void ReportError(string theErrorText, Exception theException)
    {
      Console.WriteLine(theErrorText + " - " + theException.Message);
    }

    static void ReportStreamError(int theStreamIndex, string theErrorText, Exception theException)
    {
      Console.WriteLine("Stream#" + theStreamIndex + ": " + theErrorText +
                        " - " + theException.Message);
    }
  }
}
