/*
 * FILE NAME:  sysv/prossysv.c
 *
 * System V printer driver for Axis print servers using the PROS 2.1 protocol.
 *
 * (C) Copyright 1994, Axis Communications AB, LUND, SWEDEN.
 *
 * Version 1.4 April 25, 1994 RW
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/wait.h>
#include <signal.h>
#include <time.h>

#define PROS_PORT       35
#define MAX_LOOPS       200
#define MAX_STATBUF     4096
#define BUFSIZE         4096
#define MAX_OUTBUF      256

/* PROS protocol 2.1 */

/* server -> host error messages */

#define PROSERR_HDR    0  /* header error */
#define PROSERR_MEM    1  /* out of memory */
#define PROSERR_NOA    2  /* No access */
#define PROSERR_POC    3  /* Printer occupied */
#define PROSERR_BAN    4  /* Bad printer name */
#define PROSERR_OLD    5  /* Old protocol */
#define PROSERR_NOI    6  /* Printer not installed */
#define PROSERR_OFL    7  /* Printer off line */
#define PROSERR_EOP    8  /* Printer out of paper */
#define PROSERR_BSY    9  /* Printer busy */
#define PROSERR_PRO   10  /* Protocol error */
#define PROSERR_UND   11  /* Undefined error */

/* host -> server informational messages */

#define PROSMSG_EOF   32   /* End-of-file: no more print data */
#define PROSMSG_UID   33   /* user name */
#define PROSMSG_HST   34   /* host name */
#define PROSMSG_PRN   35   /* printer name */
#define PROSMSG_PAS   36   /* password */
#define PROSMSG_DTP   37   /* data to printer */
#define PROSMSG_NOP   38   /* NOP - ignore */

/* server -> host informational messages */

#define PROSMSG_JOK   48   /* Job ended ok */
#define PROSMSG_JST   49   /* Job started */
#define PROSMSG_ACC   50   /* accounting data */
#define PROSMSG_DFP   51   /* data from printer */

/* status byte bits */

#define PROSBIT_FATAL  0x40   /* fatal error bit */
#define PROSBIT_DATA   0x80   /* data record follows */

int sock;
int mothid;
time_t current_time;
char indata[BUFSIZE+3];
char outdata[MAX_OUTBUF];
unsigned char statbuf[MAX_STATBUF];

char *boxname, *printername, *userid, *hostname, *current_filename;

void cancel();

void cancel(dummy)
int dummy;
{
  int status;

  wait(&status);
  close(sock);
  if (status & 0xff)
    exit(2);
  else
    exit(status >> 8);
}

main(argc, argv)
int argc;
char **argv;
{
  int ok, loops;
  int stat_len;
  struct sockaddr_in server;
  struct hostent *hp, *gethostbyname();
  char *outdata_data;
  int doit, daughter, status;
  char *password;
  unsigned char status_code;
  char eof_marker = PROSMSG_EOF;
  int infile;
  int bytes_read;
  int copies, arg_idx;

  boxname = BOXNAME;
  printername = PRINTERNAME;

#ifdef HOSTNAME
  hostname = HOSTNAME;
#else
  hostname = "some host";
#endif

#ifdef USERNAME
  userid = USERNAME;
#else
  userid = argv[2];
#endif

#ifdef PASSWORD
  password = PASSWORD;
#else
  password = "netprinter";
#endif

  copies = atoi(argv[4]);

  while (copies > 0)
  {
    arg_idx = 6;
    while (argv[arg_idx])
    {
      current_filename = argv[arg_idx];
      infile = open(current_filename, O_RDONLY);
      if (infile < 0)
      {
        notify();
        fprintf(stderr, "Error opening input file.\n");
        exit(1);
      }

      doit = fork();
      if (doit != 0) /* grandmother */
      {
        if (doit == -1) /* fork failed */
        {
          notify();
          fprintf(stderr, "Main fork failed - insufficient resources!\n");
          exit(1);
        }
        wait(&status);
        close(infile);
        if (status & 0xff00)
        {
          exit(status >> 8);
        }
        /* fall through down to bottom of main */
      }
      else /* mother; open connection to print server, then read from infile */
      {
        server.sin_family = AF_INET;
        hp = gethostbyname(boxname);
        if (hp == 0)
        {
          notify();
          fprintf(stderr, "%s: unknown host\n", boxname);
          exit(2);
        }
        memcpy((char *) &server.sin_addr, (char *) hp->h_addr, hp->h_length);
        server.sin_port = htons(PROS_PORT);
        ok = 0;
        loops = 0;
        do
        {
          sock = socket(AF_INET, SOCK_STREAM, 0);
          if (sock < 0)
          {
            notify();
            fprintf(stderr, "Error creating socket\n");
            exit(1);
          }
          if (connect(sock, (struct sockaddr *) & server, sizeof server) < 0)
          {
#ifndef TRY_FOREVER
            loops++;
#endif
            if (errno == ECONNREFUSED)
            {
              close(sock);
              sleep(10);
            }
            else if (errno == ETIMEDOUT)
            {
              close(sock);
            }
            else
            {
              notify();
              fprintf(stderr, "Unable to contact printer server; Unix errno is %d\n",
              errno);
              exit(2);
            }
          }
          else
            ok++;
        } while (!ok && loops < MAX_LOOPS);
        if (!ok)
        {
          notify();
          fprintf(stderr, "Connection refused; print server might be hung\n");
          exit(1);
        }
        mothid = getpid();
        daughter = fork();
        if (daughter != 0)
        {
          if (daughter == -1)
          {
            notify();
            fprintf(stderr, "Fork failed - insufficient resources!\n");
            exit(1);
          }
          (void) signal(SIGINT, cancel);
          (void) signal(SIGPIPE, cancel);
          send_hdr(sock, PROSMSG_HST, hostname);
          send_hdr(sock, PROSMSG_UID, userid);
          send_hdr(sock, PROSMSG_PRN, printername);
          send_hdr(sock, PROSMSG_PAS, password);
            
          bytes_read = read(infile, indata+3, BUFSIZE);
          if (bytes_read < 0)
          {
            notify();
            fprintf(stderr, "Error reading input file!\n");
            exit(1);
          }
          outdata_data = indata;
          while (bytes_read > 0)
          {
            outdata_data[0] = PROSMSG_DTP | PROSBIT_DATA;
            outdata_data[1] = bytes_read >> 8;
            outdata_data[2] = bytes_read & 255;
            if (write(sock, outdata_data, bytes_read + 3) < 0)
            {
              notify();
              fprintf(stderr, "Error writing data to socket\n");
              exit(1);
            }
            bytes_read = read(infile, indata + 3, BUFSIZE);
            if (bytes_read < 0)
            {
              notify();
              fprintf(stderr, "Can't read from infile!");
              break;
            }
          }
          if (write(sock, &eof_marker, 1) < 0)
          {
            notify();
            fprintf(stderr, "Error writing eof to socket\n");
            exit(1);
          }
          while (1)
          {
            sleep(60);
            outdata[0] = PROSMSG_NOP;
            if (write(sock, outdata, 1) < 0)
            {
              fprintf(stderr, "Error writing NOP to socket\n");
              exit(1);
            }
          }
        }
        else
        {
          while (1)
          {
            get_stat(&status_code, 1);
            if (status_code & PROSBIT_DATA)
            {
              get_stat(statbuf, 2);
              stat_len = (statbuf[0] << 8) + statbuf[1];
              get_stat(statbuf, stat_len);
              statbuf[stat_len] = 0;
            }
            else
            {
              strcpy((char *)statbuf, "<no message>");
              stat_len = 0;
            }
       
            if ( status_code == (PROSMSG_DFP | PROSBIT_DATA) )
              fprintf(stdout, "%s", statbuf);
              
            if ((status_code & 0x3f) == PROSMSG_JOK)
            {
              kill(mothid, SIGINT);
              exit(0);
            }
            if (status_code & PROSBIT_FATAL)
            {
              notify();
              fprintf(stderr, "Msg: (%d, l=%d) %s%s\n",
                      status_code, stat_len, statbuf,
                      (status_code & PROSBIT_FATAL) ? " (FATAL)" : "");
              kill(mothid, SIGINT);
              exit(2);
            }
          }
        }
      } /* end if 1. fork */
      arg_idx++; /* next file */
    } /* end of while files */
    copies--;
  } /* end while copies */
  exit(0);
}  
    
send_hdr(sock, msg_code, msg_data)
int sock;
int msg_code;
char *msg_data;
{
  int str_len;

  str_len = strlen(msg_data);
  outdata[0] = msg_code | PROSBIT_DATA;
  outdata[1] = str_len >> 8;
  outdata[2] = str_len & 255;
  strcpy(&outdata[3], msg_data);

  if (write(sock, outdata, str_len + 3) < 0)
  {
    notify();
    fprintf(stderr, "Error writing header to socket\n");
    exit(1);
  }
}

get_stat(buffer, len)
char *buffer;
int len;
{
  int rval;

  do
  {
    if ((rval = read(sock, buffer, len)) < 0)
    {
      notify();
      fprintf(stderr, "Error reading from print server\n");
      kill(mothid, SIGINT);
      exit(2);
    }
    else if (rval == 0)
    {
      notify();
      fprintf(stderr, "Unexpected close by print server\n");
      kill(mothid, SIGINT);
      exit(2);
    }
    len -= rval;
    buffer += rval;
  } while (len > 0);
}

notify()
{
  fprintf(stderr, "\nError while printing file %s on print server %s:%s\n",
          current_filename, boxname, printername);
  fprintf(stderr, "(job initiated by %s@%s)\n", userid, hostname);
}
