/*
 * Copyright (c) 2005-2006 Brocade Communications Systems, Inc.
 * All rights reserved.
 *
 * Description:
 *     Global defines and function prototypes for the
 *     Message Passing Interface (MPI).
 *
 *     The MPI is based upon a circular queue (CQ) transport medium.
 *     The CQs provide a point to point connection between two peers.
 *     CQs are paired in a Rx/Tx fashion, forming a single channel.
 *     A peer transmits over one queue and receives over the other queue.
 *     There may be more than one channel that comprises a connection
 *     between peers.
 *
 *     Single CQ configuration:
 *
 *      Peer A
 *
 *      Tx   Rx
 *     |  | |  |
 *     |  | |  |
 *     |  | |  |
 *     |  | |  |
 *      Rx   Tx
 *
 *      Peer B
 *
 *
 *      Multiple CQ configuration for load balancing, priorities, etc.:
 *
 *      Peer A
 *
 *        CQ1         CQ2                        CQn
 *      Tx   Rx     Tx   Rx                    Tx   Rx
 *     |  | |  |   |  | |  |                  |  | |  |
 *     |  | |  |   |  | |  |                  |  | |  |
 *     |  | |  |   |  | |  |                  |  | |  |
 *     |  | |  |   |  | |  |       . . .      |  | |  |
 *     |  | |  |   |  | |  |                  |  | |  |
 *      Rx   Tx     Rx   Tx                    Rx   Tx
 *
 *      Peer B
 *
 */

#ifndef __MPI_H__
#define __MPI_H__

#include <mpi/mpi_mach_dep.h>

#ifdef __cplusplus
extern "C" {
#endif


/* Max number of Rx/Tx circular queues pairs */
#define MPI_MAX_CQ			32

/* Max number of pages */
#define MPI_MAX_PAGES		((64 * 1024) - 1)

/* Max number of registered SAP handlers */
#define MPI_MAX_SAP_HDLRS	32

/* Check if CQ has messages */
#define MPI_RX_CQ_NOT_EMPTY(hdl, cq)	(mpi_get_cq_sz(hdl, (cq)) > 0)

/* Increment a CQ index */
#define MPI_WRAP(val_, len_, amt_)		(((val_) + (amt_)) % (len_))
#define MPI_CQ_INCR(val_, len_, amt_)	val_ = MPI_WRAP(val_, len_, amt_)


/*
 * Abstract data type representing an MPI instance.  This handle is created
 * when the MPI is instantiated.  It is subsequently passed as a parameter to
 * invocations of MPI methods detailed below.
 */
typedef void *mpi_handle_t;


/* Circular queue number */
typedef unsigned mpi_cq_num_t;


/* Bitmap indicating which CQs have enqueued messages */
typedef uint32_t mpi_cq_map_t;


/* Abstract data type for scatter/gather list buffer */
typedef void *mpi_sgl_buf_t;


/* Type of message to be queued */
typedef enum {
	MPI_MSG_TYPE_INV = 0,	/* invalid: reserved initialized val */
	MPI_MSG_TYPE_USER,		/* msg_buf points to user-provided buffer */
	MPI_MSG_TYPE_DYN,		/* msg_buf dynamically allocated by MPI */
	MPI_MSG_TYPE_SGL,		/* msg_buf organized as scatter/gather list */
	MPI_MSG_TYPE_LAST
} mpi_msg_type_t;


/*
 * MPI message type which is used for both sending and receiving.
 *
 * The "type" member indicates how the message buffer (or payload)
 * is constructed.
 *
 * Types MPI_MSG_TYPE_USER and MPI_MSG_TYPE_DYN have the same layout but
 * the message buffer is allocated differently .  With MPI_MSG_TYPE_DYN,
 * the payload is allocated by MPI and is suitable for HW assisted transfers
 * (i.e. DMA).  With MPI_MSG_TYPE_USER the payload is user-supplied and
 * therefore incurs an additional copy to transfer the payload to CQ memory.
 *
 * +-------------------------+
 * |type | msg_len | msg_buf |
 * +-------------------------+
 *                      |
 *                      |   +----------+
 *                      +-> | payload  |
 *                          +----------+
 *
 * Type MPI_MSG_TYPE_SGL has a payload comprised of a Scatter/Gather List (SGL).
 *
 * +-------------------------+
 * |type | msg_len | msg_buf |
 * +-------------------------+
 *                     |
 *                     |   +------+  +------+  +------+       +------+
 *                     +-> | buf0 |->| buf1 |->| buf1 |->...->| bufN |
 *                         +------+  +------+  +------+       +------+
 *
 */
typedef struct mpi_msg {
	mpi_msg_type_t	type;			/* MPI message type */
	int				msg_len;		/* msg payload length in bytes */
	void			*msg_buf;		/* ptr to msg buffer */
} mpi_msg_t;


/*
 * Pointer to function that is called when a message is sent or received.
 * This function could perform asynchronous notification to the peer.
 *
 * 		void my_notify_func(mpi_cq_num_t cq, void *arg);
 */
typedef void (*mpi_notify_fn_t) (mpi_cq_num_t cq, void *arg);


/*
 * Pointer to function that is called when a user message (MPI_MSG_TYPE_USER)
 * is enqueued onto the destination CQ.  This allows the client to deallocate
 * the user-provided buffer, if needed.
 *
 * 		void my_tx_cb_func(void *user_buf);
 */
typedef void (*mpi_tx_cb_fn_t) (void *user_buf);


/*
 * Pointer to function that is called when a message of a matching SAP is
 * received.  The function has the prototype:
 *
 * 		int my_rx_cb_func(mpi_msg_t *msg, void *arg);
 */
typedef void (*mpi_rx_cb_fn_t) (mpi_msg_t *msg, void *arg);


typedef uint8_t mpi_sap_t;

/* SAP to application handler mapping */
typedef struct mpi_sap_app_map {
	mpi_sap_t		sap;				/* Service Access Point */
	mpi_rx_cb_fn_t	cb;					/* appl handler call back funct */
	void			*arg;				/* arg to pass to call back */
} mpi_sap_app_map_t;


/*
 * Statistics that are tracked per CQ.
 */
typedef struct mpi_stats {
	unsigned rx_msgs[MPI_MAX_CQ];		/* total msgs received */
	unsigned tx_msgs[MPI_MAX_CQ];		/* total msgs sent */
	unsigned rx_bytes[MPI_MAX_CQ];		/* total bytes received */
	unsigned tx_bytes[MPI_MAX_CQ];		/* total bytes sent */
	unsigned tx_cq_full[MPI_MAX_CQ];	/* total Tx queue full errors */
} mpi_stats_t;

#ifdef DEBUG

extern uint32_t	mpi_debug;

#endif

/*
 *-----------------------------------------------------------------------------
 *                     MPI Initialization and Cleanup
 *-----------------------------------------------------------------------------
 */


/*
 * Function: mpi_init()
 *
 * Description:
 *    Initializes the MPI with the following platform-specific parameters:
 *
 *    	- Define the number and extent of the CQs.
 *    	- Base address of the CQ memory region.
 *    	- MPI storage page size
 *    	- Number of pages to allocate
 *    	- Flag that indicates whether this instance owns the CQ memory.
 *    	- Platform-specific DMA info such as for Linux the PCI bus/slot info
 *    	- A Rx notification function that will be called every time a
 *    	  message is received.
 *    	- A Tx notification function that will be called every time a
 *    	  message is sent.  This provides MPI a way to invoke a platform-
 *    	  specific routine to asynchronously inform its peer of message
 *    	  availability.
 *
 * Input:
 *    num_cq			Number of CQ Rx/Tx pairs to create
 *
 *    cq_sz				Number of entries in each CQ Rx/Tx pair
 *
 *    page_sz			Size of MPI storage page in bytes
 *
 *    num_pages			Number of pages in free page pool
 *
 *    cq_base_addr		Virtual base address of the CQ memory and storage
 *
 *    cq_mem_owner		Flags if CQ memory resides on this platform
 *
 *    dma_info			Pointer to a platform-dependent structure that provides
 *    					the necessary info to perform DMA operations.
 *    					This definition is contained in mpi_plat_dep.h
 *
 *    rx_notify_fn		Pointer to function to call upon receiving a message.
 *    					This function can be used to perform CQ flow control.
 *    					If notification is not needed, pass in NULL.
 *
 *    rx_notify_arg		Argument to be passed to the rx_notify_fn when called.
 *    					This is for the MPI client's use to indicate the
 *    					particular MPI instance.
 *
 *    tx_notify_fn		Pointer to function to call upon sending a message.
 *    					This function performs asynchronous notification of the
 *    					peer that a new message is available.  If notification
 *    					is not needed, pass in NULL.
 *
 *    tx_notify_arg		Argument to be passed to the tx_notify_fn when called.
 *    					This is for the MPI client's use to indicate the
 *    					particular MPI instance.
 *
 * Output:
 *	  new_hdl			Pointer to the handle that is to be used in subsequent
 *						calls to MPI methods.
 *
 * Returns:
 *    0 				On success
 *
 *    -EINVAL			Invalid num_cq, num_cq_entries, page_size, num_pages,
 *    					cq_base_addr, cq_mem_sz, dma_info, or new_hdl pointer
 *
 *    -ENOMEM			Failed to allocate new MPI handle
 */
extern int mpi_init(
	int					num_cq,
	int 				cq_sz,
	int					page_sz,
	int					num_pages,
	void 				*cq_base_addr,
	int					cq_mem_owner,
	mpi_dma_info_t 		*dma_info,
	mpi_notify_fn_t		rx_notify_fn,
	void				*rx_notify_arg,
	mpi_notify_fn_t		tx_notify_fn,
	void				*tx_notify_arg,
	mpi_handle_t 		*new_hdl);


/*
 * Function: mpi_cleanup()
 *
 * Description:
 *    Releases system resources allocated when MPI was instantiated.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		Invalid handle
 */
extern int mpi_cleanup(mpi_handle_t hdl);


/*
 *-----------------------------------------------------------------------------
 *                   Message Management (alloc/free)
 *-----------------------------------------------------------------------------
 */

/*
 * Function: mpi_alloc_msg()
 *
 * Description:
 *    Allocates a new Tx message.  All send messages must be allocated by
 *    this method.  Messages are automatically freed by MPI when they are
 *    enqueued to the destination CQ, with the exception of type
 *    MPI_MSG_TYPE_USER.  No attempt is made to free the user-provided buffer.
 *    An optional callback function may be specified to handle this.  If
 *    specfied, the callback is invoked following message enqueue.  This
 *    callback function can then perform any necessary deallocation of
 *    the user-provided message buffer.
 *
 *    Once the message is allocated, the buffer can be accessed by
 *    dereferencing the "msg_buf" member for types MPI_MSG_TYPE_USER and
 *    MPI_MSG_TYPE_DYN.  (Of course the original user-provided buffer may
 *    be accessed in the case of MPI_MSG_TYPE_USER).  For MPI_MSG_TYPE_SGL
 *    the payload is constructed piecemeal using the mpi_alloc_sgl_buf()
 *    and mpi_add_sgl_buf() methods.  The "msg_buf" member should not be
 *    accessed directly.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *
 *    type			Type of message to allocate
 *
 *    user_buf		Pointer to user provided msg buffer if type is 
 *                  MPI_MSG_TYPE_USER, NULL otherwise.
 *
 *    user_cb		For MPI_MSG_TYPE_USER type messages, this is an optional
 *    				callback function that is invoked when the message is
 *    				enqueued to allow, for example, freeing of the user buffer.
 *                  If callback is not needed, pass in NULL.  If type is not
 *					MPI_MSG_TYPE_USER, this parameter is ignored.
 *
 *	  msg_len		Message payload length in bytes
 *
 * Output:
 *	  new_msg		Pointer to the newly allocated message.
 *
 * Returns:
 *    0 			On success
 *    -ENOMEM		Unable to allocate message
 *    -EMSGSIZE		msg_len is invalid
 *    -EINVAL		Invalid hdl, type, new_msg pointer
 *    				Invalid user_buf if type is MPI_MSG_TYPE_USER
 */
extern int
mpi_alloc_msg(
	mpi_handle_t	hdl,
	mpi_msg_type_t	type,
	void			*user_buf,
	mpi_tx_cb_fn_t	user_cb,
	uint32_t		msg_len,
	mpi_msg_t		**new_msg);


/*
 * Function: mpi_free_msg()
 *
 * Description:
 *    Deallocates an Rx message or an unsent Tx message.
 *
 *    *** NOTE ***
 *    This function can also be called to free a previously allocated Tx
 *    message provided it has not been sent yet or the send resulted in a
 *    queue full error.  The mpi_send_msg() method normally frees messages
 *    once they have been successfully enqueued.  If the queue was full,
 *    the message is not freed, giving the sender a chance to resend it.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    msg			Pointer to message to be freed
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		Invalid hdl or msg
 */
extern int mpi_free_msg(mpi_handle_t hdl, mpi_msg_t *msg);


/*
 * Function: mpi_add_msg_hdr()
 *
 * Description:
 *    Adds a header buffer to a previously allocated message of any type.
 *    The header is used to hold any protocol data that should precede
 *    the payload contained in msg_buf.  When the message is
 *    enqueued, the header is stored first before msg_buf.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    msg			Pointer to the previously allocated message
 *    hdr_len		Header length in bytes
 *
 * Output:
 *	  hdr_buf		Pointer to new header buffer which is accessed like
 *	  				plain memory, e.g. memset(), memcpy(), etc.
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		Invalid hdl, msg or hdr_buf pointers.
 *    -ENOMEM		Failed to allocate header.
 */
extern int mpi_add_msg_hdr(mpi_handle_t hdl, mpi_msg_t *msg,
						   int hdr_len, void **hdr_buf);


/*
 * Function: mpi_alloc_sgl_buf()
 *
 * Description:
 *    Allocates a new buffer that is to be added to a previously allocated
 *    scatter/gather list message of type MPI_MSG_TYPE_SGL.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *	  buf_len		Buffer length in bytes
 *
 * Output:
 *	  ret_buf		Pointer to newly allocated SGL buffer
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		hdl invalid.
 *    -ENOMEM		Unable to allocate memory for SGL buffer.
 *    -EMSGSIZE		buf_len is invalid.
 */
extern int
mpi_alloc_sgl_buf(mpi_handle_t hdl, int buf_len, mpi_sgl_buf_t *ret_buf);


/*
 * Function: mpi_free_sgl_buf()
 *
 * Description:
 *    Deallocate a previously allocated scatter/gather list buffer.
 *
 *    *** NOTE ****
 *    This function should only be called to free a buffer that has NOT
 *    been added to a message.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *	  buf			Pointer to SGL buffer
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		Invalid hdl or buf pointers.
 */
extern int mpi_free_sgl_buf(mpi_handle_t hdl, mpi_sgl_buf_t buf);


/*
 * Function: mpi_add_sgl_buf()
 *
 * Description:
 *    Adds a buffer to an SGL message that was previously allocated.
 *    The new buffer is added to the end of the current list of buffers.
 *
 *    *** NOTE ***
 *    - A buffer may not be added to more than one message instance.
 *
 *    - If an allocated buffer is not added to a message, it must be freed
 *      by calling the mpi_free_sgl_buf() method.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    msg			Pointer to the previously allocated message
 *	  buf			Pointer to allocated SGL buffer
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		Invalid hdl, msg, msg type, or buf pointers.
 *    -ENOSPC		Max number of buffers exceeded.
 */
extern int
mpi_add_sgl_buf(mpi_handle_t hdl, mpi_msg_t *msg, mpi_sgl_buf_t buf);


/*
 * Function: mpi_num_sgl_bufs()
 *
 * Description:
 *    Return the number of buffers currently allocated for
 *    the specified scatter/gather list message.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    msg			Pointer to the previously allocated message
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    >= 0			Number of buffers on success
 *    -EINVAL		Invalid hdl, invalid msg pointer, or msg not SGL type
 */
extern int mpi_num_sgl_bufs(mpi_handle_t hdl, mpi_msg_t *msg);


/*
 * Function: mpi_get_sgl_buf_addr()
 *
 * Description:
 *    Return the address of a scatter/gather list buffer.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *	  buf			Pointer to buffer instance.
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    Address of buffer on success.
 *    NULL on invalid hdl or buf.
 */
extern void *mpi_get_sgl_buf_addr(mpi_handle_t hdl, mpi_sgl_buf_t buf);


/*
 * Function: mpi_get_sgl_buf_sz()
 *
 * Description:
 *    Get the current size of the specified buffer.  At creation, the size
 *    is set to the allocated size but may be changed by mpi_set_sgl_buf_sz().
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *	  buf			Pointer to buffer instance.
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    Size of buffer on success
 *    -EINVAL		Invalid hdl or buf.
 */
extern int mpi_get_sgl_buf_sz(mpi_handle_t hdl, mpi_sgl_buf_t buf);


/*
 * Function: mpi_set_sgl_buf_sz()
 *
 * Description:
 *    Sets the current size of the specified SGL buffer.  At creation, the size
 *    is set to the allocated size.
 *
 *    *** NOTE ***
 *    The new size may not be greather than the original allocated size.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *	  buf			Pointer to buffer instance.
 *	  size			Size to set to in bytes.
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0 			On success
 *    -EINVAL		Invalid hdl, buf, or size
 */
extern int mpi_set_sgl_buf_sz(mpi_handle_t hdl, mpi_sgl_buf_t buf, int size);


/*
 *-----------------------------------------------------------------------------
 *                          Message Rx/Tx
 *-----------------------------------------------------------------------------
 */

/*
 * Function: mpi_send_msg()
 *
 * Description:
 *    Sends the previously allocated message to the destination CQ.
 *    The send is performed in an asynchronous manner in that this
 *    function returns as soon as the message is queued.  It does
 *    not wait until the message is actually dequeued by its peer.  Once
 *    enqueued, the message is automatically freed.
 *
 *    *** NOTE ***
 *    If the send fails because of a queue full condition, the message
 *    is NOT freed.  This allows the client to retry sending.  If no
 *    retry is successful (or if it is not attempted), then the message
 *    must be manually freed by calling mpi_free_msg().
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq			Circular queue to enqueue message to
 *    msg			Pointer to a previously allocated message
 *
 * Output:
 *	  None.
 *
 * Returns:
 *	  0				On success
 *    -EINVAL		Invalid hdl, cq number, or msg pointer.
 *    -ENOSPC		Unable to queue msg for transmission.  The message is
 *    				NOT freed so that the caller can attempt retransmission.
 *    -ECONNRESET	Memory owner reset.  Shared memory no longer valid.
 */
extern int mpi_send_msg(mpi_handle_t hdl, mpi_cq_num_t cq, mpi_msg_t *msg);


/*
 * Function: mpi_recv_msg()
 *
 * Description:
 *    Receives a message from the specified CQ.
 *
 *    *** NOTE ***
 *    The received message must be freed by calling mpi_free_msg().
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq			Circular queue to receive message from
 *
 * Output:
 *    msg			Pointer to received message
 *
 * Returns:
 *	  0				On success.
 *    -EINVAL		Invalid hdl, cq number, or msg pointer.
 *	  -ENOMEM		No messages available to receive from specified cq.
 *    -ECONNRESET	Memory owner reset.  Shared memory no longer valid.
 */
extern int mpi_recv_msg(mpi_handle_t hdl, mpi_cq_num_t cq, mpi_msg_t **msg);


/*
 *-----------------------------------------------------------------------------
 *                        CQ Management and Status
 *-----------------------------------------------------------------------------
 */

/*
 * Function: mpi_flush_cq()
 *
 * Description:
 *    Flush the contents of the specified CQ.  Any pending receive messages
 *    are discarded and not processed.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq			Circular queue to flush
 *
 * Output:
 *    None.
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid hdl or cq number
 *    -ECONNRESET	Memory owner reset.  Shared memory no longer valid.
 */
extern int mpi_flush_cq(mpi_handle_t hdl, mpi_cq_num_t cq);


/*
 * Function: mpi_msgs_avail()
 *
 * Description:
 *    Determines if there are any messages queued on any of the CQs.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq_map		Pointer to return bitmap indicating which CQs have
 *    				messages queued.  Bit 0 = CQ 0, 1 = CQ 1, etc.
 *    				This parameter may be omitted by passing NULL.
 *
 * Output:
 *    None.
 *
 * Returns:
 *    -EINVAL		Invalid handle
 *    -ECONNRESET	Memory owner reset.  Shared memory no longer valid.
 *    0				No messages queued
 *	  >0			Total number of messages queued
 */
extern int mpi_msgs_avail(mpi_handle_t hdl, mpi_cq_map_t *cq_map);


/*
 * Function: mpi_send_capacity()
 *
 * Description:
 *    Determine if there is capacity in the specified CQ to send messages.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq			Circular queue to inspect
 *
 * Output:
 *    None.
 *
 * Returns:
 *    -EINVAL		Invalid handle or CQ number
 *    -ECONNRESET	Memory owner reset.  Shared memory no longer valid.
 *    0				CQ is full
 *	  >0			Number of messages that may be enqueued
 */
extern int mpi_send_capacity(mpi_handle_t hdl, mpi_cq_num_t cq);


/*
 * Function: mpi_get_cq_sz()
 *
 * Description:
 *    Returns the number of messages queued in the specified CQ.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq			Circular queue to check
 *
 * Output:
 *    None.
 *
 * Returns:
 *    -EINVAL		Invalid handle or CQ number
 *    -ECONNRESET	Memory owner reset.  Shared memory no longer valid.
 *    Number of receive messages queued
 */
extern int mpi_get_cq_sz(mpi_handle_t hdl, mpi_cq_num_t cq);


/*
 * Function: mpi_num_cq()
 *
 * Description:
 *    Returns the number of CQ's allocated to the specified MPI instantiation.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *
 * Output:
 *    None.
 *
 * Returns:
 *    -EINVAL		Invalid handle
 *    Number of CQ's allocated
 */
extern int mpi_num_cq(mpi_handle_t hdl);


/*
 *-----------------------------------------------------------------------------
 *                   Application Interface Functions
 *-----------------------------------------------------------------------------
 */

/*
 * Function: mpi_app_reg_hdlr()
 *
 * Description:
 *    Register a handler function for the specified SAP.  Upon receipt of
 *    a message, the MPI can consult an internal lookup table that maps
 *    SAP to callback function and if there is one registered, pass the
 *    message to it.  Note that it is allowable to map more than one
 *    SAP to a given callback function but it is assumed that the callback
 *    routine will demultiplex on SAP.
 *
 *    *** NOTE ***
 *    The application handler must call the mpi_free_msg() function once
 *    it is done with received messages.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    sap			Service Access Point to register.  Specifying 0 means
 *    				this handler is a default one that will be invoked
 *    				when no other handler could be matched.
 *    cb			Pointer to callback function
 *    arg			Argument that is passed to callback function to allow
 *    				application to pass back private data/context.
 *
 * Output:
 *    None.
 *
 * Returns:
 *	  0				On success.
 *    -EINVAL		Invalid handle or callback function
 *    				Already a handler registered for sap
 */
extern int 
mpi_app_reg_hdlr(mpi_handle_t hdl, mpi_sap_t sap, mpi_rx_cb_fn_t cb, void *arg);


/*
 * Function: mpi_app_send_msg()
 *
 * Description:
 *    Application level message send method which implements the platform-
 *    dependent logic to perform message parameter (e.g. SAP, OP) binding
 *    to CQ.  This function will invoke mpi_send_msg() with a CQ number that
 *    is based on platform-specific ordering and prioritization policies.
 *    The send message is created by calling mpi_alloc_msg() as usual.
 *
 *    *** NOTE ***
 *    When send fails because of a queue full condition, the message
 *    is NOT freed.  This allows the client to retry sending.  If no
 *    retry is successful (or if it is not attempted), then the message
 *    must be manually freed by calling mpi_free_msg().
 *
 *    Also note that this method serves as a virtual function, the
 *    implementation of which will be platform dependent.  Platforms that
 *    deal with CQs directly and do not need application layering do not
 *    have to provide an implementation for this method.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    msg			Pointer to message to send which was constructed
 *    				by calling the mpi_alloc_msg() method.
 *
 * Output:
 *    None.
 *
 * Returns:
 *	  0				On success.
 *    -EINVAL		Invalid hdl or msg.
 *    -ENOENT		Could not determine destination CQ.  Message is freed.
 *    -ENOSPC		Unable to queue msg for transmission.  The message is
 *    				NOT freed so that the caller can attempt retransmission.
 */
extern int mpi_app_send_msg(mpi_handle_t hdl, mpi_msg_t *msg);


/*
 * Function: mpi_dispatch_app_msgs()
 *
 * Description:
 *    Application level method that reads messages from the CQs and dispatches
 *    them to applications based on previous application cb registrations
 *    made by calling mpi_app_reg_hdlr().  This is a platform-dependent
 *    routine that will have the necessary understanding of message payload
 *    format to extract the SAP.  It will also apply any required ordering
 *    and prioritization of CQ servicing.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    cq_map		Optional CQ bitmap returned from mpi_msgs_avail()
 *    				that indicates CQs with pending messages.  If 0, 
 *    				all CQs should be examined.
 *
 * Output:
 *    None.
 *
 * Returns:
 *    -EINVAL		Invalid hdl
 *    -EIO			Warning: messages with no matching SAP handler were
 *    				encoutered and dropped.  All others were dispatched.
 *	  >= 0			Number of messages dispatched
 */
extern int mpi_dispatch_app_msgs(mpi_handle_t hdl, mpi_cq_map_t cq_map);


/*
 * Function: mpi_lookup_app_hdlr()
 *
 * Description:
 *    Given SAP, return the corresponding application handler mapping
 *    which provides call back function pointer and argument.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    sap			Service Access Point
 *
 * Output:
 *	  map			Pointer to mapping.  If mapping not required and just
 *	  				lookup is to be performed, pass in NULL.
 * Returns:
 *    0				On success.
 *    -EINVAL		Invalid hdl.
 *    -ENOENT		No mapping for specified sap
 */
extern int
mpi_lookup_app_hdlr(mpi_handle_t hdl, mpi_sap_t sap, mpi_sap_app_map_t **map);


#ifdef __cplusplus
}
#endif

#endif
