/*
 * Copyright (c) 2005-2007 Brocade Communications Systems, Inc.
 * All rights reserved.
 *
 * Description:
 *     Machine dependent Message Passing Interface (MPI) defines and function 
 *	   prototypes.
 *
 *	   This file contains the following CPU/OS combinations:
 *
 *	   1. PowerPC 4xx / Linux
 *	   2. BCM 1125/1250 / Presto
 *	   3. BCM 1125/1250 / SOS (SAS Operating System)
 *	   4. Solaris (host based unit test)
 *	   5. BCM 1250 / QNX
 */

#ifndef __MPI_MACH_DEP_H__
#define __MPI_MACH_DEP_H__

#ifdef	__KERNEL__
/* -------------------------------------------------------------------------- */
/*                             P P C  L I N U X                               */
/* -------------------------------------------------------------------------- */
#include <linux/kernel.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <asm/semaphore.h>
#if defined(CONFIG_4xx) || defined(CONFIG_MPC8548)

#ifdef	DEBUG

#define MPI_DBG(args...)	\
	{	\
		if (mpi_debug)	\
			printk(args);	\
	}

#else

#define MPI_DBG(args...)

#endif

#define MPI_MEMCPY	memcpy

#define MPI_MEMSET	memset

#define MPI_ADDR_CAST		unsigned int

typedef int mpi_dma_info_t;

/* Semaphore operations */
typedef struct semaphore mpi_sem_type_t;
#if 0
#define MPI_SEM_INIT(sem)	init_MUTEX(sem)
#define MPI_SEM_LOCK(sem)	down(sem)
#define MPI_SEM_UNLOCK(sem)	up(sem)
#else
#define MPI_SEM_INIT(sem)
#define MPI_SEM_LOCK(sem)
#define MPI_SEM_UNLOCK(sem)
#endif

#else
#error "Unsupported architecture!"
#endif	/* ifdef _CONFIG_4xx */

/* -------------------------------------------------------------------------- */
/*                 L I N U X  U S E R  U N I T  T E S T                       */
/* -------------------------------------------------------------------------- */
#elif defined(__LINUX_MPI__)	/* host based testing */
#include <stdio.h>
#include <sys/types.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>

#define MPI_ADDR_CAST		unsigned int

#define MPI_DBG(format,args...)	printf(format, ## args)

#define MPI_MEMCPY	memcpy

#define MPI_MEMSET	memset

typedef pthread_mutex_t mpi_sem_type_t;
#define MPI_SEM_INIT(sem)	pthread_mutex_init(sem, NULL)
#define MPI_SEM_LOCK(sem)	pthread_mutex_lock(sem)
#define MPI_SEM_UNLOCK(sem)	pthread_mutex_unlock(sem)

typedef int mpi_dma_info_t;


/* -------------------------------------------------------------------------- */
/*                           S O L A R I S                                    */
/* -------------------------------------------------------------------------- */
#elif defined(__SOLARIS_MPI__)	/* host based testing */
#include <stdio.h>
#include <inttypes.h>
#include <string.h>
#include <errno.h>
#if MPI_PTHREADS
#include <pthread.h>
#else
#include <thread.h>
#include <synch.h>
#endif

#define MPI_ADDR_CAST		unsigned int

#define MPI_DBG(format,args...)	printf(format, ## args)

#define MPI_MEMCPY	memcpy

#define MPI_MEMSET	memset

/* Semaphore operations */
#ifdef MPI_PTHREADS
typedef pthread_mutex_t mpi_sem_type_t;
#define MPI_SEM_INIT(sem)	pthread_mutex_init(sem, NULL)
#define MPI_SEM_LOCK(sem)	pthread_mutex_lock(sem)
#define MPI_SEM_UNLOCK(sem)	pthread_mutex_unlock(sem)
#else
typedef mutex_t mpi_sem_type_t;
#define MPI_SEM_INIT(sem)	mutex_init(sem, NULL, NULL)
#define MPI_SEM_LOCK(sem)	mutex_lock(sem)
#define MPI_SEM_UNLOCK(sem)	mutex_unlock(sem)
#endif

typedef int mpi_dma_info_t;


/* -------------------------------------------------------------------------- */
/*                    B R O A D C O M  1125 / 1250    Q N X                   */
/* -------------------------------------------------------------------------- */
#elif defined(__QNX__)
#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <pthread.h>
#include <errno.h>

#define MPI_ADDR_CAST		unsigned int

#ifdef	DEBUG

#define MPI_DBG(format,args...)	printf(format, ## args)

#else

#define MPI_DBG(format,args...)

#endif

#define MPI_MEMCPY	memcpy

#define MPI_MEMSET	memset

/* Semaphore operations */
typedef pthread_mutex_t mpi_sem_type_t;
#define MPI_SEM_INIT(sem)	pthread_mutex_init(sem, NULL)
#define MPI_SEM_LOCK(sem)	pthread_mutex_lock(sem)
#define MPI_SEM_UNLOCK(sem)	pthread_mutex_unlock(sem)

typedef int mpi_dma_info_t;


/* -------------------------------------------------------------------------- */
/*               B R O A D C O M  1125 / 1250    P R E S T O                  */
/* -------------------------------------------------------------------------- */
#else	/* assume it's BCM 1125/1250 Presto */
#include <stdio.h>
#include <pro_types.h>
#include <presto.h>
#include <presto_internal.h>
#include <lib_string.h>
#include <lib_spinlock.h>
#include <errno.h>

/* Debug logging */
#define MPI_DBG(format,args...)	printf(format, ## args)

#define MPI_MEMCPY	pro_memcpy

#define MPI_MEMSET	pro_memset

#define MPI_ADDR_CAST		unsigned int

/* Semaphore operations */
typedef spinlock_t mpi_sem_type_t;
#define MPI_SEM_INIT(sem)	spinlock_init(sem)
#define MPI_SEM_LOCK(sem)	spinlock_acquire(sem)
#define MPI_SEM_UNLOCK(sem)	spinlock_release(sem)

typedef int mpi_dma_info_t;

#endif	/* __KERNEL__ */


/*
 * When volatile memory is defined it means that the non-memory owner may
 * see shared memory become inaccessible at anytime.  Therefore additional
 * checks should be done before referencing CQ management datastructures.
 */
#if	defined(__KERNEL__) && defined(CONFIG_SILKWORM8548)
#define MPI_VOLATILE_MEM		1
#endif

/*
 * The following indicates that on the non-memory owner side receive
 * messages should not be copied to a heap buffer since no intermediate
 * operations will be performed.  The rx message will only be copied out.
 * The one exception is if the message is composed of multiple linked
 * memory blocks (i.e. SGL).  In that case a heap buffer is still required
 * to create a final contiguous message payload.
 */
#if	defined(__KERNEL__) && defined(CONFIG_SILKWORM8548)
#define MPI_NON_OWNER_NO_RX_HEAP	1
#endif

#endif	/* __MPI_MACH_DEP_H__ */
