/*
 *    Copyright (c) 2003-2007 Brocade Communications Systems, Inc.
 *    All rights reserved.
 *
 *    Description:
 *      Definitions of data structures in 1250's DRAM that gets exposed as PCI 
 *      memory mapped IO region(PMMR) to CP. These data structures are used to
 *      communicate between CP and 1250.  Also includes definitions for
 *      1250 DRAM that is used for message passing with the BP.
 *
 *      Instead of using 1250 as the prefix for all datastructures, we have
 *      choosen to use SAS (IP Processor) prefix. This will hopefully enable
 *      us to move to new chips (from 1250 family) without causing 
 *      code-readability issues.
 *
 *      Abbreviations used:
 *      	SAS 	- IP processor (1250 at present)
 *      	PMMR 	- PCI Memory Mapped Region
 *      	CQID	- Circular Queue ID
 *      	CI		- Consumer Index
 *      	PI		- Producer Index
 */

#ifndef __SAS_PMMR_H__
#define __SAS_PMMR_H__

#ifdef __KERNEL__
#include <linux/types.h>
#include <linux/string.h>
#elif defined __SOS__
#include "sos.h"
#include "types.h"
#else
#include <sys/types.h>
#endif /* __KERNEL__ */

#include <mpi/mpi_mem_sz.h>			/* for MPI defines */
#include <sas/sas_public.h>			/* for sas events and structs	*/

#include "sflipper_reg_def.h"
#include "anz_shrmem.h"

/*
 * #############################################################################
 * Handy Macros:
 * #############################################################################
 */
/*
 * These macros are to be used in the register bit definitions so that we think
 * only of bit positions as shown in condor.pdf instead of 1 in a sea of 0's.
 * That should prevent human errors of translating bit positions into masks.
 *
 * An alternate form of "#define RBIT(bit) (1 << (bit))" is not used because
 * we think that it's too compiler dependent - where some compilers may decide
 * not to pre-process the shift into a fixed constant and instead do the
 * shifting during execution time.
 */
#define	RBIT_31	0x80000000
#define	RBIT_30	0x40000000
#define	RBIT_29	0x20000000
#define	RBIT_28	0x10000000
#define	RBIT_27	0x08000000
#define	RBIT_26	0x04000000
#define	RBIT_25	0x02000000
#define	RBIT_24	0x01000000
#define	RBIT_23	0x00800000
#define	RBIT_22	0x00400000
#define	RBIT_21	0x00200000
#define	RBIT_20	0x00100000
#define	RBIT_19	0x00080000
#define	RBIT_18	0x00040000
#define	RBIT_17	0x00020000
#define	RBIT_16	0x00010000
#define	RBIT_15	0x00008000
#define	RBIT_14	0x00004000
#define	RBIT_13	0x00002000
#define	RBIT_12	0x00001000
#define	RBIT_11	0x00000800
#define	RBIT_10	0x00000400
#define	RBIT_9 	0x00000200
#define	RBIT_8	0x00000100
#define	RBIT_7	0x00000080
#define	RBIT_6	0x00000040
#define	RBIT_5	0x00000020
#define	RBIT_4	0x00000010
#define	RBIT_3	0x00000008
#define	RBIT_2	0x00000004
#define	RBIT_1	0x00000002
#define	RBIT_0	0x00000001

/*
 * This defines a bit-field.
 * Offset - right most bit position, Width - field width.
 */
#define BITFLD(offset,width)    ( ((1<<(width))-1) << (offset))

/*
 * #############################################################################
 * Begin -- Defines & Data Structures for NPSAS driver
 * #############################################################################
 */

#define	NPSAS_RTE_DOMN_TBL_SZ		240
#define	NPSAS_RTE_AREA_TBL_SZ		256
#define NPSAS_RTE_EGID_ENTRY_SZ 	512
#define	NPSAS_RTE_DP_ENTRY_SZ		8

#define	NPSAS_BI_VC_MIN				1	/* min VC# used by the BI port */
#define	NPSAS_BI_VC_MAX				8	/* max VC# used by the BI port */
#define NPSAS_ANZ_VC_PER_EGID		2	/* Two VC per 1250 EGID */
#define NPSAS_EVEN_VC_START			2
#define NPSAS_ODD_VC_START			1

#define	NPSAS_MAX_ANZIO_DPORTS		8

#define	NPSAS_VALID_EGID(egid)	(((egid) >= 0) && ((egid) < SAS_EGID_SZ))
/* Valid bit definitions for (egid_sas_t *)->valid */
#define	NPSAS_RTE_INVALID			0	/* ISW_INVALID_ENTRY */
#define	NPSAS_RTE_VALID				1	/* Route entry is valid */
/* Valid bit definitions for (egid_sas_t *)->egid[] */
#define	NPSAS_RTE_EMB				RBIT_10			/* ISW_EMB_ENTRY */

/*
 * BITFLD(offset, width) If new fields are added, update SAS_RTE_ID_MASK also.
 */
#define NPSAS_RTE_ID_MASK			BITFLD(0,11)	/* entire route ID mask */
#define	NPSAS_RTE_EMB_OFFSET		10				/* ISW_EMB_OFFSET */
#define NPSAS_RTE_EMB_MASK			BITFLD(10,1)	/* ISW_EMB_MASK */
#define	NPSAS_RTE_LSW_OFFSET		9				/* Local SW: don't route to */
#define NPSAS_RTE_LSW_MASK			BITFLD(9,1)		/* Flsaser C_Port */
#define NPSAS_RTE_EGID_OFFSET		0				/* ISW_EGID_OFFSET */
#define NPSAS_RTE_EGID_MASK			BITFLD(0,9)		/* ISW_EGID_MASK */

/*
 * Bit definition for the "flags" word of the npsas_rte_pol_t block. The
 * external bit of the flags word (rte_ioc_policy_t) is not carried over
 * since it's not applicable.
 */
#define	NPSAS_RTE_HASH_XOR    		0	/* XOR folding bit */
#define	NPSAS_RTE_HASH_CRC10  		1	/* CRC 10 */
#define	NPSAS_RTE_HASH_RR			2  	/* round-robin */

/* RTE policy structure that defines the policy parameters */
typedef volatile struct npsas_rte_pol_s {
	uint32_t	hash_mode;		/* refer above for various modes */
	uint32_t	seed;			/* 10-bits seed */
	uint32_t	rxport_mask;	/* 5-bits mask */
	uint32_t	stag_mask;		/* 16-bits mask */
	uint32_t	dtag_mask;		/* 16-bits mask */
	uint32_t	sid_mask;		/* 24-bits mask */
	uint32_t	did_mask;		/* 24-bits mask */
	uint32_t	oxid_mask;		/* 16-bits mask */
	uint32_t	rxid_mask;		/* 16-bits mask */
	uint32_t	reserved[5];
} npsas_rte_pol_t;

/* Definition for EGID type */
typedef uint32_t		egid_t;

typedef volatile struct npsas_rte_dpsentry_s {
	uint32_t   	valid;              /* 0 == invalid, non-zero == valid */
	egid_t   	egids[NPSAS_RTE_DP_ENTRY_SZ];	/* bits 0-8 - EGID */
	uint32_t	num_paths;		/* no. of valid EGIDs in the egid[] */
	uint32_t	reserved[6];
} npsas_rte_dpsentry_t;

/*
 * Per Anzio port route related information. The routes are
 * from condor-to-Anzio port to any user port on the switch.
 */
typedef volatile struct npsas_rte_pmmr_s {
	npsas_rte_dpsentry_t	domn[NPSAS_RTE_DOMN_TBL_SZ]; /* domain table */
	npsas_rte_dpsentry_t	area[NPSAS_RTE_AREA_TBL_SZ]; /* area table */
	uint32_t				stag;
	uint32_t				reserved[5];
} npsas_rte_pmmr_t;

/*
 * #############################################################################
 * End -- Defines & Data Structures for NPSAS driver
 * #############################################################################
 */

#define	SAS_MAGIC			0xdeadbead	/* wr by 1250 per fwdl ok */
#define SAS_PMMR_SRE_NAME	"sas_pmmr"	/* PMMR shared region name */
#define SAS_MPI_SRE_NAME	"sas_mpi"	/* MPI shared region name */
#define SAS_MAX_PORTS		10			/* num of ports supported */
#define SAS_NUM_CQS			2			/* num of CQ RX/TX pairs */
#define SAS_CQ_ENTRIES		64			/* num of entries in each CQ pair */
#define SAS_CQ_PG_SZ		128			/* size of MPI storage page in bytes */
#define SAS_CQ_NUM_PG		1024		/* num of pages in free page pool */

#define SAS_MPI_SIZE	\
	(MPI_MEM_SZ(SAS_NUM_CQS, SAS_CQ_ENTRIES, SAS_CQ_PG_SZ, SAS_CQ_NUM_PG))

/*
 * CQID assignments:
 *	These cannot be changed without changing both the FCIP asic driver and
 *	the 1250 microcode.
 */
#define SAS_CQID_CORE0_CTL		0		/* CQ_ID for core0 ctrl cmds & events */
#define SAS_CQID_CORE1_CTL		1		/* CQ_ID for core1 ctrl cmds & events */

/*
 * SAP assignments:
 *	These cannot be changed without changing both the FCIP asic driver and
 *	the 1250 microcode.
 */
/*
 * SAPs used by CPSAS Driver to send messages to 1250
 */
#define SAS_SAP_ANZIO_CTL		0		/* SAP for Anzio ports' cmds */
#define SAS_SAP_SFLIPPER_CTL	1		/* SAP for Sflipper ports' cmds */
#define SAS_SAP_NPSAS0_CTL		2		/* SAP for core0 htbt, NPSAS cmds */
#define SAS_SAP_NPSAS1_CTL		3		/* SAP for core1 htbt */

/*
 * SAPs used by 1250 drivers to send messages to CPSAS
 */
#define SAS_SAP_CPSAS_CTL		0		/* SAP for CPSAS events */

/*
 * BP to 1250 MPI defines.  These instances will be used by DPC on the 1250s
 * to communicate with internal SAC running on the BP.
 */
#define SAS_NP_BP_SRE_NAME		"np_bp_mpi"	/* MPI shared region name */
#define SAS_NP_BP_NUM_CQS		1
#define SAS_NP_BP_CQ_ENTRIES	0x2000	/* 8K per SAS team sizing */
#define SAS_NP_BP_PG_SZ			256
#define SAS_NP_BP_NUM_PG		0x4000	/* 16K per sas team sizing */

/*
 * Port numbers for Anzio
 */
#define SAS_PORT_ANZIO0			0		/* Anzio port 0 */
#define SAS_PORT_ANZIO1			1		/* Anzio port 1 */
#define SAS_PORT_ANZIO2			2		/* Anzio port 2 */
#define SAS_PORT_ANZIO3			3		/* Anzio port 3 */
#define SAS_PORT_ANZIO4			4		/* Anzio port 4 */
#define SAS_PORT_ANZIO5			5		/* Anzio port 5 */
#define SAS_PORT_ANZIO6			6		/* Anzio port 6 */
#define SAS_PORT_ANZIO7			7		/* Anzio port 7 */
#define SAS_PORT_ANZIO8			8		/* Anzio port 8 */

/*
 * Port numbers for Sflipper
 */
#define SAS_PORT_SFLIPPER0		9		/* Sflipper port 0 */


/*
 * chip.info.chip_status codes from 1250 during the init sequence
 */
#define SAS_STATUS_NO_INIT				0x0
#define SAS_STATUS_INIT					0x1		/* app took ctl from CFE */
#define SAS_STATUS_MSG_WAIT				0x2		/* waits for HTBT msg from CP */
#define SAS_STATUS_ACTIVE				0x4		/* HTBT msg rx, & response tx */
#define SAS_STATUS_INIT_FAILED			0x8
#define SAS_STATUS_MSG_WAIT_FAILED		0x10
#define SAS_STATUS_TX_FAILED			0x20
#define SAS_STATUS_FAULT				0x40
#define SAS_STATUS_SFPR_CLK150_FAILED	0x80
#define SAS_STATUS_SFPR_CLK125_FAILED	0x100
#define SAS_STATUS_SFPR_GBUS_FAILED		0x200
#define SAS_STATUS_FAILED				(SAS_STATUS_INIT_FAILED | \
								 SAS_STATUS_MSG_WAIT_FAILED | \
							 	SAS_STATUS_TX_FAILED | \
								 SAS_STATUS_FAULT | \
								SAS_STATUS_SFPR_CLK150_FAILED | \
								SAS_STATUS_SFPR_CLK125_FAILED | \
								SAS_STATUS_SFPR_GBUS_FAILED)

/*
 * Trace buffer defines
 */
#define SAS_TRACE_MAX_SZ	(5 *1024) /* Max no entires in 1250 Trace Buufer */
#define SAS_TRACE_STR_SZ	92		  /* Size of info in each trace entry	*/

/*
 * Basic configuration parameters pushed from CP to 1250
 * Currently, GE ports are not enabled.  This MAC address is just left
 * as place holder.
 */

typedef enum {
    SAS_BCFG_INVALID = 0,
    SAS_BCFG_VALID   = 1
} sas_bcfg_valid_t;

typedef volatile struct sas_bcfg {
    sas_bcfg_valid_t valid;   		/* validity of configuration info       */
	/*
	 * The following struct mimics and used as a substitute for ieee48_t
	 * which is used to pass down the MAC address from the blade layer.
	 * Unfortunately the 1250/IPS port code world does not understand
	 * ieee48_t and it's decided not to create this new define in that world.
	 *
	 * Per src/sys/include/fc/wwn.h
	 *	IEEE (48-bit) Universal LAN Address (ULA)
	 *		typedef struct ieee48 {
	 *			u_char      ie_ula_hi[3];
	 *			u_char      ie_ula_lo[3];
	 *		} ieee48_t;
	 */
#ifdef TO_BE_FIXED
	struct ge_mac_mimic_ieee48_t {
		u_char		ie_ula_hi[3];
		u_char		ie_ula_lo[3];
	} ge_mac;
#endif	/* TO_BE_FIXED */
	u_char			pad[2];
	uint32_t		pad2[32];
} sas_bcfg_t;


typedef struct  sfpr_csi_intr_stats_s {
   	uint32_t    num_125_dcm_intr    ;
   	uint32_t    num_150_dcm_intr     ;
    uint32_t    num_fps_stats_intr  ;
    uint32_t    num_gpio_read_chg_intr   ;
    uint32_t    num_comp_intr       ;
    uint32_t    num_txq_intr        ;
    uint32_t    num_fpl_intr        ;
    uint32_t    num_fcip_intr       ;
    uint32_t    num_gpio_bus_par_err_intr;
} sfpr_csi_intr_stats_t;
/* Interrupt statistics from TXQ/CR Block */

typedef struct  sfpr_txq_intr_stats_s {
	uint32_t    num_txq_over_run                ;
	uint32_t    num_txq_p1_over_run             ;
	uint32_t    num_misalign_detect_from_1250   ;
	uint32_t    num_p1_misalign_detect_from_1250   ;
	uint32_t    num_crc_err_from_1250           ;
	uint32_t    num_p1_crc_err_from_1250        ;
	uint32_t    num_sf_fifo_oversize_frm        ;
	uint32_t    num_p1_sf_fifo_oversize_frm     ;
	uint32_t    num_par_err_comp_sf_fifo        ;
	uint32_t    num_p1_par_err_comp_sf_fifo     ;
    uint32_t    num_txq_par_err     ;
    uint32_t    num_runt_from_fc    ;
    uint32_t    num_runt_from_1250  ;
    uint32_t    num_tx_buf_collison ;
    uint32_t    num_rx_buf_overrun  ;
    uint32_t    num_tx_buf_oversize ;
    uint32_t    num_ctrl_word_err   ;
    uint32_t    num_runt_from_1250_2  ;
    uint32_t    num_tx_buf_collison_2 ;
    uint32_t    num_tx_buf_oversize_2 ;
    uint32_t    num_ctrl_word_err_2   ;
    uint32_t    num_par_err_rx_mem  ;
    uint32_t    num_par_err_tx_mem  ;
} sfpr_txq_intr_stats_t;

typedef struct	sfpr_fpl_intr_stats_s {
	uint32_t		num_efifo_overflow;
	uint32_t		num_efifo_underflow;
	uint32_t		num_serdes_tx_fifo_error;
} sfpr_fpl_intr_stats_t;

typedef struct sfpr_frame_stats_t {
#ifdef __SOS__
	atomic_t    num_crc_err;
	atomic_t    num_dscr_err;
	atomic_t    num_bad_frame_err;
	atomic_t	num_sfpr_bad_tagcsum;
	atomic_t    num_sfpr_bad_crc;
	atomic_t    num_sfpr_inv_code;
	atomic_t    num_sfpr_inv_eof;
	atomic_t    num_sfpr_perr;
	atomic_t    num_sfpr_too_long;
#else
	uint32_t    num_crc_err;
	uint32_t    num_dscr_err;
	uint32_t    num_bad_frame_err;
	uint32_t	num_sfpr_bad_tagcsum;
	uint32_t    num_sfpr_bad_crc;
	uint32_t    num_sfpr_inv_code;
	uint32_t    num_sfpr_inv_eof;
	uint32_t    num_sfpr_perr;
	uint32_t    num_sfpr_too_long;
#endif
} sfpr_frm_stats_t;

typedef struct sfpr_sw_stat_s {
	sfpr_csi_intr_stats_t csi;
	/* Interrupt statistics from TXQ/CR Block */
	sfpr_txq_intr_stats_t txq;
	sfpr_fpl_intr_stats_t fpl;
	sfpr_frm_stats_t	frm;
} sfpr_sw_stats_t;

/* sflipper port structs */
typedef volatile struct sfpr_pmmr_s {
	/* Registers */
	sfpr_csi_reg_t		csi;
	sfpr_fps_reg_t		fps;
	sfpr_comp_decomp_reg_t comp;
	sfpr_fcip_reg_t		fcip;
	sfpr_txq_reg_t		txq;
	sfpr_bbc_reg_t		bbc;
	sfpr_fpl_reg_t		fpl;

	uint32_t    	    pt_status;	/* BI Port status */
	uint32_t			pad[31];
	sfpr_sw_stats_t		sw_stats;	/* software error stats */
	uint32_t			pad2[45];
} sfpr_pmmr_t;


#ifdef LATER_CPSAS
typedef struct port_cfg {
	uint32_t 		ha_spad; 		/* used across failover 	*/
	uint32_t 		excess_ptintr_thresh;	/* per-port config */
	uint32_t 		rttov;			/* per-port config */
	uint32_t 		busybuf_tov;		/* per-port config */
	uint32_t 		non_crit_par_thresh;	/* per-chip config */
	uint32_t 		excess_chintr_thresh;	/* per-chip config */
	uint32_t 		err_sample_period;	/* per-chip config */
	uint32_t		sw_err_thresh;
	uint32_t		data_par_thresh;
	uint32_t		ctrl_par_thresh;
	uint32_t		pad[32];
} port_cfg_t;

typedef struct _credit_model {
	uint16_t	num_vcs; /* number of VCs supported */
	uint16_t 	min_bufs; /* min buffers that has to be allocated to port */
	/*
	 * Following field is used dependent on the port type (FE vs BE)
	 * FE : define the minimum buffers per VC
	 * BE : define the num_vcs and vc_alloc for self 
	 * in the blade definition and later during attach
	 * gets replaced by their respective peer port values before 
	 * being pushed down to the ASIC driver
	 */
	uint16_t	*vc_alloc; /* min buffers per VC */
	uint16_t	*norm_vc_alloc; /* normal buffers per VC : ONLY for FE */
	uint16_t	cr_shar_bufs; /* number of buffers that can be shared */
} cr_model_t;

/* Arguments for sending the port property */
typedef struct sas_prop_arg {
    	cr_model_t  	cr_model;	/* credit model */
    	link_cap_t  		link_cap;	/* link capabilities */
    	port_cfg_t	 	port_cfg;	/* port configurations */
} sas_prop_arg_t;   

/* SAS payload  */
typedef struct sas_payld {
	uint32_t    	cmd;		/*  command/event; must be 1st field */
	int         	port_num;	/*  port number */
	union {
		sas_prop_arg_t prop;		/* arguments for the port property */
		sas_cmd_arg_t 	cmd_arg;	/* generic command arguments */
	}args;
} sas_payld_t;
#endif	/*  LATER_CPSAS */


/*
 * Structure for the individual entry in the Trace Buffer to commuicate between
 * 1250 and CP
 */
typedef volatile struct {
	uint32_t			sid;
	char				tdata[SAS_TRACE_STR_SZ];
} sas_tbuffentr_t;

#ifdef TO_BE_FIXED	/* Exist in ../ipp/ipp_pmmr.h */
/*
 * Trace buffer Register sub-structs
 */
typedef volatile struct tb_cfg {
	uint8_t				console;	/* flag for console	*/
	uint8_t				buffer;		/* flag for pci shared memory buffer	*/
	uint8_t				flag[15];	/* trace flag for each of 1250 modules	*/
} tb_cfg_t;

typedef volatile struct tb_info {
	uint32_t			cur_entr;	/* index for storing next trace entry	*/
	uint32_t			seq_no;		/* overall seqence no of entry	*/
	uint32_t			roll_over;	/* no of roll over in the buffer */
} tb_info_t;

typedef volatile struct tb_pmmr {
	tb_cfg_t			cfg;		/* fields written by CP   */
	tb_info_t			info;		/* fields written by 1250 */
} tb_pmmr_t;

#endif	/* TO_BE_FIXED */

typedef volatile struct pt_pmmr {
	uint32_t				ha_spad; /* per-port scratch pad word */
	uint32_t				status;	 /* port status on 1250 side */
	npsas_rte_pmmr_t		rte;	 /* per-port routing table information */
	union {
		anz_pmmr_t		anzio;	/* Anzio reg fields copied from shared mem */
		sfpr_pmmr_t		sfpr;	/* Sflipper reg fields copied to shared mem */
	} pt_info;
} pt_pmmr_t;

/*
 * Chip level sub-structs
 */
typedef volatile struct saschip_cfg {
	uint32_t			cp_status;	 /* reflects CP's current status */
	sas_bcfg_t   		basic_cfg;   /* basic configuration for 1250 */

	npsas_rte_pol_t		rte_pol;	 /* RTE config policy and routing info */

	uint32_t	rx_intr_enable; 	 /* Bitmap used by CP to request a doorbell
									  * interrupt from the 1250 whenever it 
									  * queues a message on a CQ. 
									  *
									  * This field will be cleared during 
									  * doorbell interrupt handling on CP
									  *
									  * When this flag is set, 1250 need
									  * not assert interrupt line
									  */ 
	uint32_t 	tx_intr_enable; 	 /* A Bitmap set by CP when it detects 
									  * Tx Queue Full condition on a CQ.
									  *
									  * On 1250 side, if tx_intr_enable is set,
									  * on consuming an Rx msg, 1250 will:
									  * 	o set tx_intr_status[CQ_ID]
									  * 	o trigger the doorbell interrupt to 
									  * 	  CP
									  *
									  * This field should be cleared by CP 
									  * after processing the tx_intr_status 
									  * based interrupt indication.
									  *
									  * Reverse logic is not needed on 1250 
									  * side because it will just use polling 
									  * to detect space in Tx CQ 
									  */

	uint32_t	rte_uc_port2vc[2];	/* portID to VC mapping per-chip registers */
										
	uint32_t	pad[32];
} sas_chip_cfg_t;

typedef volatile struct saschip_info {
	uint32_t	rx_intr_status[SAS_NUM_CQS];
									/* will be set to 1 after queuing a msg,
									 * if rx_intr_enable was set.
									 *
									 * This field should be cleared by CP
									 * (rx_intr_enable should be unset at that
									 * time) on processing the interrupt
									 */
	uint32_t	tx_intr_status[SAS_NUM_CQS];
									/* see tx_intr_enable comments also.
									 * 
									 * This flag should be cleared by CP after
									 * processing the tx_intr (should ensure
									 * that tx_intr_enable is unset at that
									 * time). 
									 *
									 * Should be cleared on processing the 
									 * interrupt
									 */
	uint32_t    chip_status;     	/* reflects 1250's current status	*/
	uint32_t    cpu_status[2];		/* for two CPUs status */
	uint32_t    sfp_status;     	/* SFP states. Msg will (also) be sent on 
									 * status change 
								 	 */
	uint32_t    rev_bootprom_np;	/* 1250 boot prom version */
	uint32_t    rev_image_np;		/* 1250 image version */
	uint32_t    rev_bootprom_ipsec;	/* 4350 boot prom version */
	uint32_t    rev_image_ipsec;	/* 4350 image version */
	uint32_t    rev_sflipper;		/* flipper fpga version */
	uint32_t    rev_anzio;			/* anzio fpga version */
	uint32_t	pad[27];
	uint32_t	pad2[32];
} sas_chip_info_t;

typedef volatile struct saschip_pmmr {
	sas_chip_cfg_t	cfg;		/* fields written by CP; read-only for 1250 */
	sas_chip_info_t	info; 		/* fields written by 1250; read-only for CP */
} sas_chip_pmmr_t;

/*
 * 1250 Trace Buffer
 */
typedef volatile struct sas_tbuff {
	sas_tbuffentr_t		tbuffer[SAS_TRACE_MAX_SZ];
} sas_tbuff_t;

/*
 * 1250 Core dump fields
 */
#ifdef TO_BE_FIXED	/* already defined in ../ipp/ipp_pmmr.h */
#define CP_MAX_CONN          2
#define CP_MAX_PENDING_FRAME 1

#define SAS_STACK_TRACE_SZ	(10 * 1024)
#define SAS_GLOBALS_SZ		(2  * 1024)

typedef struct cpu_context_s {
	uint64_t ef_regs[32];
	uint64_t ef_cause;
	uint64_t epc_reg;
	uint32_t stack_trace[SAS_STACK_TRACE_SZ];
	uint32_t globals[SAS_GLOBALS_SZ];
} cpu_context_t;

typedef struct coredump_s {
	uint32_t	coredump_magic;	/* magic ID for the Core */
	uint32_t	coredump_flags;	/* Core dump Valid/Invalid */
	uint32_t	coredump_type;	/* type exception WD vs SW error */
	uint32_t	coredump_id;	/* valid length of Core Dump */
	cpu_context_t	cpu[2];		/* core information	*/
} coredump_t;

#define RUNTIME_VERSION_SUPPORT 1 
#define RUNTIMEIMAGE_FIRMWARE  0x1000
#define RUNTIMEIMAGE_FCIP      0x2000

typedef volatile struct fixed_pmmr {
  uint32_t pci_fixed_pad[1023];
  uint64_t runtime_imagetype; 
  uint64_t bcm_firmware;
  uint64_t brcd_firmware;
  uint64_t bcm_software;
  uint64_t brcd_software;
  uint64_t core_status;	/* Core dump status	*/
  coredump_t coredump;	/* Core dump data	*/
} fixed_pmmr_t;

#endif	/* TO_BE_FIXED */
/* 
 *
 * Main structure to interpret 1250's memory mapped region (through PCI)
 *
 */

#define PCI_STRUCT_VERSION 0x0000000000000002
 
typedef volatile struct sas_pmmr {
	uint32_t			magic;		/* magic # by 1250 per successful fwdl */
#if (RUNTIME_VERSION_SUPPORT == 1)
    fixed_pmmr_t		fixed_area; /* DO NOT MOVE THIS STRUCT */
#endif /* RUNTIME_VERSION_SUPPORT */
	uint64_t			version;
	pt_pmmr_t			port[SAS_MAX_PORTS]; /* port info */
	sas_chip_pmmr_t		chip;		/* chip info */
	uint32_t			dump_in_progress;	/* flag for panic dump state */
#ifdef TO_BE_FIXED		/* exist in ../ipp/ipp_pmmr.h */
	tb_pmmr_t			tb;			/* trace buffer registers */
#endif
	sas_tbuff_t			tbuff;		/* 1250 trace buffer */
	uint32_t			pad[32];
} sas_pmmr_t;

#endif /* __SAS_PMMR_H__ */
