#ifndef _CSCN_H_
#define _CSCN_H_

#ifdef _KERNEL
#include <sys/types.h>
#include <machine/param.h>
#include <sys/mbuf.h>
#include <netinet/in.h>
#else
#include <stdtypes.h>
#include <stdlib.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/un.h>
#include <sys/uio.h>
#include <sys/time.h>
#include <syslog.h>
#include <sys/time.h>
#include <signal.h>
#endif /* !_KERNEL */

#include "cs.h"
#include "csq.h"
#include "csdbg.h"
#include "csmsg.h"


#ifdef _KERNEL
#define CS_CN_LOCALHOST             "127.0.0.1"
#else
#define CS_CN_LOCALHOST             "localhost"
#endif

#define CS_CN_IOV_MAX               4

#define CS_CN_HOSTNAME_MAX          256
#define CS_CN_FILENAME_MAX          256

#define CS_CN_RETRY_INFINITE        -1

/* return values  */
typedef enum {
    CS_CN_RVAL_SUCCESS              =  0,
    CS_CN_RVAL_CLOSE                =  1,
    CS_CN_ERR_CN_ALLOC              = -1,
    CS_CN_ERR_RETRY_OVER            = -2,
    CS_CN_ERR_LISTEN                = -3,
    CS_CN_ERR_ACCEPT                = -4,
    CS_CN_ERR_FCNTL_NONBLK          = -5,
    CS_CN_ERR_TMR_CREATE            = -6,
    CS_CN_ERR_IO                    = -7,
    CS_CN_ERR_BUF_OVERFLOW          = -8,
    CS_CN_ERR_GET_SOCKNAME          = -9,
    CS_CN_ERR_CONNRESET             = -10
} cs_cn_rval_t;

/* message handle */
typedef void    *cs_cn_msg_hdl_t;

/* timer  handle */
typedef void    *cs_cn_tmr_hdl_t;

/* connection handle */
typedef void    *cs_cn_hdl_t;


/* message switch table */
typedef struct {
    void   (*fn)(void *arg);
    void   *arg;
} cs_cn_msg_sw_t;

typedef enum {
    CS_CN_MSG_WAITING = 1,
    CS_CN_MSG_WAIT_COMP,
} cs_cn_msg_wait_state_t;

/* wait queue struct */
typedef struct {
    cs_q_t                  qe;
    cs_cn_msg_sw_t          sw;
    cs_cn_msg_wait_state_t  state;
} cs_cn_msg_wait_qe_t;

#define cs_cn_msg_wait_cb_set(_wq, _fn, _arg)                                \
                                        {(_wq)->sw.fn = _fn;                 \
                                        (_wq)->sw.arg = (void *)_arg;}
                                        
typedef struct {
    int count;
    int bytes;
    struct iovec  iov[CS_CN_IOV_MAX];
} cn_iov_t;

/* This represents the size of the xpmsg_buffer within cn_msg_hdl_t */
#define CS_CN_MAX_XPMSG_BUF 56

/* message structure */
typedef struct {
    cs_q_t          qe;
    unsigned int    mn1;
    unsigned int    mn2;
    cs_cn_msg_sw_t  sw;
    int             status;
    struct {
        unsigned int is_iov               :1;
        unsigned int is_msgl              :1;
        unsigned int hdr_read_complete    :1;
        unsigned int is_xbar              :1;
    } flags;
    union {
        /* generic messaging header struct */
        struct  {
            int hdr_off;
            int hdr_len;
            int pl_off;
            int pl_len;
        } msg;
        struct  {
            int rd_off;
            int rd_len;
            int rd2_off;
            int rd2_len;
        } msg_csmsg_xbar;
        /* iovec messaging header struct */
        struct  {
            cn_iov_t    usr_iov;
            cn_iov_t    cn_iov;
            uint8_t     xpmsg_buffer[CS_CN_MAX_XPMSG_BUF];
        } msgv;
    } u;
    void                (*last_recv)(void *);
} cn_msg_hdl_t;


#define cs_cn_msg_status_get(_m)                                              \
                    (((cn_msg_hdl_t *)_m)->status)


#define cs_cn_msg_cb_set(_m, _fn, _arg)                                       \
                    {((cn_msg_hdl_t *)_m)->sw.fn = _fn;                       \
                    ((cn_msg_hdl_t *)_m)->sw.arg = (void *)_arg;}

#define cs_cn_msg_buf_get(_m)                                                 \
                    (char *)(((cn_msg_hdl_t *)_m) + 1)

#define cs_cn_txt_msg_len_get(_m)                                             \
                    (((cn_msg_hdl_t *)_m)->u.txt_msg_len)

#define cs_cn_msgv_count_set(_m, _cnt)                                        \
                    (((cn_msg_hdl_t *)_m)->u.msgv.usr_iov.count = _cnt)

#define cs_cn_msgv_iov_set(_m, _i, _base, _len)                               \
            {((cn_msg_hdl_t *)_m)->u.msgv.usr_iov.iov[_i].iov_base =_base;    \
            ((cn_msg_hdl_t *)_m)->u.msgv.usr_iov.iov[_i].iov_len =_len;}

/* timer struct */
typedef struct cn_tmr_hdl_s {
    cs_q_t  qe;
    void    (*fn)(void *arg);
    void    *arg;
    long    msecs;
    struct  {
        unsigned int    periodic    :1;
        unsigned int    inuse       :1;
    } flags;
    struct  timeval tv;
    struct  cn_tmr_hdl_s *nextp;
} cn_tmr_hdl_t;

/* connection switch table */
typedef struct cn_sw_s {
    int     recv_hdr_len;
    int     (*recv_pl_len)  (cs_cn_hdl_t ch, cs_cn_msg_hdl_t msg);
    void    (*recv)         (cs_cn_hdl_t ch, cs_cn_msg_hdl_t msg);
    void    (*accept)       (cs_cn_hdl_t ch);
    void    (*connect)      (cs_cn_hdl_t ch, cs_cn_rval_t error);
    void    (*error)        (cs_cn_hdl_t ch, cs_cn_rval_t error);
} cs_cn_sw_t;

typedef struct cs_cn_sw_rw {
    void    (*read_ready)(void *uctx, cs_cn_hdl_t ch);
    void    (*write_ready)(void *uctx, cs_cn_hdl_t ch);
} cs_cn_sw_rw_t;

typedef struct cn_hdl_s cn_hdl_t;
/* connection struct */
struct cn_hdl_s {
    cs_q_t              qe;
    struct {
        unsigned int is_read         :1;
        unsigned int is_write        :1;
        unsigned int is_inprogress   :1;
        unsigned int is_txt          :1;
        unsigned int error           :1;
        unsigned int msg_in_waitq    :1;
        unsigned int msgl_reqd       :1;
        unsigned int is_xbar         :1;
        unsigned int is_csmsg_xbar   :1;
    } flags;
    int                 type;
    int                 fd;
    int                 qnum;
    cs_cn_sw_t          sw;
    cs_cn_sw_rw_t       rw_sw;
    void                *uctx;
    int                 retry_cnt;
    int                 retry_max;
    long                retry_msecs;
    cn_tmr_hdl_t        *retry_tmr;
    bool_t              is_tcp;
    union {
        struct sockaddr_in  saddr_tcp;        
        struct sockaddr_un  saddr_uds;    /* for UDS */
    } saddr;

    union {
        struct sockaddr_in  caddr_tcp;
        struct sockaddr_un  caddr_uds;    /* for UDS */ 
    } caddr;

    char                host[CS_CN_HOSTNAME_MAX];
    char                file[CS_CN_FILENAME_MAX];
    cs_cn_msg_wait_qe_t waitqe;
    cn_msg_hdl_t        *rmsg, *smsg;
    cs_q_t              smsgq;
    int                 nsmsg;
#ifdef _KERNEL
    cs_q_t              mbuf_q;
#endif
};


/* api's to get info from ch */
#define cs_cn_host_get(_ch)           (((cn_hdl_t *)_ch)->host)
#define cs_cn_server_in_addr_get(_ch) (((cn_hdl_t *)_ch)->saddr.saddr_tcp.sin_addr)
#define cs_cn_client_in_addr_get(_ch) (((cn_hdl_t *)_ch)->caddr.caddr_tcp.sin_addr)
#define cs_cn_server_ip_get(_ch)      (((cn_hdl_t *)_ch)->saddr.saddr_tcp.sin_addr.s_addr)
#define cs_cn_client_ip_get(_ch)      (((cn_hdl_t *)_ch)->caddr.caddr_tcp.sin_addr.s_addr)
#define cs_cn_server_port_get(_ch)    (ntohs(((cn_hdl_t *)_ch)->saddr.saddr_tcp.sin_port))
#define cs_cn_client_port_get(_ch)    (ntohs(((cn_hdl_t *)_ch)->caddr.caddr_tcp.sin_port))
#define cs_cn_queue_number_get(_ch)   (((cn_hdl_t *)_ch)->qnum)
#define cs_cn_uctx_get(_ch)           (((cn_hdl_t *)_ch)->uctx)


/* config defaults */
#define CS_CN_MSG_HDR_LEN_MAX         sizeof(cn_msg_hdl_t)
#define CS_CN_MSG_FC_PL_LEN_MAX       4*1024 /* this need to be >= SBMSG_MAX_SIZE  */
                                             /* defined in /sys/dev/switch/pcmsg.h */
#define CS_CN_MSG_PL_LEN_MAX          65*1024 /* arbitrary for now */

#define CS_CN_CONFIG_MSG_NUM_INIT     50
#define CS_CN_CONFIG_MSG_GROWBY       50
#define CS_CN_CONFIG_MSG_MAX          250

#define CS_CN_CONFIG_LMSG_NUM_INIT    1
#define CS_CN_CONFIG_LMSG_GROWBY      1
#define CS_CN_CONFIG_LMSG_MAX         3

#define CS_CN_CONFIG_CN_MAX           16


typedef struct {
        int init;
        int growby;
        int max;
        int alloc_len;
} msg_cfg_t;

/* config struct */
typedef struct {
    msg_cfg_t   msg;
    msg_cfg_t   msgl;
    struct {
        int     max;
        struct {
            unsigned int tcp_nodelay       :1;
        } flags;
    } cn;
    struct {
        uint32_t debug : 1;
        uint32_t reserved : 31;
    } flags;
} cs_cn_cfg_t;


/* connection messaging api's */

/* initialization and configuration */
void cs_cn_config_get(cs_cn_cfg_t *cfg);
extern void cs_cn_config (cs_cn_cfg_t *cfg);
extern int  cs_cn_init (bool_t is_daemon, char *ident, int logopt,
                        void (*sigfn)(int sig));
void cs_cn_stop(void);
void cs_cn_destroy(void);

/* call attach if desired after accept cb is called to change sw
 * else listen sw is copied over to new connection
 */
extern void cs_cn_attach  (cs_cn_hdl_t ch, cs_cn_sw_t *sw, void *uctx);

/* call attach if desired after accept cb is called to change sw
 * to read-write sw
 */
extern void cs_cn_attach_rw(cs_cn_hdl_t ch, cs_cn_sw_rw_t *rw_sw, void *uctx);

/* call setsockopt if desired after connect or accept to change
 * scoket options
 */
int cs_cn_setsockopt(cs_cn_hdl_t ch, int level, int optname, void *optval,
                                                        socklen_t optlen);
/* start new connections */
extern cs_cn_hdl_t cs_cn_listen  (cs_cn_sw_t *sw, void *uctx, int port,
                                char *host);
/* The default protocol for connect is tcp */
extern cs_cn_hdl_t cs_cn_connect (cs_cn_sw_t *sw, void *uctx, int port,
                                  char *host, int retry_max, long retry_msecs);
extern cs_cn_hdl_t cs_cn_connect_uds(cs_cn_sw_t *sw, void *uctx, int port, 
                                     char *host, int retry_max, 
                                     long retry_msecs);

extern cs_cn_hdl_t cs_cn_listen_tcp  (cs_cn_sw_t *sw, void *uctx, int port, 
                                      char *host);
extern cs_cn_hdl_t cs_cn_listen_uds(cs_cn_sw_t *sw, void *uctx, int port, 
                                    char *host);
#ifdef XBAR
extern cs_cn_hdl_t cs_cn_connect_xbar (cs_cn_sw_t *sw, void *uctx, char *fname,
                                       bool_t is_hipriority);
#endif

#ifndef _KERNEL
extern cs_cn_hdl_t cs_cn_file_open (cs_cn_sw_t *sw, void *uctx, char *file,
                                    int flags, mode_t mode);
extern int  cs_cn_ioctl     (cs_cn_hdl_t ch, unsigned long request, void *arg);
extern int  cs_cn_getservport(const char *name);
#endif

/* messaging - must set cb in msg hdr using cs_cn_msg[v]_cb_set()
 * will call cb->fn(cb->arg) on send completion
 */
extern void cs_cn_msg_send  (cs_cn_hdl_t ch, cs_cn_msg_hdl_t msg, int len);
extern void cs_cn_msg_sendv (cs_cn_hdl_t ch, cs_cn_msg_hdl_t msgv);

#ifdef XBAR
extern int  cs_cn_xbar_hdr_get(cs_cn_hdl_t ch, csmsg_hdr_t *csmsg,
                               uint32_t len, uint32_t dap);
#endif
extern int  cs_cn_smsgq_cnt_get(cs_cn_hdl_t ch);


/* close - can be called anytime */
extern void cs_cn_close (cs_cn_hdl_t ch);

/* main loop - never returns */
extern void cs_cn_main (void);

/* messaging pool mgmt api's */
extern cs_cn_msg_hdl_t  cs_cn_msg_alloc (void);
extern void             cs_cn_msg_free (cs_cn_msg_hdl_t msg);
extern int              cs_cn_msg_used_len_get(cs_cn_msg_hdl_t msg);
extern int              cs_cn_msg_unused_len_get(cs_cn_msg_hdl_t msg);
extern void             cs_cn_msg_wait (cs_cn_msg_wait_qe_t *mwqe);
extern void             cs_cn_msg_wait_cancel (cs_cn_msg_wait_qe_t *mwqe);

/* timer api's */
extern cs_cn_tmr_hdl_t  cs_cn_tmr_create (long msecs, bool_t periodic,
                                            void (*fn)(void *arg), void *arg);
extern void             cs_cn_tmr_cancel (cs_cn_tmr_hdl_t th);
extern void             cs_cn_tmr_reset (cs_cn_tmr_hdl_t th);

/* common utl api's */
extern void cs_cn_utl_setsig (int32_t sig, void (*fn)(int sig));


#endif /* _CSCN_H_ */
