/*****************************************************************************
 *
 * Copyright 2000 Rhapsody Networks Inc
 *
 * Module Name:
 *     csq.h
 *
 * Abstract:
 *     Implements common queueing macros.
 *
 * Revision History:
 *     $Revision: 1.3 $
 *
 ****************************************************************************/

#ifndef _csq_h
#define _csq_h

#include "./csdbg.h"

typedef struct cs_q {
    struct cs_q *next;
    struct cs_q *prev;
} cs_q_t;


#define cs_q_next(_qe)  (((cs_q_t *) (_qe))->next)
#define cs_q_prev(_qe)  (((cs_q_t *) (_qe))->prev)

/*
 * cs_q_init - to initialize a queue
 */
#define cs_q_init(_q) {                                                   \
    cs_q_next(_q) = (cs_q_t *) (_q);                                      \
    cs_q_prev(_q) = (cs_q_t *) (_q);                                      \
}

/*
 * cs_q_qe_init - to initialize a queue element
 */
#define cs_q_qe_init(_qe) {                                               \
    cs_q_next(_qe) = (cs_q_t *) NULL;                                     \
    cs_q_prev(_qe) = (cs_q_t *) NULL;                                     \
}

/*
 * cs_q_enq - enqueue an element at the end of the list
 */
#define cs_q_enq(_q, _qe) {                                             \
    cs_assert((cs_q_next(_qe) == NULL) && (cs_q_prev(_qe) == NULL));    \
    cs_q_next(_qe) = (cs_q_t *) (_q);                                   \
    cs_q_prev(_qe) = cs_q_prev(_q);                                     \
    cs_q_next(cs_q_prev(_q)) = (cs_q_t *) (_qe);                        \
    cs_q_prev(_q) = (cs_q_t *) (_qe);                                   \
}

/*
 * cs_q_enq_head - enqueue an element at the head of queue
 */
#define cs_q_enq_head(_q, _qe) {                                        \
    cs_assert((cs_q_next(_qe) == NULL) && (cs_q_prev(_qe) == NULL));    \
    cs_q_next(_qe) = cs_q_next(_q);                                     \
    cs_q_prev(_qe) = (cs_q_t *) (_q);                                   \
    cs_q_prev(cs_q_next(_q)) = (cs_q_t *) (_qe);                        \
    cs_q_next(_q) =  (cs_q_t *) (_qe);                                  \
}

/*
 * cs_q_is_empty - returns TRUE if queue is empty
 */
#define cs_q_is_empty(_q)                                                 \
    (cs_q_next(_q) == ((cs_q_t *) (_q)))

/*
 * cs_q_enq_q - enqueue another queue at the tail
 */
#define cs_q_enq_q(_dstq, _srcq) {                                      \
    if (!cs_q_is_empty(_srcq)) {                                        \
        cs_q_next(cs_q_prev(_srcq)) = (cs_q_t *) (_dstq);               \
        cs_q_prev(cs_q_next(_srcq)) = cs_q_prev(_dstq);                 \
        cs_q_next(cs_q_prev(_dstq)) = cs_q_next(_srcq);                 \
        cs_q_prev(_dstq) = cs_q_prev(_srcq);                            \
        cs_q_init(_srcq);                                               \
    }                                                                   \
}

/*
 * cs_q_enq_q_head - enqueue another queue at the head
 */
#define cs_q_enq_q_head(_dstq, _srcq) {                                 \
    if (!cs_q_is_empty(_srcq)) {                                        \
        cs_q_next(cs_q_prev(_srcq)) = cs_q_next(_dstq);                 \
        cs_q_prev(cs_q_next(_srcq)) = (cs_q_t *) (_dstq);               \
        cs_q_prev(cs_q_next(_dstq)) = cs_q_prev(_srcq);                 \
        cs_q_next(_dstq) = cs_q_next(_srcq);                            \
        cs_q_init(_srcq);                                               \
    }                                                                   \
}

/*
 *  cs_q_qe_deq - dequeue a queue element from a queue
 */
#define cs_q_qe_deq(_qe) {                                              \
    cs_assert(cs_q_next(_qe) && cs_q_prev(_qe));                        \
    cs_q_next(cs_q_prev(_qe)) = cs_q_next(_qe);                         \
    cs_q_prev(cs_q_next(_qe)) = cs_q_prev(_qe);                         \
    CS_Q_DBG_INIT(_qe);                                                 \
}

/*
 * cs_q_deq - dequeue an element from head of the queue
 */
#define cs_q_deq(_q, _qe) {                                             \
    if (!cs_q_is_empty(_q)) {                                           \
        (*((cs_q_t **) (_qe))) = cs_q_next(_q);                         \
        cs_q_prev(cs_q_next(*((cs_q_t **) _qe))) = (cs_q_t *) (_q);     \
        cs_q_next(_q) = cs_q_next(*((cs_q_t **) _qe));                  \
        CS_Q_DBG_INIT(*((cs_q_t **) _qe));                              \
    } else {                                                            \
        *((cs_q_t **) (_qe)) = (cs_q_t *) NULL;                         \
    }                                                                   \
}

/*
 * cs_q_deq_tail - dequeue an element from tail of the queue
 */
#define cs_q_deq_tail(_q, _qe) {                                        \
    if (!cs_q_is_empty(_q)) {                                           \
        *((cs_q_t **) (_qe)) = cs_q_prev(_q);                           \
        cs_q_next(cs_q_prev(*((cs_q_t **) _qe))) = (cs_q_t *) (_q);     \
        cs_q_prev(_q) = cs_q_prev(*(cs_q_t **) _qe);                    \
        CS_Q_DBG_INIT(*((cs_q_t **) _qe));                              \
    } else {                                                            \
        *((cs_q_t **) (_qe)) = (cs_q_t *) NULL;                         \
    }                                                                   \
}

/*
 * cs_q_mv - move all queue elements from one to another queue
 */
#define cs_q_mv(_srcq, _dstq) {                                         \
    cs_assert(cs_q_is_empty(_dstq));                                    \
                                                                        \
    if (!cs_q_is_empty(_srcq)) {                                        \
        cs_q_next(_dstq) = cs_q_next(_srcq);                            \
        cs_q_prev(_dstq) = cs_q_prev(_srcq);                            \
        cs_q_prev(cs_q_next(_srcq)) = (cs_q_t *) _dstq;                 \
        cs_q_next(cs_q_prev(_srcq)) = (cs_q_t *) _dstq;                 \
        cs_q_init(_srcq);                                               \
    }                                                                   \
}

extern int cs_q_is_on_q_func(cs_q_t *q, cs_q_t *qe);

#ifdef CS_DEBUG

#define CS_Q_DBG_INIT(_qe)      cs_q_qe_init(_qe)
#define cs_q_is_queued(_qe)     (cs_q_next(_qe) && cs_q_prev(_qe))
#define cs_q_is_on_q(_q, _qe)   cs_q_is_on_q_func(_q, (cs_q_t *)(_qe))

#else

#define CS_Q_DBG_INIT(_qe)

#endif


#endif /* _csq_h */

