#!/bin/sh

# httpd - This shell script starts/restart/stops the Apache Server

BASE=/fabos/webtools/bin
CONF=$BASE/httpd.conf
CTRL=$BASE/apachectl

RM=/bin/rm
SED=/bin/sed
LN=/bin/ln
PIDOF=/sbin/pidof
MKDIR=/bin/mkdir
LOGDIR=/tmp/webtools
PIDDIR=/tmp/webtools
OUTDEVICE=$PIDDIR/webtools.alert
LOGFILE=$PIDDIR/httpd.log
WDIR=/fabos/webtools/htdocs
WEBLINKER=weblinker.fcg
STATUSPAGE=$WDIR/serverstatus.html
TOUCHDIR=/bin/touch

# Apache options

HTTPD=/usr/apache/bin/httpd
HTTPDOPTS="-f"

logMessage() {
  msg=$*
	dt=`date '+%d%b%y_%H%M%S'`

	# Output to console always
  echo $msg

	# Output headers for first time
	if [ ! -f $OUTDEVICE ]; then
		echo "Webtools not available alert" >> $OUTDEVICE
		echo "----------------------------" >> $OUTDEVICE
		echo "Creation date: $dt" >> $OUTDEVICE
		echo "" >> $OUTDEVICE
		echo "Switch Instance Id -> $INST_ID" >> $OUTDEVICE
		echo "---------------------------" >> $OUTDEVICE
		echo "Reason: " >> $OUTDEVICE
		echo "------- " >> $OUTDEVICE
		echo "" >> $OUTDEVICE
	fi
	echo [$dt]: $msg >> $OUTDEVICE
}

vlog () {
  msg=$*
	dt=`date '+%d%b%y_%H%M%S'`
  if [ ! -d $PIDDIR ] ; then
    $MKDIR -p $PIDDIR
	  chmod 1777 $PIDDIR
	fi
	if [ -f $LOGFILE ]; then
	  echo [$dt]: $msg >> $LOGFILE
	fi
}

checkPaths()
{
  # This directory must be created since it is not saved in the flash
	if [ -x $MKDIR ] ; then
		$MKDIR -p $LOGDIR ;
	    chmod 1777 $LOGDIR
	    if [ -x $TOUCHDIR ] ; then
			$TOUCHDIR $OUTDEVICE
			$TOUCHDIR $LOGFILE  
		fi
	else 
		# Assuming tmp must exist
		OUTDEVICE=/tmp/webtools.alert
		logMessage "Unable to create $LOGDIR directory, $MKDIR not found." ;
		exit 0 ;
	fi

	if [ ! -x $SED ] ; then
		logMessage "$SED not found, http server not started." ;
		exit 0 ;
	fi

	if [ ! -x $RM ] ; then
		logMessage "Warning: $RM not found" ;
	fi

  if [ ! -d $SOCKDIR ] ; then
    $MKDIR -p $SOCKDIR
	  chmod -R 1777 $SOCKDIR
	fi

	# bail out if httpd does not exist

  if [ ! -f $HTTPD ] ; then 
		logMessage "Missing executable file: $HTTPD"
		exit 1;
	else
		if [ ! -L $HTTPD.$INST_ID ]; then
			vlog "Creating symbolic link: $HTTPD.$INST_ID"
			$LN -s $HTTPD $HTTPD.$INST_ID 2>/dev/null
		else
			vlog "Symbolic link $HTTPD.$INST_ID already exist."
		fi
	fi

	if [ ! -L $HTTPD.$INST_ID ]; then
		logMessage "Missing executable link: $HTTPD.$INST_ID"
		exit 1;
	fi

	# bail out if weblinker does not exist

	winst=$INST_ID.$WEBLINKER
	wsrc=$WDIR/$WEBLINKER
	wtar=$WDIR/$winst

  if [ ! -f $wsrc ] ; then 
		logMessage "Missing fcgi executable file: $wsrc"
		exit 1;
	else
		if [ ! -L $wtar ]; then
			vlog "Creating symbolic link: $wtar"
			$LN -s $wsrc $wtar 2>/dev/null
		else
			vlog "Symbolic link $wtar already exist."
		fi
	fi

	vlog "Running server as user: $RUNASUSER"

	if [ "x$RUNASUSER" = "xroot" ] ; then
		chmod 755 $wsrc
	else
		chmod 4755 $wsrc
	fi

	if [ ! -L $wtar ]; then
		logMessage "Missing fcgi executable link: $wtar"
		exit 1;
	fi

} 


# Set up configuration files for starting multiple instances

setupConfig()
{
	INST_CONF=`echo $CONF'.'$INST_ID`

  $SED -e '
    /^Include/s|^.*|Include '"$CONFFILE"'|g
    /FastCgiIpcDir/s|^.*|FastCgiIpcDir '"$SOCKDIR"'|g
    /FastCgiServer/s|-idle-timeout [ ]*[0-9]*|-idle-timeout '"$FCGITIMEOUT"'|g
		/weblinker.fcg/s|weblinker.fcg|'"$INST_ID"'.weblinker.fcg|g
   ' $CONF > $INST_CONF
	 return 0;
}

cleanup()
{
	INST_CONF=`echo $CONF'.'$INST_ID`
	INST_CTRL=`echo $CTRL'.'$INST_ID`

	if [ -f $INST_CONF ] ; then
		vlog Removing file $INST_CONF ...
		$RM $INST_CONF ;
	fi

 	if [ -f $INST_CTRL ] ; then
		vlog Removing file $INST_CTRL ...
		$RM $INST_CTRL ;
	fi

	if [ -d $SOCKDIR ] ; then
		vlog Removing file $SOCKDIR ...
		$RM -rf $SOCKDIR ;
	fi
}

# This clean up is required if apache shutdown was not graceful
# Pid's stored in the files can be recycled for another process
# that will cause script to think httpd is running (via kill -0)

cleanupApacheFiles()
{
  INST_PID=`echo $PIDDIR/httpd.pid.$INST_ID`
  INST_LOCK=`echo $PIDDIR/httpd.lock.$INST_ID`

	if [ -f $INST_PID ] ; then
		PID=`cat $INST_PID 2>/dev/null`
		if [ "x$PID" != "x" ] && kill -0 $PID 2>/dev/null ; then
			vlog "Process with pid [$PID] is still running ..."
		  vlog "File $INST_PID not removed." 
		else
		  vlog Removing file $INST_PID ...
		  $RM $INST_PID 2>/dev/null;
			if [ -f $INST_LOCK ] ; then
				vlog Removing file $INST_LOCK ...
				$RM $INST_LOCK 2>/dev/null;
			fi
			cleanup
		fi
	fi

	# remove after debugging

	if [ -f /core ] ; then
		vlog "httpd not started (crash) ?"
		$RM -f /core ;
	fi

	# these files are created by mmlib
	vlog "Removing /usr/apache/logs files ..."
	$RM -rf /usr/apache/logs/*
}


# Check apache (http) status
# return 1, if still running, 0 otherwise

checkApacheStatus()
{
  INST_PID=`echo $PIDDIR/httpd.pid.$INST_ID`
  INST_LOCK=`echo $PIDDIR/httpd.lock.$INST_ID`

	if [ -f $INST_PID ] ; then
		PID=`cat $INST_PID 2>/dev/null` 
		if [ "x$PID" != "x" ] && kill -0 $PID 2>/dev/null ; then
			vlog "Process with pid [$PID] is running ..."
			return 1;
		fi
	fi

	return 0;
}

# Start http server
# return 0 on success, 1 on failure

startApache()
{
	INST_CFG=$CONF.$INST_ID

	if [ ! -f $INST_CFG ]; then 
		logMessage "Missing configuration file: $INST_CFG"
		return 1
	fi

	# Launch Apache

	vlog "Starting http server [$INST_ID] ..."
	vlog $HTTPD.$INST_ID $HTTPDOPTS $INST_CFG

	$HTTPD.$INST_ID $HTTPDOPTS $INST_CFG 
	sleep 2;
	checkApacheStatus

	if [ $? -ne 1 ] ; then
		vlog "httpd may NOT be started."
		return 1;
	else
		vlog "httpd started."
  fi

	return 0;
}

# Stop http server

stopApache()
{
  INST_PID=`echo $PIDDIR/httpd.pid.$INST_ID`
  INST_LOCK=`echo $PIDDIR/httpd.lock.$INST_ID`
	if [ -f $INST_PID ] ; then
		PID=`cat $INST_PID 2>/dev/null` 
		if [ "x$PID" != "x" ] && kill -0 $PID 2>/dev/null ; then
			vlog "Stopping http server with pid [$PID] ..."
			kill -TERM $PID
		fi
	fi
	return 0;
}


killApache()
{
	INST_PID=`echo $PIDDIR/httpd.pid.$INST_ID`
	INST_LOCK=`echo $PIDDIR/httpd.lock.$INST_ID`
		if [ -f $INST_PID ] ; then
			PID=`cat $INST_PID 2>/dev/null`
			if [ "x$PID" != "x" ] && kill -0 $PID 2>/dev/null ; then
				vlog "Stopping(sigkill) http server with pid [$PID] ..."
				kill -KILL $PID
			fi
		fi
	return 0;
}

retryStopApache()
{
	vlog "retry stop apache"
	a=0
	checkApacheStatus
	if [ $? -eq 0 ] ; then
		vlog "httpd is NOT running"
		return 1
	fi
	while [ $a -lt 4 ]; do
		stopApache
		cleanupApacheFiles
		checkApacheStatus
		if [ $? -eq 0 ] ; then
			vlog "httpd stopped"
			return 0
		else
			sleep 1;
			checkApacheStatus
			if [ $? -eq 0 ] ; then
				vlog "httpd stopped"
				return 0
			fi
			a=`expr $a + 1`
		fi
	done
	vlog "try to stop apache with SIGKILL on a final attempt"
	killApache
}


# Restart http server

restartApache()
{
  INST_PID=`echo $PIDDIR/httpd.pid.$INST_ID`
  INST_LOCK=`echo $PIDDIR/httpd.lock.$INST_ID`

	if [ -f $INST_PID ] ; then
		PID=`cat $INST_PID 2>/dev/null` 
		if [ "x$PID" != "x" ] && kill -0 $PID 2>/dev/null ; then
			vlog "Sending HUP to http server with pid [$PID] ..."
			if kill -HUP $PID ; then
			  vlog "http server restarted."
			fi
		fi
	fi
	return 0;
}


# Recover HTTP if possible

recoverHttp()
{
	httpInst=httpd.$INST_ID
	weblinkerInst=$INST_ID.$WEBLINKER

	vlog "Stopping all $httpInst process..."
	httplist=`$PIDOF $httpInst`  2>/dev/null
	vlog "httpd pid list: $httplist"
	for i in $httplist
	do
		vlog "Stopping http process $i ..."
		kill -TERM $i 2>/dev/null
	done
	sleep 3;

	weblinkerlist=`$PIDOF $weblinkerInst`  2>/dev/null
	vlog "weblinker pid list: $weblinkerlist"

	for i in $httplist
	do
		if kill -0 $i 2>/dev/null ; then
			vlog "Process $i still running (possibly zombie), force termination..."
			kill -9 $i 2>/dev/null
		fi
	done
	sleep 3;
	for i in $weblinkerlist
	do
		if kill -0 $i 2>/dev/null ; then
			vlog "Process weblinker $i still running (possibly zombie), force termination..."
			kill -9 $i 2>/dev/null
		fi
	done

	return 0;
}

execCmd()
{
	checkPaths

	case $CMD in
		start)  
			 # start server
			checkApacheStatus
			if [ $? -ne 1 ] ; then
				setupConfig 
				if [ $? -eq 0 ] ; then
					startApache
				else
					vlog "HTTP Configuration incomplete, server not started."
				fi
			else
				vlog "httpd is already running."
			fi
			;; 
		stop)
			# stop server
			retryStopApache
			;;
		restart)
			# restart server
			checkApacheStatus
			if [ $? -eq 1 ] ; then
				setupConfig
				if [ $? -eq 0 ] ; then
			  	restartApache
				else
					vlog "HTTP Configuration incomplete, server not restarted."
				fi
			else
				vlog "No http server, HUP will not be sent"
				startApache
			fi
			;;
		recover)
			# Recover http
			recoverHttp
			;;
		none)
			# usage
			usage
			;;
	esac
}

usage()
{
	echo "Usage: $0 -i <instance> -c <cmd> -f <conffile> -d <http_define> -t <fcgitimeout> -r <user>"
	exit 0;
}

showEnv()
{
	echo "CMD = $CMD"
	echo "INST_ID = $INST_ID"
	echo "HTTPDOPTS = $HTTPDOPTS"
	echo "CONFFILE = $CONFFILE"
	echo "FCGITIMEOUT = $FCGITIMEOUT"
	echo "RUNASUSER = $RUNASUSER"
}

parseCmdlineArgs()
{
	while getopts i:c:f:d:t:r: opt $*
	do
		case $opt in
			i) INST_ID=$OPTARG;;
			c) CMD=$OPTARG;;
			f) CONFFILE=$OPTARG;;
			t) FCGITIMEOUT=$OPTARG;;
			d) HTTPDEFINES=$OPTARG
				 HTTPDOPTS="-D$HTTPDEFINES $HTTPDOPTS"
			;;
			r) RUNASUSER=$OPTARG;;
			\?) usage;;
		esac
	done
}

shift `expr $OPTIND - 1`
vlog "cmd: $*"
parseCmdlineArgs $*

INST_ID=${INST_ID:="0"}
CONFFILE=${CONFFILE:="/dev/null"}
FCGITIMEOUT=${FCGITIMEOUT:="90"}
CMD=${CMD:="none"}
RUNASUSER=${RUNASUSER:="nobody"}

SOCKDIR=$PIDDIR/tmp_$INST_ID
FABOS_SWITCHNO=$INST_ID

# Create a dummy file for status check

if [ ! -f $STATUSPAGE ]; then 
	vlog "Creating dummy page: $STATUSPAGE"
	/bin/touch $STATUSPAGE 2>/dev/null
fi

# Fabric OS env

SWITCHHOME=/fabos
TZENVFILE=/fabos/cliexec/tsSetEnvTZ

if [ -f $TZENVFILE ]; then
	source $TZENVFILE
fi

TZ=${TZ:="UTC"}
export SWITCHHOME FABOS_SWITCHNO TZ

execCmd $CMD
