# Helper utilities for shell script fault logging
#
# This file should be "source"d by callers to pick up the functions.
#

FAULTDIR=/mnt/sysfs/fault
FAULTLOG=$FAULTDIR/fault.log

APPFS_VERSION_FILE=/mnt/apps/version
ROOTFS_VERSION_FILE=/version

# -----------------------------------------------------------------------------
initialize_fault_log() {
    # Ensure flash log exists with correct permissions
    # (must be done after filesystems are mounted)

    # See if the fault log is getting too full, and trim it if so.
    if [ -f "$FAULTLOG" ] ; then
        size=$(ls -l $FAULTLOG | awk '{print $5}')
        if [ "$size" -gt "204800" ] ; then
            # Assume that the caller has sourced /ciena/scripts/utils.sh
            # (to provide print_app_line and print_result).
            print_app_line "archiving fault log"
	    /ciena/scripts/fault_log_compress
            print_result $?
            rm -f /tmp/fault.log             # if the move fails, rm tmpfs file
        fi
    fi

    # Force creation of a fault log file (if necessary), and fix permissions.
    # Note that this will not affect the log contents if it already exists.
    /bin/mkdir -p $FAULTDIR
    /bin/touch $FAULTLOG
    /bin/chmod 666 $FAULTLOG
}

# -----------------------------------------------------------------------------
generate_fault_banner() {
    echo "************** FAULT REPORT ******************" >>$FAULTLOG
    echo "$(date)" >>$FAULTLOG

    if [ -f "$APPFS_VERSION_FILE" ] ; then
        echo "appfs version  : $(cat $APPFS_VERSION_FILE)" >>$FAULTLOG
    fi

    if [ -f "$ROOTFS_VERSION_FILE" ] ; then
        echo "rootfs version : $(cat $ROOTFS_VERSION_FILE)" >>$FAULTLOG
    fi
}

# -----------------------------------------------------------------------------
generate_core_log() {

    local exec_name=$1   ; shift
    local pid=$1         ; shift
    local timestamp=$1   ; shift
    local file_name=$1   ; shift
    local thread_name=$1 ; shift
    local write_error=$1 ; shift
    local fancy_name

    # check if the thread_name is different from the executable name
    if [ "$exec_name" == "$thread_name" ] ; then
        fancy_name="$exec_name($pid)"
    else
        fancy_name="$thread_name, $exec_name($pid)"
    fi

    # write an entry to the fault log
    mkdir -m 775 -p $FAULTDIR 2>/dev/null
    generate_fault_banner
    echo "Core dump from $fancy_name$write_error" >>$FAULTLOG

    if [ -f "$file_name" ]; then
        ls -al "$file_name" >>$FAULTLOG 2>&1
        /ciena/bin/evt_log CRIT "$fancy_name: core file $file_name$write_error"
        /ciena/bin/evt_copy ram0 flash0
    else
        if [ -z "$write_error" ] ; then
            echo "(corefile max reached)" >>$FAULTLOG
        fi
    fi
}

