# Helper utilities for init scripts
#
# This file should be "source"d by callers to pick up the functions.
#

source /ciena/scripts/kernel_arg.sh
source /ciena/scripts/board_lib.sh

SUCCESS_STR="OK"
FAIL_STR="FAILED"

# -----------------------------------------------------------------------------
print_app_line() {
    printf "%-8.8s   %-50s " "$(basename $0)" "$1"
}

# -----------------------------------------------------------------------------
print_app_line_cr() {
    print_app_line "$*"
    echo
}

# -----------------------------------------------------------------------------
bg_run() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    $exec_name $* &
    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_and_wait_timeout() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    timeout=$1    ; shift

    print_app_line "$app_string"
    /bin/$exec_name $* &
    /ciena/bin/waitfor -t $timeout $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_and_wait() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    pid_file="/var/run/$(basename $exec_name).pid"

    print_app_line "$app_string"
    $exec_name $* &
    /ciena/bin/waitfor $pid_file

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
valgrind_bg_run_and_wait() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    /usr/bin/valgrind --log-socket=$VALGRIND_IP:$VALGRIND_PORT /bin/$exec_name $* &
    /ciena/bin/waitfor -t 60 $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_log_and_wait() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    echo "*** Log of $exec_name output ($(date))" >> /var/log/$exec_name.log
    /bin/$exec_name $* >> /var/log/$exec_name.log 2>&1 &
    /ciena/bin/waitfor $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_log_and_wait_timeout() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    timeout=$1    ; shift

    print_app_line "$app_string"
    echo "*** Log of $exec_name output ($(date))" >> /var/log/$exec_name.log
    /bin/$exec_name $* >> /var/log/$exec_name.log 2>&1 &
    /ciena/bin/waitfor -t $timeout $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_and_queue() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    /bin/$exec_name $* &
    wait_list="$wait_list $exec_name.pid"
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
bg_run_and_save_pid() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    /bin/$exec_name $* &
    echo "$!" > $exec_name.pid
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
valgrind_bg_run_and_save_pid() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    port=$1       ; shift

    print_app_line "$app_string"
    /usr/bin/valgrind --log-socket=$VALGRIND_IP:$VALGRIND_PORT /bin/$exec_name $* &
    echo "$!" > $exec_name.pid
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
valgrind_bg_run_and_queue() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    port=$1       ; shift

    print_app_line "$app_string"
    /usr/bin/valgrind --log-socket=$VALGRIND_IP:$VALGRIND_PORT /bin/$exec_name $* &
    wait_list="$wait_list $exec_name.pid"
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
bg_run_log_and_save_pid() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    echo "*** Log of $exec_name output ($(date))" >> /var/log/$exec_name.log
    /bin/$exec_name $* >> /var/log/$exec_name.log 2>&1 &
    echo "$!" > $exec_name.pid
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
fg_run_no_output() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    $exec_name $* > /dev/null 2>&1

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return "$rc"
}

# -----------------------------------------------------------------------------
fg_run() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    $exec_name $*

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return "$rc"
}

# -----------------------------------------------------------------------------
print_result() {
    if [ "$1" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR $1"
    fi

    return $1
}

# -----------------------------------------------------------------------------
insmod_wrapper() {
    mod_name=$1
    shift

    print_app_line "Loading $mod_name module"

    /sbin/modprobe $mod_name $*

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
rmmod_wrapper() {
    mod_name=$1
    shift

    print_app_line "Unloading $mod_name module"

    /sbin/modprobe -r $mod_name

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
exit_on_error_parm() {
    if [ "$1" != "0" ] ; then
        exit_wrapper $1
    fi
}

# -----------------------------------------------------------------------------
exit_on_error() {
    rc="$?"
    if [ "$rc" != "0" ] ; then
        exit_wrapper $rc
    fi
    return "$rc"
}

# -----------------------------------------------------------------------------
exit_wrapper() {
    if [ -n "$EXIT_OVERRIDE" ] ; then
        $EXIT_OVERRIDE
    else
        exit $1
    fi
}

# -----------------------------------------------------------------------------
kill_from_dir() {

    count=0

    for pid_file in $(ls ./*.pid 2> /dev/null) ; do
        print_app_line "Killing $(basename $pid_file .pid)"
        pid=$(< $pid_file)
        kill $pid &> /dev/null
        rc=$?
        if [ "$rc" != "0" ] ; then
            echo "FAILED $rc (pid $pid)"
        else
            echo "$pid"
        fi
        rm $pid_file
        ((count += 1))
    done

    if [ "$count" = "0" ] ; then
        echo "Nothing to kill"
    fi
}

# -----------------------------------------------------------------------------
cd_to_pid_dir() {
    this_file=$(basename $1)
    pid_dir="/var/run/$this_file.d"
    mkdir -p $pid_dir
    cd $pid_dir
}

# -----------------------------------------------------------------------------
guard_file_add() {
    death_action=$1 ; shift
    exec_name=$1    ; shift
    args="$*"

    outfile=/tmp/guardian_config

    # If there is already an entry for this executable in the config file,
    # then do not make a new one.
    if [ -e $outfile ] ; then
        if [ "$(grep -c $exec_name $outfile)" != "0" ] ; then
            return
        fi
    fi

    echo "$0 $death_action $exec_name $args" >> $outfile
}

# -----------------------------------------------------------------------------
get_rootfs_rw() {

    /bin/touch /no_touch 2> /dev/null
    if [ -e "/no_touch" ] ; then
        /bin/rm /no_touch
        echo "writable"
    else
        echo "read-only"
    fi
}

# -----------------------------------------------------------------------------
get_rootfs_location() {
    /bin/mount | grep -qE "/dev/root|ubi"
    if [ "$?" -ne "0" ] ; then
        echo "initramfs"
    else
        case "$(kernel_arg root)" in

        /dev/nfs)
            echo "nfs"
            ;;
        /dev/mtdblock* | \
        ubi*)
            echo "flash"
            ;;
        *)
            echo "unknown device ($(kernel_arg root))"
            ;;
        esac
    fi
}

# -----------------------------------------------------------------------------
cookie_clean() {
    local cookie_number="$1" ; shift
    local cookie_name="$1"   ; shift

    /mnt/apps/bin/saparam read "$cookie_number" > /dev/null 2>&1
    if [ "$?" -ne "0" ] ; then
        fg_run "RTFD cookie $cookie_number ($cookie_name) missing" /bin/true
    else
        fg_run_no_output "RTFD cookie $cookie_number ($cookie_name) erase " \
            /mnt/apps/bin/saparam erase $cookie_number
    fi
}

# -----------------------------------------------------------------------------
check_reset_to_factory_defaults() {
    # read factory reset cookie - delete files if it is successful
    /mnt/apps/bin/saparam read 259 > /dev/null 2>&1

    if [ "$?" -eq "0" ] ; then # 0x103 (259) is the factory reset cookie
        # the following files used to be deleted in kernelApi.c
        fg_run "RTFD deleting system directory" /bin/rm -rf /mnt/sysfs/system/*
        fg_run "RTFD deleting software directory" /bin/rm -rf /mnt/sysfs/software/*
        fg_run "RTFD deleting log directory"    /bin/rm -rf /mnt/sysfs/log/*
        fg_run "RTFD deleting config directory" /bin/rm -rf /mnt/sysfs/config/*
        fg_run "RTFD deleting ssh directory"    /bin/rm -rf /mnt/sysfs/ssh/*
        fg_run "RTFD deleting corefiles"       /bin/rm -rf /mnt/log/corefiles/*
        
        # now erase the the cookies
        cookie_clean 259 "factory reset"
        cookie_clean   9 "serial console"
        cookie_clean 260 "IP address"
        cookie_clean 261 "subnet mask"
        cookie_clean 262 "gateway"
        cookie_clean 299 "power DC"
        cookie_clean 2 "system mode"
        cookie_clean 11 "local interface"
        cookie_clean 1 "full POST 1"
        cookie_clean 3 "full POST 2"
        cookie_clean 16 "POST packets"
        cookie_clean 17 "POST loopback mask"
        cookie_clean 18 "POST port mask"
        cookie_clean 19 "POST repeat"
        cookie_clean 22 "POST reset"
        cookie_clean 31 "POST skip"
        cookie_clean 329 "POST bypass"
        cookie_clean 313 "Guardian Disable"
        cookie_clean 314 "Guardian Max Reboots"
    fi
}

# -----------------------------------------------------------------------------
bank_status_cookie_clean() {
    local cookie_number="$1" ; shift
    local cookie_name="$1"   ; shift

    /mnt/apps/bin/saparam read "$cookie_number" > /dev/null
    if [ "$?" -ne "0" ] ; then
        fg_run "cookie $cookie_number ($cookie_name) missing" /bin/true
    else
        fg_run_no_output "cookie $cookie_number ($cookie_name) erase " \
            /mnt/apps/bin/saparam erase $cookie_number
    fi
}

# -----------------------------------------------------------------------------
clear_bank_status() {
    # read factory reset cookie - delete files if it is successful
    /mnt/apps/bin/saparam read 302 > /dev/null 2>&1

    if [ "$?" -eq "0" ] ; then # 0x12E (302) is the bank status cookike
        # now erase the cookies
        bank_status_cookie_clean 302 "bank status"
        bank_status_cookie_clean 303 "bank 0 status check time"
        bank_status_cookie_clean 304 "bank 1 status check time"
    fi
}

# -----------------------------------------------------------------------------
mount_log_partition() {
    local partition_number=$1 ; shift

    print_app_line "mount logs via fstab"
    /bin/mount /mnt/log > /dev/null 2> /dev/null
    print_result "$?"
    if [ "$?" -ne "0" ] ; then
        fg_run "clearing log partition (this will take a while)" \
            /usr/sbin/flash_eraseall -j -q /dev/mtd$partition_number
        fg_run "mount logs via fstab" /bin/mount /mnt/log
    fi
}

# -----------------------------------------------------------------------------
read_u32_cookie() {
    local cookie_id=$1; shift

    /mnt/apps/bin/saparam read $cookie_id | awk '/^0000:/ {print "0x"$2$3$4$5}'
}

# -----------------------------------------------------------------------------
capture_reboot_cookie() {
    local reason_cookie
    local reason="unspecified"
    local count_cookie

    reason_cookie="$(read_u32_cookie 65287)"
    count_cookie="$(read_u32_cookie 65286)"

    if [ -n "$reason_cookie" ] ; then
        #
        # The constants here are derived from the ResetType typedef in
        # src/leos/common/src/genericSwitch/include/genericSwitchCommon.h.
        # As these are values written into cookies, the assignments should
        # never change.
        #
        case "$reason_cookie" in
            "0x00000000") reason="Unknown"           ;;
            "0x00000001") reason="Snmp"              ;;
            "0x00000002") reason="PwrFail"           ;;
            "0x00000003") reason="AppLoad"           ;;
            "0x00000004") reason="ErrorHandler"      ;;
            "0x00000005") reason="WatchDog"          ;;
            "0x00000006") reason="Upgrade"           ;;
            "0x00000007") reason="Cli"               ;;
            "0x00000008") reason="ResetButton"       ;;
            "0x00000009") reason="ServiceModeChange" ;;
            *)            reason="$reason_cookie"    ;;
        esac
    fi

    if [ -z "$count_cookie" ] ;  then
        count_cookie="unspecified"
    fi

    /ciena/bin/evt_log ALERT "reboot $count_cookie, reason was $reason"
}

get_console_tty()
{
  _tty=`kernel_arg console | awk -F, '{print $1}'`
  if [ -z "${_tty}" ]; then
    echo '/dev/ttyS0'
  else
    echo "/dev/${_tty}"
  fi
}

# -----------------------------------------------------------------------------
get_base_mac()
{
    mac=$(dd if=$(get_eeprom_file) bs=1 skip=34 count=12 2>/dev/null)

    # insert ':' between octets and set to uppercase
    mac=$(echo $mac |                        \
          sed -e 's|..|\0:|g' -e 's|:$||' |  \
          awk '{print toupper($1)}')

    echo $mac
}
