# This file contains platform specific functions some of which require utils.sh
# to be sourced before this file. To avoid a circular reference, users should
# source utils.sh instead of this file to be sure that all references will be
# resolved.

source /ciena/scripts/paramApi.sh

# -----------------------------------------------------------------------------
get_board_id()
{
    cat /sys/bus/platform/devices/board_info/id
}

#------------------------------------------------------------------------------
is_fpga_gx30 ()
{
    gx50='/sys/bus/platform/devices/board_info/gx50'
    [ -r "${gx50}" ] && [ "$(<${gx50})" == "0" ]
}

# -----------------------------------------------------------------------------
get_board_name()
{
    board_id=$(get_board_id)

    case "$board_id" in
        1)
            echo '5150' ;;
        2)
            echo '3916' ;;
        3)
            echo '3930' ;;
        4)
            echo '3931' ;;
        5)
            echo '3932' ;;
        6)
            echo '5142' ;;
        7)
            echo '5160' ;;
        8)
            echo '3942' ;;
        9)
            echo '3938' ;;
        *)
            echo 'unknown' ;;
    esac
}

# -----------------------------------------------------------------------------
get_board_name_pretty()
{
    board_name=$(get_board_name)

    # Handle special cases where the same board id is used for multiple
    # board variants.  Regular cases get the unmodified result from
    # the get_board_name result.
    #
    case "$board_name" in

        "3932")
            if ! pwe_supported ; then
                echo "3930ET"
                return
            fi
            ;;

        "3916")
            if cr3916 ; then
                echo "3916CR"
                return
            fi
            ;;

    esac

    echo $board_name
}

# -----------------------------------------------------------------------------
get_local_if()
{
    board=$(get_board_name)
    if [ $board == '5150' ]; then
        echo 'eth0'
    else
        echo 'mgmt0'
    fi
}

# -----------------------------------------------------------------------------
board_has_rtc()
{
    [ "$(get_board_name)" = "5150" -o \
      "$(get_board_name)" = "3930" -o \
      "$(get_board_name)" = "3931" ]
}

# -----------------------------------------------------------------------------
board_has_push_button_interrupt()
{
    [ "$(get_board_name)" = "3932" -o \
      "$(get_board_name)" = "3938" -o \
      "$(get_board_name)" = "3942" -o \
      "$(get_board_name)" = "5142" -o \
      "$(get_board_name)" = "5160" ]
}

# -----------------------------------------------------------------------------
mount_log_partition()
{
    local size=0
    local vol_id=3
    local target_size=$((195*1024*1024))
    local ubinfo=""
    local free_space=0
    local log_backup="/tmp/log_bkup.$$"
    local evt_backup="/tmp/evt_bkup.$$"

    # if the ubi log partition exists, then find the size and volume id
    ubinfo=$(ubinfo /dev/ubi0 -N logs)
    if [ "$?" -eq "0" ] ; then
        size=$(echo "$ubinfo" | awk '/^Size:/ {print $4}' | sed s/\(// )
        vol_id=$(echo "$ubinfo" | awk '/^Volume ID:/ {print $3}' )
    fi

    # check how much free space is available
    free_space=$(ubinfo /dev/ubi0 -a | \
        awk '/^Amount of available logical eraseblocks:/ {print $7}' | \
        sed s/\(// )

    # include any currently allocated space in the calculation
    free_space=$((free_space + size))

    # reduce target size if it exceeds the space available on the device
    if [ "$free_space" -lt "$target_size" ] ; then
        # use all of free space less one LEB
        target_size=$((free_space - 128*1024))
    fi

    # If the partition is already at least as big as we need, then
    # mount it and return from this function.
    #
    if [ "$size" -ge "$target_size" ] ; then
        fg_run "mount logs via fstab" /bin/mount /mnt/log

        if [ "$?" -ne "0" ] ; then
            # If we fail to mount, attempt to recover by reformatting
            #
            print_app_line_cr "attempting to create an empty log partition"
            fg_run "remove volume" ubirmvol /dev/ubi0 -N logs
            print_app_line_cr "create new volume"
            ubimkvol /dev/ubi0 -s $target_size -n $vol_id -N logs
            print_result "$?"
            fg_run "format ubifs" mkfs.ubifs /dev/ubi0_$vol_id
            
            fg_run "mount logs via fstab" /bin/mount /mnt/log
        fi

        return
    fi

    # If we get here, then we need to resize the log partition

    # Note that this script runs before the watchdog is serviced, so we
    # need to ensure that these operations are relatively quick.
    # We run at this point due to guardian/cookie/reboot-reason ordering
    # dependencies in rcS.

    fg_run "resize log partition (from $size to $target_size)" \
        ubirsvol /dev/ubi0 -N logs -s $target_size

    fg_run "mount resized logs via fstab" /bin/mount /mnt/log
}

# -----------------------------------------------------------------------------
EXIT_BANKA=1
EXIT_BANKB=2
EXIT_NFS=3
EXIT_UNKNOWN=25
get_running_bank()
{
    case "$(kernel_arg root)" in
        /dev/nfs)  return $EXIT_NFS     ;;
        ubi0:appA) return $EXIT_BANKA   ;;
        ubi0:appB) return $EXIT_BANKB   ;;
        *)         return $EXIT_UNKNOWN ;;
    esac
}

# -----------------------------------------------------------------------------
get_eeprom_file()
{
    case $(get_board_name) in
        '3932' | \
        '3938' | \
        '3942' | \
        '5142' | \
        '5160')
            echo /dev/mtd/mfg-eeprom
            ;;
        *)
            echo /sys/bus/i2c/devices/0-0054/eeprom
            ;;
    esac
}

# ------------------------------------------------------------------------------
get_data_fpga_file()
{
    case $(get_board_name) in
        '3916' | \
        '3930' | \
        '3931' | \
        '3932')  echo "grey_top_gx30.rbf" ;;
        '3938')  echo "galdin_top.rbf"  ;;
        '3942')  echo "galdin_top.rbf"  ;;
        '5142')  
            if  is_fpga_gx30 ; then  
                echo "langwell_top.rbf"  
            else
                echo "galdin_top.rbf"  
            fi ;;
        '5150')  echo "sind_top.rbf.gz"   ;;
        '5160')  echo "tag_top.rbf"       ;;
    esac
}

# -----------------------------------------------------------------------------
show_fpga_reg ()
{
    local decoder="$1"
    local regname="$2"
    local regex=$regname'\s+(0[xX].*)'
    local x=$($decoder $regname)
    if [[ $x =~ $regex ]]
    then
        echo ${BASH_REMATCH[0]}
    fi
}

# ------------------------------------------------------------------------------
show_fpga_version()
{
    case $(get_board_name) in
        '3916' | \
        '3930' | \
        '3931' | \
        '3932')
            show_fpga_reg greylinreg GREYLIN_REG_REVISION
            show_fpga_reg greylinreg GREYLIN_REG_BUILD_REVISION
            ;;
        '3938')
            show_fpga_reg galdinreg GALDIN_REG_REVISION
            show_fpga_reg galdinreg GALDIN_REG_BUILD_REVISION
            ;;
        '3942')
            show_fpga_reg galdinreg GALDIN_REG_REVISION
            show_fpga_reg galdinreg GALDIN_REG_BUILD_REVISION
            ;;
        '5142')
            if  is_fpga_gx30 ; then  
                show_fpga_reg langwellreg LANGWELL_REG_REVISION
                show_fpga_reg langwellreg LANGWELL_REG_BUILD_REVISION
            else
                show_fpga_reg galdinreg GALDIN_REG_REVISION
                show_fpga_reg galdinreg GALDIN_REG_BUILD_REVISION
            fi
            ;;
        '5160')
            show_fpga_reg taglinreg TAGLIN_REG_REVISION
            show_fpga_reg taglinreg TAGLIN_REG_BUILD_REVISION
            ;;
    esac
}

# ------------------------------------------------------------------------------
pwe_supported()
{
    pwe_info='/sys/bus/platform/devices/board_info/pwe'
    [ -r "${pwe_info}" ] && [ "$(<${pwe_info})" == "1" ]
}

# ------------------------------------------------------------------------------
pwe_in_reset()
{
    [ $(/ciena/bin/balinreg BALIN_RESET.PWE_RESET | tail -1 | sed -e 's/ */ /g' | cut -d' ' -f6) == "0x1" ]
}

# ------------------------------------------------------------------------------
setup_pwe()
{
    pwe_supported || return

    if [ "$(get_board_name)" == "3932" ] ; then
        # PWE will be in reset after power-up or reboot.
        # However we need to apply reset to PWE if SAOS is restarting
        if ! pwe_in_reset; then
            fg_run "Reset PWE"                                     /ciena/bin/balinreg BALIN_RESET.PWE_RESET 1
            sleep 1
        fi
        fg_run "Removing RESET on PWE"                             /ciena/bin/balinreg BALIN_RESET.PWE_RESET 0
        fg_run "Configuring eth2"                                  /ciena/bin/cavium52xx_PCSreg PCS0_MISC2_CTL_REG 0x1805  # first disable it
        fg_run "Flush eth2 RX queue"                               /ciena/bin/cavium52xx_GMXreg GMX0_PRT2_CFG 0x4
        fg_run "Ensure eth2 mode set to SGMII and sampling of 5"   /ciena/bin/cavium52xx_PCSreg PCS0_MISC2_CTL_REG
        fg_run "Set eth2 to 100Mbps, autoneg, full-duplex"         /ciena/bin/cavium52xx_PCSreg PCS0_MR2_CONTROL_REG 0x3100
        fg_run "Reset eth2"                                        /ciena/bin/cavium52xx_PCSreg PCS0_MR2_CONTROL_REG 0xb100
        fg_run "Ensure eth2 reset is complete"                     /ciena/bin/cavium52xx_PCSreg PCS0_MR2_CONTROL_REG
        fg_run "Check eth2 link up"                                /ciena/bin/cavium52xx_PCSreg PCS0_MR2_STATUS_REG
        fg_run "Check eth2 RX and TX idle"                         /ciena/bin/cavium52xx_GMXreg GMX0_PRT2_CFG
        fg_run "Enable eth2 interrupts"                            /ciena/bin/cavium52xx_PCSreg PCS0_INT2_EN_REG 0xffffffff
        fg_run "Enable eth2 RX interrupts"                         /ciena/bin/cavium52xx_GMXreg GMX0_RX2_INT_EN  0xffffffff
        fg_run "Enable eth2 TX interrupts"                         /ciena/bin/cavium52xx_GMXreg GMX0_TX_INT_EN   0xffffffff
        fg_run "Any pending eth2 interrupts"                       /ciena/bin/cavium52xx_GMXreg GMX0_TX_INT_REG
        fg_run "Enable RX on eth2"                                 /ciena/bin/cavium52xx_GMXreg GMX0_PRT2_CFG 0x5
        fg_run "Enable TX on eth2"                                 /ciena/bin/cavium52xx_PCSreg PCS0_MISC2_CTL_REG 0x1005

        # Assign MAC and IP address for Cavium-PWE mgmt link which is over eth0
        fg_run "assign MAC and IP address to eth2 - pwe mgmt link" \
        /sbin/ifconfig eth2 hw ether 02:0a:0b:0c:0d:01 169.254.1.1 netmask 255.255.255.0 up

        # bootps (udhcpd) specific file required to be present
        /bin/touch    /tmp/etc/udhcpd.leases
    fi
}

# ------------------------------------------------------------------------------
spiboot_supported()
{
    spiboot_info='/sys/bus/platform/devices/board_info/spiboot'
    [ -r "${spiboot_info}" ] && [ "$(<${spiboot_info})" == "1" ]
}

# ------------------------------------------------------------------------------
cr3916()
{
    cr_info='/sys/bus/platform/devices/board_info/cr'
    [ -r "${cr_info}" ] && [ "$(<${cr_info})" == "1" ]
}

# ------------------------------------------------------------------------------
copy_kernel() {
    # For boards that boot from SPI flash the kernel isn't in a separate partition.
    ! spiboot_supported && ! cr3916
}

# ------------------------------------------------------------------------------
spiboot_ack()
{
    case $(get_board_name) in
        '3932') reg=BALIN_CPU_BOOT_ATTEMPTS ;;
        '3938') reg=VANYAR_CPU_BOOT_ATTEMPTS ;;
        '3942') reg=LINDIR_CPU_BOOT_ATTEMPTS ;;
        '5142') reg=GAMIL_CPU_BOOT_ATTEMPTS ;;
        '5160') reg=NORIN_CPU_BOOT_ATTEMPTS ;;
        *) echo "$FUNCNAME called on unsupported board"; return ;;
    esac

    /ciena/bin/bsfreg $reg 0
}

# ------------------------------------------------------------------------------
spiboot_bank_set_to_running()
{
    get_running_bank
    case $? in
        $EXIT_BANKA) bank=0 ;;
        $EXIT_BANKB) bank=1 ;;    
    esac
    spiboot_bank_set $bank
}

# ------------------------------------------------------------------------------
spiboot_bank_get()
{
    local cfg=/dev/mtd/fpga-cfg
    dd if=$cfg bs=1 count=1 2>/dev/null | hexdump -ve '1/1 "%u"'
}

# ------------------------------------------------------------------------------
spiboot_bank_set()
{
    local bank=$1
    local cfg=/dev/mtd/fpga-cfg

    # always set attempts to a default
    local attempts=3

    case $(get_board_name) in
        '3932') reg=BALIN_CPU_BOOT_BANK_VALID ;;
        '3938') reg=VANYAR_CPU_BOOT_BANK_VALID ;;
        '3942') reg=LINDIR_CPU_BOOT_BANK_VALID ;;
        '5142') reg=GAMIL_CPU_BOOT_BANK_VALID ;;
        '5160') reg=NORIN_CPU_BOOT_BANK_VALID ;;
        *) echo "$FUNCNAME called on unsupported board"; return ;;
    esac

    # invalidate the cached bank value in the FPGA to cause it to refresh from flash
    /ciena/bin/bsfreg $reg 0

    # FPGA config flash location contains bank information
    # Format:
    #   0x00: bank[3:0]
    #   0x01: attempts[3:0]

    # Check the existing bank setting to avoid a program/erase cycle
    # on the flash if nothing is changing.
    [ $bank == $(spiboot_bank_get) ] && return

    flash_erase -q $cfg 0 0

    echo -ne "\x"$bank"\x"$attempts > $cfg

}

# ------------------------------------------------------------------------------
get_preferred_bank()
{
    if spiboot_supported; then
        bank=$(spiboot_bank_get)
    else
        # Stored in ParamType_BootCookie
        bank=$(/mnt/apps/bin/saparam tofile 282 /proc/self/fd/1 | hexdump -n 4 -e '1/4 "%u"')
    fi

    case "$bank" in
        '0') return $EXIT_BANKA   ;;
        '1') return $EXIT_BANKB   ;;
        *)   return $EXIT_UNKNOWN ;;
    esac
}

#--------------------------------------------------------------------
xftp_partition_supported()
{
    [ "$(get_board_name)" == "3930" ]
}

# ------------------------------------------------------------------------------
load_fpga_setup()
{
    case $(get_board_name) in
        '5160')
            zl30150reg ZL30150_Freq_Config.ref0_freq_multiple      0x03e8
            zl30150reg ZL30150_Freq_Config.ref2_base_freq          0xa861
            zl30150reg ZL30150_Freq_Config.ref2_freq_multiple      0x6a18
            zl30150reg ZL30150_Freq_Config.ref3_base_freq          0xa861
            zl30150reg ZL30150_Freq_Config.ref3_freq_multiple      0x6a18
            zl30150reg ZL30150_DPLL_Config.dpll0_ref_priority3_2   0x77
            zl30150reg ZL30150_DPLL_Config.dpll0_ref_priority1_0   0x70
            zl30150reg ZL30150_DPLL_Config.dpll0_mode_refsel       0xe
            zl30150reg ZL30150_DPLL_Config.dpll1_ref_priority3_2   0x10
            zl30150reg ZL30150_DPLL_Config.dpll1_ref_priority1_0   0x77
            zl30150reg ZL30150_DPLL_Config.dpll1_mode_refsel       0x8f
            zl30150reg ZL30150_Synth_Config.synth0_base_freq       0x800c
            zl30150reg ZL30150_Synth_Config.synth0_freq_multiple   0x5962
            zl30150reg ZL30150_Synth_Config.synth2_base_freq       0xa861
            zl30150reg ZL30150_Synth_Config.synth2_freq_multiple   0xc409
            zl30150reg ZL30150_Synth_Config.output_synth_drive_pll 0x50
            zl30150reg ZL30150_Synth_Config.output_synthesizer_en  0x7
            zl30150reg ZL30150_Synth0_Config.synth0_post_div_A     0x40000
            zl30150reg ZL30150_Synth1_Config.synth1_post_div_A     0x080000
            zl30150reg ZL30150_Synth1_Config.synth1_post_div_B     0xa0000
            zl30150reg ZL30150_Synth1_Config.synth1_post_div_C     0x7d0000
            zl30150reg ZL30150_Synth2_Config.synth2_post_div_A     0x140000
            zl30150reg ZL30150_Ref_Select.hp_diff_en               0xf1
            zl30150reg ZL30150_Ref_Select.hp_cmos_en               0x4
            zl30150reg ZL30150_Ref_Select.config_output_mode_3_0   0x1
            zl30150reg ZL30150_Ref_Select.synth1_0_stop_clk        0x10
            zl30150reg ZL30150_Interrupts.ref_config               0x10
            zl30143reg ZL30143_APLL_Config.apll_run                0x7f
            zl30143reg ZL30143_Diff_Output_Config.diff_clk_sel     0x66
            zl30143reg ZL30143_DPLL1_Control.dpll1_modesel         0x02
            ;;
        '5142')
            zl30143reg ZL30143_Diff_Output_Config.diff_clk_sel 0x66
            zl30143reg ZL30143_DPLL1_Control.dpll1_modesel     0x02
            zl30143reg ZL30143_APLL_Config.apll_enable         0x81
            zl30143reg ZL30143_APLL_Config.apll_run            0x7d
            zl30143reg ZL30143_APLL_Config.apll_clk_freq       0x01
            ;;
        '3938')
            zl30143reg ZL30143_Diff_Output_Config.diff_clk_sel 0x66
            zl30143reg ZL30143_DPLL1_Control.dpll1_modesel     0x02
            zl30143reg ZL30143_APLL_Config.apll_enable         0x81
            zl30143reg ZL30143_APLL_Config.apll_run            0x7d
            zl30143reg ZL30143_APLL_Config.apll_clk_freq       0x01
            zl30150reg ZL30150_Ref_Select.hp_diff_en           0xff
            zl30150reg ZL30150_Synth1_Config.synth1_post_div_A 0x320000
            zl30150reg ZL30150_Synth1_Config.synth1_post_div_B 0x320000
            ;;
        '3930' | '3931')
            synce=`octopusreg OCT_MPU_VARIANT.SYNCE |grep SYNCE |awk '{print $5}'`
            if [ "${synce}" = "0x1" ]; then
                /mnt/apps/bin/saparam read "${ParamType_BypassPost}" > /dev/null 2>&1 
                if [ "$?" -ne "0" ] ; then # (305) by pass not true
                    octopusreg OCT_MPU_RESET.ZL30143_RST 0x1
                    octopusreg OCT_MPU_RESET.ZL30143_RST 0x0
                    sleep 0.01 # it should only need to wait for 5 ms
                fi
            fi
            ;;
        '5150')
            /mnt/apps/bin/saparam read "${ParamType_BypassPost}" > /dev/null 2>&1 
            if [ "$?" -ne "0" ] ; then # (305) by pass not true
                vinmarreg VIN_MPU_RESET.ZL143_RST 0x1
                vinmarreg VIN_MPU_RESET.ZL143_RST 0x0
                sleep 0.01 # it should only need to wait for 5 ms
            fi
            ;;
        *)  ;;
    esac
}

load_fpga_pre()
{
    case $(get_board_name) in
        '5150')
            # if it was a warm start, then we need to remove the modules
            if [ -f /sys/bus/pci/devices/0000\:02\:02.0/0000\:04\:00.0/remove ]
            then 
                echo 1 > /sys/bus/pci/devices/0000\:02\:02.0/0000\:04\:00.0/remove
            fi

            if [ -f /sys/bus/pci/devices/0000\:02\:03.0/0000\:05\:00.0/remove ]
            then
                echo 1 > /sys/bus/pci/devices/0000\:02\:03.0/0000\:05\:00.0/remove
            fi

            # Disable the PCIe ports to the modules, as loading the FPGAs mysteriously
            # takes down a lane between the PEX and the Triumph
            # redirect stderr due to NAK on last expected byte of transaction from PEX
            plx_i2c_write 0 0x234 0xC049C 2> /dev/null
            plx_i2c_write 2 0x78 0x20110010 2> /dev/null
            plx_i2c_write 3 0x78 0x20110010 2> /dev/null

            ;;
        '5160')
            taglinreg TAGLIN_REG_PHY_RESET 1
            ;;
        *)  ;;
    esac
}

load_fpga_post()
{
    case $(get_board_name) in
        '5150')
            # re-enable the PCIe ports to the modules
            plx_i2c_write 2 0x78 0x20110000 2> /dev/null
            plx_i2c_write 3 0x78 0x20110000 2> /dev/null
            plx_i2c_write 0 0x234 0x49C  2> /dev/null
            ;;
        '5160')
            taglinreg TAGLIN_REG_PHY_RESET 0
            ;;
        *)  ;;
    esac
}

mgmt_interface_reset()
{
	# only 5150 requires the mgmt phy to be reset after a warm restart
    case $(get_board_name) in
        '5150')
            resetReason=`/mnt/apps/bin/saparam read 65287 | awk '/^0000:/ { print $5 }'`
            snmpWarmStart="0C"
            upgradeWarmStart="0D"
            cliWarmStart="0E"

            case "$resetReason" in
                "$snmpWarmStart"    |  \
                "$upgradeWarmStart" |  \
                "$cliWarmStart")
                    vinmarreg VIN_MPU_RESET.RST_PHY_MGMT 1
                    vinmarreg VIN_MPU_RESET.RST_PHY_MGMT 0
                    ;;
                *)
                    ;;
            esac
            ;;
        *)
        ;;
	esac
}

#--------------------------------------------------------------------
enable_fpga_flash_driver()
{
    return
}

#--------------------------------------------------------------------
board_has_fpga_preloader()
{
    false
}

# ------------------------------------------------------------------------------
dataplane_fpga_init()
{
    load_fpga_pre

    /ciena/scripts/load_FPGA.sh

    load_fpga_post

    # Rescan the PCI bus after loading the FPGA
    echo 1 > /sys/bus/pci/rescan
}

# ------------------------------------------------------------------------------
insmod_uio_ciena_pdrv_shared_irq()
{
    case $(get_board_name) in
        '3938' | \
        '3942' | \
        '5142' | \
        '5160')
            insmod_wrapper uio_ciena_pdrv_shared_irq
            ;;
        *) ;;
    esac

}

# ------------------------------------------------------------------------------
insmod_board_specific()
{
    return
}
