# This file contains platform specific functions some of which require utils.sh
# to be sourced before this file. To avoid a circular reference, users should
# source utils.sh instead of this file to be sure that all references will be
# resolved.

source /ciena/scripts/paramApi.sh

# -----------------------------------------------------------------------------
get_board_id ()
{
    cat /proc/device-tree/board_id.?
}

# -----------------------------------------------------------------------------
get_board_name()
{
    cat /proc/device-tree/model
}

# -----------------------------------------------------------------------------
get_board_name_pretty()
{
    get_board_name
}

# -----------------------------------------------------------------------------
get_local_if()
{
    echo 'eth0'
}

# -----------------------------------------------------------------------------
board_has_rtc()
{
    false
}

# -----------------------------------------------------------------------------
board_has_push_button_interrupt()
{
    true
}

# -----------------------------------------------------------------------------
mount_log_partition()
{
    fg_run "mount logs via fstab" /bin/mount /mnt/log
}

# -----------------------------------------------------------------------------
EXIT_BANKA=1
EXIT_BANKB=2
EXIT_NFS=3
EXIT_UNKNOWN=25
get_running_bank()
{
    case "$(kernel_arg root)" in
        /dev/nfs)  return $EXIT_NFS     ;;
        ubi0:appA) return $EXIT_BANKA   ;;
        ubi0:appB) return $EXIT_BANKB   ;;
        *)         return $EXIT_UNKNOWN ;;
    esac
}

# -----------------------------------------------------------------------------
get_eeprom_file()
{
    echo /dev/mtd/mfg-eeprom
}

# ------------------------------------------------------------------------------
get_data_fpga_file()
{
    true
}

# -----------------------------------------------------------------------------
show_fpga_reg ()
{
    # FIX THIS - should this be common?
    local decoder="$1"
    local regname="$2"
    local regex=$regname'\s+(0[xX].*)'
    local x=$($decoder $regname)
    if [[ $x =~ $regex ]]
    then
        echo ${BASH_REMATCH[0]}
    fi
}

# ------------------------------------------------------------------------------
show_fpga_version()
{
    echo FIX THIS - need to implement show_fpga_version
}

# ------------------------------------------------------------------------------
load_fpga_required()
{
    false
}

# ------------------------------------------------------------------------------
pwe_supported()
{
    false
}

# ------------------------------------------------------------------------------
setup_pwe()
{
    return
}

# ------------------------------------------------------------------------------
spiboot_supported()
{
    true
}

# ------------------------------------------------------------------------------
cr3916()
{
    false
}

# ------------------------------------------------------------------------------
copy_kernel() {
    false
}

# ------------------------------------------------------------------------------
spiboot_ack()
{
    /ciena/bin/bsfreg AMROTH_CPU_BOOT_ATTEMPTS 0
}

# ------------------------------------------------------------------------------
spiboot_bank_set_to_running()
{
    get_running_bank
    case $? in
        $EXIT_BANKA) bank=0 ;;
        $EXIT_BANKB) bank=1 ;;    
    esac
    spiboot_bank_set $bank
}

# ------------------------------------------------------------------------------
spiboot_bank_get()
{
    local cfg=/dev/mtd/fpga-cfg
    dd if=$cfg bs=1 count=1 2>/dev/null | hexdump -ve '1/1 "%u"'
}

# ------------------------------------------------------------------------------
spiboot_bank_set()
{
    local bank=$1
    local cfg=/dev/mtd/fpga-cfg

    # always set attempts to a default
    local attempts=3

    # invalidate the cached bank value in the FPGA to cause it to refresh from flash
    /ciena/bin/bsfreg AMROTH_CPU_BOOT_BANK_VALID 0

    # FPGA config flash location contains bank information
    # Format:
    #   0x00: bank[3:0]
    #   0x01: attempts[3:0]

    # Check the existing bank setting to avoid a program/erase cycle
    # on the flash if nothing is changing.
    [ $bank == $(spiboot_bank_get) ] && return

    flash_erase -q $cfg 0 0

    echo -ne "\x"$bank"\x"$attempts > $cfg

}

# ------------------------------------------------------------------------------
get_preferred_bank()
{
    local bank=$(spiboot_bank_get)

    case "$bank" in
        '0') return $EXIT_BANKA   ;;
        '1') return $EXIT_BANKB   ;;
        *)   return $EXIT_UNKNOWN ;;
    esac
}

#--------------------------------------------------------------------
xftp_partition_supported()
{
    false
}

# ------------------------------------------------------------------------------
load_fpga_setup()
{
    local board="$(get_board_name)"

    case "$board" in
        "3903")
            # No clocks to set up in this case
            # amroth temperature registers
            amrothreg AMROTH_SHUTDOWN_TEMPERATURE 0x70
            amrothreg AMROTH_OVERTEMP_THRESHOLD 0x63
            amrothreg AMROTH_MAX_STARTUP_TEMPERATURE 0x55
            ;;
        "3904" | "3905")
            # go to read the cookie to see if this is cold restart
            /mnt/apps/bin/saparam read "${ParamType_BypassPost}" > /dev/null 2>&1 
            if [ "$?" -ne "0" ] ; then # (305) by pass not true
                amrothreg AMROTH_RESET.ZL30143_RESET 0x1
                amrothreg AMROTH_RESET.ZL30143_RESET 0x0
                sleep 0.01 # it should only need to wait for 5 ms
            fi

            zl30156reg ZL30156_Page_Pointer.page_register      0x00
            zl30156reg ZL30156_Interrupts.ref_config           0x20
            # ref1 config
            zl30156reg ZL30156_Freq_Config.ref1_base_freq      0x9c40  ## 0x61a8
            zl30156reg ZL30156_Freq_Config.ref1_freq_multiple  0x0271  ## 0x03e8

            #DPLL config
            zl30156reg ZL30156_DPLL_Config.dpll1_ref_priority3_2    0x77
            zl30156reg ZL30156_DPLL_Config.dpll1_ref_priority1_0    0x07
            zl30156reg ZL30156_DPLL_Config.dpll1_mode_refsel        0x42
    
            # synth1 config
            zl30156reg ZL30156_Synth_Config.synth1_base_freq        0xa861
            zl30156reg ZL30156_Synth_Config.synth1_freq_multiple    0x350C
   
            zl30156reg ZL30156_Page_Pointer.page_register      0x01
            zl30156reg ZL30156_Synth1_Config.synth1_post_div_A 0x0A0000
            zl30156reg ZL30156_Synth1_Config.synth1_post_div_B 0x320000
            zl30156reg ZL30156_Ref_Select.hp_diff_en           0x50

            zl30143reg ZL30143_APLL_Config.apll_enable 0x80 #enable APLL
            zl30143reg ZL30143_APLL_Config.apll_run 0x70 #enable ethernet frequencies
            zl30143reg ZL30143_DPLL1_Control.dpll1_modesel 0x02
            zl30143reg ZL30143_Diff_Output_Config.diff_clk_ctrl 0xa1
            zl30143reg ZL30143_Diff_Output_Config.diff_clk_sel 0x06 #25 MHz
            ;;
        *)
            echo "board ($board) not supported"
            ;;
    esac
}

#--------------------------------------------------------------------
mgmt_interface_reset()
{
    return
}

#--------------------------------------------------------------------
enable_fpga_flash_driver()
{
    # amroth_flash driver is required for cookie operations
    insmod_wrapper amroth_flash
}

#--------------------------------------------------------------------
board_has_fpga_preloader()
{
    true
}

# ------------------------------------------------------------------------------
dataplane_fpga_init()
{
    load_fpga_setup
}

# ------------------------------------------------------------------------------
insmod_board_specific()
{
    insmod_wrapper i2c_fpga_mux
}

# ------------------------------------------------------------------------------
insmod_uio_ciena_pdrv_shared_irq()
{
    return
}
