# Helper utilities for shell scripts
#
# This file should be "source"d by callers to pick up the functions.
#

source /ciena/scripts/paramApi.sh

SAOS_PATH=/mnt/apps/bin/leos
SAOS_PIDFILE="/var/run/$(basename $SAOS_PATH).pid"
SAOS_RUNMODE_FILE="/var/run/$(basename $SAOS_PATH).RUNMODE"
SAOS_NOCONFIG_FILE="/var/run/$(basename $SAOS_PATH).NOCONFIG"
SAOS_CONFIG_GOOD_FILE="/var/run/$(basename $SAOS_PATH).CONFIG_GOOD"
SAOS_PENDING_REBOOT_FILE="/var/run/pending_reboot"

# -----------------------------------------------------------------------------
saos_server_pid()
{
    # This function returns the PID of the SAOS server process,
    # or 0 if the server is not running.

    if [ -r "$SAOS_PIDFILE" ] ; then
        cat $SAOS_PIDFILE
    else
        echo 0
    fi
}

# -----------------------------------------------------------------------------
kill_saos_server()
{
    # If the SAOS server is running, this function will kill the server
    # and any child processes.

    pid=$(saos_server_pid)

    if [ "$pid" -ne "0" ] ; then
        kill -TERM -$pid 2>/dev/null
    fi
}

# -----------------------------------------------------------------------------
saos_server_running()
{
    # This function can be used in conditional statements to determine
    # if the SAOS server is running.

    [ -d "/proc/$(saos_server_pid)" ]
}

# -----------------------------------------------------------------------------
saos_server_runmode()
{
    # This function returns the most recent runmode of the SAOS server
    # process or "missing" if the runmode file can not be found.

    if [ -r "$SAOS_RUNMODE_FILE" ] ; then
        cat $SAOS_RUNMODE_FILE
    else
        echo "missing"
    fi
}

# -----------------------------------------------------------------------------
saos_server_pending_reboot()
{
    # This function can be used in conditional statements to determine
    # if the SAOS server has started a reboot sequence.

    [ -f "$SAOS_PENDING_REBOOT_FILE" ]
}

# -----------------------------------------------------------------------------
param_cmd()
{
    # This function runs the appropriate cookie command depending
    # on the state of the SAOS server

    if $(saos_server_running) ; then
        param $*
    else
        /ciena/bin/saparam $*
    fi
}

# -----------------------------------------------------------------------------
read_u32_cookie() {
    local cookie_id=$1

    echo $(param_cmd tofile $cookie_id /proc/self/fd/1 \
        | hexdump -n 4 -e '1/4 "0x%08x"')
}

# -----------------------------------------------------------------------------
write_cookie()
{
    param_cmd swrite $1 $2
}

# -----------------------------------------------------------------------------
capture_reboot_cookie() {
    local reason_cookie
    local reason="unspecified"
    local count_cookie

    reason_cookie="$(read_u32_cookie $ParamType_LastResetReason)"
    count_cookie="$(read_u32_cookie $ParamType_TotalRebootCount)"

    if [ -n "$reason_cookie" ] ; then
        #
        # The constants here are derived from the ResetType typedef in
        # src/leos/common/src/genericSwitch/include/genericSwitchCommon.h.
        # As these are values written into cookies, the assignments should
        # never change.
        #
        case $((reason_cookie)) in
            $((ResetReason_Unknown))           ) reason="Unknown"             ;;
            $((ResetReason_SnmpColdStart))     ) reason="SNMP cold-start"     ;;
            $((ResetReason_PwrFail))           ) reason="PwrFail"             ;;
            $((ResetReason_AppLoad))           ) reason="AppLoad"             ;;
            $((ResetReason_ErrorHandler))      ) reason="ErrorHandler"        ;;
            $((ResetReason_WatchDog))          ) reason="WatchDog"            ;;
            $((ResetReason_UpgradeColdStart))  ) reason="Upgrade cold-start"  ;;
            $((ResetReason_CliColdStart))      ) reason="CLI cold-start"      ;;
            $((ResetReason_ResetButton))       ) reason="ResetButton"         ;;
            $((ResetReason_ServiceModeChange)) ) reason="ServiceModeChange"   ;;
            $((ResetReason_Guardian))          ) reason="Guardian"            ;;
            $((ResetReason_GuardianSaosRestart)))reason="GuardianSaosRestart" ;;
            $((ResetReason_SnmpWarmStart))     ) reason="SNMP warm-start"     ;;
            $((ResetReason_UpgradeWarmStart))  ) reason="Upgrade warm-start"  ;;
            $((ResetReason_CliWarmStart))      ) reason="CLI warm-start"      ;;
            *)                                   reason="$reason_cookie"      ;;
        esac
    fi

    if [ -z "$count_cookie" ] ;  then
        count_cookie="unspecified"
    fi

    /ciena/bin/evt_log ALERT "reboot $count_cookie, reason was $reason"
}

