#!/bin/sh
#
# This script is launched from /etc/inittab to in turn launch the
# serial console session in the appropriate (normal/debug) mode.
# We use ttyS0 and not the console, even though these are nominally
# the same device, because the console is unable to act like a
# controlling terminal.  (Without that ^C won't work, etc.)
#
# When in debug mode a root shell is invoked immediately, with no
# authentication, and who (et al.) won't see it (no utmp entry).
# If leos is configured not to run or isn't running anymore in a
# non-respawn environment then we also drop directly into a shell.
#
# If the serial port is disabled a slow poll will be started
# until the port is enabled.  If the root account is not enabled
# the getty will also be withheld until leos has created some
# user accounts.  (If the root account is enabled getty is started
# immediately.)
#
# Compound conditionals are difficult to do right (perhaps ash has
# some problems in this area?) so the conditionals are blown out
# into a more discrete form, resulting in duplicate action lines.
# (It _is_ easier to read this way.)
#

source /ciena/scripts/kernel_arg.sh
source /ciena/scripts/utils.sh
source /ciena/scripts/saos_utils.sh

# -----------------------------------------------------------------------------
drop_to_shell()
{
    echo "starting raw shell"
    home=$(cat /etc/passwd | grep ^root: | cut -d: -f6)
    HOME=$home SHELL=/bin/bash exec /bin/bash --login -i +s <$TTY >$TTY 2>&1
}

# -----------------------------------------------------------------------------
login_prompt()
{
    # Try to get the baudrate from the kernel command line.  If we get a
    # value, use it, otherwise default to 9600.
    baudrate=$(kernel_arg console | awk -F, '{print $2}')
    if [ -z "$baudrate" ] ; then
        baudrate="9600"
    fi

    exec /sbin/getty -L $baudrate $TTY
}

# -----------------------------------------------------------------------------
waste_time()
{
    sleep 3
}

# --- main --------------------------------------------------------------------

wait_message=0
TTY=$(get_console_tty)
console_arg=$(kernel_arg console)

while :; do     # Retry this forever, avoid formal respawn.  (More efficient?)

    source /ciena/scripts/krn getenv

    if [ "$KRN_MODE" == "debug" ] ; then 
        echo "KRN debug mode"
        drop_to_shell
    elif [ "$KRN_AUTO" != "enable" ] ; then
        echo "SAOS autostart disabled"
        drop_to_shell
    elif [ "$KRN_RESPAWN" == "disable" ] ; then
        if ! $(saos_server_running) ; then
            echo "SAOS death, no respawn"
            drop_to_shell
        fi
    fi

    # Spin while we are told there is no serial login
    if [ -f /tmp/noSerialLogin ] ; then
        while [ -f /tmp/noSerialLogin ] ; do
            waste_time
        done
    fi

    if [ -f /mnt/sysfs/system/enableRoot ] ; then
        login_prompt
    fi

    if [ "$(saos_server_runmode)" == "run" ] ; then
        echo > $TTY
        login_prompt
    else
        if [ -z "$console_arg" ] ; then
            if [ "$wait_message" -eq "0" ] ; then
                echo -n "Loading SAOS " > $TTY
                wait_message=1
            fi
            echo -n "." > $TTY
        fi
        waste_time
    fi
done
