# This file contains platform specific functions some of which require utils.sh
# to be sourced before this file. To avoid a circular reference, users should
# source utils.sh instead of this file to be sure that all references will be
# resolved.

# -----------------------------------------------------------------------------
get_board_id ()
{
    local board_id=$(ls -d /sys/bus/platform/devices/board_id.*)
    local regex='\.(.+)$'
    if [[ $board_id =~ $regex ]]
    then
        echo ${BASH_REMATCH[1]}
    else
        echo '0'
    fi
}

# -----------------------------------------------------------------------------
get_board_name()
{
    board3940=/sys/bus/platform/devices/board_id.1
    board3960=/sys/bus/platform/devices/board_id.2

    if [ -e "$board3940" ]; then
        echo '3940'
    elif [ -e "$board3960" ]; then
        echo '3960'
    else
        echo 'unknown'
    fi
}

# -----------------------------------------------------------------------------
get_board_name_pretty()
{
    get_board_name
}

# -----------------------------------------------------------------------------
get_local_if()
{
    echo 'eth0'
}

# -----------------------------------------------------------------------------
board_has_rtc()
{
    [ "$(get_board_name)" = "3960" ]
}

# -----------------------------------------------------------------------------
board_has_push_button_interrupt()
{
    false
}

# -----------------------------------------------------------------------------
mount_log_partition()
{
    mount_log_partition_on_jffs2
}

# -----------------------------------------------------------------------------
EXIT_BANKA=1
EXIT_BANKB=2
EXIT_NFS=3
EXIT_UNKNOWN=-1
get_running_bank()
{
    case "$(kernel_arg root)" in
        /dev/nfs)  return $EXIT_NFS     ;;
        mtd:app-a) return $EXIT_BANKA   ;;
        mtd:app-b) return $EXIT_BANKB   ;;
        *)         return $EXIT_UNKNOWN ;;
    esac
}

# -----------------------------------------------------------------------------
get_eeprom_file()
{
    echo /sys/bus/i2c/devices/1-0054/eeprom
}

# ------------------------------------------------------------------------------
get_data_fpga_file()
{
    case "$(get_board_name)" in
        '3940')
            if test_cpld_reg LE450CPLD_VER 0x11
            then
                echo "lune_top_revAB14.rbf.gz"  # rev 1
            else
                echo "lune_top.rbf.gz"          # rev 2
            fi
            ;;
        '3960')
            echo "siri_top.rbf.gz"
            ;;
    esac
}

# -----------------------------------------------------------------------------
show_fpga_reg ()
{
    local decoder="$1"
    local regname="$2"
    local regex=$regname'\s+(0[xX].*)'
    local x=$($decoder $regname)
    if [[ $x =~ $regex ]]
    then
        echo ${BASH_REMATCH[0]}
    fi
}

# ------------------------------------------------------------------------------
show_fpga_version()
{
    case "$(get_board_name)" in
        '3940')
            show_fpga_reg lunereg LUNE_VER
            ;;
        '3960')
            show_fpga_reg sirionreg0 SRN_VER
            show_fpga_reg sirionreg1 SRN_VER
            ;;
    esac
}

# ------------------------------------------------------------------------------
test_cpld_reg ()
{
    local regname="$1"
    local value="$2"
    local regex=$regname'\s+(0[xX][[:xdigit:]]+)'
    local x=$(le450cpldreg $regname)
    if [[ $x =~ $regex ]]
    then
        (( ${BASH_REMATCH[1]} == $value ))
        return $?
    fi
    return 1
}

# ------------------------------------------------------------------------------
setup_pwe()
{
    # Do nothing for brego
    return
}

# ------------------------------------------------------------------------------
spiboot_supported()
{
    # Not supported
    return 1
}

# ------------------------------------------------------------------------------
copy_kernel()
{
    # This function must agree with copy_kernel() in the stand-alone
    # software_install script.

    # Always required
    return 0
}

# ------------------------------------------------------------------------------
mount_kernel()
{
    # Always mount the kernel partitions on brego boards.
    return 0
}

# ------------------------------------------------------------------------------
get_preferred_bank()
{
    # Stored in ParamType_BootCookie
    bank=$(/mnt/apps/bin/saparam tofile 282 /proc/self/fd/1 | hexdump -n 4 -e '1/4 "%u"')
    case "$bank" in
        '0') return $EXIT_BANKA   ;;
        '1') return $EXIT_BANKB   ;;
        *)   return $EXIT_UNKNOWN ;;
    esac
}

#--------------------------------------------------------------------
xftp_partition_supported()
{
    # Not supported
    return 1
}

# ------------------------------------------------------------------------------
load_fpga_setup()
{
    return 0
}

# ------------------------------------------------------------------------------
mgmt_interface_reset()
{
    return 0
}

# ------------------------------------------------------------------------------
enable_fpga_flash_driver()
{
    return
}

# ------------------------------------------------------------------------------
board_has_fpga_preloader()
{
    false
}

# ------------------------------------------------------------------------------
dataplane_fpga_init()
{
    /ciena/scripts/load_FPGA.sh

    # Rescan the PCI bus after loading the FPGA
    echo 1 > /sys/bus/pci/rescan
}

# ------------------------------------------------------------------------------
insmod_board_specific()
{
    return
}

# ------------------------------------------------------------------------------
insmod_uio_ciena_pdrv_shared_irq()
{
    return
}
