# This file contains platform specific functions some of which require utils.sh
# to be sourced before this file. To avoid a circular reference, users should
# source utils.sh instead of this file to be sure that all references will be
# resolved.

source /ciena/scripts/paramApi.sh
source /ciena/scripts/saos_utils.sh
source /ciena/scripts/fault.sh

# -----------------------------------------------------------------------------
get_board_id ()
{
    cat /proc/device-tree/board_id.?
}

# -----------------------------------------------------------------------------
get_board_name()
{
    cat /proc/device-tree/model
}

# -----------------------------------------------------------------------------
get_board_name_pretty()
{
    get_board_name
}

# -----------------------------------------------------------------------------
get_local_if()
{
    echo 'eth0'
}

# -----------------------------------------------------------------------------
board_has_rtc()
{
    # The hardware RTC should work at 3928 rev3 and 3926 rev2.  Trying to use
    # it in earlier variants should not fail, but the clock will not keep
    # time.
    true
}

# -----------------------------------------------------------------------------
board_has_push_button_interrupt()
{
    true
}

# -----------------------------------------------------------------------------
mount_log_partition()
{
    fg_run "mount logs via fstab" /bin/mount /mnt/log
}

# -----------------------------------------------------------------------------
EXIT_BANKA=1
EXIT_BANKB=2
EXIT_NFS=3
EXIT_UNKNOWN=25
get_running_bank()
{
    case "$(kernel_arg root)" in
        /dev/nfs)       return $EXIT_NFS     ;;
        /dev/mmcblk0p5) return $EXIT_BANKA   ;;
        /dev/mmcblk0p6) return $EXIT_BANKB   ;;
        *)              return $EXIT_UNKNOWN ;;
    esac
}

# -----------------------------------------------------------------------------
get_eeprom_file()
{
    echo /mnt/boot/eeprom0.dat
}

# ------------------------------------------------------------------------------
get_data_fpga_file()
{
    true
}

# -----------------------------------------------------------------------------
show_fpga_reg ()
{
    # FIX THIS - should this be common?
    local decoder="$1"
    local regname="$2"
    local regex=$regname'\s+(0[xX].*)'
    local x=$($decoder $regname)
    if [[ $x =~ $regex ]]
    then
        echo ${BASH_REMATCH[0]}
    fi
}

# ------------------------------------------------------------------------------
show_fpga_version()
{
    case $(get_board_name) in
        "3926" | \
        "3928" )
            show_fpga_reg wfldreg WFLD_TOP_MJR
            show_fpga_reg wfldreg WFLD_TOP_MNR
            show_fpga_reg wfldreg WFLD_TOP_BLD
            ;;
    esac
}

# ------------------------------------------------------------------------------
load_fpga_required()
{
    false
}

# ------------------------------------------------------------------------------
pwe_supported()
{
    false
}

# ------------------------------------------------------------------------------
setup_pwe()
{
    return
}

#--------------------------------------------------------------------
wait_for_file_content()
{
    # wait for file $1 to contain $2, optionally for $3 seconds
    local now_time end_time
    local SYSLOG_INFO="logger -s -p USER.INFO  -t $FUNCNAME"

    if [ "${3}" != "" ]; then
        end_time=$(( $(/bin/date +%s) + ${3} ))
    fi

    if [ ! -r "${1}" ]; then
        $SYSLOG_INFO "Cannot read ${1}."
        return -1
    fi

    while [ $(/bin/cat "${1}") != "${2}" ]; do
        now_time=$(/bin/date +%s)
        if [ ${end_time:-${now_time}} -lt ${now_time} ]; then
            $SYSLOG_INFO "${1} $(cat ${1}) != ${2} after ${3} seconds."
            return -1
        fi
        sleep 0.1
    done

    return 0
}

# --- eredan only --------------------------------------------------------------
eredan_power_down()
{
    local SYSLOG_CRIT="logger -s -p USER.CRIT  -t $FUNCNAME"

    # Turn off both power supplies.
    /ciena/bin/sarosreg SAROS_SBD_MISC_CTL.PSL_OFF 1
    /ciena/bin/sarosreg SAROS_SBD_MISC_CTL.PSR_OFF 1

    sleep 5

    # Still here? Then try a board reset.
    ${SYSLOG_CRIT} "Still running after power down - resetting the board."
    sleep 1
    /ciena/bin/sarosreg SAROS_BRD_RESET 0xa5a5
}

# --- eredan only --------------------------------------------------------------
init_3928_rev1()
{
    local i2cset="/usr/sbin/i2cset"

    print_app_line_cr "3928 rev1 init hack"

    # Note that this sequence must be performed after the i2c mux driver
    # has been installed.

    # This script sets up the clocking on the Broadcom Metrolite chip.
    # This allows the device to show up on the PCI bus.
    #
    # The magic settings that follow are consistently NAK-ed. Just sweep
    # the failures under the carpet; this is a temporary work-around.
    #
    $i2cset -y 25 0x44 0x00 0x01 0x00 0x98 0x62 0x03 0x40 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x01 0x00 0x9c 0x60 0x37 0x75 0x31 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x01 0x00 0xa0 0x00 0x00 0x00 0x31 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x0c 0x34 0x48 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x10 0x02 0x00 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x14 0x00 0x00 0x00 0x60 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x0c 0x34 0x08 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x10 0x32 0x00 0x03 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x14 0x00 0x00 0x00 0x04 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x14 0x74 0x00 0x00 0x00 0x40 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x00 0x00 0x00 0x00 0x03 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x00 0x00 0x00 0x00 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x04 0x00 0x08 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x08 0x00 0x00 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x0c 0x2c 0x08 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x10 0x32 0x00 0x03 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x14 0x18 0x00 0xfc 0x08 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x00 0x00 0x00 0x00 0x01 i 2>/dev/null

    sleep 1
}

# --- eredan only --------------------------------------------------------------
gate_3926_si5341_out7()
{
    local i2cset="/usr/sbin/i2cset"
    print_app_line_cr "3926 gate out7 on si5341"
    # set page to page 1
    $i2cset -y 26 0x76 0x01 0x01 2>/dev/null
    # set the register 0x12B to value 0x00 to disable output 7
    $i2cset -y 26 0x76 0x2B 0x00 2>/dev/null
    # set the page number back to 0x0
    $i2cset -y 26 0x76 0x01 0x00 2>/dev/null
    sleep 1
}


# --- eredan only --------------------------------------------------------------
init_392x_metrolite()
{
    local board="${1}"
    local board_rev="${2}"
    local clock_hack=0
    local -a pci_devs_before=( /sys/bus/pci/devices/* )
    local -a pci_devs_after
    local SYSLOG_CRIT="logger -s -p USER.CRIT  -t $FUNCNAME"

    # Clock hack only applies to 3928 rev1
    if [ "${board}" = "3928" ] ; then
        if [[ "${board_rev}" -eq "1" ]] ; then
            clock_hack=1
        fi
    fi

    print_app_line_cr "${board} metrolite pci staging"

    # Reset the Metrolite before probing the PCI bus.

    echo 371 > /sys/class/gpio/export
    echo out > /sys/class/gpio/gpio371/direction

    # Hold the Metrolite PCIe interface in reset.
    echo 0 > /sys/class/gpio/gpio371/value
    # Wait for the PCIe link to go down (link_status == 2 means no PCIe link).
    wait_for_file_content /sys/bus/platform/drivers/nwl-pcie/*/link_status 2 10

    # Hold the main Metrolite in reset,
    echo switch > /dev/device-reset/assert

    if [ "${clock_hack}" -eq "1" ] ; then
        echo dpll > /dev/device-reset/assert
    fi

    # There is no proper status to check here; just sleep.
    sleep 0.5

    if [ "${clock_hack}" -eq "1" ] ; then
        echo dpll > /dev/device-reset/deassert
    fi

    # Release the main Metrolite from reset.
    echo switch > /dev/device-reset/deassert
    # Idem.
    sleep 0.5

    if [ "${clock_hack}" -eq "1" ] ; then
        init_3928_rev1
    fi

    # 3926 and rev not "1", gate the silab output7 clock
    if [ "${board}" = "3926" ] ; then
        if [[ "${board_rev}" -gt "1" ]] ; then
            gate_3926_si5341_out7
        fi
    fi


    # Releasse the Metrolite PCIe interface from reset.
    echo 1 > /sys/class/gpio/gpio371/value
    # Link_status == 3 means the PCIe link is up.
    wait_for_file_content /sys/bus/platform/drivers/nwl-pcie/*/link_status 3 10

    echo 371 > /sys/class/gpio/unexport

    echo 1 > /sys/bus/pci/rescan

    # If the Metrolite was staged successfully, there will be more PCI
    # devices on the bus.
    pci_devs_after=( /sys/bus/pci/devices/* )
    if [ "${#pci_devs_before[@]}" -ge "${#pci_devs_after[@]}" ]; then
        ${SYSLOG_CRIT} "before pci rescan: ${pci_devs_before[@]}"
        ${SYSLOG_CRIT} "after pci rescan: ${pci_devs_after[@]}"
        ${SYSLOG_CRIT} "power cycling board"
        sleep 1
        eredan_power_down
    fi
}

# --- eredan only --------------------------------------------------------------
eredan_hardware_init()
{
    local board="$(get_board_name)"
    local board_rev="$(/ciena/bin/sarosreg SAROS_REV | awk '{print $3}')"

    case "$board" in
        "3926" | \
        "3928" )
            init_392x_metrolite ${board} ${board_rev}
            ;;
     esac
}

# ------------------------------------------------------------------------------
spiboot_supported()
{
    # Not really spiboot, but this prevents other code from trying to use
    # cookies to control the boot bank.
    true
}

# ------------------------------------------------------------------------------
cr3916()
{
    false
}

# ------------------------------------------------------------------------------
copy_kernel() {
    false
}

# ------------------------------------------------------------------------------
mount_kernel() {
    false
}

# ------------------------------------------------------------------------------
spiboot_ack()
{
    if [ ! -f "/sys/ciena/eredan/bootcount" ] ; then
        insmod_wrapper ciena_sysfs
    fi

    /ciena/bin/root bash -c "echo 0 > /sys/ciena/eredan/bootcount"
}

# ------------------------------------------------------------------------------
spiboot_bank_set_to_running()
{
    get_running_bank
    case $? in
        $EXIT_BANKA) bank=0 ;;
        $EXIT_BANKB) bank=1 ;;    
    esac
    spiboot_bank_set $bank
}

# ------------------------------------------------------------------------------
spiboot_bank_get()
{
    cat /mnt/param/bootbank.def
}

# ------------------------------------------------------------------------------
spiboot_bank_set()
{
    local bank=$1

    case "$bank" in 
        0) bank_alpha="a" ;;
        1) bank_alpha="b" ;;
    esac

    # acknowledge the current boot chain
    spiboot_ack

    # set the persistent bank
    /ciena/bin/root bash -c "echo $bank_alpha > /sys/ciena/eredan/bank"

    # Check the existing bank setting to avoid a program/erase cycle
    # on the flash if nothing is changing.
    [ $bank == $(spiboot_bank_get) ] && return

    /ciena/bin/root bash -c "echo -n $bank > /mnt/param/bootbank.def"
}

# ------------------------------------------------------------------------------
get_preferred_bank()
{
    local bank=$(spiboot_bank_get)

    case "$bank" in
        '0') return $EXIT_BANKA   ;;
        '1') return $EXIT_BANKB   ;;
        *)   return $EXIT_UNKNOWN ;;
    esac
}

#--------------------------------------------------------------------
xftp_partition_supported()
{
    false
}

# ------------------------------------------------------------------------------
load_fpga_setup()
{
    local board="$(get_board_name)"

    case "$board" in
        "3928" | "3926" )
            # No clocks to set up in this case
            ;;
        *)
            echo "board ($board) not supported"
            ;;
    esac
}

#--------------------------------------------------------------------
mgmt_interface_reset()
{
    return
}

#--------------------------------------------------------------------
enable_fpga_flash_driver()
{
    return
}

#--------------------------------------------------------------------
board_has_fpga_preloader()
{
    false
}

# ------------------------------------------------------------------------------
dataplane_fpga_init()
{
    load_fpga_setup
}

# ------------------------------------------------------------------------------
insmod_board_specific()
{
    insmod_wrapper ciena_sysfs
    insmod_wrapper ciena_board_id
    insmod_wrapper i2c_fpga_mux
    insmod_wrapper ciena_saros_hp_cic
    insmod_wrapper ciena_saros_hp_gpio
    insmod_wrapper reset-fpga
    insmod_wrapper reset-sysfs
    insmod_wrapper dwc3_of_simple
    insmod_wrapper dwc3
    eredan_hardware_init
}

# ------------------------------------------------------------------------------
insmod_uio_ciena_pdrv_shared_irq()
{
    return
}

# ------------------------------------------------------------------------------
prepare_cookies()
{
    local readonly cookie_dir="/mnt/sysfs/cookie"
    local readonly old_archive="${cookie_dir}/archives.bin"
    local readonly new_archive="/mnt/param/archives.bin"
    local link_dest

    # ensure that the cookie directory exists
    mkdir -p $cookie_dir

    # Verify that the cookie file exists - create it if necessary
    if [ -f "${new_archive}" ] ; then
        # cookie file exists - this is good
        :
    else
        # no cookie file
        if [ -f "${old_archive}" ] ; then
            # Use the cookie file from the incorrect location to seed
            # the correct one.
            fg_run "transferring cookies to /mnt/param" \
                mv "${old_archive}" "${new_archive}"
        else
            # No cookies anywhere, create an empty file
            fg_run "creating empty cookie file" touch ${new_archive}
        fi
    fi

    # Verify that the cookie symlink points to the correct location
    if [ -L "${old_archive}" ] ; then
        link_dest=$(readlink "${old_archive}")
        if [ "$link_dest" != "${new_archive}" ] ; then
            # Remove the incorrect symlink and fall through to the
            # next check to create the symlink
            fg_run "removing bad cookie link to $link_dest" \
                rm "${old_archive}"
        fi
    fi

    # Verify that the cookie symlink exists
    if [ ! -L "${old_archive}" ] ; then
        # No symlink - remove any cruft where the symlink should be
        rm -rf "${old_archive}"
        # create the symlink to the correct location
        fg_run "creating cookie symlink" \
            ln -s "${new_archive}" "${old_archive}"
    fi
}

# ------------------------------------------------------------------------------
get_push_button_rtfd_state()
{
    # Most of our systems set the RTFD cookie in u-boot.  The eredan platform
    # can not do this, as the SAROS can not be easily reached from u-boot, so
    # we grab the value and clear down the FPGA bit here.

    local hold3s=$(/ciena/bin/sarosreg SAROS_PB_STATUS.PHELD \
        | awk '/PHELD/ {print $5}')

    if [ "$hold3s" == "0x1" ] ; then
        write_cookie $ParamType_FactoryReset 1
        /ciena/bin/sarosreg SAROS_PB_STATUS.PHELD 0x0
        print_app_line_cr "RTFD via push button detected"
    fi
}

# ------------------------------------------------------------------------------
default_console_tty()
{
    echo '/dev/ttyPS0'
}

# ------------------------------------------------------------------------------
extra_reboot_info()
{
    # The CONSOLE_COLLECTOR variable will be set if we are called from rcS
    local console_path=$CONSOLE_COLLECTOR

    # load the ciena_sysfs module if necessary
    if [ ! -f "/sys/ciena/eredan/bootcount" ] ; then
        if [ -z "$console_path" ] ; then
            console_path="/dev/null"
        fi
        # Redirect the output of this command so that it doesn't end up
        # as part of our reboot log.
        insmod_wrapper ciena_sysfs > $console_path
    fi

    local csubr="$(cat /sys/ciena/eredan/csubr)"
    local pmupb="$(cat /sys/ciena/eredan/pmupb)"
    local fault_log_required=0
    local output_str

    if [ "$csubr" == "0x00000000" ] ; then
        csubr=""
    else
        csubr=" csubr=$csubr"
        echo 0 > /sys/ciena/eredan/csubr
        fault_log_required=1
    fi

    if [ "$pmupb" == "0x00000000" ] ; then
        pmupb=""
    else
        pmupb=" pmupb=$pmupb"
        echo 0 > /sys/ciena/eredan/pmupb
        fault_log_required=1
    fi

    output_str=$(printf " (bank=%s,%s reason=%s error=%s,%s multi=%s%s%s)" \
        "$(cat /sys/ciena/eredan/bank)" \
        "$(cat /sys/ciena/eredan/bootcount)" \
        "$(cat /sys/ciena/eredan/reset_reason)" \
        "$(cat /sys/ciena/eredan/error1)" \
        "$(cat /sys/ciena/eredan/error2)" \
        "$(cat /sys/ciena/eredan/multiboot)" \
        "$csubr" \
        "$pmupb")

    if [ $fault_log_required == "1" ] ; then
        generate_fault_banner
        echo "firmware fault detected" >>$FAULTLOG
        echo "$output_str" >>$FAULTLOG
    fi

    echo -n "$output_str"
}

# ------------------------------------------------------------------------------
log_fpga_diag_state()
{
    return
}

