# Helper utilities for init scripts
#
# This file should be "source"d by callers to pick up the functions.
#

source /ciena/scripts/kernel_arg.sh
source /ciena/scripts/board_lib.sh

SUCCESS_STR="OK"
FAIL_STR="FAILED"

# -----------------------------------------------------------------------------
print_app_line() {
    printf "%-8.8s   %-50s " "$(basename $0)" "$1"
}

# -----------------------------------------------------------------------------
print_app_line_cr() {
    print_app_line "$*"
    echo
}

# -----------------------------------------------------------------------------
print_time_stamp() {
    if [ -z "$1" ]
    then
        echo "Need argument for time stamping"
        return
    fi
    now_time=`date "+%s"`
    print_app_line_cr "$1: ciena_time [ $now_time ]"
}
# -----------------------------------------------------------------------------
bg_run() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    $exec_name $* &
    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_and_wait_timeout() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    timeout=$1    ; shift

    print_app_line "$app_string"
    /bin/$exec_name $* &
    /ciena/bin/waitfor -t $timeout $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_and_wait() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    pid_file="/var/run/$(basename $exec_name).pid"

    print_app_line "$app_string"
    $exec_name $* &
    /ciena/bin/waitfor $pid_file

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
valgrind_bg_run_and_wait() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    /usr/bin/valgrind --log-socket=$VALGRIND_IP:$VALGRIND_PORT /bin/$exec_name $* &
    /ciena/bin/waitfor -t 60 $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_log_and_wait() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    echo "*** Log of $exec_name output ($(date))" >> /var/log/$exec_name.log
    /bin/$exec_name $* >> /var/log/$exec_name.log 2>&1 &
    /ciena/bin/waitfor $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_log_and_wait_timeout() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    timeout=$1    ; shift

    print_app_line "$app_string"
    echo "*** Log of $exec_name output ($(date))" >> /var/log/$exec_name.log
    /bin/$exec_name $* >> /var/log/$exec_name.log 2>&1 &
    /ciena/bin/waitfor -t $timeout $exec_name.pid

    rc=$?

    if [ "$rc" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR (timeout)"
    fi

    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
bg_run_and_queue() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    /bin/$exec_name $* &
    wait_list="$wait_list $exec_name.pid"
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
bg_run_and_save_pid() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    /bin/$exec_name $* &
    echo "$!" > $exec_name.pid
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
valgrind_bg_run_and_save_pid() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    port=$1       ; shift

    print_app_line "$app_string"
    /usr/bin/valgrind --log-socket=$VALGRIND_IP:$VALGRIND_PORT /bin/$exec_name $* &
    echo "$!" > $exec_name.pid
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
valgrind_bg_run_and_queue() {
    app_string=$1 ; shift
    exec_name=$1  ; shift
    port=$1       ; shift

    print_app_line "$app_string"
    /usr/bin/valgrind --log-socket=$VALGRIND_IP:$VALGRIND_PORT /bin/$exec_name $* &
    wait_list="$wait_list $exec_name.pid"
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
bg_run_log_and_save_pid() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    echo "*** Log of $exec_name output ($(date))" >> /var/log/$exec_name.log
    /bin/$exec_name $* >> /var/log/$exec_name.log 2>&1 &
    echo "$!" > $exec_name.pid
    echo "$SUCCESS_STR"
}

# -----------------------------------------------------------------------------
fg_run_no_output() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    $exec_name $* > /dev/null 2>&1

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return "$rc"
}

# -----------------------------------------------------------------------------
fg_run() {
    app_string=$1 ; shift
    exec_name=$1  ; shift

    print_app_line "$app_string"
    $exec_name $*

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return "$rc"
}

# -----------------------------------------------------------------------------
print_result() {
    if [ "$1" = "0" ] ; then
        echo "$SUCCESS_STR"
    else
        echo "$FAIL_STR $1"
    fi

    return $1
}

# -----------------------------------------------------------------------------
insmod_wrapper() {
    mod_name=$1
    shift

    print_app_line "Loading $mod_name module"

    /sbin/modprobe $mod_name $*

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
rmmod_wrapper() {
    mod_name=$1
    shift

    print_app_line "Unloading $mod_name module"

    /sbin/modprobe -r $mod_name

    rc=$?
    print_result $rc
    exit_on_error_parm $rc

    return $rc
}

# -----------------------------------------------------------------------------
exit_on_error_parm() {
    if [ "$1" != "0" ] ; then
        exit_wrapper $1
    fi
}

# -----------------------------------------------------------------------------
exit_on_error() {
    rc="$?"
    if [ "$rc" != "0" ] ; then
        exit_wrapper $rc
    fi
    return "$rc"
}

# -----------------------------------------------------------------------------
exit_wrapper() {
    if [ -n "$EXIT_OVERRIDE" ] ; then
        $EXIT_OVERRIDE
    else
        exit $1
    fi
}

# -----------------------------------------------------------------------------
kill_from_dir() {

    count=0

    for pid_file in $(ls ./*.pid 2> /dev/null) ; do
        print_app_line "Killing $(basename $pid_file .pid)"
        pid=$(< $pid_file)
        kill $pid &> /dev/null
        rc=$?
        if [ "$rc" != "0" ] ; then
            echo "FAILED $rc (pid $pid)"
        else
            echo "$pid"
        fi
        rm $pid_file
        ((count += 1))
    done

    if [ "$count" = "0" ] ; then
        echo "Nothing to kill"
    fi
}

# -----------------------------------------------------------------------------
cd_to_pid_dir() {
    this_file=$(basename $1)
    pid_dir="/var/run/$this_file.d"
    mkdir -p $pid_dir
    cd $pid_dir
}

# -----------------------------------------------------------------------------
guard_file_add() {
    death_action=$1 ; shift
    exec_name=$1    ; shift
    args="$*"

    outfile=/tmp/guardian_config

    # If there is already an entry for this executable in the config file,
    # then do not make a new one.
    if [ -e $outfile ] ; then
        if [ "$(grep -c $exec_name $outfile)" != "0" ] ; then
            return
        fi
    fi

    echo "$0 $death_action $exec_name $args" >> $outfile
}

# -----------------------------------------------------------------------------
get_rootfs_rw() {

    /bin/touch /no_touch 2> /dev/null
    if [ -e "/no_touch" ] ; then
        /bin/rm /no_touch
        echo "writable"
    else
        echo "read-only"
    fi
}

# -----------------------------------------------------------------------------
get_rootfs_location() {
    /bin/mount | grep -qE "nfs|ubi|mtd|ext4"
    if [ "$?" -ne "0" ] ; then
        echo "initramfs"
    else
        case "$(kernel_arg root)" in

        /dev/nfs)
            echo "nfs"
            ;;
        /dev/mmc* | \
        mtd* | \
        ubi*)
            echo "flash"
            ;;
        *)
            echo "unknown device ($(kernel_arg root))"
            ;;
        esac
    fi
}

# -----------------------------------------------------------------------------
cookie_clean() {
    local cookie_number="$1" ; shift
    local cookie_name="$1"   ; shift

    /mnt/apps/bin/saparam read "$cookie_number" > /dev/null 2>&1
    if [ "$?" -ne "0" ] ; then
        fg_run "RTFD cookie $cookie_number ($cookie_name) missing" /bin/true
    else
        fg_run_no_output "RTFD cookie $cookie_number ($cookie_name) erase " \
            /mnt/apps/bin/saparam erase $cookie_number
    fi
}

# -----------------------------------------------------------------------------
kick_watchdog() {
    # Under normal circumstances, the watchdog is kicked by the guardian
    # process.  If we need to kick the watchdog prior to the guardian
    # starting, this will kick the watchdog and allow it to be closed
    # gracefully.  Note that this will not work if the NOWAYOUT option
    # is selected in the kernel config.
    echo -n 'V' > /dev/watchdog
}

# -----------------------------------------------------------------------------
check_reset_to_factory_defaults() {

    get_push_button_rtfd_state

    # read factory reset cookie - delete files if it is successful
    /mnt/apps/bin/saparam read $ParamType_FactoryReset > /dev/null 2>&1

    if [ "$?" -eq "0" ] ; then # 0x103 (259) is the factory reset cookie
        # the following files used to be deleted in kernelApi.c
        fg_run "RTFD deleting system directory"   /bin/rm -rf /mnt/sysfs/system/*
        fg_run "RTFD deleting software directory" /bin/rm -rf /mnt/sysfs/software/*
        fg_run "RTFD deleting log directory"      /bin/rm -rf /mnt/sysfs/log/*
        fg_run "RTFD deleting config directory"   /bin/rm -rf /mnt/sysfs/config/*
        fg_run "RTFD deleting ssh directory"      /bin/rm -rf /mnt/sysfs/ssh/*
        fg_run "RTFD deleting fpga images"        /bin/rm -rf /mnt/sysfs/fpga/*
        fg_run "RTFD deleting corefiles"          /bin/rm -rf /mnt/log/corefiles/*

        kick_watchdog

        # now erase the the cookies
        cookie_clean   9 "serial console"
        cookie_clean 260 "IP address"
        cookie_clean 261 "subnet mask"
        cookie_clean 262 "gateway"
        cookie_clean 299 "power DC"
        cookie_clean   2 "system mode"
        cookie_clean  11 "local interface"
        cookie_clean   1 "full POST 1"
        cookie_clean   3 "full POST 2"
        cookie_clean  16 "POST packets"
        cookie_clean  17 "POST loopback mask"
        cookie_clean  18 "POST port mask"
        cookie_clean  19 "POST repeat"
        cookie_clean  22 "POST reset"
        cookie_clean  31 "POST skip"
        cookie_clean 305 "POST bypass"
        cookie_clean 313 "Guardian Disable"
        cookie_clean 314 "Guardian Max Reboots"
        cookie_clean 315 "PBT Mode"
        cookie_clean 319 "Disable Enhanced CFM FPGA Port"

        # Clear the RTFD cookie last so that if we get interrupted in
        # the middle of this operation, we will come back and finish
        # the job.
        cookie_clean $ParamType_FactoryReset "factory reset"

        kick_watchdog
    fi
}

# -----------------------------------------------------------------------------
bank_status_cookie_clean() {
    local cookie_number="$1" ; shift
    local cookie_name="$1"   ; shift

    /mnt/apps/bin/saparam read "$cookie_number" > /dev/null
    if [ "$?" -ne "0" ] ; then
        fg_run "cookie $cookie_number ($cookie_name) missing" /bin/true
    else
        fg_run_no_output "cookie $cookie_number ($cookie_name) erase " \
            /mnt/apps/bin/saparam erase $cookie_number
    fi
}

# -----------------------------------------------------------------------------
clear_bank_status() {
    # read factory reset cookie - delete files if it is successful
    /mnt/apps/bin/saparam read 302 > /dev/null 2>&1

    if [ "$?" -eq "0" ] ; then # 0x12E (302) is the bank status cookike
        # now erase the cookies
        bank_status_cookie_clean 302 "bank status"
        bank_status_cookie_clean 303 "bank 0 status check time"
        bank_status_cookie_clean 304 "bank 1 status check time"
    fi
}

# -----------------------------------------------------------------------------
mount_log_partition_on_jffs2() {
    print_app_line "mount logs via fstab"
    /bin/mount /mnt/log > /dev/null 2> /dev/null
    print_result "$?"
    if [ "$?" -ne "0" ] ; then
        fg_run "clearing log partition (this will take a while)" \
            /usr/sbin/flash_eraseall -j -q /dev/mtd/logs
        fg_run "mount logs via fstab" /bin/mount /mnt/log
    fi
}

# -----------------------------------------------------------------------------
get_console_tty()
{
  _tty=`kernel_arg console | awk -F, '{print $1}'`
  if [ -z "${_tty}" ]; then
    echo $(default_console_tty)
  else
    echo "/dev/${_tty}"
  fi
}

# -----------------------------------------------------------------------------
console_enabled()
{
    [ -n "$(kernel_arg console)" ]
}

# -----------------------------------------------------------------------------
set_epoch_time()
{
    local epoch=$1 ; shift

    # Set date to Jan 1 2000 if the date is 1970-01-01
    # When running MIPS kernels 2.6.24 and higher, the date will default to the
    # Unix epoch (Jan 1, 1970).  This is represented as a date of 0.  When PAM
    # sees password entries that have been set on date 0, it assumes that the
    # entries have been administratively expired (and this prevents LEOS logins).
    # To avoid this, force the date to something non-zero.
    # On systems with an insane HW RTC, the time can be less than the epoch,
    # so treat these the same way.
    if [ "$(date -I | cut -d- -f1)" -le "1970" ] ; then
        print_app_line_cr "Setting date: $(/bin/date -s $epoch)"
        # linux driver will return 01-01-1970 00:00:00 if the date in the
        # RTC is bad, therefore if the date is 1970-01-01,
        # then the RTC should be set to a valid date too
        if board_has_rtc; then
            hwclock -w
        fi
    fi
}

# -----------------------------------------------------------------------------
get_base_mac()
{
    mac=$(dd if=$(get_eeprom_file) bs=1 skip=34 count=12 2>/dev/null)

    # insert ':' between octets and set to uppercase
    mac=$(echo $mac |                        \
          sed -e 's|..|\0:|g' -e 's|:$||' |  \
          awk '{print toupper($1)}')

    echo $mac
}

# -----------------------------------------------------------------------------
running_fpga_image_version()
{
    /ciena/bin/bsfreg | awk '
        BEGIN {
            build = "0x0000"
            bank  = "unknown"
            revision = "0x0000"
        }
        /VIN_MPU_VERSION/       { revision = $3 }
        /VIN_MPU_BOARD_ID/      { board_id = $3 }
        /OCT_MPU_VERSION /      { revision = $3 }
        /OCT_MPU_BUILD_VERSION/ { build = $3 }
        /OCT_MPU_RESET_BUTTON/  { bank = and($3, 8) ? "user" : "golden" }
        /OCT_MPU_FPGA_DATE_0/   { date_0 = $3 }
        /OCT_MPU_FPGA_DATE_1/   { date_1 = $3 }
        /OCT_MPU_BOARD_ID/      { board_id = $3 }
        /OCT_MPU_FPGA_ID/       { id = $3 }
        /FARIN_REVISION/        { revision = $3 }
        /FARIN_BUILD_REVISION/  { build = $3 }
        /FARIN_PB_STATE/        { bank = and($3, 8) ? "user" : "golden" }
        /FARIN_FPGA_DATE0/      { date_0 = $3 }
        /FARIN_FPGA_DATE1/      { date_1 = $3 }
        /FARIN_BOARD_ID/        { board_id = $3 }
        /FARIN_FPGA_ID/         { id = $3 }
        /GAMIL_VERSION/         { revision = $3 }
        /GAMIL_BUILD_VERSION/   { build = $3 }
        /GAMIL_RESET_BUTTON/    { bank = and($3, 8) ? "user" : "golden" }
        /GAMIL_FPGA_DATE_0/     { date_0 = $3 }
        /GAMIL_FPGA_DATE_1/     { date_1 = $3 }
        /GAMIL_BOARD_ID/        { board_id = $3 }
        /GAMIL_FPGA_ID/         { id = $3 }
        /LINDIR_VERSION/        { revision = $3 }
        /LINDIR_BUILD_VERSION/  { build = $3 }
        /LINDIR_RESET_BUTTON/   { bank = and($3, 8) ? "user" : "golden" }
        /LINDIR_FPGA_DATE_0/    { date_0 = $3 }
        /LINDIR_FPGA_DATE_1/    { date_1 = $3 }
        /LINDIR_BOARD_ID/       { board_id = $3 }
        /LINDIR_FPGA_ID/        { id = $3 }
        /VANYAR_VERSION/        { revision = $3 }
        /VANYAR_BUILD_VERSION/  { build = $3 }
        /VANYAR_RESET_BUTTON/   { bank = and($3, 8) ? "user" : "golden" }
        /VANYAR_FPGA_DATE_0/    { date_0 = $3 }
        /VANYAR_FPGA_DATE_1/    { date_1 = $3 }
        /VANYAR_BOARD_ID/       { board_id = $3 }
        /VANYAR_FPGA_ID/        { id = $3 }
        /BALIN_VERSION/         { revision = $3 }
        /BALIN_BUILD_VERSION/   { build = $3 }
        /BALIN_RESET_BUTTON/    { bank = and($3, 8) ? "user" : "golden" }
        /BALIN_FPGA_DATE_0/     { date_0 = $3 }
        /BALIN_FPGA_DATE_1/     { date_1 = $3 }
        /BALIN_BOARD_ID/        { board_id = $3 }
        /BALIN_FPGA_ID/         { id = $3 }
        /NORIN_VERSION/         { revision = $3 }
        /NORIN_BUILD_VERSION/   { build = $3 }
        /NORIN_RESET_BUTTON/    { bank = and($3, 8) ? "user" : "golden" }
        /NORIN_FPGA_DATE_0/     { date_0 = $3 }
        /NORIN_FPGA_DATE_1/     { date_1 = $3 }
        /NORIN_BOARD_ID/        { board_id = $3 }
        /NORIN_FPGA_ID/         { id = $3 }
        /AMROTH_VARIANT/        { variant = $3 }
        /AMROTH_VERSION/        { revision = $3 }
        /AMROTH_BUILD_VERSION/  { build = $3 }
        /AMROTH_RESET_BUTTON/   { bank = and($3, 8) ? "user" : "golden" }
        /AMROTH_FPGA_DATE/      { date_0 = $3 }
        /AMROTH_BOARD_ID/       { board_id = $3 }
        /AMROTH_FPGA_ID/        { id = $3 }
        /SAROS_REV/             { variant = $3 }
        /SAROS_MJR/             { revision = sprintf("0x%04x",or(revision,lshift(and($3,0xff),8))) }
        /SAROS_MNR/             { revision = sprintf("0x%04x",or(revision,and($3,0xff))) }        
        /SAROS_PB_STATUS/       { bank = and($3, 8) ? "user" : "golden" }
        /SAROS_BLD/             { build = sprintf("0x%02x",and($3, 0xff)) }
        /SAROS_BID/             { board_id = $3 }
        /SAROS_FID/             { id = $3 }
        END {
            print "revision=" revision " build=" build " bank=" bank " date_0=" date_0 " date_1=" date_1 " board_id=" board_id " id=" id " variant=" variant ;
        }
    '
}

#--------------------------------------------------------------------
find_mtd_device()
{
    # $1: partition name i.e. isfuser/isfgold

    for x in /sys/class/mtd/mtd*; do
        if [ -f $x/name ] && [ "$(<$x/name)" == "$1" ]; then
            basename $x
            return 0
        fi
    done

    return 1
}

#--------------------------------------------------------------------
find_ubi_volume()
{
    # $1: volume name
    for x in /sys/class/ubi/ubi*; do
        if [ -f $x/name ] && [ "$(<$x/name)" == "$1" ]; then
            basename $x
            return 0
        fi
    done

    return 1
}

#--------------------------------------------------------------------
create_ubi_volume()
{
    # $1: partition name
    # $2: partition size
    local name=$1
    local size=$2
    local dev=/dev/ubi0

    if [ ! -c $dev ]; then
        echo "UBI device, $dev, not found"
        return 1
    fi

    ubimkvol $dev -s $size -N $name >/dev/null
}

#--------------------------------------------------------------------
mount_xftp_partition()
{
    # The xftp partition is maintained for backwards compatibility.
    # If we need to create the partition, then we make a minimally
    # sized partition.  If the partition already exists, then we will
    # shrink it down to the desired size if it is unused.
    #
    local name=xftp
    local target_size=10579968 # = 82 * (126 * 1024) = 82 LEBs ~= 10MB
    local mnt=/mnt/xftp

    find_ubi_volume $name >/dev/null
    if [ $? != 0 ]; then
        create_ubi_volume $name $target_size
        if [ $? != 0 ]; then
            echo "Failed to create UBI volume $name"
            return 1
        fi
    fi

    mount $mnt

    # this directory is dedicated for the FTP/TFTP server feature in 6.10.2
    # it is owned by ftp user and leosadmin group. 503 is leosadmin's group id
    # which will be created at run time.

    chown ftp:503 $mnt

    # Shrink the partition if it is unused and larger than our target size
    
    local ubinfo=""
    local free_space=0

    # if the xftp  partition exists, then find the size and volume id

    ubinfo=$(ubinfo /dev/ubi0 -N $name)
    if [ "$?" -ne "0" ] ; then
        # ubinfo failed - give up
        return
    fi

    local size=$(echo "$ubinfo" | awk '/^Size:/ {print $4}' | sed s/\(// )
    local vol_id=$(echo "$ubinfo" | awk '/^Volume ID:/ {print $3}' )

    if [ "$size" -le "$target_size" ] ; then
        # partition is already small enough
        return
    fi

    if [ "$(du -s $mnt | cut -f1)" -ne "0" ] ;  then
        # partition is not empty - leave it alone
        return
    fi

    # Create a new smaller xftp partition
    echo
    print_app_line_cr "shrink $name volume to $((target_size/1024/1024))M"
    fg_run "unmount $mnt" umount $mnt
    fg_run "remove $name volume" \
        ubirmvol /dev/ubi0 -N $name

    print_app_line_cr "create new $name volume"
    ubimkvol /dev/ubi0 -N $name -n $vol_id -s $target_size
    print_result "$?"

    print_app_line "mount resized $mnt"
    mount $mnt
}

#--------------------------------------------------------------------
set_local_interface_mac()
{
    local ifname=$(get_local_if)
    local base_mac=$(get_base_mac)
    local link_show=$(ip link show $ifname)
    local curr_mac=$(echo -ne "$link_show" |
                     grep 'link/ether' |\
                     cut -c16-32 |\
                     tr a-z A-Z)

    # If the correct MAC is already programmed, then there is no need
    # to do anything here.
    if [ "$base_mac" == "$curr_mac" ] ; then
        return
    fi

    local state=$(echo -ne "$link_show" | grep -c UP)

    # The MAC can only be set if the link is down, so bring the link
    # down if it was UP when we got here.  Put the state back the way
    # it should be once we are done.

    if [ "$state" != "0" ] ; then
        ip link set dev $ifname down
    fi

    fg_run_no_output "setting local interface MAC" \
        ip link set dev $ifname address $base_mac

    if [ "$state" != "0" ] ; then
        ip link set dev $ifname up
    fi
}

#--------------------------------------------------------------------
verify_reset_reason()
{
    local SYSLOG_ERR="logger -p USER.ERR  -t $FUNCNAME"
    local SYSLOG_INFO="logger -p USER.INFO  -t $FUNCNAME"

    # Get the reset reason
    local reset_reason="$(read_u32_cookie $ParamType_LastResetReason)"

    case $(get_board_name) in
        '3903'| \
        '3904'| \
        '3905'| \
        '3906') reg=AMROTH_SCRATCHPAD ;;
        '3916') reg=FARIN_SCRATCHPAD ;;
        '3926'| \
        '3928') reg=SAROS_SCR ;;
        '3930') reg=OCT_MPU_SCRATCHPAD ;;
        '3931') reg=OCT_MPU_SCRATCHPAD ;;
        '3932') reg=BALIN_SCRATCHPAD ;;
        '3938') reg=VANYAR_SCRATCHPAD ;;
        '3940') return ;;
        '3942') reg=LINDIR_SCRATCHPAD ;;
        '3960') return ;;
        '5142') reg=GAMIL_SCRATCHPAD ;;
        '5150') reg=VIN_MPU_TEST_0 ;;
        '5160') reg=NORIN_SCRATCHPAD ;;
        *) echo "$FUNCNAME called on unsupported board"; return ;;
    esac

    # Get the value in the scratchpad register
    val=`/ciena/bin/bsfreg $reg | awk '{ print $3 }' `

    # IF: the scratchpad value is 0xf, then there was no power fail
    #     so if the reset reason cookie is "pwrFail", then it was
    #     likely a kernel issue.
    # ELSE IF: the scratchpad value is 0x0, it was probably a power
    #          fail so if the reset reason cookie is not "pwrFail",
    #          that is an error
    # ELSE IF: the scratchpad value is 0x8000, the previously running
    #          software did not have this logic, so log that we don't
    #          know.
    # ELSE: this shouldn't happen, so log it.
    if [[ "$val" -eq 0xf ]] ; then
        if [[ "$reset_reason" -eq $ResetReason_PwrFail ]] ; then
            $SYSLOG_ERR "Reset reason cookie error; no power fail detected."
            write_cookie $ParamType_LastResetReason $ResetReason_Unknown
        fi
    elif [[ "$val" -eq 0x0 ]] ; then
        if [[ "$reset_reason" -ne $ResetReason_PwrFail ]] ; then
            $SYSLOG_ERR "Reset reason cookie error; FPGA upgrade likely."
        fi
    elif [[ "$val" -eq 0x8000 ]] ; then
        $SYSLOG_INFO "Cannot detect accuracy of reset reason cookie due to "\
                     "lack of support in previously running software."
    else
        $SYSLOG_INFO "Unexpected value in scratchpad register."
    fi

    # set the scratchpad register to a known value
    /ciena/bin/bsfreg $reg 0xf
}

#--------------------------------------------------------------------
bootarg_get()
{
    # the boot arguments are strings separated by spaces: use "printf"
    # to split them into multiple lines, then grep to pick up:
    #    <start of line>"<sought arg>"<either "=" or end-of-line>
    printf "%s\n" $(cat /proc/device-tree/chosen/bootargs) | \
        grep ^"${1}"'\(=.*\|$\)'
}
