#!/bin/bash -x

_ACTION=${1}
_GPIO=${2}
_CIC=${3}
_DEV_DIR=/dev/cic_gpio
_SYSLOG_ERR="/usr/bin/logger -p USER.ERR -t ${BASH_SOURCE}"

function find_softlink()
{
    local _src_dir="${1}" _dst="${2}" _full="${3:+-f}" _i

    if [ ! -e "${_src_dir}" ]; then
        ${_SYSLOG_ERR} "source directory ${_src_dir} does not exist"
        return -1
    fi

    for _i in $(/usr/bin/find ${_src_dir}/ -type l -maxdepth 1); do
        # fully resolve the soft link only when a device is added
        # when a device is removed, the link is already stale and a full resolve will fail
        if [ "$(/usr/bin/readlink ${_full}${_full:+ }${_i})" = "${_dst}" ]; then
            echo "${_i}"
            return 0
        fi
    done

    ${_SYSLOG_ERR} "cannot find softlink to ${_dst} under ${_src_dir}"
    return -1
}

case ${_ACTION} in

    add)
        #
        # a gpio pin had been added as:
        #    /sys/devices/foo/cic/bar/gpiox
        #
        # the cic driver is supposed to have created a softlink:
        #     /sys/devices/foo/cic/irq_name -> /sys/devices/foo/cic/bar/gpiox
        #
        # our job is to replicate the soft link as:
        #     /dev/cic_gpio/irq_name -> /sys/devices/foo/cic/bar/gpiox
        #
        _LINK="$(find_softlink ${_GPIO%/${_CIC}/*}/${_CIC} ${_GPIO} full)"
        if [ "$?" != "0" ]; then
            exit -1
        fi
        mkdir -p ${_DEV_DIR}
        /bin/ln -sf "${_GPIO}" "${_DEV_DIR}/${_LINK##*/}"
        if [ ! -h "${_DEV_DIR}/${_LINK##*/}" ]; then
            ${_SYSLOG_ERR} "did not create soft link ${_DEV_DIR}/${_LINK##*/} -> ${_GPIO}"
            exit -1
        fi
        ;;

    remove)
        #
        # a gpio pin had been removed:
        #    /sys/devices/foo/cic/bar/gpiox
        #
        # a previous udev add rule is supposed to have created a soft link:
        #     /dev/cic_gpio/irq_name -> /sys/devices/foo/cic/bar/gpiox
        #
        # and our job is to remove it
        #
        _LINK="$(find_softlink ${_DEV_DIR} ${_GPIO})"
        if [ "$?" != "0" ]; then
            exit -1
        fi
        /bin/rm -f "${_LINK}"
        ;;

    *)
        ${_SYSLOG_ERR} "unknown action ${_ACTION}"

esac
