#!/bin/bash

## This script fixes incorrect checksum values in power supply EEPROMs that
## contain manufacturing data. An incorrect checksum will cause a POST warning.
##
## The checksum value can be verified with the SAOS command "chassis show power",
## which will display the data (and the checksum value) if the checksum is
## correct, or list the power supply as "Unequipped" if it is incorrect.

source /ciena/scripts/board_lib.sh

SCRIPT_VERSION='1.5'

OPTS=$(getopt -o 'ndw:hf' -l 'dry-run,debug,help,force' -- "$@")
eval set -- "$OPTS"

unset A B DRY DEBUG OVERRIDE_CKSUM HELP FORCE

while true; do
	case "$1" in
		-n|--dry-run) DRY=yes ; shift ;;
		-d|--debug) DEBUG=yes ; shift ;;
		-w) OVERRIDE_CKSUM="$2" ; shift 2 ;;
		-f|--force) FORCE=yes ; shift ;;
		-h|--help) HELP=yes ; shift ;;
		--) shift ; break ;;
		*) echo "Argument parsing error" ; exit 1 ;;
	esac
done

RET=0
for arg; do
	case "$arg" in
		A) A=A ; shift ;;
		B) B=B ; shift ;;
		*) echo "Invalid power supply ${arg}; must be A or B" ; RET=1 ; HELP=yes ;;
	esac
done

if [ -n "$HELP" ]; then
	echo "Usage: $0 [options] [A] [B]"
	echo ''
	echo 'Options:'
	echo '-n|--dry-run	Dry run (Nothing will be written)'
	echo '-w <checksum>	Use [checksum] as the new checksum instead of calculating it'
	echo '-f|--force	Force writing checksum (overrides -n)'
	echo '-h|--help	Display this text'
	echo ''
	echo 'Specify A and/or B on the command line to select which power supply to operate'
	echo 'on. If neither is given, both power supplies are checked but nothing will be'
	echo 'written.'
	echo ''
	echo "Version: $SCRIPT_VERSION"
	exit $RET
fi

if [ -n "$OVERRIDE_CKSUM" ]; then
	if [ ${#OVERRIDE_CKSUM} -ne 2 ]; then
		echo "-w argument must be 2 characters long"
		exit 1
	fi
fi

board_name=$(get_board_name)
# Mappings are from src/software/saos-sds/leos/platform/caliondo/src/i2cPlatform.c
case "$board_name" in
	5150)
		PS0_DEV='58-0050'
		PS1_DEV='59-0051'
		;;
	3930)
		PS0_DEV='12-0050'
		PS1_DEV='13-0051'
		;;
	3931)
		PS0_DEV='12-0050'
		PS1_DEV='13-0051'
		;;
	3932)
		PS0_DEV='13-0050'
		PS1_DEV='14-0051'
		;;
	3942)
		PS0_DEV='2-0050'
		PS1_DEV='3-0051'
		;;
	5142)
		PS0_DEV='2-0050'
		PS1_DEV='3-0051'
		;;
	5160)
		PS0_DEV='2-0050'
		PS1_DEV='3-0051'
		;;
	3926)
		PS0_DEV='20-0050'
		PS1_DEV='19-0051'
		;;
	*)
		echo "No known power supply EEPROM for $board_name"
		exit 1
		;;
esac

PS0_EEPROM="/sys/bus/i2c/devices/$PS0_DEV/eeprom"
PS1_EEPROM="/sys/bus/i2c/devices/$PS1_DEV/eeprom"

choose_eeprom() {
	if [ $1 = A ]; then
		EEPROM="$PS0_EEPROM"
	elif [ $1 = B ]; then
		EEPROM="$PS1_EEPROM"
	else
		echo "Invalid power supply ${1}" 1>&2
		exit 1
	fi
}

read_eeprom() {
	dd if="$1" $2 2>/dev/null
}

# Get the decimal offset of the first character of the specified field name ($2)
# in the specified EEPROM ($1)
get_field_offset() {
	local off=$(read_eeprom "$1" | strings -o | grep "$2:" | awk '{print $1}')
	if [ -z "$off" ]; then
		if [ -n "$DEBUG" ]; then
			echo "Failed to get field offset for $2 from $1" 1>&2
		fi
		exit 1
	fi
	# `strings -o` gives offsets in octal
	echo $((0${off}))
}

# Read the specified EEPROM ($1) and get the value of the specified field ($2)
get_field() {
	local field=$(read_eeprom "$1" | strings | grep "^$2:")
	if [ -z "$field" ]; then
		echo "Failed to get field $2 from $1" 1>&2
		exit 1
	fi
	printf "%s" "$field" | cut -d' ' -f2
}

# $1: EEPROM character device
# $2: Bytes to checksum
# Calculates a one-byte checksum (formatted as two hex digits) of the first $2
# bytes of the file at $1. The checksum is calculated by calculating the sum of
# the bytes and taking the last byte of the resulting sum.
calculate_checksum() {
	read_eeprom "$1" "bs=1 count=$2" | \
	hexdump -v -e '/1 "%d\n"' | \
	(
	while read num; do
		total=$((total+num))
	done
	printf "%04X" $total | cut -b3-
	)
}

if [ -z "$A" ] && [ -z "$B" ]; then
	DRY=yes
	DO_PS="A B"
else
	DO_PS="$A $B"
fi

for ps in $DO_PS; do
	choose_eeprom $ps

	if [ ! -e "$EEPROM" ]; then
		echo "WARNING: $EEPROM (PS $ps) does not exist"
	else
		if [ ! -r "$EEPROM" ] || [ ! -w "$EEPROM" ]; then
			echo "WARNING: No read/write permission for $EEPROM (PS $ps)"
		fi
	fi

	if ! offset=$(get_field_offset "$EEPROM" 'CHECKSUM'); then
		echo "Failed to read EEPROM $EEPROM for power supply ${ps}; skipping."
		continue
	fi
	check_bytes=$(($offset-1))
	checksum_offset=$(($offset+$(printf 'CHECKSUM: ' | wc -c)))
	if [ -n "$DEBUG" ]; then
		printf 'Checksum value offset: 0x%x\n' $checksum_offset
	fi

	old_checksum=$(get_field "$EEPROM" 'CHECKSUM')
	new_checksum=$(calculate_checksum "$EEPROM" $check_bytes)

	echo "Current power supply $ps checksum: $old_checksum"
	if [ -z "$OVERRIDE_CKSUM" ]; then
		echo "Correct power supply $ps checksum: $new_checksum"
	else
		echo "Correct power supply $ps checksum (actual): $new_checksum"
		new_checksum=$OVERRIDE_CKSUM
		echo "Correct power supply $ps checksum (overridden): $new_checksum"
	fi

	# Case insensitive comparison
	if [ -n "$FORCE" ] || [ "${old_checksum,,}" != "${new_checksum,,}" ]; then
		if [ -n "$FORCE" ] || [ -z "$DRY" ]; then
			echo -n $new_checksum | \
			dd of="$EEPROM" count=2 seek=$checksum_offset bs=1 2>/dev/null
			if [ $? -ne 0 ]; then
				echo "Failed to write checksum to power supply $ps"
			else
				echo "Wrote new checksum to power supply $ps"
			fi
		else
			echo "Would write new checksum to power supply $ps"
		fi
	else
		echo "Power supply $ps checksum not modified"
	fi
done
