#!/bin/bash

source /ciena/scripts/utils.sh

upgrade_version="1.0"
page_size=264

#--------------------------------------------------------------------

die()
{
    echo "$1"
    exit 1
}

#--------------------------------------------------------------------

now()
{
    date '+%s'
}


run_cmd()
{
    # $1: Title
    # $2: Command

    printf '  * %-40.40s ' "$1"

    local _start=$(now)
    eval "$2 &>/dev/null" || die 'FAILED'
    local _diff=$(($(now) - _start))

    [ ${_diff} -eq 0 ] && _diff='< 1'
    echo "OK [${_diff}s]"
}

#--------------------------------------------------------------------
input_file_cat()
{
    if [ "${1%%.gz}" = "${1}" ]; then
        cat "${1}"
    else
        zcat "${1}"
    fi
}

#--------------------------------------------------------------------
input_file_size()
{
    # $1: Partition input file
    (input_file_cat "${1}" | wc -c) 2>/dev/null
}

#--------------------------------------------------------------------
erase_block_count()
{
    # $1: Device Name
    # $2: Partition input file
    local eb_size=0x$(grep ^$1: /proc/mtd | cut -f3 -d\ )
    local im_size=$(input_file_size "$2") || \
        die 'FAILED to get image size'

    echo $(( $(( ${im_size} + ${eb_size} - 1 )) / ${eb_size} ))
}

#--------------------------------------------------------------------

erase_isf_part()
{
    # $1: Device Name
    # $2: Partition input file

    flash_erase -q /dev/${1} 0 $(erase_block_count ${1} ${2})
}

#--------------------------------------------------------------------

write_isf_part()
{
    # $1: Device Name
    # $2: Partition input file

    input_file_cat "${2}" | \
        dd bs=${option_wbsize:-${page_size}} of=/dev/$1
}

#--------------------------------------------------------------------

verify_isf_part()
{
    # $1: Device name
    # $2: Partition input file

    local image_size

    if [ -n "${option_size}" ]; then
        image_size="${option_size}"
    else
        image_size=$(input_file_size "$2") || return 1
    fi

    input_file_cat "${2}" | cmp --bytes=$image_size /dev/$1
}

#--------------------------------------------------------------------

md5sum_isf_part()
{
    # $1: Device name
    # $2: Size

    dd bs=$2 count=1 if=/dev/$1 2>/dev/null | md5sum | cut -f1 -d' '
}

#--------------------------------------------------------------------

erase_altera_part()
{
    # $1: Device Name
    # $2: Partition input file

    flash_erase -q /dev/${1} 0 $(erase_block_count ${1} ${2})
}

#--------------------------------------------------------------------

write_altera_part()
{
    # $1: Device Name
    # $2: Partition input file

    input_file_cat "${2}" | /ciena/bin/reverse > /dev/$1
}

#--------------------------------------------------------------------

verify_altera_part()
{
    # $1: Device name
    # $2: Partition input file

    local image_size

    if [ -n "${option_size}" ]; then
        image_size="${option_size}"
    else
        image_size=$(stat -c'%s' "$2" 2> /dev/null) || return 1
    fi

    input_file_cat "${2}" | /ciena/bin/reverse | \
        cmp --bytes=$image_size /dev/$1
}

#--------------------------------------------------------------------

md5sum_altera_part()
{
    # $1: Device name
    # $2: Size

    dd bs=$2 count=1 if=/dev/$1 2>/dev/null | /ciena/bin/reverse | md5sum | cut -f1 -d' '
}

#--------------------------------------------------------------------
combo_tdm_fru_reconfigure()
{
    echo "option --reconfigure is not supported for the ${1} FPGA"
    echo "to reconfigure, use:"
    echo "    rc.combo_tdm_fru stop"
    echo "    rc.fru_power restart"
    echo "    rc.combo_tdm_fru start"
    exit 1
}

#--------------------------------------------------------------------
validate_image_fname()
{
    local _fpga="${1}" _bank="${2}" _fname="${3}"
    local _brd=$(get_board_name)
    local _usually

    if [ "main_bsf" = "${_fpga}" ]; then
        _usually="$(grep ^${_brd}\ *${_bank} /ciena/fpga/manifest)"
        # the FPGA name is the last field in the record
        _usually="${_usually##* }"
    else
        _usually="${_fpga}"
    fi

    if [ "${_fname}" = "${_fname/${_usually}}" ]; then
        # print a friendly warning if users are about to shoot
        # themselves in the foot
        echo "WARNING: ${_fpga}/${_bank} images are usually named \"${_usually}\""
    fi
}

#--------------------------------------------------------------------

reconfigure_fpga()
{

    board=$(get_board_name)

    case $board in
        '5150')           cmd="/ciena/bin/vinmarreg    VIN_MPU_FPGA_ICAP_CTRL 0xa5a5";
                          rstcmd="/ciena/bin/vinmarreg VIN_MPU_SW_RESET 0xa5a5" ;;
        '3916')           cmd="/ciena/bin/farinreg     FARIN_FPGA_ICAP_CTRL   0xa5a5";
                          rstcmd="/ciena/bin/farinreg  FARIN_BRD_RESET 0xa5a5" ;;
        '3922' | '3924' | '3926' | '3928')\
                          cmd="/ciena/bin/sarosreg     SAROS_RCFG_CMD 0x5a5a";
                          rstcmd="/ciena/bin/sarosreg  SAROS_BRD_RESET 0xa5a5" ;;
        '3930' | '3931')  cmd="/ciena/bin/octopusreg   OCT_MPU_FPGA_ICAP_CTRL 0xa5a5";
                          rstcmd="/ciena/bin/octopusreg OCT_MPU_BRD_RESET 0xa5a5" ;;
        '3932')           cmd="/ciena/bin/balinreg     BALIN_FPGA_RECONFIG    0x5a5a";
                          rstcmd="/ciena/bin/balinreg  BALIN_MPU_BRD_RESET 0xa5a5" ;;
        '3938')           cmd="/ciena/bin/vanyarreg     VANYAR_FPGA_RECONFIG    0x5a5a";
                          rstcmd="/ciena/bin/vanyarreg  VANYAR_MPU_BRD_RESET 0xa5a5" ;;
        '3942')           cmd="/ciena/bin/lindirreg    LINDIR_FPGA_RECONFIG    0x5a5a";
                          rstcmd="/ciena/bin/lindirreg LINDIR_MPU_BRD_RESET 0xa5a5" ;;
        '5142')           cmd="/ciena/bin/gamilreg     GAMIL_FPGA_RECONFIG    0x5a5a";
                          rstcmd="/ciena/bin/gamilreg  GAMIL_MPU_BRD_RESET 0xa5a5" ;;
        '5160')           cmd="/ciena/bin/norinreg     NORIN_FPGA_RECONFIG    0x5a5a";
                          rstcmd="/ciena/bin/norinreg  NORIN_MPU_BRD_RESET 0xa5a5" ;;
        '3906' | '3905' | '3904' | '3903')\
                          cmd="/ciena/bin/amrothreg    AMROTH_FPGA_RECONFIG 0x5a5a";
                          rstcmd="/ciena/bin/amrothreg AMROTH_MPU_BRD_RESET 0xa5a5" ;;
	*)       echo "unsupported board = \"$board\"" && exit 1 ;;
    esac

    /ciena/bin/evt_log ALERT "Rebooting due to board support FPGA reconfiguration request"

    sync && sleep 1

    eval $cmd

    eval $rstcmd
}

#--------------------------------------------------------------------

usage()
{
    echo
    echo "Usage:"
    echo -e "\t $0    [options] [--] [image filename]"
    echo
    echo "Upgrade or verify a board support FPGA image"
    echo
    echo "Options:"
    echo -e "\t-b,--bank=<bank>\tSpecify which bank to target, user (default) or golden."
    echo -e "\t-c,--compare \t\tVerify the fpga image againt the given image."
    echo -e "\t-f,--fpga=<name>\tOverride the FPGA name."
    echo -e "\t-m,--md5     \t\tRun md5sum on the image. --size required."
    echo -e "\t-r,--reconfigure\tTrigger the fpga to recongifure with the user image."
    echo -e "\t-s,--size=<size>\tImage size in bytes."
    echo -e "\t-w,--wbsize=<size>\tWrite block size (default 264)."
    echo -e "\t-h,--help    \t\tPrint usage."
    echo -e "\t-v,--version \t\tPrint script version."
    echo

    exit $1
}

#--------------------------------------------------------------------

version() {
        echo "version $upgrade_version"
        exit 0
}

#--------------------------------------------------------------------

#
# Get Boardconfig
#

board=$(get_board_name)
fpga_name="main_bsf"

case $board in
    '5150' | '3916' | '3930' | '3931') \
        interface="isf"; mtduser_name="isfuser"; mtdgold_name="isfgold" ;;

    '3922' | '3924' | '3926' | '3928' | '3932' | '3938' | '3942' | '5142' | '5160' | '3906' | '3905' | '3904' | '3903') \
        interface="altera"; mtduser_name="isfuser"; mtdgold_name="isfgold" ;;

    *)  echo "unsupported board is \"$board\"" && exit 1 ;;
esac


#
# Process options
#

longoptions="bank:,compare,debug,fpga:,help,md5,reconfigure,size:,version,wbsize:"
shortoptions="b:cdf:hmrs:vw:"

opts=`getopt --name=$0 --options="$shortoptions" --longoptions="$longoptions" -- "$@"`

[ $? -eq 0 ] || usage 1

eval set -- "$opts"

while true; do
    case "$1" in
    -b | --bank)         option_bank=$2;        shift; shift ;;
    -c | --compare)      option_compare=1;      shift ;;
    -f | --fpga)         option_fpga=$2;        shift; shift ;;
    -m | --md5)          option_md5=1;          shift ;;
    -r | --reconfigure)  option_reconfigure=1;  shift ;;
    -s | --size)         option_size=$2;        shift; shift ;;
    -w | --wbsize)       option_wbsize=$2;      shift; shift ;;
    -h | --help)         usage 0;               shift ;;
    -v | --version)      version;               shift ;;
    --debug)             option_debug=1;        shift ;;
    --)                                         shift; break ;;
    *)  die 4 "Unhandled option $1";            shift ;;
    esac
done

option_image=$1

#
# First and foremost, check if the FPGA name is overridden.
#
if [ -n "$option_fpga" ]; then

    fpga_name="${option_fpga}"

    case "$option_fpga" in
        "codechip" | \
        "riffelhorn")
            interface="isf"
            ;;
        "norborn")
            fpga_name="nrbn"
            interface="isf"
            ;;
        "sarn")
            interface="altera";
            ;;
        *)
            echo "unknown FPGA is \"$option_fpga\""
            exit 1
            ;;
    esac

    mtduser_name="${option_fpga}-user"
    mtdgold_name="${option_fpga}-gold"
    [ -n "$option_reconfigure" ] && combo_tdm_fru_reconfigure "${option_fpga}"

fi

[ -n "$option_reconfigure" ] && reconfigure_fpga

#
# Sanitize bank option
#

[ -n "$option_bank" ]  || option_bank="user"

case "$option_bank" in
    user)   mtdname=$mtduser_name ;;
    golden) mtdname=$mtdgold_name ;;
    *)      die "Invalid bank \"$option_bank\"" ;;
esac

dev=$(find_mtd_device $mtdname) || die "Unable to locate mtd device"

#
# md5sum on specified bank. --size option is required
#

if [ "$option_md5" ]; then
    [ -n "$option_size" ]       || die "Missing --size"
    [ -z "$option_compare" ]    || die "--compare and --md5 are incompatible"

    md5sum_${interface}_part $dev $option_size

    exit $?
fi

#
# Sanitize image file
#

[ -n "$option_image" ] || usage 1
[ -f "$option_image" ] || die "Image file $1 does not exist"
validate_image_fname "${fpga_name}" "${option_bank}" "${option_image}"

#
# Compare/Verify. Image file is required
#

if [ "$option_compare" ]; then
    run_cmd "Verifying $option_bank bank ($dev)"  "verify_${interface}_part $dev $option_image"

    exit $?
fi

#
# Write/Update image
#


# Note: Verify stage might be superflous if the kernel config
# has CONFIG_MTD_DATAFLASH_WRITE_VERIFY set

run_cmd "Erase  $option_bank bank ($dev)"  "erase_${interface}_part  $dev $option_image"
run_cmd "Write  $option_bank bank ($dev)"  "write_${interface}_part  $dev $option_image"
run_cmd "Verify $option_bank bank ($dev)"  "verify_${interface}_part $dev $option_image"

