#!/bin/bash

source /ciena/scripts/utils.sh
source /ciena/scripts/paramApi.sh
source /ciena/scripts/saos_utils.sh

debugswitch="/mnt/sysfs/system/skip-boardsupport-upgrade"
fpgapath="/ciena/fpga"
manifest="$fpgapath/manifest"
blacklist="$fpgapath/blacklist"
bsf_name="Board Support"
upgrade="/ciena/scripts/bsf_upgrade.sh"

reconfigure_on_exit="no"

# When dryrun is set the upgrading of the images are skipped.
#dryrun=0


upgrade_log()
{
    local level=$1
    shift
    /ciena/bin/evt_log $level "$*"
    echo "$*"
}

reconfigure_required()
{
    reconfigure_on_exit='yesplease'
}

reconfigure_abort()
{
    reconfigure_on_exit="no"
    exit $1
}

upgrade_abort()
{
    reconfigure_on_exit="no"
    echo "Skipping ${bsf_name} FPGA upgrade ($1)" && exit 1
}

updateResetReasonFromWarmToCold()
{
    local lastResetReason=$(read_u32_cookie $ParamType_LastResetReason)
    lastResetReason=$(printf "%d\n" $lastResetReason)
    local newResetReason=$lastResetReason

    case $lastResetReason in
        $ResetReason_SnmpWarmStart) newResetReason=$ResetReason_SnmpColdStart ;;
        $ResetReason_UpgradeWarmStart) newResetReason=$ResetReason_UpgradeColdStart ;;
        $ResetReason_CliWarmStart) newResetReason=$ResetReason_CliColdStart ;;
    esac

    if [ $lastResetReason != $newResetReason ] ; then
        write_cookie $ParamType_LastResetReason $newResetReason
    fi

}

exit_trap()
{
    if [ -z "${use_fpga}" ] && \
       [ "$reconfigure_on_exit" == "yesplease" ] ; then

        #
        # ack first boot, so we stay on this bank
        #

        if [ "$swbank" -ne $EXIT_NFS ]; then
            /ciena/scripts/bootack || upgrade_abort "Failed to ack boot"

            if spiboot_supported; then
                spiboot_bank_set_to_running
            fi
        fi

        #
        # reboot
        #

        [ -n  "$dryrun" ] && upgrade_abort "\$dryrun is set"

        updateResetReasonFromWarmToCold
        $upgrade --reconfigure

        upgrade_log NOTICE "Error: ${bsf_name} FPGA upgrade - failed to reboot"
    fi
}


prefered_fpga_image_info()
{
    local board=$1
    local bank=$2

    awk -v board="$board" -v bank="$bank" '
        BEGIN { found = 0 }

        /^#/ || /^$/ { next }

        1 {
            if ($1 == board && $2 == bank) {
                print "revision=" $3 " build=" $4 " image=" $5;
                found = 1;
                last
            }
        }

        END {
            if (! found) {
                print "unknown"
            }
        }
    ' $manifest || upgrade_abort "internal error"
}


golden_upgrade_required()
{
    # Return true if an upgrade is recomended
    # $1: board

    local board=$1

    [ -z "$board" ] && return 1

    awk -v board="$board" -v listfile="$blacklist" '
        BEGIN { md = 0;  rc = "Good" }

        /^[[:space:]]*#/ || !NF { next }

        # board <x> length <n> [#.*]
        /^board/ {
            md = 0

            if ($2 == board) {
                c = "/ciena/scripts/bsf_upgrade.sh ${opt_fpga} --bank golden --md5 --size " $4
                #print "Running \"" c "\" ..."
                c | getline md
                close( c );

                if (md !~ /^[[:xdigit:]]{32}$/) {
                    print "Warning: malformed md5 \"" md "\""
                }
            }

            next
        }

        /^.[[:space:]]*[[:xdigit:]]{32}[[:space:]]*#?/ {
            if (md && $1 == "-" && $2 == md) {
                rc = "update recomended (" NR ")"; last
            }

            next
        }

        1 { print "Warning: " listfile " line " NR ": malformed input" }

        END {
            print "'"${bsf_name}"' FPGA Golden Image - " rc
            exit rc ~ /^update/ ? 0 : 1
        }

    ' $blacklist
}


upgrade()
{
    local attempts
    local bank=$1
    local img=$2

    [ -n "$dryrun" ] && return $dryrun

    attempts=3
    while [ $attempts -gt 0 ] ; do

        # For systems that may take a long time to upgrade the FPGA
        # image, add to the guardian initialization timer to prevent
        # the system from rebooting while we are upgrading.
        /ciena/bin/guardian extend-init 400

        $upgrade ${opt_wbsize} ${opt_fpga} --bank $bank "$img" && break
        attempts=$((attempts-1))
    done

    if [ $attempts -le 0 ] ; then
        false # pass an error to the the caller
        return
    fi

    if board_has_fpga_preloader ; then
        attempts=3
        while [ $attempts -gt 0 ] ; do
            /ciena/scripts/update_fpga_preloader --bank $bank --no-prompt && break
            attempts=$((attempts-1))
        done
    fi

    if [ $attempts -le 0 ] ; then
        false # pass an error to the the caller
        return
    fi
}


upgrade_golden_image()
{
    golden_upgrade_required $1 || return 0

    local golden_image_info=$(prefered_fpga_image_info $1 golden)

    [ "$golden_image_info" == "unknown" ] && return

    for x in $golden_image_info; do
        eval "local golden_$x"
    done

    [ -n "$golden_image" ] || return

    local img="$fpgapath/$golden_image"

    [ -r "$img" ] || return

    upgrade "golden" "$img"

    if [ "$?" -ne  0 ]; then
        upgrade_log EMERG "Error: ${bsf_name,,*} FPGA - golden upgrade failed" \
                          "image \"$img\""

        reconfigure_abort 3
    fi

    upgrade_log ALERT "${bsf_name} FPGA - golden upgraded to" \
                       "revision $golden_revision" \
                       "build $golden_build"

    reconfigure_required

    return 0
}


#--------------------------------------------------------------------

usage()
{
    echo
    echo "Usage:"
    echo -e "\t $0    [options]"
    echo
    echo "Upgrade or verify a board support FPGA image"
    echo
    echo "Options:"
    echo -e "\t-f,--fpga=<name>\tOverride the FPGA name."
    echo -e "\t-s,--size=<size>\tCompare size before declaring images equal."
    echo -e "\t-w,--wbsize=<size>\tWrite block size (default 264)."
    echo

    exit $1
}

#-------------------------------------------------------------------------

trap exit_trap EXIT

[ -e $debugswitch ] && upgrade_abort "$debugswitch exists"

get_running_bank
swbank=$?

opts=`getopt --name=$0 \
             --options="f:s:w:" \
             --longoptions="fpga:,size:,wbsize:" \
             -- "$@"`

[ $? -eq 0 ] || usage 1

eval set -- "$opts"

#
# allow the user to override the FPGA name and sizes
#
declare -l use_fpga opt_fpga
declare opt_size check_size
while [ $# -ne 0 ]; do
    case "$1" in
        -f | --fpga)
            bsf_name=${2^*}
            export use_fpga=$2
            export opt_fpga="--fpga ${use_fpga}"
            shift 2
            ;;
        -s | --size)
            # opt_size must be larger or equal to the /ciena/fpga/blacklist size
            export opt_size="--size $2"
            shift 2
            ;;
        -w | --wbsize)
            # use large write blocks to ease off the write overhead
            export opt_wbsize="--wbsize $2"
            shift 2
            ;;
        --)
            shift
            ;;
        *)
            echo "Unhandled option $1" && exit 10
            shift
            ;;
    esac
done

[ -z "$dryrun" ] && [ -z "$use_fpga" ] && [ "$swbank" -eq $EXIT_NFS ] && \
    upgrade_abort "Running from nfs"

#
# Let the following variables be coaxed into lowercase
#

declare -l revision build fpga_revision fpga_build fpga_bank

#
# Check the board
#

board=${use_fpga:-$(get_board_name)}

[ "$board" == "unknown" ] && upgrade_abort "unknown board"

#
# Gather Information regarding filesystem FPGA image
#

[ -f $manifest ] || upgrade_abort "missing FPGA image manifest ($manifest)"

image_info=$(prefered_fpga_image_info $board user)

[ "$image_info" == "unknown" ] && upgrade_abort "$board board not supported"

eval $image_info

[ -n "$revision" ] || upgrade_abort "internal error on line $LINENO"
[ -n "$build" ]    || upgrade_abort "internal error on line $LINENO"
[ -n "$image" ]    || upgrade_abort "internal error on line $LINENO"

#
# Gather Running image information
#

fpga_info=$(running_fpga_image_version $board)

for x in $fpga_info; do
    eval "fpga_$x"
done

[ -n "$fpga_build" ]    || fpga_build="0x0000"
[ -n "$fpga_revision" ] || upgrade_abort "internal error on line $LINENO"
[ -n "$fpga_bank" ]     || upgrade_abort "internal error on line $LINENO"


#
# On the 5150 we do not want to upgrade alpha boards,
# and we shall force the bank to user.
#

if [ "$board" == "5150" ]; then
 	[[ "$fpga_revision" -lt 0x0200 ]]  && upgrade_abort "vinmar revision 0x01"
	[ "$fpga_bank" == "unknown" ] && fpga_bank="user"
fi

#
# If the versions and bank match, there is nothing to do
#

if [ "$build" == "$fpga_build" ]               \
        && [ "$revision" == "$fpga_revision" ] ; then

    if [ "$fpga_bank" == "user" ] ; then
        echo "${bsf_name} FPGA is up-to-date (revision $revision build $fpga_build)"
        # make sure we don't require a gold upgrade before exiting
        golden_upgrade_required $board || exit 0
    elif [ "$fpga_bank" == "unknown" ] ; then
        # When the versions match but the FPGA cannot say from which
        # bank it booted, things are probably fine. If the caller
        # supplied a size override, use it to prevent a check that may
        # take minutes.
        check_size="${opt_size}"
    fi

fi

#
# Compare the image to the isf one, this should be quick when they differ.
# Abort if the images are already identical and the golden image
# is not blacklisted - something fishy is afoot.
#

user_image=$fpgapath/$image

[ -r "$user_image" ] || upgrade_abort "Missing fpga image $user_image"

$upgrade ${check_size} ${opt_fpga} --bank user --compare "$user_image"

if [ $? -eq 0 ]; then

    #
    # First, make sure gold is not on the black list
    #

    upgrade_golden_image $board && exit

    #
    # Then complain
    #

    upgrade_log CRIT \
        "Error: board support FPGA ($board) - image mismatch."

    upgrade_log CRIT \
        "Error: running rev $fpga_revision build $fpga_build from $fpga_bank." \
        "Expecting rev $revision build $build ($user_image)"

    reconfigure_abort 1
fi

#
# Update the fpga
#

upgrade "user" $fpgapath/$image

if [ "$?" -ne  0 ]; then
    upgrade_log CRIT "Error: ${bsf_name} FPGA - user upgrade failed" \
                      "image \"$fpgapath/$image\""
    reconfigure_abort 2
fi

upgrade_log ALERT  "${bsf_name} FPGA - upgraded to" \
                   "revision $revision build $build" \
                   "from rev $fpga_revision build $fpga_build"

reconfigure_required

#
# Finally, check golden bank
#

upgrade_golden_image $board


## The end
