#-------------------------------------------------------------------------------
source /ciena/scripts/utils.sh

#-------------------------------------------------------------------------------
#
# usage: vm_gaps [ -a ] [ <pid> | <program> ]
#
# options:
#   -a: display gaps in ascending address order
#
# This function identifies the "gaps" of unassigned virtual memory
# inside a given process memory map. The gaps are displayed in
# ascending size order or, with the "-a" option, in the order in which
# they appear in the memory map.
#
# If this function is invoked with no pid or program name, it collects
# the gaps of the main leos server.
#
function vm_gaps()
{
    local vm_start vm_end vm_addr pg_offset
    local pexe pmap vmap proc_list proc_id
    local ranges rrr r_start r_end r_tail
    local gap num_gaps total_gaps
    local order=/usr/bin/sort
    local elf_mach elf_class
    local out_hdrw=10 out_fmtw=8

    if [ "$1" = "-a" ]; then
       order=/bin/cat
       shift
    fi

    # figure out which process to query, default to the leos server
    case "$#" in
       0) proc_list="$(pgrep -f '/mnt/apps/bin/leos.*-s.*' 2> /dev/null)";;
       1) proc_list="$1 $(pgrep $1 2> /dev/null)";;
       *) proc_list="";;
    esac

    if [ "${proc_list}" = "" -o "$1" = "--help" ]; then
       echo "usage: vm_gaps [ -a ] [ <pid> | <program> ]" 1>&2; return 1
    fi

    # get the process memory map
    for proc_id in ${proc_list}; do
       pexe="/proc/${proc_id}/exe"
       vmap="/proc/${proc_id}/maps"
       if [ -f "${vmap}" -a -r "${vmap}" ]; then
          if [ "${pmap}" != "" ]; then
             echo "vm_gaps: more than one process found, use pid" 1>&2
             return 2
          fi
          pmap=${vmap}
       fi
    done

    if [ -z "${pmap}" ]; then
       echo "vm_gaps: no process map found for $1" 1>&2
       return 2
    fi

    # compute the proper VM range
    vm_start=$(cat /proc/sys/vm/mmap_min_addr)

    # the top of the VM space is a HW-dependent quirk
    elf_mach=$(hexdump -s18 -n2 -e '/2 "%d\n"' ${pexe})
    case "${elf_mach}" in
       8)  # mips (can be 32 or 64-bit)
           elf_class=$(hexdump -s4 -n1 -e '/1 "%d\n"' ${pexe})
           case "${elf_class}" in
              1)  vm_end=0x80000000;;         # mips 32 bits kuseg
              2)  vm_end=0x100000000000       # mips 64 bits xkuseg
                  out_hdrw=13;out_fmtw=11;;
              *)  echo "vm_gaps: unknown EI_CLASS ${elf_class}" 1>&2; return 3;;
           esac;;
       40) # arm (32-bit only)
           let pg_offset=$(zcat /proc/config.gz | grep CONFIG_PAGE_OFFSET 2>/dev/null | cut -f2 -d=)
           let vm_end=${pg_offset}-0x1000000  # arm 32 bits
           vm_end=$(printf "0x%08x" ${vm_end});;
       *)  echo "vm_gaps: unknown E_MACHINE ${elf_mach}" 1>&2; return 4;;
    esac

    # awk would be a natural choice for this work, but busybox awk
    # does not handle string to 32-bit unsigned integers easily; use
    # bash arrays instead
    ranges=($(cat ${pmap} | cut -f1 -d\ ))
    ranges+=(${vm_end/0x}-${vm_end/0x})

    # now use vm_end as an integer
    let vm_end=${vm_end}

    printf "%*s @ %*s\n" ${out_hdrw} "gap" ${out_hdrw} "address"
    printf "%*s   %*s\n" ${out_hdrw} "---" ${out_hdrw} "-------"
    let r_end=${vm_start}
    (
      for rrr in ${ranges[@]}; do
         let r_start=0x${rrr/-*}
         if [ ${r_start} -gt ${vm_end} ]; then
            let r_start=${vm_end}
         fi
         let r_tail=0x${rrr/*-}
         if [ ${r_tail} -gt ${vm_end} ]; then
            let r_tail=${vm_end}
         fi
         if [ ${r_start} -ge ${r_end} ]; then
            if [ ${r_start} -gt ${r_end} ]; then
               let gap=${r_start}-${r_end}
               let total_gaps+=${gap}
               let num_gaps+=1
               printf "0x%0*x @ 0x%0*x\n" ${out_fmtw} ${gap} ${out_fmtw} ${r_end}
            fi
            let r_end=${r_tail}
         fi
      done
      printf "0x%0*x bytes in %d gaps"  ${out_fmtw} ${total_gaps} ${num_gaps}
      printf " (pid: %d, %s)\n" ${proc_id} $(cat /proc/${proc_id}/cmdline | xargs -n1 echo)
    ) | ${order}
}

#-------------------------------------------------------------------------------
#
# usage: proc_heap [ <pid> | <program> ]
#
# This function uses m2 (the malloc tracer tool) to collect the total
# heap usage for a given process (or for all processes if invoked with
# no parameter).
#
function proc_heap()
{
    local m2_raw m2_pids pid_list proc_list a_proc a_pid
    local -A m2_totals m2_names

    proc_list="${@}"
    for a_proc in ${proc_list}; do
       a_pid=$(pidof ${a_proc} 2>/dev/null)
       if [ "0" = "$?" ]; then
          pid_list+="${pid_list:+ }${a_pid}"
       else
          pid_list+="${pid_list:+ }${a_proc}"
       fi
    done

    m2_raw=($(/ciena/bin/m2 -b | sed -e 's/^ *//g' -e 's/\([a-zA-Z]\) \([a-zA-Z]\)/\1_\2/g' |
            tr -s " " | grep ^[0-9] | cut -f1-3,6 -d" "))

    while [ "${#m2_raw[@]}" -ge 4 ]; do
       let m2_totals[${m2_raw[1]}]+=${m2_raw[3]}
       if [ "${m2_raw[1]}" = "${m2_raw[0]}" ]; then
          m2_names[${m2_raw[1]}]="${m2_raw[2]}"
          m2_pids+="${m2_pids:+ }${m2_raw[1]}"
       fi
       m2_raw=(${m2_raw[@]:4})
    done

    pid_list="${pid_list:-${m2_pids}}"

    for a_pid in ${pid_list}; do
       if [ "${m2_names[${a_pid}]}" != "" ]; then
          printf "%15s(%d): %u bytes\n" ${m2_names[${a_pid}]} ${a_pid} ${m2_totals[${a_pid}]}
       fi
    done
}
