# This file contains platform specific functions some of which require utils.sh
# to be sourced before this file. To avoid a circular reference, users should
# source utils.sh instead of this file to be sure that all references will be
# resolved.

source /ciena/scripts/paramApi.sh
source /ciena/scripts/saos_utils.sh
source /ciena/scripts/fault.sh

#
# PCI ID definitions, to avoid repeating them elsewhere
#
METROLITE_B270_PCI_ID="14E4:B270"
METROLITE_B271_PCI_ID="14E4:B271"
CIENA_NORBORN_PCI_ID="16FC:031A"
AV_RIFFELHORN_PCI_ID="16FC:031B"
ARRV_CODECHIP_PCI_ID="16FC:031C"

PLX8604REG="/ciena/bin/plx8604reg"
PLX8606REG="/ciena/bin/plx8606reg"
PLX8714REG="/ciena/bin/plx8714reg"

# -----------------------------------------------------------------------------
get_board_id ()
{
    cat /proc/device-tree/board_id.?
}

# -----------------------------------------------------------------------------
get_board_name()
{
    cat /proc/device-tree/model
}

# -----------------------------------------------------------------------------
get_board_name_pretty()
{
    get_board_name
}

# -----------------------------------------------------------------------------
get_board_rev()
{               
    /ciena/bin/sarosreg SAROS_REV | awk '{print $3}'               
}

# -----------------------------------------------------------------------------
init_3926_plx8xxx()
{
if [ "3926" = "$(get_board_name)" ] && [ "4" = "$(get_board_id)" ] && \
	[[ ( "$(get_board_rev)" -eq "7" ) || ( "$(get_board_rev)" -eq "8" ) ]]; then
	PLX8XXX_FRU_LINK_REG="PLX87XX_LINK_STATUS_AND_CONTROL_1"
	PLX8XXX_FRU_SLOT_REG="PLX87XX_SLOT_STATUS_AND_CONTROL_1"
	PLX8XXX_MEM_BASE_AND_LIMIT_1="PLX87XX_MEM_BASE_AND_LIMIT_1"
	PLX8XXX_PREFETCH_MEM_UPPER_BASE_ADDR_1="PLX87XX_PREFETCH_MEM_UPPER_BASE_ADDR_1"
	PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1="PLX87XX_PREFETCH_MEM_BASE_AND_LIMIT_1"
	PLX8XXX_PREFETCH_MEM_UPPER_LIMIT_ADDR_1="PLX87XX_PREFETCH_MEM_UPPER_LIMIT_ADDR_1"
	PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1="PLX87XX_PREFETCH_MEM_BASE_AND_LIMIT_1"
	PLX8XXX_PCI_COMMAND_STATUS_1="PLX87XX_PCI_COMMAND_STATUS_1"
	PLX8XXXREG="/ciena/bin/plx8714reg"
elif [ "3926" = "$(get_board_name)" ] && [ "4" = "$(get_board_id)" ]; then
	PLX8XXX_FRU_LINK_REG="PLX86XX_LINK_STATUS_AND_CONTROL_1"
	PLX8XXX_FRU_SLOT_REG="PLX86XX_SLOT_STATUS_AND_CONTROL_1"
	PLX8XXX_MEM_BASE_AND_LIMIT_1="PLX86XX_MEM_BASE_AND_LIMIT_1"
	PLX8XXX_PREFETCH_MEM_UPPER_BASE_ADDR_1="PLX86XX_PREFETCH_MEM_UPPER_BASE_ADDR_1"
	PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1="PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT_1"
	PLX8XXX_PREFETCH_MEM_UPPER_LIMIT_ADDR_1="PLX86XX_PREFETCH_MEM_UPPER_LIMIT_ADDR_1"
	PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1="PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT_1"
	PLX8XXX_PCI_COMMAND_STATUS_1="PLX86XX_PCI_COMMAND_STATUS_1"
	PLX8XXXREG="/ciena/bin/plx8604reg"
fi
}

# -----------------------------------------------------------------------------
get_local_if()
{
    echo 'eth0'
}

# -----------------------------------------------------------------------------
board_has_rtc()
{
    # The hardware RTC should work at 3928 rev3 and 3926 rev2.  Trying to use
    # it in earlier variants should not fail, but the clock will not keep
    # time.
    true
}

# -----------------------------------------------------------------------------
board_has_push_button_interrupt()
{
    true
}

# -----------------------------------------------------------------------------
mount_log_partition()
{
    fg_run "mount logs via fstab" /bin/mount /mnt/log
}

# -----------------------------------------------------------------------------
EXIT_BANKA=1
EXIT_BANKB=2
EXIT_NFS=3
EXIT_UNKNOWN=25
get_running_bank()
{
    case "$(kernel_arg root)" in
        /dev/nfs)       return $EXIT_NFS     ;;
        /dev/mmcblk0p5) return $EXIT_BANKA   ;;
        /dev/mmcblk0p6) return $EXIT_BANKB   ;;
        *)              return $EXIT_UNKNOWN ;;
    esac
}

# -----------------------------------------------------------------------------
get_eeprom_file()
{
    echo /mnt/boot/eeprom0.dat
}

# ------------------------------------------------------------------------------
get_data_fpga_file()
{
    true
}

# -----------------------------------------------------------------------------
show_fpga_reg ()
{
    # FIX THIS - should this be common?
    local decoder="$1"
    local regname="$2"
    local regex=$regname'\s+(0[xX].*)'
    local x=$($decoder $regname)
    if [[ $x =~ $regex ]]
    then
        echo ${BASH_REMATCH[0]}
    fi
}

# ------------------------------------------------------------------------------
show_fpga_version()
{
    case $(get_board_name) in
        "3922" | \
        "3924" | \
        "3926" | \
        "3928" )
            show_fpga_reg wfldreg WFLD_TOP_MJR
            show_fpga_reg wfldreg WFLD_TOP_MNR
            show_fpga_reg wfldreg WFLD_TOP_BLD
            ;;
    esac
}

# ------------------------------------------------------------------------------
load_fpga_required()
{
    false
}

# ------------------------------------------------------------------------------
pwe_supported()
{
    false
}

# ------------------------------------------------------------------------------
setup_pwe()
{
    return
}

#--------------------------------------------------------------------
wait_for_file_content()
{
    # wait for file $1 to exist, contain $2, optionally for $3 seconds
    # if $2 is "--", file existence is sufficient
    local now_time end_time
    local SYSLOG_INFO="logger -s -p USER.INFO  -t $FUNCNAME"

    if [ "${3}" != "" ]; then
        end_time=$(( $(/bin/date +%s) + ${3} ))
    fi

    while [ ! -r "$(echo ${1})" ] || \
          ([ "--" != "${2}" ] && \
           [ "$(/bin/cat $(echo ${1}) 2>/dev/null)" != "${2}" ]); do
        now_time=$(/bin/date +%s)
        if [ ${end_time:-${now_time}} -lt ${now_time} ]; then
            if [ ! -r ${1} ]; then
                $SYSLOG_INFO "Cannot read ${1} after ${3} seconds."
            else
                $SYSLOG_INFO "${1} $(cat $(echo ${1}) 2>/dev/null)" \
                    " != ${2} after ${3} seconds."
            fi
            return -1
        fi
        sleep 0.1
    done

    return 0
}

# -----------------------------------------------------------------------------
regdecode_field()
{
    local command="${1}" reg="${2}" field="${3}" value="${4}"
    local value_now

    value_now=$(${command} ${reg}.${field} 2>/dev/null)
    value_now=${value_now##* ${field} }

    echo "${value_now}"
}

# -----------------------------------------------------------------------------
wait_for_regdecode_field()
{
    local command="${1}" reg="${2}" field="${3}" value="${4}"
    local value_now
    local now_time end_time
    local SYSLOG_ERR="logger -s -p USER.ERR  -t $FUNCNAME"

    end_time=$(( $(date +%s) + ${5} ))

    [ "--" = "${field}" ] && field=""

    while /bin/true; do

        now_time=$(date +%s)
        value_now=$(regdecode_field "${command}" "${reg}" "${field}")

        if [ $(( ${value} )) -eq $(( ${value_now} )) ]; then
            return 0
        fi

        if [ ${end_time} -lt ${now_time} ]; then
            $SYSLOG_ERR "${reg}.${field} still ${value_now}, not ${value}"
            return -1
        fi

        /bin/sleep 0.1

    done
}

# --- eredan only --------------------------------------------------------------
eredan_power_down()
{
    local SYSLOG_CRIT="logger -s -p USER.CRIT  -t $FUNCNAME"

    # Turn off both power supplies.
    /ciena/bin/sarosreg SAROS_SBD_MISC_CTL.PSL_OFF 1
    /ciena/bin/sarosreg SAROS_SBD_MISC_CTL.PSR_OFF 1

    sleep 5

    # Still here? Then try a board reset.
    ${SYSLOG_CRIT} "Still running after power down - resetting the board."
    sleep 1
    /ciena/bin/sarosreg SAROS_BRD_RESET 0xa5a5
}

# --- eredan only --------------------------------------------------------------
eredan_reset_reason()
{
    local _reason

    case $(cat /sys/ciena/eredan/reset_reason) in
        0x0001) _reason="EXTERNAL_POR";;
        0x0002) _reason="INTERNAL_POR";;
        0x0004) _reason="PMU_SYS";;
        0x0008) _reason="PS_ONLY";;
        0x0010) _reason="EXTERNAL_SYS";;
        0x0020) _reason="SOFTWARE";;
        0x0040) _reason="DEBUGGER";;
        *)      _reason="unknown";;
    esac

    echo "${_reason}"
}

# --- eredan only --------------------------------------------------------------
fru_model_type()
{
    local _fru_type

    case $(/ciena/bin/idp read module-eeprom MP 2>/dev/null) in
        170-012*) _fru_type="NFV";;
        170-013*) _fru_type="TDM";;
        170-017*) _fru_type="OC48_TDM";;
        *)        _fru_type="unknown";;
    esac

    echo "${_fru_type}"
}

# --- eredan only --------------------------------------------------------------
extend_por_to_fru()
{
    local _board="${1}"
    local _sbd_present _sbd_powered

    # 3926 power-on resets do not extend to the FRU. This is fine for
    # I2C devices, but tricky when PCI is involved.

    if [ "3926" != "${_board}" ] || [ "4" != "$(get_board_id)" ]; then
        # Not dealing with a PCI FRU host, bail out.
        return 0
    fi

    if [ "EXTERNAL_POR" != "$(eredan_reset_reason)" ]; then
        # This was not a power-on reset, bail out.
        return 0
    fi

    _sbd_present=$(regdecode_field sarosreg SAROS_SBD_INTR_STS SBD_PRSNT)
    _sbd_powered=$(regdecode_field sarosreg SAROS_SBD_INTR_STS SBD_PWR_GOOD)

    if [ 1 -eq $(( ${_sbd_present} )) ] && \
       [ 1 -eq $(( ${_sbd_powered} )) ] && \
       [ "OC48_TDM" = "$(fru_model_type)" ]; then
        # There is an FRU, it is powered up, and it lives on the PCI
        # bus. Run the FRU POR now.
        /ciena/scripts/rc.fru_power restart
    fi
}

# --- eredan 3926 only ---------------------------------------------------------
eredan_fru_poweroff()
{
    local sarncmd="/ciena/bin/sarn-i2creg" _msg="up"

    [ "1" == "${1}" ] && _msg="down"

    fg_run "powering ${_msg} FRU" \
        /ciena/bin/sarosreg SAROS_SBD_MISC_CTL.sbd_pwr_off "${1}"

    sleep 0.5
}

# --- eredan 3926 only ---------------------------------------------------------
init_3926_plx8604()
{
    # pre-program the plx8604 config space to assign bus 0000:04 to
    # the main board metrolite, and push the FRU devices into the
    # 0000:[08-0f] bus range
    #
    # plx8604 upstream port: bus 0000:01, leading to buses 0000:[02-0f]
    ${PLX8604REG} PLX86XX_BUS_NUMBER.PRIMARY_BUS_NUMBER 0x1
    ${PLX8604REG} PLX86XX_BUS_NUMBER.SECONDARY_BUS_NUMBER 0x2
    ${PLX8604REG} PLX86XX_BUS_NUMBER.SUBORDINATE_BUS_NUMBER 0x0f
    # plx8604 downstream port 1: leads to buses 0000:[08-0f] (FRU)
    ${PLX8604REG} PLX86XX_BUS_NUMBER_1.PRIMARY_BUS_NUMBER 0x2
    ${PLX8604REG} PLX86XX_BUS_NUMBER_1.SECONDARY_BUS_NUMBER 0x8
    ${PLX8604REG} PLX86XX_BUS_NUMBER_1.SUBORDINATE_BUS_NUMBER 0xf
    # plx8604 downstream port 4: leads to bus 0000:04 (on-board metrolite)
    ${PLX8604REG} PLX86XX_BUS_NUMBER_4.PRIMARY_BUS_NUMBER 0x2
    ${PLX8604REG} PLX86XX_BUS_NUMBER_4.SECONDARY_BUS_NUMBER 0x4
    ${PLX8604REG} PLX86XX_BUS_NUMBER_4.SUBORDINATE_BUS_NUMBER 0x4
    # plx8604 downstream port 5: leads to bus 0000:05
    ${PLX8604REG} PLX86XX_BUS_NUMBER_5.PRIMARY_BUS_NUMBER 0x2
    ${PLX8604REG} PLX86XX_BUS_NUMBER_5.SECONDARY_BUS_NUMBER 0x5
    ${PLX8604REG} PLX86XX_BUS_NUMBER_5.SUBORDINATE_BUS_NUMBER 0x5

    # adjust the top bus number in the root complex for the [00-0f] range
    #
    /usr/sbin/setpci -s 00:00.0 SUBORDINATE_BUS.B=0xf

    # open bridge windows wide enough to accommodate the FRU devices
    #
    # xilinx-nwl host bridge: 0xe0800000-0xefffffff (32-bit)
    #                         0x600000000-0x6ffffffff (prefetchable, 64-bit)
    /usr/sbin/setpci -s 00:00.0 MEMORY_BASE.W=0xe080
    /usr/sbin/setpci -s 00:00.0 MEMORY_LIMIT.W=0xeff0
    /usr/sbin/setpci -s 00:00.0 PREF_BASE_UPPER32.L=0x6
    /usr/sbin/setpci -s 00:00.0 PREF_LIMIT_UPPER32.L=0x6
    /usr/sbin/setpci -s 00:00.0 PREF_MEMORY_BASE.W=0x1
    /usr/sbin/setpci -s 00:00.0 PREF_MEMORY_LIMIT.W=0xfff1
    #
    # plx8604 upstream port: 0xe1000000-0xefffffff (32-bit)
    #                        0x600000000-0x6ffffffff (prefetchable, 64-bit)
    ${PLX8604REG} PLX86XX_MEM_BASE_AND_LIMIT.MEM_BAR[31:20] 0xe10
    ${PLX8604REG} PLX86XX_MEM_BASE_AND_LIMIT.MEM_LIMIT[31:20] 0xeff
    ${PLX8604REG} PLX86XX_PREFETCH_MEM_UPPER_BASE_ADDR.PBUP[63:32] 0x6
    ${PLX8604REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PMEM_BAR[31:20] 0x00
    ${PLX8604REG} PLX86XX_PREFETCH_MEM_UPPER_LIMIT_ADDR.PLIMUP[63:32] 0x6
    ${PLX8604REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PMEM_LIMIT[31:20] 0xfff
    ${PLX8604REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PREFETCH_MEM_BASE_CAP 0x1
    ${PLX8604REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PREFETCH_MEM_LIMIT_CAP 0x1
    #
    # the other windows can take care of themselves whenever the PCI
    # bus is rescanned
    #
    # enable hotplug and surprise removals on the FRU port
    #
    ${PLX8604REG} PLX86XX_SLOT_CAP_1.HOT_PLUG_SURPRISE 0x1
    ${PLX8604REG} PLX86XX_SLOT_CAP_1.HOT_PLUG_CAPABLE 0x1
}

# --- eredan 3926-903 rev7 and 3926-904 rev8 only ------------------------------
init_3926_plx8714()
{
    # pre-program the plx8714 config space to assign bus 0000:04 to
    # the main board metrolite, and push the FRU devices into the
    # 0000:[08-0f] bus range
    #
    # plx8714 upstream port: bus 0000:01, leading to buses 0000:[02-0f]
    ${PLX8714REG} PLX87XX_BUS_NUMBER.PRIMARY_BUS_NUMBER 0x1
    ${PLX8714REG} PLX87XX_BUS_NUMBER.SECONDARY_BUS_NUMBER 0x2
    ${PLX8714REG} PLX87XX_BUS_NUMBER.SUBORDINATE_BUS_NUMBER 0x0f
    # plx8714 downstream port 1: leads to buses 0000:[08-0f] (FRU)
    ${PLX8714REG} PLX87XX_BUS_NUMBER_1.PRIMARY_BUS_NUMBER 0x2
    ${PLX8714REG} PLX87XX_BUS_NUMBER_1.SECONDARY_BUS_NUMBER 0x8
    ${PLX8714REG} PLX87XX_BUS_NUMBER_1.SUBORDINATE_BUS_NUMBER 0xf
    # plx8714 downstream port 2: leads to bus 0000:02
    ${PLX8714REG} PLX87XX_BUS_NUMBER_2.PRIMARY_BUS_NUMBER 0x2
    ${PLX8714REG} PLX87XX_BUS_NUMBER_2.SECONDARY_BUS_NUMBER 0x5
    ${PLX8714REG} PLX87XX_BUS_NUMBER_2.SUBORDINATE_BUS_NUMBER 0x5
    # plx8714 downstream port 3: leads to bus 0000:03
    ${PLX8714REG} PLX87XX_BUS_NUMBER_3.PRIMARY_BUS_NUMBER 0x2
    ${PLX8714REG} PLX87XX_BUS_NUMBER_3.SECONDARY_BUS_NUMBER 0x3
    ${PLX8714REG} PLX87XX_BUS_NUMBER_3.SUBORDINATE_BUS_NUMBER 0x3
    # plx8714 downstream port 4: leads to bus 0000:04 (on-board metrolite)
    ${PLX8714REG} PLX87XX_BUS_NUMBER_4.PRIMARY_BUS_NUMBER 0x2
    ${PLX8714REG} PLX87XX_BUS_NUMBER_4.SECONDARY_BUS_NUMBER 0x4
    ${PLX8714REG} PLX87XX_BUS_NUMBER_4.SUBORDINATE_BUS_NUMBER 0x4

    # adjust the top bus number in the root complex for the [00-0f] range
    #
    /usr/sbin/setpci -s 00:00.0 SUBORDINATE_BUS.B=0xf

    # open bridge windows wide enough to accommodate the FRU devices
    #
    # xilinx-nwl host bridge: 0xe0400000-0xefffffff (32-bit)
    #                         0x600000000-0x6ffffffff (prefetchable, 64-bit)
    /usr/sbin/setpci -s 00:00.0 MEMORY_BASE.W=0xe040
    /usr/sbin/setpci -s 00:00.0 MEMORY_LIMIT.W=0xeff0
    /usr/sbin/setpci -s 00:00.0 PREF_BASE_UPPER32.L=0x6
    /usr/sbin/setpci -s 00:00.0 PREF_LIMIT_UPPER32.L=0x6
    /usr/sbin/setpci -s 00:00.0 PREF_MEMORY_BASE.W=0x1
    /usr/sbin/setpci -s 00:00.0 PREF_MEMORY_LIMIT.W=0xfff1
    #
    # plx8714 upstream port: 0xe1000000-0xefffffff (32-bit)
    #                        0x600000000-0x6ffffffff (prefetchable, 64-bit)
    ${PLX8714REG} PLX87XX_MEM_BASE_AND_LIMIT.MEM_BAR[31:20] 0xe10
    ${PLX8714REG} PLX87XX_MEM_BASE_AND_LIMIT.MEM_LIMIT[31:20] 0xeff
    ${PLX8714REG} PLX87XX_PREFETCH_MEM_UPPER_BASE_ADDR.PBUP[63:32] 0x6
    ${PLX8714REG} PLX87XX_PREFETCH_MEM_BASE_AND_LIMIT.PMEM_BAR[31:20] 0x00
    ${PLX8714REG} PLX87XX_PREFETCH_MEM_UPPER_LIMIT_ADDR.PLIMUP[63:32] 0x6
    ${PLX8714REG} PLX87XX_PREFETCH_MEM_BASE_AND_LIMIT.PMEM_LIMIT[31:20] 0xfff
    ${PLX8714REG} PLX87XX_PREFETCH_MEM_BASE_AND_LIMIT.PREFETCH_MEM_BASE_CAP 0x1
    ${PLX8714REG} PLX87XX_PREFETCH_MEM_BASE_AND_LIMIT.PREFETCH_MEM_LIMIT_CAP 0x1
    #
    # the other windows can take care of themselves whenever the PCI
    # bus is rescanned
    #
    # enable hotplug and surprise removals on the FRU port
    #
    ${PLX8714REG} PLX87XX_SLOT_CAP_1.HOT_PLUG_SURPRISE 0x1
    ${PLX8714REG} PLX87XX_SLOT_CAP_1.HOT_PLUG_CAPABLE 0x1
}
# --- eredan 3926 only ---------------------------------------------------------
init_3926_combo_tdm_fru_bridges()
{
    # pre-program the plx8604/plx8714 port leading to the combo TDM FRU and
    # the FRU plx8606 bridge, so the FPGAs can fit inside the existing
    # 32-bit non-prefetchable and 64-bit prefetchable windows
    #
    # plx8604/plx8714 FRU port: 0xe1000000-0xefffffff (32-bit)
    #                   0x601400000-0x6027fffff (prefetchable, 64-bit)
    ${PLX8XXXREG} ${PLX8XXX_MEM_BASE_AND_LIMIT_1}.MEM_BAR[31:20] 0xe10
    ${PLX8XXXREG} ${PLX8XXX_MEM_BASE_AND_LIMIT_1}.MEM_LIMIT[31:20] 0xeff
    ${PLX8XXXREG} ${PLX8XXX_PREFETCH_MEM_UPPER_BASE_ADDR_1}.PBUP[63:32] 0x6
    ${PLX8XXXREG} ${PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1}.PMEM_BAR[31:20] 0x014
    ${PLX8XXXREG} ${PLX8XXX_PREFETCH_MEM_UPPER_LIMIT_ADDR_1}.PLIMUP[63:32] 0x6
    ${PLX8XXXREG} ${PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1}.PMEM_LIMIT[31:20] 0x027
    ${PLX8XXXREG} ${PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1}.PREFETCH_MEM_BASE_CAP 0x1
    ${PLX8XXXREG} ${PLX8XXX_PREFETCH_MEM_BASE_AND_LIMIT_1}.PREFETCH_MEM_LIMIT_CAP 0x1
    #
    # plx8606 upstream port: 0xe1800000-0xefffffff (32-bit)
    #                        0x601400000-0x6027fffff (prefetchable, 64-bit)
    ${PLX8606REG} PLX86XX_MEM_BASE_AND_LIMIT.MEM_BAR[31:20] 0xe18
    ${PLX8606REG} PLX86XX_MEM_BASE_AND_LIMIT.MEM_LIMIT[31:20] 0xeff
    ${PLX8606REG} PLX86XX_PREFETCH_MEM_UPPER_BASE_ADDR.PBUP[63:32] 0x6
    ${PLX8606REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PMEM_BAR[31:20] 0x014
    ${PLX8606REG} PLX86XX_PREFETCH_MEM_UPPER_LIMIT_ADDR.PLIMUP[63:32] 0x6
    ${PLX8606REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PMEM_LIMIT[31:20] 0x027
    ${PLX8606REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PREFETCH_MEM_BASE_CAP 0x1
    ${PLX8606REG} PLX86XX_PREFETCH_MEM_BASE_AND_LIMIT.PREFETCH_MEM_LIMIT_CAP 0x1
    #
    # enable memory accesses on our newly opened apertures
    #
    ${PLX8XXXREG} ${PLX8XXX_PCI_COMMAND_STATUS_1}.MEM_ACCESS_ENABLE 1
}

# --- eredan only --------------------------------------------------------------
configure_and_rescan_392x_bus()
{
    if [ "${board}" = "3926" ] && \
       [ "4" = "$(get_board_id)" ] && [[ ( "$(get_board_rev)" -eq "7" ) || ( "$(get_board_rev)" -eq "8" ) ]]; then
        init_3926_plx8714
    elif [ "${board}" = "3926" ] && \
       [ "4" = "$(get_board_id)" ]; then
        init_3926_plx8604
    fi

    # Common for both 8604 and 8714
    if [ "${board}" = "3926" ] && \
       [ "4" = "$(get_board_id)" ]; then
        # disable the link and hotplug interrupts, lest the FRU
        # sprouts in at unexpected moments
        ${PLX8XXXREG} ${PLX8XXX_FRU_SLOT_REG}.PRESENCE_DETECT_ENABLE 0
        ${PLX8XXXREG} ${PLX8XXX_FRU_SLOT_REG}.HOT_PLUG_ENABLE 0
        ${PLX8XXXREG} ${PLX8XXX_FRU_LINK_REG}.LINK_DISABLE 1
    fi

    echo 1 > /sys/bus/pci/rescan
}

# --- eredan only --------------------------------------------------------------
init_3928_rev1()
{
    local i2cset="/usr/sbin/i2cset"

    print_app_line_cr "3928 rev1 init hack"

    # Note that this sequence must be performed after the i2c mux driver
    # has been installed.

    # This script sets up the clocking on the Broadcom Metrolite chip.
    # This allows the device to show up on the PCI bus.
    #
    # The magic settings that follow are consistently NAK-ed. Just sweep
    # the failures under the carpet; this is a temporary work-around.
    #
    $i2cset -y 25 0x44 0x00 0x01 0x00 0x98 0x62 0x03 0x40 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x01 0x00 0x9c 0x60 0x37 0x75 0x31 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x01 0x00 0xa0 0x00 0x00 0x00 0x31 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x0c 0x34 0x48 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x10 0x02 0x00 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x14 0x00 0x00 0x00 0x60 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x0c 0x34 0x08 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x10 0x32 0x00 0x03 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x14 0x00 0x00 0x00 0x04 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x10 0x00 0x00 0x00 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x14 0x74 0x00 0x00 0x00 0x40 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x00 0x00 0x00 0x00 0x03 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x00 0x00 0x00 0x00 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x04 0x00 0x08 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x08 0x00 0x00 0x00 0x01 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x0c 0x2c 0x08 0x02 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x10 0x32 0x00 0x03 0x00 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x14 0x18 0x00 0xfc 0x08 i 2>/dev/null
    $i2cset -y 25 0x44 0x00 0x03 0x16 0x00 0x00 0x00 0x00 0x01 i 2>/dev/null

    sleep 1
}

# --- eredan only --------------------------------------------------------------
gate_3926_si5341_out7()
{
    local i2cset="/usr/sbin/i2cset"
    print_app_line_cr "3926 gate out7 on si5341"
    # set page to page 1
    $i2cset -y 26 0x76 0x01 0x01 2>/dev/null
    # set the register 0x12B to value 0x00 to disable output 7
    $i2cset -y 26 0x76 0x2B 0x00 2>/dev/null
    # set the page number back to 0x0
    $i2cset -y 26 0x76 0x01 0x00 2>/dev/null
    sleep 1
}


# --- eredan only --------------------------------------------------------------
init_392x_metrolite()
{
    local board="${1}"
    local board_rev="${2}"
    local clock_hack=0
    local -a pci_devs_before
    local -a pci_devs_after
    local SYSLOG_CRIT="logger -s -p USER.CRIT  -t $FUNCNAME"

    # Clock hack only applies to 3928 rev1
    if [ "${board}" = "3928" ] ; then
        if [[ "${board_rev}" -eq "1" ]] ; then
            clock_hack=1
        fi
    fi

    print_app_line_cr "${board} metrolite pci staging"

    # Reset the Metrolite before probing the PCI bus.

    echo 371 > /sys/class/gpio/export
    echo out > /sys/class/gpio/gpio371/direction

    # Hold the Metrolite PCIe interface in reset.
    echo 0 > /sys/class/gpio/gpio371/value

    # Hold the main Metrolite in reset,
    echo switch > /dev/device-reset/assert

    if [ "${clock_hack}" -eq "1" ] ; then
        echo dpll > /dev/device-reset/assert
    fi

    # There is no proper status to check here; just sleep.
    sleep 0.5

    # The root complex comes up disabled in the device tree.
    # Enable it now.
    echo -n c:eredan_pcie_enable > /dev/ciena_dts_overlay

    # Wait for the PCIe link status (link_status == 2 means no PCIe link).
    wait_for_file_content /sys/bus/platform/drivers/nwl-pcie/*/link_status 2 10
    pci_devs_before=( /sys/bus/pci/devices/* )

    if [ "${clock_hack}" -eq "1" ] ; then
        echo dpll > /dev/device-reset/deassert
    fi

    # Release the main Metrolite from reset.
    echo switch > /dev/device-reset/deassert
    # Idem.
    sleep 0.5

    if [ "${clock_hack}" -eq "1" ] ; then
        init_3928_rev1
    fi

    # 3926 and rev not "1", gate the silab output7 clock
    if [ "${board}" = "3926" ] ; then
        if [[ "${board_rev}" -gt "1" ]] ; then
            gate_3926_si5341_out7
        fi
    fi

    # Releasse the Metrolite PCIe interface from reset.
    echo 1 > /sys/class/gpio/gpio371/value
    # Link_status == 3 means the PCIe link is up.
    wait_for_file_content /sys/bus/platform/drivers/nwl-pcie/*/link_status 3 10

    echo 371 > /sys/class/gpio/unexport

    configure_and_rescan_392x_bus ${board} ${board_rev}

    # If the Metrolite was staged successfully, there will be more PCI
    # devices on the bus.
    pci_devs_after=( /sys/bus/pci/devices/* )
    if [ "${#pci_devs_before[@]}" -ge "${#pci_devs_after[@]}" ]; then
        ${SYSLOG_CRIT} "before pci rescan: ${pci_devs_before[@]}"
        ${SYSLOG_CRIT} "after pci rescan: ${pci_devs_after[@]}"
        ${SYSLOG_CRIT} "power cycling board"
        sleep 1
        eredan_power_down
    fi
}

# --- eredan only --------------------------------------------------------------
eredan_hardware_init()
{
    local board="$(get_board_name)"
    local board_rev="$(/ciena/bin/sarosreg SAROS_REV | awk '{print $3}')"

    case "$board" in
        "3922" | \
        "3924" | \
        "3926" | \
        "3928" )
            init_3926_plx8xxx
            extend_por_to_fru ${board}
            init_392x_metrolite ${board} ${board_rev}
            ;;
    esac
}

# ------------------------------------------------------------------------------
spiboot_supported()
{
    # Not really spiboot, but this prevents other code from trying to use
    # cookies to control the boot bank.
    true
}

# ------------------------------------------------------------------------------
cr3916()
{
    false
}

# ------------------------------------------------------------------------------
copy_kernel() {
    false
}

# ------------------------------------------------------------------------------
mount_kernel() {
    false
}

# ------------------------------------------------------------------------------
spiboot_ack()
{
    if [ ! -f "/sys/ciena/eredan/bootcount" ] ; then
        insmod_wrapper ciena_sysfs
    fi

    /ciena/bin/root bash -c "echo 0 > /sys/ciena/eredan/bootcount"
}

# ------------------------------------------------------------------------------
spiboot_bank_set_to_running()
{
    get_running_bank
    case $? in
        $EXIT_BANKA) bank=0 ;;
        $EXIT_BANKB) bank=1 ;;    
    esac
    spiboot_bank_set $bank
}

# ------------------------------------------------------------------------------
spiboot_bank_get()
{
    cat /mnt/param/bootbank.def
}

# ------------------------------------------------------------------------------
spiboot_bank_set()
{
    local bank=$1

    case "$bank" in 
        0) bank_alpha="a" ;;
        1) bank_alpha="b" ;;
    esac

    # acknowledge the current boot chain
    spiboot_ack

    # set the persistent bank
    /ciena/bin/root bash -c "echo $bank_alpha > /sys/ciena/eredan/bank"

    # Check the existing bank setting to avoid a program/erase cycle
    # on the flash if nothing is changing.
    [ $bank == $(spiboot_bank_get) ] && return

    /ciena/bin/root bash -c "echo -n $bank > /mnt/param/bootbank.def"
}

# ------------------------------------------------------------------------------
get_preferred_bank()
{
    local bank=$(spiboot_bank_get)

    case "$bank" in
        '0') return $EXIT_BANKA   ;;
        '1') return $EXIT_BANKB   ;;
        *)   return $EXIT_UNKNOWN ;;
    esac
}

#--------------------------------------------------------------------
xftp_partition_supported()
{
    false
}

# ------------------------------------------------------------------------------
load_fpga_setup()
{
    local board="$(get_board_name)"

    case "$board" in
        "3928" | "3926" | "3924" | "3922")
            # No clocks to set up in this case
            ;;
        *)
            echo "board ($board) not supported"
            ;;
    esac
}

#--------------------------------------------------------------------
mgmt_interface_reset()
{
    return
}

#--------------------------------------------------------------------
enable_fpga_flash_driver()
{
    return
}

#--------------------------------------------------------------------
board_has_fpga_preloader()
{
    false
}

# ------------------------------------------------------------------------------
dataplane_fpga_init()
{
    load_fpga_setup
}

# ------------------------------------------------------------------------------
allocate_vfio_dma_pool()
{
    #
    # Each eredan metrolite expects 8 MB worth of DMA memory. Every
    # board type requires this much, except for the 3926-902 which can
    # have two metrolites and hence requires 16 MB.
    #
    echo > /dev/vfiodma-8MB

    case $(get_board_name) in
        "3926")
            case $(get_board_id) in
                "4") echo > /dev/vfiodma-8MB;;
            esac
            ;;
    esac
}

# ------------------------------------------------------------------------------
launch_vfiouser_group_server()
{
    if [ -x "/ciena/bin/vfiouser_group_server" ] ; then
        bg_run "vfiouser iommu group server" /ciena/bin/vfiouser_group_server
    fi
}

# ------------------------------------------------------------------------------
register_vfio_pci_ids()
{
    local SYSLOG_WARN="logger -s -p USER.WARN  -t $FUNCNAME"
    local _pci_id

    for _pci_id in "${METROLITE_B270_PCI_ID}" \
                   "${METROLITE_B271_PCI_ID}" \
                   "${CIENA_NORBORN_PCI_ID}" \
                   "${AV_RIFFELHORN_PCI_ID}" \
                   "${ARRV_CODECHIP_PCI_ID}"
    do
        echo "${_pci_id/:/ }" > /sys/bus/pci/drivers/vfio-pci/new_id || \
            ${SYSLOG_WARN} "failed to register PCI ID ${_pci_id}"
    done
}

# ------------------------------------------------------------------------------
eredan_vfio_init()
{
    # This sounds scarier than it is. The eredan SMMU does not protect
    # the host and VMs against MSI cross-talk from another malicious
    # (or clueless) VM. But neither does UIO.
    echo Y > /sys/module/vfio_iommu_type1/parameters/allow_unsafe_interrupts

    insmod_wrapper vfiouser_dma
    allocate_vfio_dma_pool
    launch_vfiouser_group_server
    register_vfio_pci_ids
}

# ------------------------------------------------------------------------------
insmod_board_specific()
{
    insmod_wrapper ciena_sysfs
    insmod_wrapper ciena_board_id
    insmod_wrapper ciena_dts_overlay
    insmod_wrapper ciena_i2c_adapter
    insmod_wrapper i2c_fpga_mux
    insmod_wrapper i2c_plx86xx
    insmod_wrapper i2c_raw_chardev
    insmod_wrapper ciena_norborn_cic
    insmod_wrapper ciena_norborn_gpio
    insmod_wrapper ciena_saros_hp_cic
    insmod_wrapper ciena_saros_hp_gpio
    insmod_wrapper ciena_sarn_cic
    insmod_wrapper ciena_sarn_gpio
    insmod_wrapper reset-fpga
    insmod_wrapper reset-sysfs
    insmod_wrapper dwc3_of_simple
    insmod_wrapper dwc3
    eredan_vfio_init

    # replay the udev rules to ensure everyone is up to date
    /usr/bin/udevadm trigger
    /usr/bin/udevadm settle

    # and now do some useful HW bring-up
    eredan_hardware_init
}

# ------------------------------------------------------------------------------
insmod_uio_ciena_pdrv_shared_irq()
{
    return
}

# ------------------------------------------------------------------------------
prepare_cookies()
{
    local readonly cookie_dir="/mnt/sysfs/cookie"
    local readonly old_archive="${cookie_dir}/archives.bin"
    local readonly new_archive="/mnt/param/archives.bin"
    local link_dest

    # ensure that the cookie directory exists
    mkdir -p $cookie_dir

    # Verify that the cookie file exists - create it if necessary
    if [ -f "${new_archive}" ] ; then
        # cookie file exists - this is good
        :
    else
        # no cookie file
        if [ -f "${old_archive}" ] ; then
            # Use the cookie file from the incorrect location to seed
            # the correct one.
            fg_run "transferring cookies to /mnt/param" \
                mv "${old_archive}" "${new_archive}"
        else
            # No cookies anywhere, create an empty file
            fg_run "creating empty cookie file" touch ${new_archive}
        fi
    fi

    # Verify that the cookie symlink points to the correct location
    if [ -L "${old_archive}" ] ; then
        link_dest=$(readlink "${old_archive}")
        if [ "$link_dest" != "${new_archive}" ] ; then
            # Remove the incorrect symlink and fall through to the
            # next check to create the symlink
            fg_run "removing bad cookie link to $link_dest" \
                rm "${old_archive}"
        fi
    fi

    # Verify that the cookie symlink exists
    if [ ! -L "${old_archive}" ] ; then
        # No symlink - remove any cruft where the symlink should be
        rm -rf "${old_archive}"
        # create the symlink to the correct location
        fg_run "creating cookie symlink" \
            ln -s "${new_archive}" "${old_archive}"
    fi
}

# ------------------------------------------------------------------------------
get_push_button_rtfd_state()
{
    # Most of our systems set the RTFD cookie in u-boot.  The eredan platform
    # can not do this, as the SAROS can not be easily reached from u-boot, so
    # we grab the value and clear down the FPGA bit here.

    local hold3s=$(/ciena/bin/sarosreg SAROS_PB_STATUS.PHELD \
        | awk '/PHELD/ {print $5}')

    if [ "$hold3s" == "0x1" ] ; then
        write_cookie $ParamType_FactoryReset 1
        /ciena/bin/sarosreg SAROS_PB_STATUS.PHELD 0x0
        print_app_line_cr "RTFD via push button detected"
    fi
}

# ------------------------------------------------------------------------------
default_console_tty()
{
    echo '/dev/ttyPS0'
}

# ------------------------------------------------------------------------------
extra_reboot_info()
{
    # The CONSOLE_COLLECTOR variable will be set if we are called from rcS
    local console_path=$CONSOLE_COLLECTOR

    # load the ciena_sysfs module if necessary
    if [ ! -f "/sys/ciena/eredan/bootcount" ] ; then
        if [ -z "$console_path" ] ; then
            console_path="/dev/null"
        fi
        # Redirect the output of this command so that it doesn't end up
        # as part of our reboot log.
        insmod_wrapper ciena_sysfs > $console_path
    fi

    local csubr="$(cat /sys/ciena/eredan/csubr)"
    local pmupb="$(cat /sys/ciena/eredan/pmupb)"
    local fault_log_required=0
    local output_str

    if [ "$csubr" == "0x00000000" ] ; then
        csubr=""
    else
        csubr=" csubr=$csubr"
        echo 0 > /sys/ciena/eredan/csubr
        fault_log_required=1
    fi

    if [ "$pmupb" == "0x00000000" ] ; then
        pmupb=""
    else
        pmupb=" pmupb=$pmupb"
        echo 0 > /sys/ciena/eredan/pmupb
        fault_log_required=1
    fi

    output_str=$(printf " (bank=%s,%s reason=%s error=%s,%s multi=%s%s%s)" \
        "$(cat /sys/ciena/eredan/bank)" \
        "$(cat /sys/ciena/eredan/bootcount)" \
        "$(cat /sys/ciena/eredan/reset_reason)" \
        "$(cat /sys/ciena/eredan/error1)" \
        "$(cat /sys/ciena/eredan/error2)" \
        "$(cat /sys/ciena/eredan/multiboot)" \
        "$csubr" \
        "$pmupb")

    if [ $fault_log_required == "1" ] ; then
        generate_fault_banner
        echo "firmware fault detected" >>$FAULTLOG
        echo "$output_str" >>$FAULTLOG
    fi

    echo -n "$output_str"
}

# ------------------------------------------------------------------------------
start_mmc_refresh_scanner()
{
    if [ -x "/ciena/scripts/mmc_refresh_scan" ] ; then
        bg_run "eMMC refresh scanner" /ciena/scripts/mmc_refresh_scan
    fi
    return
}

# ------------------------------------------------------------------------------
log_fpga_diag_state()
{
    return
}

