/*
 *       $RCSfile: cfgfmt.c,v $
 *       $Revision: 1.5 $
 *       $Date: 2002-04-05 14:46:55-08 $
 */
 
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <math.h>
#ifdef _WIN32
#include <fcntl.h>
#endif

#define CFG_ENCRYPTKEY		61

/* special tags: */
#define CFG_UPGRADECODE		0x1100
#define CFG_UPGRADELANG		0x1101
#define CFG_UPGRADELOGO		0x1103
#define CFG_UPGRADEXMLCARDS	0x1104
#define CFG_SUMLEN		0x7ffe

/* input tag format flags */
#define TFMT_MAC	0x0001
#define	TFMT_IP		0x0002
#define	TFMT_INT32	0x0004
#define	TFMT_DIGITS	0x0008
#define	TFMT_ALPHA	0x0010
#define	TFMT_BOOLEAN	0x0020
#define	TFMT_SHORTA	0x0040
#define	TFMT_IP2	0x0080
#define	TFMT_TONE	0x0100
#define	TFMT_UINT32	0x0200

/* tag descriptors */
#define MAX_PTAG_NAME_STR	32
typedef struct PTagFmt {
  struct PTagFmt *next;
  unsigned short tag;
  unsigned short fmt;
  unsigned int size;
  char name[MAX_PTAG_NAME_STR];
} PTagFmt;


extern PTagFmt* findPTagFmt(unsigned int tag);

char *pTagFile = "ptag.dat";
#define ATA_MAGIC	"#ata"

unsigned int gNoInputFileEncryptKey;
unsigned char *gEncryptKey;

/*----------------------------------------------------------------------*/
void usage()
{
  printf("usage: cfgfmt [-eRc4passwd] [-E] [-tPtagFile] input output\n"
      "\t-eRc4Passwd -- use Rc4Passwd to encrypt or decrypt input\n"
      "\t-E -- do not use EncryptKey parameter's value in the input\n"
      "\t      text file to encrypt the output binary file\n"
      "\t-tPtagFile -- specify an alternate PtagFile path\n"
      );
  exit(1);
}

/*----------------------------------------------------------------------*/
typedef struct rc4Key 
{ 
  unsigned char state[256]; 
  unsigned char x; 
  unsigned char y; 
} rc4Key; 

#define swapByte(x,y) t = *(x); *(x) = *(y); *(y) = t 

void rc4(rc4Key *key, unsigned char *buf, int len)
{ 
  unsigned char t; 
  unsigned char x; 
  unsigned char y; 
  unsigned char *state; 
  unsigned char xorIndex; 
  int i;

  x = key->x; 
  y = key->y; 
  state = &key->state[0]; 
  for(i = 0; i < len; i++) 
  { 
    x = (x + 1) % 256; 
    y = (state[x] + y) % 256; 
    swapByte(&state[x], &state[y]); 
    xorIndex = (state[x] + state[y]) % 256; 
    buf[i] ^= state[xorIndex]; 
  } 
  key->x = x; 
  key->y = y; 
} 

/*----------------------------------------------------------------------*/
void setRC4Key(rc4Key *key, char *keyStr)
{
  char seed[32]; 
  char data[32];
  char digit[8]; 

  int i, j, n; 
  unsigned char t; 
  unsigned char index1; 
  unsigned char index2; 
  unsigned char* state; 

  strcpy(data,keyStr); 
  n = strlen(data); 
  if (n&1) 
  { 
    strcat(data,"0"); 
    n++; 
  } 
  n/=2; 
  digit[0] = '0'; digit[1] = 'x';
  digit[2] = digit[3] = 'A'; digit[6] = 0;
  for (i=0;i<n;i++) 
  { 
    digit[4] = data[i*2]; 
    digit[5] = data[i*2+1]; 
    seed[i] = strtoul(digit,0,0);
  } 


  state = &key->state[0]; 
  for(j = 0; j < 256; j++) 
  state[j] = j; 
  key->x = 0; 
  key->y = 0; 
  index1 = 0; 
  index2 = 0; 
  for(j = 0; j < 256; j++) 
  { 
    index2 = (seed[index1] + state[j] + index2) % 256; 
    swapByte(&state[j], &state[index2]); 
    index1 = (index1 + 1) % n; 
  } 
} 

/*----------------------------------------------------------------------*/
int stricmp(char *s1, char *s2)
{
  while ((toupper(*s1) == toupper(*s2)) && (*s1)) ++s1, ++s2;
  return *s1-*s2;
}

/*----------------------------------------------------------------------*/
char *itoa(int value, char *string, int radix)
{
  int     i, d;
  int     flag = 0;
  char    *ptr = string;

  /* This implementation only works for decimal numbers. */
  if (radix != 10) {
    *ptr++ = 0;
    return string;
    }

  if (!value) {
    *ptr++ = 0x30;
    *ptr++ = 0;
    return string;
    }

  /* if this is a negative value insert the minus sign. */
  if (value < 0) {
    *ptr++ = '-';

    /* Make the value positive. */
    value *= -1;
    }

  for (i = 10000; i > 0; i /= 10) {
    d = value / i;

    if (d || flag) {
      *ptr++ = d + 0x30;
      value -= (d * i);
      flag = 1;
      }
    }

  /* Null terminate the string. */
  *ptr++ = 0;

  return string;

} /* itoa */


/*----------------------------------------------------------------------*/
unsigned long swapLong(unsigned long x)
{
#if defined(_I386) || defined(_WIN32) || defined(_HOST_LITTLE_ENDIAN)
  unsigned long i;
  unsigned char *p = (unsigned char *)&x;
  i = ((unsigned long)p[0]) << 24;
  i |= ((unsigned long)p[1]) << 16;
  i |= ((unsigned long)p[2]) << 8;
  i |= ((unsigned long)p[3]);
  return i;
#else
  return x;
#endif
}

/*----------------------------------------------------------------------*/
unsigned short swapShort(unsigned short x)
{
#if defined(_I386) || defined(_WIN32) || defined(_HOST_LITTLE_ENDIAN)
  unsigned short i;
  unsigned char *p = (unsigned char *)&x;
  i = ((unsigned long)p[0]) << 8;
  i |= ((unsigned long)p[1]);
  return i;
#else
  return x;
#endif
}

/*----------------------------------------------------------------------*/ 
/* getInt -- return the 32bit integer from the input -- function needed
 * 	to access potentially misaligned data in big endian order.
 */
int getInt(unsigned char *p)
{
  unsigned long i;
  i = ((unsigned long)p[0]) << 24;
  i |= ((unsigned long)p[1]) << 16;
  i |= ((unsigned long)p[2]) << 8;
  i |= ((unsigned long)p[3]);
  return (int)i;
}

/*----------------------------------------------------------------------*/ 
/* getShort -- return the short integer from the input -- function needed
 * 	to access potentially misaligned data in big endian order.
 */
short getShort(unsigned char *p)
{
    unsigned short i;
    i = ((unsigned long)p[0]) << 8;
    i |= ((unsigned long)p[1]);
    return (short)i;
}

/*----------------------------------------------------------------------*/ 
/* get1or2 -- return the short integer using one or two bytes of the input.
 * 	1 byte of input is used if the first input byte is less than 0x80,
 * 	otherwise, two byte of input is used in the following way:
 * 		((input_byte1 & 0x7f) << 8) | input_byte2
 */
unsigned short get1or2(unsigned char **p)
{
  unsigned short c;
  unsigned short v;

  c = **p; *p += 1;

  if (c & 0x80) { /* higher bit set, means 1 more byte to follow */
    v = ((c & 0x7f) << 8) | **p;
    *p += 1;
    return v;
    }
  else return c;
}

/*----------------------------------------------------------------------*/
/* move_var2(val, p)  -- move val into p as either 1 or 2 bytes depending
 * 	on the size of val.
 */
unsigned char* move_var2(unsigned char *p, unsigned short val)
{
  if (val < 0x80) {
    unsigned char t = val & 0xff;
    *p++ = t;
    }
  else if (val < 0x8000) {
    unsigned char t;
    /* the higher bit of the val 1st byte tell us if it is two or
     * or byte value.
     */
    t = (val >> 8) | 0x80;
    *p++ = t;
    t = val & 0xff;
    *p++ = t;
    }
  else {
    printf("error: value too big: 0x%x\n", val);
    }
  return p;
}

/*----------------------------------------------------------------------*/ 
int cksum_n(unsigned char *p, int len)
{
  int i;
  int sum = 0;
  for (i = 0; i < len; i++) {
    sum += p[i];
    }
  return sum;
}

/*----------------------------------------------------------------------*/ 
/* StringToIpAddr(ip,ipStr,n) -- convert from ipStr of the form
 * 	xxx.xxx.xxx.xxx to binary representation of xxx in byte
 * 	array.
 * 	The conversion is done in big endian order. Note, this
 * 	function is used for handling string representation of
 * 		1. ip address
 * 		2. ip address and port (e.g. 192.168.2.3.9001)
 * 		3. mac address (e.g. 10.20.30.40.50.60).
 * 	based on the number of dots (n+1) specified (i.e. not strictly
 * 	just for IP string).
 */
int StringToIpAddr(unsigned char *ip, unsigned char *ipStr, int n)
{
  int i, nd, dots = 0;
  unsigned char *p;

  if (ipStr[0] == '0' && ipStr[1] == 0) {
    if (n == 5) n = 6; /* special ip:port */
    memset(ip, 0, n);
    return 1;
    }
  p = ipStr;
  nd = 3;
  for (i = 0; i < n; i++) {
    char c, *t, buf[8];
    int m = 0;
    t = buf;
    while ((c = *p)) {
      p++; m++;
      if (c == '.') { dots++; break; }
      if (m > nd) return 0;
      if (isdigit(c)) {
	*t++ = c;
	}
      else break;
      }
    *t = 0;
    if (t == buf) return 0;
    if (n == 5) {
      if (nd == 5) {
	unsigned x = atoi(buf);       
	ip[i] = (x >> 8) & 0xff;
	ip[i+1] = x & 0xff;
	return 1;
	}
      if (dots == 4) {
	nd = 5; /* last digits are 5-digits long */
	}
      }
    ip[i] = atoi(buf);       
    }
  return 1;
} 

/*----------------------------------------------------------------------*/ 
/* IpAddrToString(ip,ipStr,n) -- convert from big endian
 * 	representation of byte array to integer string separated
 * 	by dots. The number of dots is specified in (n+1). This is
 * 	the reverse of StringToIpAddr function.
 */
void IpAddrToString(unsigned char *ip, char *ipStr, int n)
{
  int i;

  for (i = 0; i < n; i++) {
    if (n == 5 && i == 4) {
      unsigned n;
      n = (((unsigned)ip[i]) << 8) | ip[i+1] ;
      itoa(n, ipStr, 10);
      }
    else {
      itoa(ip[i], ipStr, 10);
      }
    ipStr += strlen(ipStr);	    /* Move past the string just added. */
    if (i < (n-1)) *ipStr++ = '.';	    /* Add the dot. */
    }
  *ipStr = 0;
} 

/*----------------------------------------------------------------------*/
/* simple string tokenizer routines.
 * usage:
 * 	1. StringTokenizer t;  -- declare a handle
 * 	2. setStringTokenizer(&t, inputString, separator); -- set up
 * 		the handle (e.g. setStringTokenizer(&t, "1,2, 3", ", ");
 * 		which set the separator between each token to be one
 * 		or more of the ',' or ' ').
 * 	3. call nextStringToken or nextIntToken etc to extract the next
 * 		token.
 */
typedef struct
{
  char *next;
  char *sep;
} StringTokenizer;

void setStringTokenizer(StringTokenizer *t, char *str, char *sep)
{
  t->next = str;
  t->sep = sep;
}

void skipSep(StringTokenizer *t)
{
  char c;
  while ((c = *(t->next)) != 0) {	/* skip separators */
    if (!strchr(t->sep,c)) break;
    t->next++;
    }
}

void skipToken(StringTokenizer *t)
{
  char c;
  skipSep(t);
  while ((c = *(t->next)) != 0) {	/* tokenize till separator */
    if (strchr(t->sep,c)) break;
    t->next++;
    }
}

int nextStringTokenN(StringTokenizer *t, char *buf, unsigned int len)
{
  char c;
  char *p = buf;
  int n = 0;
  while ((c = *(t->next)) != 0) {	/* skip separators */
    if (!strchr(t->sep,c)) break;
    t->next++;
    }
  while ((c = *(t->next)) != 0) {	/* tokenize till separator */
    if (strchr(t->sep,c)) break;
    *p++ = c;
    t->next++;
    if (++n >= len) break;
    }
  *p = 0;
  return (p != buf);
}

int nextStringToken(StringTokenizer *t, char *buf)
{
  return nextStringTokenN(t,buf,0xffffffff);
}

int nextIntToken(StringTokenizer *t, int *x)
{
  char buf[32];
  int rc = nextStringToken(t,buf);
  if (rc)
    *x = atoi(buf);
  return rc;
}

int nextUintToken(StringTokenizer *t, unsigned int *x)
{
  char buf[32];
  int rc = nextStringToken(t,buf);
  if (rc)
    *x = strtoul(buf,0,0);
  return rc;
}

/*----------------------------------------------------------------------*/
unsigned long fileSize(FILE *fp)
{
  unsigned long len;
  fseek(fp, 0, 2);
  len = ftell(fp);
  fseek(fp, 0, 0);
  return len;
}

/*----------------------------------------------------------------------*/
/* strCatp -- cat and return the end of the resulting string. */
char* strCatp(char *p, char *s)
{
  while (*s) {
    *p++ =  *s++;
    }
  *p = 0;
  return p;
}

/*----------------------------------------------------------------------*/
/* moveLong -- move long array in bigendian order. */
unsigned char* moveLong(unsigned char *dst, unsigned long *dataptr, long count)
{
  int i;
  unsigned long data;
  unsigned char *p;

  p = (unsigned char *)dst;
  for (i = 0; i < count; i++) {
    data = *dataptr++;
    *p++ =  (unsigned char) ( (data>>24) & 0xff);
    *p++ = (unsigned char) ( (data>>16) & 0xff);
    *p++ = (unsigned char) ( (data>> 8) & 0xff);
    *p++ = (unsigned char) ( data & 0xff);
    }
  return p;
}

/*----------------------------------------------------------------------*/
/* moveShort -- move short array in bigendian order. */
unsigned char* moveShort(unsigned char *dst, unsigned short *dataptr, long count)
{
  int i;
  unsigned short data;
  unsigned char *p;

  p = (unsigned char *)dst;
  for (i = 0; i < count; i++) {
    data = *dataptr++;
    *p++ = (unsigned char) ( (data>> 8) & 0xff);
    *p++ = (unsigned char) ( data & 0xff);
    }
  return p;
}

/************************************************************************** 
Here is the tone parameter list for ATA tone provisioning: 
- ntone (no. of tones): 1 or 2 (integer only) 
- freq[0] (Hz): 0.00 to 4000.00  (UI should allow 2 decimal places) 
- freq[1] (Hz): 0.00 to 4000.00  (UI should allow 2 decimal places) 
- level[0] (dBm): 10.00 to -60.00 (UI should allow 2 decimal places) 
- level[1] (dBm): 10.00 to -60.00 (UI should allow 2 decimal places) 
- steady: 1 (steady tone, next 2 parameters ignored), 
   or 0 (on/off according to next 2 parameters) (integer only) 
- on-time  (ms): 1 to 0x7FFFFFFF (integer only) 
- off-time (ms): 1 to 0x7FFFFFFF (integer only) 
- total time to play tone (ms): 0 (forever), or 1 to 0x7FFFFFFF (integer only) 

For example: US Ring-back tone 
2, 440.0, 480.0, -10.0, -10.0, 0, 2000, 4000, 0 
**************************************************************************/ 

typedef struct tagTONEPAR { 
  unsigned ntone; 
  short a[2]; 
  short s0[2]; 
  unsigned int steady; 
  unsigned int ontime; 
  unsigned int offtime; 
  unsigned int limit; 
} TONEPAR; 

/* convert freq and level into "a" and "s0" ATA tone parameters */ 
void transformFreqLevel(char *freq, char *level, short *a, short *s0) 
{ 
  #define MAX_LEVEL       10.0 
  #define MIN_LEVEL       -60.0 
  #define MAX_FREQ        4000.0 
  #define MIN_FREQ        0.0 
  #define PI              3.14159265358979 
  #define SAMPLE_RATE     8000.0 

  double f = atof(freq); 
  double m = atof(level); 
  long n1, n2; 

  if(m > MAX_LEVEL) m = MAX_LEVEL; 
  else if(m < MIN_LEVEL) m = MIN_LEVEL; 

  if(f > MAX_FREQ) f = MAX_FREQ; 
  else if(f < MIN_FREQ) f = MIN_FREQ; 

  f = 2.0*PI*f/SAMPLE_RATE; 

  m = pow(10.0, (m-MAX_LEVEL)*0.05); 

  n1 = (long)(32768.0*cos(f)); 
  n2 = (long)(32768.0*m*sin(f)); 

  if (n1 > 32767) n1 = 32767; 
  else if(n1 < -32768) n1 = -32768; 

  if (n2 > 32767) n2 = 32767; 
  else if(n2 < -32768) n2 = -32768; 

  *a = (short)n1; 
  *s0 = (short)n2; 
} 

/*----------------------------------------------------------------------*/ 
PTagFmt *pTagFmts;
void readPTagFile(char *pTagFile)
{
  /* fill up pTagFmt -- link list of pTag fmt that determine how to
   * encode tag_str:val_str to tag,len,val.
   */
  int i;
  char buf[80]; /* we know the size is not going to be longer than 80,
		  otherwise just treat it as bad value. */
  FILE *fp;
 
  fp = fopen(pTagFile, "r");
  if (fp == 0) {
    fprintf(stderr, "error: can't open ptag file: '%s'\n", pTagFile);
    exit(1);
    }
  
  while (fgets(buf, 80, fp)) {
    if (buf[0] != '#') {
      char name[MAX_PTAG_NAME_STR];
      unsigned int tag, fmt, size;
      int ok;
      StringTokenizer t;
      setStringTokenizer(&t, buf, ", \t");
      ok = nextUintToken(&t, &tag) &&
	  nextUintToken(&t, &fmt) &&
	  nextStringTokenN(&t, name, MAX_PTAG_NAME_STR) &&
	  nextUintToken(&t, &size);
      if (!ok) {
	printf("warning: bad ptag %s\n", buf);
	}
      else {
	PTagFmt *t = (PTagFmt*)malloc(sizeof(PTagFmt));
	t->tag = (unsigned short)tag;
	t->fmt = (unsigned short)fmt;
	t->size = size;
	strncpy(t->name, name, MAX_PTAG_NAME_STR-1);
	t->name[MAX_PTAG_NAME_STR-1] = 0;
	t->next = pTagFmts; /* link it */
	pTagFmts = t;
	/* printf("t %s %d %x %d\n", t->name, t->tag, t->fmt, t->size); */
	}
      }
    }

  fclose(fp);
}

/*----------------------------------------------------------------------*/
int pTagTxt2Bin(PTagFmt *t, char *text, char *bin)
{
  unsigned long fmt = t->fmt;
  unsigned int dataSize = t->size;
  
  if (fmt & TFMT_BOOLEAN) {
    if (!stricmp(text, "false") || !strcmp(text,"0")) {
       *bin= 0;
       }
    else *bin = 1;
    return 1;
    }
  else if (fmt & TFMT_IP) {
    /* convert to INT32 */
    StringToIpAddr(bin, text, 4);
    return 4;
    }
  else if (fmt & TFMT_MAC) {
    /* convert to char[6] */
    StringToIpAddr(bin, text, 6);
    return 6;
    }
  else if (fmt & TFMT_IP2) {
    /* convert to char[6] -- even if I say 5 below */
    StringToIpAddr(bin, text, 5);
    return 6;
    }
  else if (fmt & TFMT_INT32) { /* not IPFMT */
    /* convert to INT32 */
    unsigned long x = strtoul(text,0,0);
    moveLong(bin, &x,  1);
    return 4;
    }
  else if (fmt & TFMT_TONE) {
    TONEPAR tp;
    char freq[2][32], level[2][32];
    /* convert to short[] */
    unsigned short x;
    unsigned short *s;
    int i;

    StringTokenizer st;
    setStringTokenizer(&st, text, ", ");

    if (!(nextUintToken(&st, &tp.ntone) &&
	nextStringTokenN(&st, freq[0], 32) &&
	nextStringTokenN(&st, freq[1], 32) &&
	nextStringTokenN(&st, level[0], 32) &&
	nextStringTokenN(&st, level[1], 32) &&
	nextUintToken(&st, &tp.steady) &&
	nextUintToken(&st, &tp.ontime) &&
	nextUintToken(&st, &tp.offtime) &&
	nextUintToken(&st, &tp.limit))) {
      printf("error: invalid tone specification\n");
      return dataSize;
      }

    if(tp.ntone < 0) tp.ntone = 0; 
    else if(tp.ntone > 2) tp.ntone = 2; 

#if 0
    printf("<%d,%s,%s,%s,%s,%d,%d,%d,%d\n",
	tp.ntone, freq[0], freq[1], level[0], level[1], tp.steady,
	tp.ontime, tp.offtime, tp.limit);
#endif

    for(i=0; i < tp.ntone; i++) 
      transformFreqLevel(freq[i], level[i], &(tp.a[i]), &(tp.s0[i])); 

#if 0
    printf(">%d,%u,%u,%u,%u,%d,%d,%d,%d\n",
	tp.ntone, tp.a[0], tp.a[1], tp.s0[0], tp.s0[1], tp.steady,
	tp.ontime, tp.offtime, tp.limit);
#endif

    s = (unsigned short*)bin;
    i= 0;
    x = (unsigned short)tp.ntone;
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.a[0];
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.a[1];
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.s0[0];
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.s0[1];
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.steady;
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.ontime;
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.offtime;
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    x = (unsigned short)tp.limit;
    moveShort((unsigned char*)&(s[i]), &x, 1); i++;
    return dataSize;
    }
  else if (fmt & TFMT_SHORTA) {
    /* convert to short[] */
    unsigned int t;
    unsigned short *s;
    int i, n;

    StringTokenizer st;
    setStringTokenizer(&st, text, ", ");
    s = (unsigned short*)bin;
    i= 0;
    n = dataSize >> 1;
    while (nextUintToken(&st, &t)) {
      /* todo: need to check for size */
      if (i < n) {
	unsigned short x = (unsigned short)t;
	moveShort((unsigned char*)&(s[i]), &x, 1);
	i++;
	}
      }
    return dataSize;
    }
  else if (fmt & (TFMT_ALPHA|TFMT_DIGITS)) {
    strcpy(bin, text);
    return strlen(bin);
    }
  else {
    printf("internal error: unhandled attribute: tag:%d fmt:0x%x)\n",
	t->tag, fmt);
    exit(1);
    }
}

/*----------------------------------------------------------------------*/
PTagFmt* findPTagFmt(unsigned int tag)
{
  PTagFmt *t = pTagFmts;
  while (t) {
    if (t->tag == tag) {
      return t;
      }
    t = t->next;
    }
  return 0;
}

/*----------------------------------------------------------------------*/
int cmdBin2Txt(unsigned int tag, char *text, unsigned char *p, int len)
{
  if (tag == CFG_SUMLEN) {
    unsigned short sum, len;
    /* this tool ignore this parameter but ata look at it */
    sum = getShort(p); p += 2;
    len = getShort(p); p += 2;
    sprintf(text, "#sum_len:%d,%d", sum, len);
    }
  else if (tag == CFG_UPGRADECODE || tag == CFG_UPGRADELANG) {
    unsigned char upgradePolicy;
    unsigned int hardwareVersion;
    unsigned short softwareType;
    unsigned short softwareVersion;
    unsigned int tftpIp;
    unsigned short tftpPort;
    unsigned int ImageId;
    unsigned char tftpFileName[32];
    unsigned char tftpIpStr[20];
    
    if (len < (1+4+2+2+4+2+4+1) /* min len */) {
      sprintf(text, "#bad len: tag 0x%x, len 0x%x", tag, len);
      return 1;
      }
    upgradePolicy = *p++;
    hardwareVersion = getInt(p); p += 4;
    softwareType = getShort(p); p += 2;
    softwareVersion = getShort(p); p += 2;
    tftpIp = getInt(p); p += 4;
    tftpPort = getShort(p); p += 2;
    ImageId = getInt(p); p += 4;
    tftpIp = swapLong(tftpIp);
    IpAddrToString((unsigned char*)&tftpIp, tftpIpStr, 4);
    strncpy(tftpFileName, p, 31); tftpFileName[31] = 0; /* truncate */
    sprintf(text, "%s:%d,0x%08x,0x%04x,0x%04x,%s,0x%04x,0x%08x,%s",
	tag==CFG_UPGRADECODE?"upgradecode":"upgradelang",
	upgradePolicy, hardwareVersion, softwareType, softwareVersion,
	tftpIpStr, tftpPort, ImageId, tftpFileName);
    }
  else if (tag == CFG_UPGRADELOGO || tag == CFG_UPGRADEXMLCARDS) {
    unsigned int imageId;
    unsigned int tftpIp;
    unsigned char tftpFileName[32];
    unsigned char tftpIpStr[20];
    
    if (len < (4+4+1) /* min len */) {
      sprintf(text, "#bad len: tag 0x%x, len 0x%x", tag, len);
      return 1;
      }
    imageId = getInt(p); p += 4;
    tftpIp = getInt(p); p += 4;
    tftpIp = swapLong(tftpIp);
    IpAddrToString((unsigned char*)&tftpIp, tftpIpStr, 4);
    strncpy(tftpFileName, p, 31); tftpFileName[31] = 0; /* truncate */
    sprintf(text, "upgrade%s:0x%08x,%s,%s",
	tag == CFG_UPGRADELOGO?"logo":"xml",
	imageId, tftpIpStr, tftpFileName);
    }
  else {
    sprintf(text, "#fixme: tag 0x%x", tag);
    }
  return 1; /* always true? */
}

/*----------------------------------------------------------------------*/
/* tmp bin for making ab pair unique. */

/* config parameters */
typedef struct
{
  unsigned char valid;
  unsigned char len;
  unsigned short tag;
  unsigned char val[256]; /* box limitation make it can't be more than 256 */
} CfgTmpVal;

typedef struct
{
  unsigned char valid;
  unsigned char len;
  char val[256];	/* box limitation make it can't be more than 256 */
} ProfTmpVal;

static CfgTmpVal cfgTmpVals[255];
static ProfTmpVal profTmpVals[255];
static int cfgTmpValIdx;

int readProfile_(char *file, FILE *fp, int nested)
{
  int line = 0;
  char attr[256];
  char val[512];
  char bval[256];
  char buf[1024];
 
  while (fgets(buf, 1023, fp)) {
    StringTokenizer st;
    PTagFmt *t;
    int found = 0;

    line++;

    if (buf[0] == '#') {
      continue;
      }
    setStringTokenizer(&st, buf, ": \n\r");
    if (!nextStringToken(&st, attr)) {
      continue; /* blank line */
      }

    skipSep(&st);
    setStringTokenizer(&st, st.next, "\n\r");
    nextStringToken(&st, val);

    if (!strcmp(attr, "include")) {
      FILE *ifp2;
      if (nested == 3) {
	printf("error: include nested too deep\n");
	continue;
	}
      /* if (debug > 2) printf("open %d %s\n", nested, val); */
      ifp2 = fopen(val, "rb");
      if (ifp2 == 0) {
	fprintf(stderr, "error: can't open profile '%s'\n", val);
	return 0;
	}
      readProfile_(val, ifp2, nested+1);
      fclose(ifp2);
      continue;
      }
    else if (!strcmp(attr, "upgradecode") || !strcmp(attr, "upgradelang")) {
      int stat;
      unsigned char *p;
      unsigned short tmpShort;
      unsigned int upgradePolicy;
      unsigned int hardwareVersion;
      unsigned int softwareType;
      unsigned int softwareVersion;
      unsigned char tftpIpStr[32];
      unsigned int tftpIp;
      unsigned int tftpPort;
      unsigned int imageId;
      int len;
      char tftpFileName[32];
      setStringTokenizer(&st, val, ", ");
      stat = nextUintToken(&st, &upgradePolicy)
	  && nextUintToken(&st, &hardwareVersion)
	  && nextUintToken(&st, &softwareType)
	  && nextUintToken(&st, &softwareVersion)
	  && nextStringToken(&st, tftpIpStr)
	  && nextUintToken(&st, &tftpPort)
	  && nextUintToken(&st, &imageId)
	  && nextStringToken(&st, tftpFileName);
      if (stat == 0) {
	printf("error: bad argument for upgrade tag\n");
	continue;
	}
      if (StringToIpAddr((unsigned char*)&tftpIp, tftpIpStr, 4) == 0) {
	printf("error: bad tftp ip addr: %s\n", tftpIpStr);
	continue;
	}
      /* now pack it into the format acceptable by ata */
#if 0
upgradePolicy_1,hardwareVersion_4,softwareType_2,softwareVersion_2,
tftpIp_4,tftpPort_2,ImageId_4,fileName_max32, 0_1
#endif
      cfgTmpVals[cfgTmpValIdx].tag = (!strcmp(attr, "upgradecode"))?
	  CFG_UPGRADECODE:CFG_UPGRADELANG;
      p = cfgTmpVals[cfgTmpValIdx].val;
      *p++ = upgradePolicy & 0xff;
      moveLong(p, &hardwareVersion,  1); p += 4;
      tmpShort = softwareType; moveShort(p, &tmpShort,  1); p += 2;
      tmpShort = softwareVersion; moveShort(p, &tmpShort,  1); p += 2;
      memcpy(p, &tftpIp,  4); p += 4; /* already in big-endian */
      tmpShort = tftpPort; moveShort(p, &tmpShort,  1); p += 2;
      moveLong(p, &imageId,  1); p += 4;
      p = strCatp(p, tftpFileName); *p++ = 0;
      cfgTmpVals[cfgTmpValIdx].len = (p - cfgTmpVals[cfgTmpValIdx].val);
      cfgTmpVals[cfgTmpValIdx].valid = 1;
      cfgTmpValIdx++;
      continue;
      }
    else if (!strcmp(attr, "upgradelogo") || !strcmp(attr, "upgradexml")) {
      int stat;
      unsigned char *p;
      unsigned char tftpIpStr[32];
      unsigned int tftpIp;
      unsigned int imageId;
      int len, tag;
      char tftpFileName[32];

      if (!strcmp(attr, "upgradelogo")) {
	tag = CFG_UPGRADELOGO;
	}
      else {
	tag = CFG_UPGRADEXMLCARDS;
	}
      setStringTokenizer(&st, val, ", ");
      stat = nextUintToken(&st, &imageId)
	  && nextStringToken(&st, tftpIpStr)
	  && nextStringToken(&st, tftpFileName);

      if (stat == 0) {
	printf("error: bad argument for upgradelogo tag\n");
	continue;
	}
      if (StringToIpAddr((unsigned char*)&tftpIp, tftpIpStr, 4) == 0) {
	printf("error: bad tftp ip addr: %s\n", tftpIpStr);
	continue;
	}
      /* now pack it into the format acceptable by ata */
#if 0
imageId_4,tftpIp_4,filename_max32
#endif
      cfgTmpVals[cfgTmpValIdx].tag = tag;
      p = cfgTmpVals[cfgTmpValIdx].val;
      moveLong(p, &imageId,  1); p += 4;
      memcpy(p, &tftpIp,  4); p += 4; /* already in big-endian */
      p = strCatp(p, tftpFileName); *p++ = 0;
      cfgTmpVals[cfgTmpValIdx].len = (p - cfgTmpVals[cfgTmpValIdx].val);
      cfgTmpVals[cfgTmpValIdx].valid = 1;
      cfgTmpValIdx++;
      continue;
      }
    t = pTagFmts;
    while (t) {
      if (stricmp(t->name, attr) == 0) {
	unsigned short tag, len;
	found = 1;
	tag = (unsigned short)t->tag;
	if (tag < 255 && val[0] != 0) {
	  /* overwrite whatever value was entered before */
	  len = pTagTxt2Bin(t, val, bval);
	  profTmpVals[tag].valid = 1;
	  profTmpVals[tag].len = len;
	  memcpy(profTmpVals[tag].val, bval, len);
	  }
	else {
	  printf("warning: empty value for attribute '%s' @line %d of %s\n",
	      attr, line, file);
	  }
	break;
	}
      t = t->next;
      }
    if (!found) {
      printf("warning: unknown attribute '%s' @line %d of %s\n",
	  attr, line, file);
      }
    }

  return 1;
}

/*----------------------------------------------------------------------*/
int cfgTxt2Bin(char *iFile, FILE *ifp, FILE *ofp)
{
  unsigned char profTmpBuf[2048];
  unsigned char *sumLenLocPtr, *firstParam, *p;
  unsigned short len, sum;
  int i;

  for(i = 0; i < 255; i++) {
    profTmpVals[i].valid = 0;
    }

  cfgTmpValIdx = 0;
  for(i = 0; i < 255; i++) {
    cfgTmpVals[i].valid = 0;
    }

  readProfile_(iFile, ifp, 0);

  p = profTmpBuf;	/* transfer the tmp structures to buf */

  memcpy(p, ATA_MAGIC, 4); p += 4;
  p = move_var2(p, CFG_SUMLEN); /* tag */
  p = move_var2(p, 4); /* len */
  sumLenLocPtr = p;
  p += 4;
  firstParam = p;

  for (i = 0; i < 255; i++) {
    if (cfgTmpVals[i].valid) {
      unsigned short len = cfgTmpVals[i].len;
      p = move_var2(p, cfgTmpVals[i].tag);
      p = move_var2(p, len);
      memcpy(p, cfgTmpVals[i].val, len);
      p += len;
      }
    }

  for (i = 0; i < 255; i++) {
    if (profTmpVals[i].valid) {
      unsigned short len = profTmpVals[i].len;
      p = move_var2(p, (unsigned short)i);
      p = move_var2(p, len);
      memcpy(p, profTmpVals[i].val, len);
      p += len;

      /* check for encryption key */
      if (i == CFG_ENCRYPTKEY &&
	  profTmpVals[i].len > 0 && !(profTmpVals[i].val[0] == '0' &&
	  profTmpVals[i].val[1] == 0)) {
	/* the one in the profile overwrite the one specified on command
	 * line
	 */
	if (gNoInputFileEncryptKey == 0)
	  gEncryptKey = profTmpVals[i].val;
	}
      }
    }

  /* patch the sumlen tag value */
  len = (unsigned short)(p - firstParam);
  sum = (unsigned short)(cksum_n(firstParam, len));

  moveShort(sumLenLocPtr, &sum, 1);
  moveShort(sumLenLocPtr+2, &len, 1);
  
  if (gEncryptKey) {
    rc4Key key;
    setRC4Key(&key, gEncryptKey);
    rc4(&key, profTmpBuf, p-profTmpBuf);
    }

  fwrite(profTmpBuf, 1, p-profTmpBuf, ofp);
}

/*----------------------------------------------------------------------*/ 
void pTagBin2Txt(PTagFmt *t, char *text, char *bin, int len)
{
  unsigned long fmt = t->fmt;
  
  if (t->size < len) {
    printf("#warning: input len > known len (%d>%d)\n", len, t->size);
    len = t->size;
    }

  if (fmt & TFMT_BOOLEAN) {
    /* boolean -- 0 or 1 */
    strcpy(text, (*bin=='0'||*bin==0)?"0":"1");
    }
  else if (fmt & TFMT_IP) {
    /* ip address of the form xxx.xxx.xxx.xxx */
    IpAddrToString((unsigned char*)bin, text, 4);
    }
  else if (fmt & TFMT_IP2) {
    /* ip address with port of the form xxx.xxx.xxx.xxx.yyyyy */
    IpAddrToString(bin, text, 5);
    }
  else if (fmt & TFMT_INT32) { /* not IPFMT bit */
    unsigned long x = getInt(bin);
    sprintf(text, (fmt&TFMT_UINT32)?"0x%08x":"%u", x);
    }
  else if (fmt & TFMT_MAC) {
    /* mac address of the form xxx.xxx.xxx.xxx.xxx.xxx */
    IpAddrToString(bin, text, 6);
    }
  else if (fmt & TFMT_SHORTA) {
    /* short integers separated by comma ',' */
    int i, n;
    unsigned short *x;
    char *p;
    p = text;
    x = (unsigned short*)bin;
    n = len >> 1;
    for (i = 0; i < n; i++) {
      char buf[32];
      char *t;
      sprintf(buf, "%u", getShort(x+i));
      t = buf;
      while (*t) *p++ = *t++;
      if (i < (n-1)) *p++ = ',';
      }
    *p = 0;
    }
  else if (fmt & (TFMT_ALPHA|TFMT_DIGITS)) {
    /* alphanumeric or digits string */
    strncpy(text, bin, len);
    text[len] = 0;
    }
  else strcpy(text, "uimp");
}

/*----------------------------------------------------------------------*/ 
int cfgBin2Txt(unsigned char *inBuf, int inSize, FILE *ofp)
{
  char text[256];
  unsigned char *p;
  
  p = inBuf;

  fprintf(ofp, "#txt\n"); /* begin with the text header */
  while (inSize > 0) {
    PTagFmt *t;
    unsigned short tag, len;
    unsigned char *p0 = p;
    tag = get1or2(&p); inSize -= (p - p0); if (inSize <= 0) goto error;
    p0 = p;
    len = get1or2(&p); inSize -= (p - p0); if (inSize < 0) goto error;
    if ((inSize - len) < 0) goto error;
    if (tag != 0) {
      t = findPTagFmt(tag);
      if (t) {
	pTagBin2Txt(t, text, p, len);
	fprintf(ofp, "%s:%s\n", t->name, text);
	}
      else {
	if (cmdBin2Txt(tag, text, p, len)) {
	  fprintf(ofp, "%s\n", text);
	  }
	else {
	  fprintf(ofp, "#warning: unknown tag(%x)\n", tag);
	  }
	}
      }
    p += len; inSize -= len;
    }

  return 1;

error:
  printf("corrupted profile\n");
  return 0;
}

/*----------------------------------------------------------------------*/ 
int main(int argc, char **argv)
{
  FILE *ifp, *ofp;
  int nargc;
  char **nargv;
  unsigned char signature[4];
  int inSize;
  int i, j;

  /* ------- parse arguments ------- */
  nargc = 0;
  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-') ++nargc;
    else {
      if (!strncmp(argv[i], "-e", 2)) {
	gEncryptKey = argv[i]+2;
	}
      else if (!strncmp(argv[i], "-E", 2)) {
	gNoInputFileEncryptKey = 1;
	}
      else if (!strncmp(argv[i], "-t", 2)) {
	pTagFile = argv[i]+2;
	}
      else {
	printf("unknown switch %s\n", argv[i]);
	}
      }
    }

  nargv = (char**)malloc(sizeof(char*)*(nargc+1));
  for (i = 1, j = 0; i < argc; i++) {
    if (argv[i][0] != '-') {
      nargv[j++] = argv[i];
      }
    }
  nargv[j] = 0;

  if (nargc < 1) { usage(); }

  /* ------- need 2 files - in and out ------- */
  if (nargc != 2) {
    usage();
    }

  readPTagFile(pTagFile);

  ifp = fopen(nargv[0], "rb");
  if (ifp == 0) {
    printf("error: can't open %s for reading\n", nargv[0]);
    exit(1);
    }

  ofp = fopen(nargv[1], "wb");
  if (ofp == 0) {
    fclose(ifp);
    printf("error: can't open %s for writing\n", nargv[1]);
    exit(1);
    }

  inSize = fileSize(ifp);
  if (inSize < 5 || inSize > 120000) { /* don't accept arbitary input size */
    printf("error: bad input size %d\n", inSize);
    goto end;
    }

  if (fread(signature, 1, 4, ifp) != 4) {
    printf("error: read error\n");
    goto end;
    }

  fseek(ifp, 0, 0);
  if (memcmp(signature,"#txt", 4) == 0) {
    /* converting text to bin fmt */
    cfgTxt2Bin(nargv[0], ifp, ofp);
    }
  else {
    /* read in the whole thing in case we need to decrypt it */
    unsigned char *inBuf = (unsigned char*)malloc(inSize);
    if (fread(inBuf, 1, inSize, ifp) != inSize) {
      printf("error: reading input file\n");
      goto end;
      }
    if (gEncryptKey) {
      rc4Key key;
      setRC4Key(&key, gEncryptKey);
      rc4(&key, inBuf, inSize);
      }
    if (memcmp(inBuf, ATA_MAGIC, 4) == 0) {
      inSize -= 4;
      /* converting bin to readable text fmt */
      cfgBin2Txt(inBuf+4, inSize, ofp);
      }
    else {
      fprintf(stderr, "error: unknown or encrypted file: '%s'\n", nargv[0]);
      }
    }

end:
  fclose(ifp);
  fclose(ofp);
  return 0; /* clean */
}

