//
//  CJGuestCallViewController.h
//  JabberGuest
//
//  Copyright (c) 2013 Cisco Systems, Inc. All Rights Reserved.
//

#import <UIKit/UIKit.h>
#import <JabberGuest/JabberGuest.h>
#import <MessageUI/MessageUI.h>

@protocol CJGuestCallViewControllerDelegate;


/**JabberGuest/JabberGuestSDK/JabberGuestSDKLib
 * The high-level Jabber Guest controller object.
 */
@interface CJGuestCallViewController : UIViewController <CJGuestCallBarViewDelegate, UIActionSheetDelegate>

/**
 * The Jabber Guest server name.
 */
@property (nonatomic, readwrite, strong) NSString * serverName;

/**
 * The URI to call.
 */
@property (nonatomic, readwrite, strong) NSString * toURI;

/**
 * The delegate that implements the CJGuestCallViewControllerDelegate protocol.
 */
@property (weak) id <CJGuestCallViewControllerDelegate> delegate;

/**
 * The number of seconds to keep the call bar visible after it is made visible.
 */
@property (nonatomic, readwrite) NSTimeInterval secondsCallBarHide;

/**
 * Returns YES if the call state is connected, connecting, or disconnecting.
 */
+ (BOOL)hasActiveCall;

/**
 * displays a tranparent window showing call statistics for an active call
 */
- (void)showCallStatistics;

/**
 * Configures the controller by parsing a URL.  The URL must follow the format as
 * defined in the Custom URL Schemes documentation.
 *
 * Typically, you would call this from your app delegate's `handleURL:` method.
 *
 * #serverName and #toURI are set to the values pulled out of the URL.
 *
 * @param url The URL to parse.
 */
- (void)configureFromURL:(NSURL *)url;

/**
 * Handles all the details of having an active call go to the background. When the app goes into 
 * the background, the video stops but the audio will continue.  If the video was muted before 
 * entering the background, it will remain muted when returning to the foreground.  Use with 
 * #enterForeground to get this behavior.
 *
 * Call this method from your app delegate's `applicationWillResignActive:` or
 * `applicationDidEnterBackground:` implementations.
 */
- (void)enterBackground;

/**
 * Handles all the details of having an active call go to the foreground.  If the video was muted 
 * before entering the background, it will remain muted when returning to the foreground.  Use with
 * #enterBackground to get this behavior.
 *
 * Call this method from your app delegate's `applicationWillEnterForeground:` or
 * `applicationDidBecomeActive:` implementations.  
 */
- (void)enterForeground;

/**
 * Handles terminating an active call when your app terminates.  
 *
 * Call this method from your app delegate's `applicationWillTerminate:` implementation.
 */
- (void)terminate;

@end


/**
 * The delegate for CJGuestCallViewController.
 */
@protocol CJGuestCallViewControllerDelegate <NSObject>

@optional

/**
 * The call is finished (end button is pressed or disconnect occurs).
 *
 * @param callController The controller whose call is finished.
 */
- (void)callFinishedForCallController:(CJGuestCallViewController *)callController;

/**
 * The more button was pressed in CJGuestCallViewController's navigation bar.
 *
 * @param callController  The controller whose button was pressed.
 */
- (void)moreButtonPressedForCallController:(CJGuestCallViewController *)callController;

/**
 * Received an invalid certificate from the Jabber Guest server.
 *
 * If this is not implemented, an alert view will be presented to the user by default.  They can 
 * choose to not place the call, or to continue.
 */
- (void) onInvalidCert:(NSString*)certFingerprint
         certSubjectCN:(NSString*)certSubjectCN
           referenceID:(NSString*)referenceID
         invalidReason:(NSArray*)invalidReason
subjectCertificateData:(NSString*)subjectCertificateData
intermediateCACertificateData:(NSArray*)intermediateCACertificateData
   invalidCertCallback:(id <CJGuestInvalidCertCallback>)invalidCertCallback;

@end
