#!/bin/bash
##
# Copyright (c) Cisco Systems, Inc.
# All rights reserved.
#
# This script troubleshoot upstream component endpoints within a proxy container.
##

SCRIPTPATH=$(dirname ${0})
source ${SCRIPTPATH}/common_utils.sh
CONTAINER_HOME=/home
USER_AGENT_HEADER="User-Agent: Mozilla/5.0 Troubleshoot Edg/1" 

function check_file_exist_or_exit() {
	if [ ! -f ${1} ]; then
		log_error "File ${1} does not exist, please ensure the file is provided to get trobleshoot results. Exiting..."
		exitall 102
	fi
}

# Function makes curl request to upstream endpoint with write-out parameter and returns then response 
function invoke_api_request() {
	if [[ "${1}" == "finesse" || "${1}" == "ids" ]]; then
		curl -k -s -m 20 -X GET -H "${USER_AGENT_HEADER}" "${2}"
	else
		curl -s -k -m 20 -I -X GET -H "${USER_AGENT_HEADER}" -w "%{http_code},%{time_total}" -o /dev/null "${2}"
	fi
}

# Function that evaluates finesse api response parameters
function evaluate_finesse_status() {
	local response=$( invoke_api_request "${1}" "${2}" )
	local finesse_status=$( echo "${response}" | sed -ne 's:.*<status>\(.*\)</status>.*:\1:p' )
	local response=$( invoke_api_request "" "${2}" )
	local response_time=$( get_response_time "${response}" )
	if [ -z "$finesse_status" ]; then
		finesse_status=$( get_http_status "${response}" )
	fi
    print_response "Internal" "$finesse_status" "${1}" "${3}" "$response_time"
}

# Function that evaluates ids api response parameters
function evaluate_ids_status() {
	local response=$( invoke_api_request "${1}" "${2}" )
	local ids_status=$( echo "${response}" | sed -En 's/.*"state":"([^"]*).*/\1/p' ) 
	local response=$( invoke_api_request "" "${2}" )
	local response_time=$( get_response_time "${response}" )
	if [ -z "$ids_status" ]; then
		ids_status=$( get_http_status "${response}" )
	fi
    print_response "Internal" "$ids_status" "${1}" "${3}" "$response_time"
}

# Function validates upstream components with provided endpoint env file
function validate_upstream() {
	print_header
    for line in $(cat "${1}")
    do
    IFS='=' read -ra ARR <<< "${line}" 
    	if [[ ${ARR[0]} == *"upstream"* ]]; then
    		local component="$(echo ${ARR[0]} | awk -F_ '{print $2}')"
    		local hostname="$( echo "${ARR[1]}" | awk -F/ '{print $3}' )"
        	case "${component}" in 
            	"finesse")
            		evaluate_finesse_status "${component}" "${ARR[1]}" "${hostname}"
            	;;
            	"ids")
            		evaluate_ids_status "${component}" "${ARR[1]}" "${hostname}"
            	;;
            	"idp-adfs3" | "cuic" | "cloudconnect" )
            		local response=$( invoke_api_request "${component}" "${ARR[1]}" )
            		local response_time=$( get_response_time "${response}" )
            		local http_status=$( get_http_status "${response}" )
            		print_response "Internal" "${http_status}" "${component}" "${hostname}" "${response_time}"
            	;;
            	# TODO: Remove below filter once endpoint is set for chat & livedata
            	"chat" | "livedata" )
            	;;
            	*)
            		echo "[WARN] Invalid parameter as component name found: ${1}, skipping...";;
        	esac
        fi
    done
}

main() {
	echo -e "\n************************* Checking Internal Configuration *************************"
    check_file_exist_or_exit ${CONTAINER_HOME}/${1}
    validate_upstream ${CONTAINER_HOME}/${1}
    cleanup ${CONTAINER_HOME}/${1} > /dev/null
    printf "\n"
}

main "$@"
