#!/bin/bash

#
# Function to check whether iptables service is running on system
#
function is_iptables_running() {
    if systemctl is-active --quiet iptables; then
        return 0
    fi
    return 1
}

#
# Function to check whether the given interface exits in the system.
# Input - network interface
# Eg. eth1
#
function check_interface_or_exit(){

  if [ -z ${1} ]; then
    log_error "External Interface is empty."
    exitall 101
  fi

  local interface_present=$(ip -br l | awk '$1 { print $1}' | grep "\<${1}\>")
  if [ -z ${interface_present} ]; then
    log_error "Configured interface ${1} is not present in host system. "
    exitall 100
  fi
}


function get_ips_of_interface() {
  local iface=${1}
  check_interface_or_exit $iface
  local ips=""
  ip_addrs=$(ip -br a show $iface 2>/dev/null | awk '{for(i=3;i<=NF;i++) if ($i ~ /^[0-9]+\./) print $i}' | cut -d'/' -f1)
  if [ -n "$ip_addrs" ]; then
    for ip in $ip_addrs; do
      ips+="$ip,"
    done
  fi
  ips=${ips%,}
  echo $ips
}


function get_ip_of_internal_interfaces() {
  local external_interface
  external_interface="${1}"
  interfaces=$(ip -br l | awk '$1 { print $1 }' | grep -v -E "docker*|podman*|lo|${external_interface}")
  local ips=""
  for iface in $interfaces;
  do
    #Iterate and find all the IP address configured in single nic
    ip_addrs=$(ip -br a show $iface 2>/dev/null | awk '{for(i=3;i<=NF;i++) if ($i ~ /^[0-9]+\./) print $i}' | cut -d'/' -f1)
    if [ -n "$ip_addrs" ]; then
      for ip in $ip_addrs; do
        ips+="$ip,"
      done
    fi
  done
  ips=${ips%,}
  echo $ips
}

# Function to add IP in iptables
function add_ip_in_iptables(){
  local source_ip=${1}
  local destination_ip=${2}
  local command="iptables -t mangle -I ALLOWED_IPS -i ${NGX_EXTERNAL_INTERFACE} -s ${source_ip} -d ${destination_ip} -j ACCEPT -m comment --comment \" Added by installer for ${CONTAINER_NAME} - Allow traffic from ${source_ip} to ${destination_ip}\""
  log_to_file "${command}" "${SCRIPTPATH}/iptables_logs/${CONTAINER_NAME}.log"
  eval "${command}"
}

# Function to delete IP from iptables using rule number
function delete_iptables_rule_using_rule_num() {
  local rule_number_list=${1}
  for rule_num in $rule_number_list
  do
    rule_num=$((rule_num+1))
    local command="iptables -t mangle -D ALLOWED_IPS ${rule_num}"
    log_to_file "${command}" "${SCRIPTPATH}/iptables_logs/${CONTAINER_NAME}.log"
    eval "${command}"
  done
}

function get_installer_added_rule_no_for_source_ip() {
  local source_ip=${1}
  local destination_ip=${2}
  local rule_numbers
  rule_numbers=$(iptables --list ALLOWED_IPS -nv --line-numbers -t mangle | awk -v ip="$source_ip" '$9 == ip' | awk -v ip="$destination_ip" '$10 == ip' | grep "Added by installer" | awk '{ print $1 }')
  echo $rule_numbers
}