#!/bin/bash

SCRIPTPATH=$(dirname ${0})
source ${SCRIPTPATH}/iptables_utilities.sh
source ${SCRIPTPATH}/common_utils.sh
source ${SCRIPTPATH}/upgrade_utils.sh
source ${SCRIPTPATH}/primary_validation_utils.sh
source ${SCRIPTPATH}/other_validation_utils.sh

INSTALLER_ENV=""

declare -a INSTALLER_ENV_FILES
declare -A IPTABLES_DISTINCT_PORTS
declare -A PORT_TO_IPTABLES_RATELIMITS_MAP
declare -A IPTABLES_RATELIMITS_TO_VALUE_MAP
DEPLOYMENT_ENV_DIR=""
PODMAN_RUN_ARGS=""
REVERSE_PROXY_OPENRESTY_CONFIGS_DIR="${SCRIPTPATH}/reverse-proxy-openresty-configs"
REVERSE_PROXY_OPENRESTY_CONTAINER_DIR="${SCRIPTPATH}/reverse-proxy-openresty-container"
REVERSE_PROXY_CONFIG_UPGRADE_SCRIPTS_DIR="${SCRIPTPATH}/migration"
REVERSE_PROXY_CONTAINER_IMAGE_TAR=${REVERSE_PROXY_CONTAINER_IMAGE_TAR:="${REVERSE_PROXY_OPENRESTY_CONTAINER_DIR}/reverse-proxy-openresty-container.tar.gz"}
LOAD_CONTAINER_IMAGE_FROM_TAR=${LOAD_CONTAINER_IMAGE_FROM_TAR:="true"}

nginx_user_userid="root"
nginx_user_usergroup="root"

# Setting this flag will automatically confirm with "Y".
AUTO_PROMT_CONFIMATION=false

declare -A TEST_ORIGINAL_PORT_TO_TEST_PORT_MAP

# Set an env to start a test container
function set_test_run_container_env() {
	set -o allexport
	export START_TEST_CONTAINER="true"
	set +o allexport
}

# Clear start test container env
function clear_test_run_container_env() {
	set -o allexport
	export START_TEST_CONTAINER="false"
	set +o allexport
}

# To get nginx username
function define_nginx_user() {
	if enforce_selinx; then
		nginx_user_userid=${NGX_USER_USERID}
		nginx_user_usergroup=${NGX_USER_USERGROUP}
	fi
	log_info "Setting nginx_user_userid as ${nginx_user_userid} and nginx_user_usergroup as ${nginx_user_usergroup}"
}

# Update the env values to start test container or normal container.
# Param: container name.
# Based on the container name, the host volume and other directory env values are exported.
function update_installer_envs() {
	local container_name=$1
	local working_dir=$2
	set -o allexport
	export CONTAINER_NAME="${container_name}"
	export HOST_WORKING_DIR=${working_dir:?}
	export HOST_CACHE_VOL=${HOST_WORKING_DIR}/cache
	export HOST_SSL_VOL=${HOST_WORKING_DIR}/ssl
	export HOST_LOGS_VOL=${HOST_WORKING_DIR}/logs
	export HOST_CONF_VOL=${HOST_WORKING_DIR}/conf
	export HOST_HTML_VOL=${HOST_WORKING_DIR}/html
	export HOST_LUA_VOL=${HOST_WORKING_DIR}/lua
	export ENV_CLONE=${HOST_WORKING_DIR}/envClone
	export FAST_CGI_TEMP=${HOST_WORKING_DIR}/fastcgi_temp
	export UWSGI_TEMP=${HOST_WORKING_DIR}/uwsgi_temp
	export SCGI_TEMP=${HOST_WORKING_DIR}/scgi_temp
	set +o allexport
}

# Function to replace test port values with the real port values.
function replace_test_port_with_real_port() {
	local NEW_CONFIG_FILE
	NEW_CONFIG_FILE=$1
	for TEST_PORT_KEY in "${!TEST_PORT_TO_ORIGINAL_PORT_MAP[@]}"; 
	do
		local org_port=${TEST_PORT_TO_ORIGINAL_PORT_MAP[${TEST_PORT_KEY}]}
		log_info "Replacing test port ${TEST_PORT_KEY} with original port ${org_port}."
		sed -i "s/${TEST_PORT_KEY}/${org_port}/g" "${NEW_CONFIG_FILE}"
	done			
}

# Function to dispaly config diffs for the given 2 files.
function display_config_diff_to_user() {
	local CURRENT_CONFIG_FILE
	local NEW_CONFIG_FILE
	local output
	CURRENT_CONFIG_FILE=$1
	NEW_CONFIG_FILE=$2

	# display the diff with existing configs - compare and display the changes
	output=$(diff -w "${CURRENT_CONFIG_FILE}" "${NEW_CONFIG_FILE}" | wc -l)
	tmp_diff_file=$(get_new_temp_file)
	set_trap_with_preserve "cleanup ${tmp_diff_file}" EXIT
	if [ "$output" -eq 0 ]; then
		log_error "There is no config change to reload. Existing."
		exitall 101
	fi
	log_warn "Find the difference between the existing config and new proposed change, and proceed."
	log_warn "Note: The hot reload will be instructing the running instance to reload its configuration without interrupting any active connections."
	diff -w "${CURRENT_CONFIG_FILE}" "${NEW_CONFIG_FILE}" > "$tmp_diff_file"
	log_warn "The difference between the current and existing configs are stored on the following file $tmp_diff_file, please check and confirm with yes to deploy."

}

function enforce_selinx() {
	local ignore_selinx=$(get_prop_val 'NGX_IGNORE_SELINUX' "${DEPLOYMENT_ENV_DIR}/core.env")
	if [ "${ignore_selinx}" == "true" ]; then
		return 1
	fi
	return 0
}

# Read from the temp file which has the test_run port to original port mapping.
function read_test_port_map_and_clear_file() {
	local fromfile
	fromfile=$1
	if [ ! -f "${fromfile}" ]; then
		log_error "There is no such file [${fromfile}] to read the test port mapping info."
		exitall 104
	fi
	log_info "Reading test port mapping info from ${fromfile}"
	while IFS= read -r input
	do
		# Split the string into key and value
		key="${input%%=*}"
		value="${input#*=}"

		# Trim leading/trailing whitespace from the key and value
		key="${key#"${key%%[![:space:]]*}"}"
		key="${key%"${key##*[![:space:]]}"}"
		value="${value#"${value%%[![:space:]]*}"}"
		value="${value%"${value##*[![:space:]]}"}"
		
		TEST_PORT_TO_ORIGINAL_PORT_MAP[${key}]="$value"
		TEST_ORIGINAL_PORT_TO_TEST_PORT_MAP[${value}]="$key"
	done < "${fromfile}"
	#:? is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
	rm -rf "${fromfile:?}"
}

# Clean up the test container and it's working directory.
function cleanup_podman_container() {
	log_info "Stopping podman container $1"
	podman stop $1
	log_info "Removing working directory of container $2"
	#It is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
	rm -rf ${2:?}
}


# Set the resouce recommendations
function set_resource_recommendation() {
	set -o allexport
	export CPU_LIMIT="$1"
	export MEM_LIMIT="${2}G"
	export MEM_SWAP_LIMIT="${3}G"
	export MEM_RES="${4}G"
	export NOFILE_LIMIT="nofile=${5}:${5}"
	set +o allexport
}

# This value cannot be configured less than 102400 for any installations.
function check_no_file_limit() {
	if [[ $(echo "${NOFILE_LIMIT}" | awk -F '=' '{print $2}' | awk -F ':' '{print $1}') -lt 102400 ]]; then
		log_error "NOFILE_LIMIT in installer.env should be more than or equal to 102400"
		exitall 215
	fi
}

# Get the value of NOFILE_LIMIT's first value.
function get_current_nofile_limit() {
	echo "${NOFILE_LIMIT}" | awk -F '=' '{print $2}' | awk -F ':' '{print $1}'
}

# Function to set the system recommended resources on the installer.env file.
function update_resouce_recommendation_in_installer_envs_file() {
	log_info "Updating installer.env file."
	sed -i '/CPU_LIMIT/d' "${INSTALLER_ENV}"
	sed -i '/MEM_LIMIT/d' "${INSTALLER_ENV}"
	sed -i '/MEM_SWAP_LIMIT/d' "${INSTALLER_ENV}"
	sed -i '/MEM_RES/d' "${INSTALLER_ENV}"
	sed -i '/NOFILE_LIMIT/d' "${INSTALLER_ENV}"
	echo CPU_LIMIT="$CPU_LIMIT" >>"${INSTALLER_ENV}"
	echo MEM_LIMIT="$MEM_LIMIT" >>"${INSTALLER_ENV}"
	echo MEM_SWAP_LIMIT="$MEM_SWAP_LIMIT" >>"${INSTALLER_ENV}"
	echo MEM_RES="$MEM_RES" >>"${INSTALLER_ENV}"
	echo NOFILE_LIMIT="$NOFILE_LIMIT" >>"${INSTALLER_ENV}"
}

# General function to confirmation from user to retain the configured properties or override with the system recommendation.
# Based on user input, it takes the user entered values or system recommended values.
function check_configs_and_get_user_confirmation() {
	local prompt_message
	prompt_message=$'Container is being configured with '"${6}"$' resources than the required. Below are the recommended configurations: \n 	CPU_LIMIT='"${1}"$'.0 \n 	MEM_LIMIT='"${2}"$'G \n 	MEM_SWAP_LIMIT='"${3}"$'G \n 	MEM_RES='"${4}"$'G \n 	NOFILE_LIMIT=nofile='"${5}"$':'"${5}"$' \nWould you like to change the proceed with the following configurations? \nPress (Y|y) to confirm to take the recommended values, else press any other key to ingore the recommendation and proceed.'
	if get_user_confirmation "${prompt_message}" "${AUTO_PROMT_CONFIMATION}"; then
		set_resource_recommendation "$1" "$2" "$3" "$4" "$5"
		update_resouce_recommendation_in_installer_envs_file
	fi
}



# This is the last time stamp of the error log.
# We are interested on all the error/crit/alert/emerg errors afters this specific time.
# If the file is not present or empty - current time is taken as start time.
function get_last_timestamp() {
	local error_file
	error_file=$1
	local current_time
	if [ ! -f "$error_file" ]; then
    	log_info "No file found in name $error_file"
		current_time=$(date +'%Y/%m/%d %H:%M:%S')
		echo "$current_time"
		return 0
	fi
	local timestamp
	timestamp=$(tail -1 ${error_file} | awk '{print $1,$2}' | xargs)
	# If some content in there in log file
	if [ -z "$my_variable" ]; then
		echo $timestamp
		return 0
	fi
	current_time=$(date +'%Y/%m/%d %H:%M:%S')
	echo "$current_time"
	return 0
}

# Function to print the error, crit, alert and emerg messages from the error logs given.
# Note: The file might contain even old stale logs, we are interested on the error messages that are happened after the script started.
function print_error_logs_after() {
	local file
	local time_after
	file=$1
	time_after=$2
	if [ ! -f "$file" ]; then
    	log_error "No file found in name $file. Exiting"
		exitall 104
	fi
	# Get the line number on which the date is present
	local line_refernce
	line_refernce=$(grep -m 1 -n "${time_after}" "${file}" | cut -d ':' -f 1 | xargs)
	if [ -z "${line_refernce}" ]; then
		line_refernce=1
	fi
	log_info "Please check error message from ngnix logs that might have caused failure."
	echo "--------------------------------------------------------------------------------"
	tail -n +${line_refernce} ${file} | grep -P '\[error\]|\[crit\]|\[alert\]|\[emerg\]'
	echo "--------------------------------------------------------------------------------"
}

#Wait for test container to comeup
function wait_for_container_to_comeup() {
	local container_name
	local is_test_container
	local wait_time
	wait_time=$2
	container_name=$1
	if [ ! -z "${3}" ]; then
		is_test_container="test "
	fi
	log_info "Waiting for ${is_test_container}reverse proxy container to come up.."
	# Wait for 30 seconds and check container in running condition, then we are sure the configs are fine.
	wait_and_print_dots $wait_time

	if ! is_container_running "$container_name"; then
		log_error "Container $container_name is not running. Exiting."
		print_error_logs_after "${HOST_LOGS_VOL}/error.log" "$START_TIME"
		exitall 209
	fi
}




# Function to set a new trap while preserving the previous trap behavior
function set_trap_with_preserve() {
  local new_trap="$1"
  local signal="$2"
  # log_info "Adding trap for signal $signal. Command to execute: $new_trap"
  # Get the previous trap command for the signal
  local previous_trap
  previous_trap=$(trap -p "$signal" | awk -F"'" '{print $2}')

  # Set the new trap and include the previous trap command
  trap "$new_trap; $previous_trap" "$signal"
}

# Function to store the openresty configs of a container in given output file
function store_container_configs_in_file() {
	local out_file
	local container_name
	container_name=$1
	out_file=$2
	log_info "Echoing the [${container_name}] container's openresty config into a file ${out_file}"
	set_trap_with_preserve "cleanup ${out_file}" EXIT
	get_openresty_config "${container_name}" "${out_file}"
}

# Start test container with given container name, working dir.
function start_test_container() {
	local test_container_name
	local test_working_dir
	local real_working_dir
	test_container_name=${1}
	test_working_dir=${2}
	real_working_dir=${3}
	set_test_run_container_env
	update_installer_envs "${test_container_name}" "${test_working_dir}"
	create_directories
	# Copy ssl files to the test container
	cp -r "${real_working_dir}"/ssl/* "${HOST_SSL_VOL}/"
	log_info "Starting test reverse proxy container. Using test container's working directory as ${test_working_dir}"
	start
}

# Reload openresty configs in exiting container container
function reload_new_openresty_configs() {
	local real_container_name
	local real_working_dir
	real_container_name=$1
	real_working_dir=$2
	clear_test_run_container_env
	log_info "Going to reload the config change."
	update_installer_envs "${real_container_name}" "${real_working_dir}"
	log_info "Updated the configs in podman container."
	# place the configs under the existing container
	configure_openresty_configs "true"
	create_ssl_cert
	# Reload the configs
	log_info "Reloading openresty configs."
	podman exec -it "${real_container_name}" /usr/local/openresty/nginx/sbin/openresty_launcher.sh reload
	# Clear cache
	log_info "Clearing cache."
	clear_cache
}

function should_auto_restart_container() {
    if [ "${AUTO_RESTART_CONTAINER}" == "1" ]; then
        return 0
    fi
    return 1
}

function is_host_network() {
	local lowercase_value="${CONTAINER_NETWORK_MODE,,}"
    if [[ "$lowercase_value" == "host" ]]; then
        return 0
    fi
    return 1
}

function run_command_or_exit() {
	"$@" 2> /dev/null
    if [ $? -ne 0 ]; then
        log_warn "Command failed: $*"
		log_warn "Setting up auto restart of container failed."
        exitall 243
    fi
}

function handle_auto_start() {
	if should_auto_restart_container; then
		log_info "Creating podman systemd unit file"
		run_command_or_exit podman generate systemd --new --name ${CONTAINER_NAME} -f 
		
		run_command_or_exit mv -f container-${CONTAINER_NAME}.service /etc/systemd/system/
		
		run_command_or_exit chcon -t systemd_unit_file_t /etc/systemd/system/container-${CONTAINER_NAME}.service
		run_command_or_exit restorecon -v /etc/systemd/system/container-${CONTAINER_NAME}.service
		
		run_command_or_exit systemctl enable podman-restart.service
		run_command_or_exit systemctl start podman-restart.service

		run_command_or_exit systemctl enable podman.service
		run_command_or_exit systemctl start podman.service

		run_command_or_exit systemctl enable container-${CONTAINER_NAME}.service
		run_command_or_exit systemctl start container-${CONTAINER_NAME}.service
		log_info "Successfully enabled auto restart of the container"
	fi
}

function create_directories() {
	log_info "Creating required directories "
	mkdir -p "${HOST_CACHE_VOL}" "${HOST_SSL_VOL}" "${HOST_LOGS_VOL}" "${HOST_CONF_VOL}" "${HOST_HTML_VOL}" "${HOST_LUA_VOL}" "${HOST_RSYSLOG_CONF_VOL}" "${HOST_RSYSLOG_CONF_VOL}/tmp" "${ENV_CLONE}" "${FAST_CGI_TEMP}" "${SCGI_TEMP}" "${UWSGI_TEMP}"
}

function get_openresty_config() {
	local out_file
	local container_name
	container_name="${1}"
	out_file="${2}"

	log_info "Creating exiting openresty config from container $container_name and placing it in $out_file" 
	podman exec -it "$container_name" openresty -T > "$out_file" 
	if [ -z "$(cat ${out_file})" ]; then
		log_error "Not able to get the openresty config from container ${container_name}. Exiting."
		exitall 210
	fi
}

function configure_openresty_configs() {
	local reload
	reload=$1
	TMP_TEST_PORT_MAPPING="$(get_new_temp_file)"
	# Create a temp file and send it as an arg to child shell to collect back the test to original mapping.
	sh -c "${REVERSE_PROXY_OPENRESTY_CONFIGS_DIR}/run.sh ${DEPLOYMENT_ENV_DIR} ${TMP_TEST_PORT_MAPPING}"
	rc=$?
	# Set up the trap to delete the temporary file on script exit
	set_trap_with_preserve "cleanup $TMP_TEST_PORT_MAPPING" EXIT
    if [ $rc -ne 0 ]; then
 		log_error "Error generating openresty configs."
		exitall 1
	fi
	# After successful completion, read the temp file to get the test to original port mapping.
	read_test_port_map_and_clear_file "${TMP_TEST_PORT_MAPPING}"
	# Creating tmp directory for rsyslog buffering
	mkdir -p "${REVERSE_PROXY_OPENRESTY_CONFIGS_DIR}"/configs_out/rsyslog.d/tmp
	for DIR in "${REVERSE_PROXY_OPENRESTY_CONFIGS_DIR}"/configs_out/*; do
		DIR_NAME=$(basename "${DIR}")
		if [[ "${DIR_NAME}" == "html" ]]; then
			# Don't clear target html directory content as it might have customer 
			# content too. Only force-copy our content and retain other things
			log_info "Copying static content from ${DIR} to ${HOST_WORKING_DIR}/${DIR_NAME}"
			\cp -fp "${DIR}"/* "${HOST_WORKING_DIR}"/"${DIR_NAME}"/
			#:? is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
			rm -rf "${DIR:?}/"
		else
			if [[ "${reload}" == "true" ]]; then
				log_info "Copying files from ${DIR} to ${HOST_WORKING_DIR}/${DIR_NAME}"
				yes | cp -Rfp "${DIR}" "${HOST_WORKING_DIR}/"
			else
				# This will copy conf and lua directories
				log_info "Moving files from ${DIR} to ${HOST_WORKING_DIR}/${DIR_NAME}"
				rm -rf "${HOST_WORKING_DIR:?}/${DIR_NAME:?}"
				mv -f "${DIR:?}/" "${HOST_WORKING_DIR:?}/${DIR_NAME:?}"
			fi
		fi
	done
	#It is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
	# Remove the existing envClone dir contents if any
	rm -rf ${ENV_CLONE:?}/*
	yes | cp -Rfp ${DEPLOYMENT_ENV_DIR}/* ${ENV_CLONE}
	# Change permission to all files recursivily. Need other users write permission as the container is going to be started with different user.
	find "${HOST_WORKING_DIR}"/ -type f -exec chmod 775 {} +
	log_info "Selinux is enforced, adding required permissions for the ngnix config files"
	rm -rf ${FAST_CGI_TEMP:?}
	rm -rf ${SCGI_TEMP:?}
	rm -rf ${UWSGI_TEMP:?}
	mkdir -p "${FAST_CGI_TEMP}" "${SCGI_TEMP}" "${UWSGI_TEMP}"
	find "${HOST_WORKING_DIR}"/ -type d -exec chown -R ${nginx_user_userid}:${nginx_user_usergroup} {} +
}

#
# Function to disable ratelimit for trusted ips.
# 1. If ip is not already added in iptables, simply add it.
# 2. If ip is already added in iptables (This could happen in multiple container scenario),
#     a) Find the rule number for that ip in iptables and increment it by 1.
#     b) Add ip in ip tables at the top.
#     c) Delete the rule at aforementioned rule number for older iptables rule.
#
function add_trusted_ips_in_iptables(){
  log_info "Allowing ips in iptables rule: ${1}"

  IFS='| ' read -r -a ips <<< ${1}
  for ip in "${ips[@]}"
  do
    local ip_rule_number
    ip_rule_number=$(get_installer_added_rule_no_for_source_ip "${ip}" "${EXTERNAL_IP}")

    add_ip_in_iptables "${ip}" "${EXTERNAL_IP}"

    if [ -n "$ip_rule_number" ]; then
      delete_iptables_rule_using_rule_num "${ip_rule_number}"
    fi
  done
}

#
# Function to add iptables rules for env files
# All *.env related validations have already run in run.sh
#
function configure_iptables_rules(){

  source ${DEPLOYMENT_ENV_DIR}/core.env

  if [ ${NGX_IPTABLES_HARDENING} -ne "1" ]; then
    return 0
  fi

  source /etc/reverse-proxy/network.env
  NGX_EXTERNAL_INTERFACE=$EXTERNAL_INTERFACE
  EXTERNAL_IP=$(get_external_ip)

  mkdir -p iptables_logs
  touch ${SCRIPTPATH}/iptables_logs/${CONTAINER_NAME}.log
  truncate -s 0 ${SCRIPTPATH}/iptables_logs/${CONTAINER_NAME}.log

  local restrict_traffic="false"

  if [ "${NGX_TRAFFIC_RESTRICTED}" == "true" ]; then
    if [ ! -z ${NGX_LOAD_BALANCER_IPS} ]; then
      restrict_traffic="true"
    else
      log_warn "NGX_LOAD_BALANCER_IPS is empty. It should contain load balancer ips when NGX_TRAFFIC_RESTRICTED is true. Ignoring NGX_TRAFFIC_RESTRICTED."
    fi
  fi

  if [ "${restrict_traffic}" == "false" ]; then
    for ENV in ${DEPLOYMENT_ENV_DIR}/*.env; do
        if [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/core.env" ] || [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/dirs.env" ]; then
          continue
        else
          configure_iptable_ports_for_env ${ENV}  #populate iptables limits map
          add_cloudconnect_client_ips_in_iptables ${ENV}
          add_idp_ip_in_iptables ${ENV}
        fi
    done

    #loop for iptables port map and add iptables rules from iptables limits map
    for port in "${!IPTABLES_DISTINCT_PORTS[@]}"
    do
      add_ratelimit_rules_for_port ${port}
    done

    add_trusted_ips_in_iptables "${NGX_RATELIMIT_DISABLE_IPS}"

  fi

  if [ ! -z ${NGX_LOAD_BALANCER_IPS} ]; then
    add_trusted_ips_in_iptables "${NGX_LOAD_BALANCER_IPS}"
  fi

  if should_auto_restart_container; then
    save_iptables_rules
  fi

}

function save_iptables_rules(){
  iptables-save -t mangle > /etc/sysconfig/iptables
  if [ $? -ne 0 ]; then
    log_error "Error saving iptables rules."
  fi
}

function add_cloudconnect_client_ips_in_iptables(){
  local ENV
  ENV=${1}
  local template_type
  template_type=$(get_prop_val '^TEMPLATE_TYPE' ${ENV})
  if [ "${template_type}" == "cloudconnect" ]; then
    local NGX_CLOUDCONNECT_CLIENT_IPS_VALUE
    NGX_CLOUDCONNECT_CLIENT_IPS_VALUE=$(get_prop_val "NGX_CLOUDCONNECT_CLIENT_IPS" "${ENV}" )
      if [ ! -z ${NGX_CLOUDCONNECT_CLIENT_IPS_VALUE} ]; then
        add_trusted_ips_in_iptables "${NGX_CLOUDCONNECT_CLIENT_IPS_VALUE}"
      fi
  fi
}


# Function to resolve and add IDP ip in iptables
function add_idp_ip_in_iptables(){
  local ENV
  ENV=${1}
  local template_type
  template_type=$(get_prop_val '^TEMPLATE_TYPE' ${ENV})
  if [ "${template_type}" == "idp-adfs3" ]; then
    local NGX_IDP_HOSTNAME_VALUE
    NGX_IDP_HOSTNAME_VALUE=$(get_prop_val "NGX_IDP_HOSTNAME" "${ENV}" )
    if [ ! -z ${NGX_IDP_HOSTNAME_VALUE} ]; then
      NGX_IDP_HOSTNAME_IP=$(get_upstream_ip ${NGX_IDP_HOSTNAME_VALUE} "${CONTAINER_DNS_RESOLVER}" )
      add_trusted_ips_in_iptables "${NGX_IDP_HOSTNAME_IP}"
    fi
  fi
}

#
# Function to add iptables rules with ratelimit 
#
function add_ratelimit_rules_for_port(){
  log_info "Adding iptables rule for ${1} port."
  local IPTABLES_PORT
  IPTABLES_PORT=${1}
  local IPTABLES_CONNECTION_LIMIT_ABOVE
  local IPTABLES_HASH_LIMIT_UPTO
  local IPTABLES_HASH_LIMIT_BURST
  local IPTABLES_LOG_LIMIT_BURST
  local IPTABLES_LOG_LIMIT
  IPTABLES_CONNECTION_LIMIT_ABOVE=${PORT_TO_IPTABLES_RATELIMITS_MAP["${IPTABLES_PORT}_IPTABLES_CONNECTION_LIMIT_ABOVE"]}
  IPTABLES_LOG_LIMIT_BURST=${PORT_TO_IPTABLES_RATELIMITS_MAP["${IPTABLES_PORT}_IPTABLES_LOG_LIMIT_BURST"]}
  IPTABLES_LOG_LIMIT=${PORT_TO_IPTABLES_RATELIMITS_MAP["${IPTABLES_PORT}_IPTABLES_LOG_LIMIT"]}
  IPTABLES_HASH_LIMIT_UPTO=${PORT_TO_IPTABLES_RATELIMITS_MAP["${IPTABLES_PORT}_IPTABLES_HASH_LIMIT_UPTO"]}
  IPTABLES_HASH_LIMIT_BURST=${PORT_TO_IPTABLES_RATELIMITS_MAP["${IPTABLES_PORT}_IPTABLES_HASH_LIMIT_BURST"]}

  tac ${SCRIPTPATH}/iptables_per_component.conf | while IFS= read line; do
    prefix="#"
    if ! [[ "$line" =~ ^"$prefix" ]]; then
    local rule
      rule=$(echo ${line} | sed -e "s~IPTABLES_EXTERNAL_INTERFACE~${NGX_EXTERNAL_INTERFACE}~g;s~EXTERNAL_IP~${EXTERNAL_IP}~g;s~IPTABLES_PORT~${IPTABLES_PORT}~g;s~IPTABLES_CONNECTION_LIMIT_ABOVE~${IPTABLES_CONNECTION_LIMIT_ABOVE}~g;s~IPTABLES_LOG_LIMIT_BURST~${IPTABLES_LOG_LIMIT_BURST}~g;s~IPTABLES_LOG_LIMIT~${IPTABLES_LOG_LIMIT}~g;s~IPTABLES_HASH_LIMIT_UPTO~${IPTABLES_HASH_LIMIT_UPTO}~g;s~IPTABLES_HASH_LIMIT_BURST~${IPTABLES_HASH_LIMIT_BURST}~g;s~CONTAINER_NAME~${CONTAINER_NAME}~g")
      log_and_evaluate "${rule}"
    fi
  done

}

#
# This function takes port as input and populates IPTABLES_DISTINCT_PORTS, PORT_TO_IPTABLES_RATELIMITS_MAP maps
# 1. IPTABLES_DISTINCT_PORTS -> This map contains the distinct ports configured in ENV files.
# 2. PORT_TO_IPTABLES_RATELIMITS_MAP : 
#     a) This map contains the key:value mapping of port_IPTableLimit:IptablesLimitValue. 
#         Eg. 8445_IPTABLES_CONNECTION_LIMIT_ABOVE:8
#             8553_IPTABLES_LOG_LIMIT:"1/min"
#     b) In case of common ports, it adds the value configured for iptable limits.
#
function populate_iptables_map(){
  local port=${1}
  local ENV=${2}
  if [ -z ${IPTABLES_DISTINCT_PORTS["${port}"]} ]; then
    log_info "Port ${port} added to IPTABLES_DISTINCT_PORTS map."
    IPTABLES_DISTINCT_PORTS["${port}"]=1
  else
    log_info "Port ${port} already present in IPTABLES_DISTINCT_PORTS map."
  fi

  IPTABLES_RATELIMITS_TO_VALUE_MAP["IPTABLES_CONNECTION_LIMIT_ABOVE"]=$(get_prop_val_clean 'IPTABLES_CONNECTION_LIMIT_ABOVE' "${ENV}")
  IPTABLES_RATELIMITS_TO_VALUE_MAP["IPTABLES_LOG_LIMIT_BURST"]=$(get_prop_val_clean 'IPTABLES_LOG_LIMIT_BURST' "${ENV}")
  IPTABLES_RATELIMITS_TO_VALUE_MAP["IPTABLES_LOG_LIMIT"]="$(get_prop_val_clean 'IPTABLES_LOG_LIMIT' "${ENV}")"
  IPTABLES_RATELIMITS_TO_VALUE_MAP["IPTABLES_HASH_LIMIT_UPTO"]="$(get_prop_val_clean 'IPTABLES_HASH_LIMIT_UPTO' "${ENV}")"
  IPTABLES_RATELIMITS_TO_VALUE_MAP["IPTABLES_HASH_LIMIT_BURST"]=$(get_prop_val_clean 'IPTABLES_HASH_LIMIT_BURST' "${ENV}")

  for key in "${!IPTABLES_RATELIMITS_TO_VALUE_MAP[@]}"
  do
    if [ -z ${PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]} ]; then
      PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]=${IPTABLES_RATELIMITS_TO_VALUE_MAP[${key}]}
    else
      if [ ${key} == "IPTABLES_LOG_LIMIT" ] || [ ${key} == "IPTABLES_HASH_LIMIT_UPTO" ]; then
        local value_in_map=$(echo ${PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]} | sed 's/\/.*//')
        local value_in_ENV_file=$(echo ${IPTABLES_RATELIMITS_TO_VALUE_MAP[${key}]} | sed 's/\/.*//')
        local time_unit=$(echo ${PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]} | sed 's/.*\///')
        PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]="$(( $value_in_map + $value_in_ENV_file ))/${time_unit}"
        continue
      fi
      PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]=$(( ${PORT_TO_IPTABLES_RATELIMITS_MAP["${port}_${key}"]} + ${IPTABLES_RATELIMITS_TO_VALUE_MAP[${key}]} ))
    fi
  done

}

#
# Function to configure iptables rules for a given env file.
# 1. It populates IPTABLES_DISTINCT_PORTS, PORT_TO_IPTABLES_RATELIMITS_MAP maps with port and iptable rule limits.
# 2. It adds NGX_CLOUDCONNECT_CLIENT_IPS to iptables in case of cloudconnect.env
#
function configure_iptable_ports_for_env(){
  local ENV
  ENV=${1}
  log_info "Configuring iptables ports for ${ENV}"

  local template_type
  template_type=$(get_prop_val '^TEMPLATE_TYPE' ${ENV})
  local result_template_type
  result_template_type=$(get_component $template_type)
  local portkey
  portkey=$(echo "NGX_PRXY_${result_template_type}_PORT")
  local portValue
  portValue=$(get_prop_val "^${portkey}" ${ENV})
  populate_iptables_map ${portValue} ${ENV}

  if [ "${template_type}" == "cuic_1261" ]; then
    local portkey
    portkey=$(echo "NGX_PRXY_${result_template_type}_DOC_PORT")
    local portValue
    portValue=$(get_prop_val "^${portkey}" ${ENV})
    populate_iptables_map ${portValue} ${ENV}
  fi

  if [ "${template_type}" == "livedata_1261" ]; then
    local portkey
    portkey=$(echo "NGX_PRXY_${result_template_type}_SCKT_IO_PORT")
    local portValue
    portValue=$(get_prop_val "^${portkey}" ${ENV})
    populate_iptables_map ${portValue} ${ENV}
  fi

  if [ "${template_type}" == "chat" ]; then
    local portList
    portList=$(grep 'NGX_CHAT_HOST._PROXY' ${ENV} | grep -v "NGX_CHAT_HOST1_PROXY" | cut -d ':' -f2 | sed 's/"$//')
    for port in ${portList[@]}
    do
      populate_iptables_map ${port} ${ENV}
    done
  fi
}

function log_and_evaluate() {
  local command=${1}
  echo `date "+%m/%d/%Y %H:%M:%S :"` "${command}" >> ${SCRIPTPATH}/iptables_logs/${CONTAINER_NAME}.log
  eval "${command}"
}



function create_ssl_cert() {
	# SSL certs will not be overwritten if already present
	# This is done to avoid regenerating certs all the time as proxy certs are 
	# uploaded to component trust store which needs to be updated if regenerated
	if [ "${CREATE_SELF_SIGNED_SSL_CERT}" == "true" ] && [ ! -f "${HOST_SSL_VOL}/${SSL_CERT_NAME}" ] && [ ! -f "${HOST_SSL_VOL}/${SSL_KEY_NAME}" ]; then
		log_info "Creating SSL certs in directory ${HOST_SSL_VOL}"
		openssl req -x509 -newkey rsa:"${SSL_CERT_KEY_LENGTH}" -nodes -out "${HOST_SSL_VOL}"/"${SSL_CERT_NAME}" -keyout "${HOST_SSL_VOL}/${SSL_KEY_NAME}" -days "${SSL_CERT_EXPIRY_IN_DAYS}" -subj "${CERTIFICATE_SUBJECT}"
		# Decrypt key
		openssl rsa -in "${HOST_SSL_VOL}/${SSL_KEY_NAME}" -out "${HOST_SSL_VOL}/${SSL_KEY_NAME}.decrypted"
		mv -f "${HOST_SSL_VOL}/${SSL_KEY_NAME}.decrypted" "${HOST_SSL_VOL}/${SSL_KEY_NAME}"
	fi
}



function configure_podman_arg_from_list() {
	local list
	local delimiter
	local podman_arg_name=
	list="${1}"
	delimiter="${2}"
	podman_arg_name="${3}"

	IFS="${delimiter} " read -r -a entries <<< ${list}
	for entry in "${entries[@]}"
	do
		PODMAN_RUN_ARGS="${PODMAN_RUN_ARGS} ${podman_arg_name} ${entry}"
	done
}

# Function to to get the port mapping info to be used in podman start command
# grep all the port values configured and form the string
function generate_port_mappings() {
    local dir="$1"
    local ip="$2"
    local private_ip="$3"
    local result=""
  	# Find all ports used for this installation and form the port mappting
    local ports=$(grep -r '^[^#]*PRXY_.*_PORT' "$dir" --exclude=core.env | sed -E 's/^[^#]*PRXY_[^=]*_PORT *= *"?([^"]*)"?/\1/' | sort | uniq)
    local chat_ports=$(grep -r '^[^#]*CHAT_.*_PROXY' "$dir" | grep -v "HOST1" | sed -E 's/^[^#]*CHAT_[^=]*_PROXY *= *"?([^"]*)"?/\1/' | cut -d ':' -f2)

    local port_list=$(echo "$ports" "$chat_ports" | tr ' ' '\n' | sort -nu)

    for port in $port_list; do
        if [ "${START_TEST_CONTAINER}" == "true" ]; then
            port=${TEST_ORIGINAL_PORT_TO_TEST_PORT_MAP[${port}]}
        fi
        check_ip_and_port_is_free_to_bind "${ip}:${port}"
        result+=" -p ${ip}:${port}:${port}"
        result+=" -p ${private_ip}:${port}:${port}"
    done

    # Skip the following when we don't have private IP
    # It was used to host proxymap api which will be bounded with private IP
    if [ -z "$private_ip" ]; then
        echo "$result"
        return 0
    fi

    # Private IP is present on the setup
    # Find the ports mentioned in core.env to host proxymap and status APIs along with the other port mapping
    ports=$(grep -r '^[^#]*PRXY_.*_PORT' "$dir/core.env" | sed -E 's/^[^#]*PRXY_[^=]*_PORT *= *"?([^"]*)"?/\1/' | sort | uniq)

    for port in $ports; do
        if [ "${START_TEST_CONTAINER}" == "true" ]; then
            port=${TEST_ORIGINAL_PORT_TO_TEST_PORT_MAP[${port}]}
        fi
        check_ip_and_port_is_free_to_bind "${private_ip}:${port}"
        result+=" -p ${private_ip}:${port}:${port}"
    done
    echo "$result"
}


# Function to check and exit if the ip:port is being used already
function check_ip_and_port_is_free_to_bind() {
	local ipport
	ipport=${1}
	ss -tln | grep -w "${ipport}" > /dev/null
    if [ $? -eq 0 ]; then
		#Check any container using that host:port - if so print the container name and exit
        local container_list=$(podman ps -q)
		for container_id in $container_list; do
			if [ -n "$container_id" ]; then
				local container_info=$(podman inspect $container_id | grep -w "${ipport}")
				if [ -n "$container_info" ]; then
					local container_name=$(podman inspect --format '{{.Name}}' $container_id)
					log_error "${ipport} is already used by the container \"${container_name}\". \nUse different IP and port to start the container. Exiting."
					exitall 295
				fi
			fi
		done
		#Check any other process is using that host:port
		local pid_info=$(ss -tlnp | grep -w "$ipport")		# -p get the process id
    	local pid=$(echo "$pid_info" | awk '{print $6}' | cut -d',' -f2 | cut -d'=' -f2)
    	local process_name=$(ps -p "$pid" -o cmd --no-headers)
    
        log_error "${ipport} is already used by the process \"${process_name}\". \nUse different IP and port to start the container. Exiting."
		exitall 295
    fi
}

function prepare_podman_run_args() {
        # Disable firewalld - harmless command if its already disabled.
	systemctl disable firewalld

	local RESTART_OPTION
	RESTART_OPTION=" --rm=true "
	if should_auto_restart_container; then
		RESTART_OPTION=" --restart unless-stopped "
	fi


	local bind_mounts=""
	local security_opts_disabled="--security-opt label=disable "
	local user_args=""
	local add_or_drop_cap=""
	local tmp_files_mapping=""
	local nginx_user_mapping=""
	source ${DEPLOYMENT_ENV_DIR}/core.env
	if [ "${NGX_IGNORE_SELINUX}" == "false" ]; then
		# Primary validations are already done.
		# Here we know selinux on the host is enforced and required rpm present
		# Bind mount params: The Z option relabels the volume so that both the container and host can safely access it without violating SELinux policies.
		# rw - container needs to both read from and write to the mounted volume.
		bind_mounts=":rw,Z"
		security_opts_disabled=""
		user_args="--user ${NGX_USER_UID}:${NGX_USER_UID} "
		add_or_drop_cap="--cap-add=NET_BIND_SERVICE --cap-drop=cap_sys_chroot --cap-drop=cap_setpcap --cap-drop=cap_setgid --cap-drop=cap_setfcap --cap-drop=cap_kill --cap-drop=cap_fsetid --cap-drop=cap_fowner --cap-drop=cap_dac_override --cap-drop=cap_chown "
		tmp_files_mapping="-v ${HOST_WORKING_DIR}/fastcgi_temp/:/usr/local/openresty/nginx/fastcgi_temp${bind_mounts} \
							-v ${HOST_WORKING_DIR}/uwsgi_temp/:/usr/local/openresty/nginx/uwsgi_temp${bind_mounts} \
							-v ${HOST_WORKING_DIR}/scgi_temp/:/usr/local/openresty/nginx/scgi_temp${bind_mounts} "
		nginx_user_mapping=" -e NGX_USER_UID=${NGX_USER_UID} -e NGX_USER_USERID=${NGX_USER_USERID}  -e NGX_USER_USERGROUP=${NGX_USER_USERGROUP} "
	fi

	local ip_port_mapping=""
	if ! is_host_network; then
		local external_ip=$(get_external_ip)
		log_info "Using ${external_ip} as proxy binding IP. From installer.env ${PROXY_BINDING_IP}."
		local private_ip=$(get_private_ip)
		if [ -z "$private_ip" ]; then
			log_warn "No private IP found, proxymap.txt and status API cannot be hosted."
		else
			log_info "Using IP ${private_ip} for hosting proxymap.txt"
		fi
		ip_port_mapping=$(generate_port_mappings "${DEPLOYMENT_ENV_DIR}" "${external_ip}" "$private_ip")

	fi


	# Only env variables names are declared in podman run args and value for same 
	# will be picked from shell environment which comes from sourced env files
	#PODMAN_RUN_ARGS="--init --rm=true --detach --name ${CONTAINER_NAME} \
        PODMAN_RUN_ARGS="--replace ${user_args} --init ${RESTART_OPTION} --detach --name ${CONTAINER_NAME} \
		${security_opts_disabled} \
		 ${add_or_drop_cap} \
	--network ${CONTAINER_NETWORK_MODE} ${ip_port_mapping} \
	--memory ${MEM_LIMIT} --memory-swap ${MEM_SWAP_LIMIT} \
	--memory-reservation ${MEM_RES} \
	--cpus ${CPU_LIMIT} --ulimit ${NOFILE_LIMIT} \
	-v /etc/localtime:/etc/localtime:ro \
	-v ${HOST_CACHE_VOL}:${NGX_CACHE_DIR}${bind_mounts} \
	-v ${HOST_SSL_VOL}:${NGX_SSL_DIR}${bind_mounts} \
	-v ${HOST_LOGS_VOL}:${NGX_LOG_DIR}${bind_mounts} \
	-v ${HOST_CONF_VOL}:${NGX_CONF_DIR}${bind_mounts} \
	-v ${HOST_HTML_VOL}:${NGX_HTML_DIR}${bind_mounts} \
	-v ${HOST_LUA_VOL}:${NGX_LUA_DIR}${bind_mounts} \
	${tmp_files_mapping} \
	-v ${HOST_RSYSLOG_CONF_VOL}:/etc/rsyslog.d${bind_mounts} \
	-e NGX_HOME -e NGX_HTML_DIR -e NGX_LUA_DIR -e NGX_CACHE_DIR \
	-e NGX_SSL_DIR -e NGX_LOG_DIR -e NGX_CONF_DIR \
	-e NGX_LIVE_LOG_SHIPPING_ENABLED \
	-e NGX_LIVE_LOG_SHIPPING_SERVER_CERT \
	-e NGX_LIVE_LOG_SHIPPING_CLIENT_CERT \
	${nginx_user_mapping} "
	# Configure container DNS settings
	configure_podman_arg_from_list "${CONTAINER_DNS_RESOLVER}" "|" "--dns"
	configure_podman_arg_from_list "${CONTAINER_DNS_SEARCH_DOMAIN}" "|" "--dns-search"
}

function configure_log_rotate() {
	log_info "Copying logrotate config to /etc/logrotate.d/${CONTAINER_NAME}/"
	cp -f ./logrotate /etc/logrotate.d/${CONTAINER_NAME}
	log_info "Updating logrotate config"
	sed -i s,NGX_LOG_DIR,${HOST_LOGS_VOL},g -- /etc/logrotate.d/${CONTAINER_NAME}
	sed -i s,CONTAINER_NAME,${CONTAINER_NAME},g -- /etc/logrotate.d/${CONTAINER_NAME}
	sed -i s,NGX_USER_USERID,${nginx_user_userid},g -- /etc/logrotate.d/${CONTAINER_NAME}
	sed -i s,NGX_USER_USERGROUP,${nginx_user_usergroup},g -- /etc/logrotate.d/${CONTAINER_NAME}
	echo "/usr/sbin/logrotate -v /etc/logrotate.d/${CONTAINER_NAME} -s ${HOST_LOGS_VOL}/nginx_log_rotate.log" >> /etc/cron.hourly/logrotate
	chmod 755 /etc/cron.hourly/logrotate
	chown root:root /etc/logrotate.d/${CONTAINER_NAME}
	chmod 755 /etc/logrotate.d/${CONTAINER_NAME}
	log_info "doing selinux rules changes"
	if enforce_selinx; then
		#On the SeLinux enabled system, logrotate cannot access the container_t files, hence need to add these rights for logrotate to work properly.
		#Compile the SELinux policy module source file (.te) into a module binary (.mod) format
		checkmodule -M -m -o logrotate_container_exec.mod selinux_rules/logrotate_container_exec.te
		#Create SELinux policy package (.pp) from the compiled module binary (.mod)
		semodule_package -o selinux_rules/logrotate_container_exec.pp -m logrotate_container_exec.mod
		#Install SELinux policy package (.pp) into the system.
		semodule -i selinux_rules/logrotate_container_exec.pp
	fi
	log_info "log rotate configured to run every hour"
}

function remove_log_rotate() {
	local escaped_value
	escaped_value=$(printf '%s\n' "${HOST_WORKING_DIR}" | sed 's/[\/&]/\\&/g')
	sed -i "\|${escaped_value}|d" /etc/cron.hourly/logrotate
	rm -rf /etc/logrotate.d/${CONTAINER_NAME}
	log_info "Removed log rotate configuration."
}

function override_podman_run_args() {
	# Method to override default podman run args
	return 0
}

function run_container() {
	if [[ "${LOAD_CONTAINER_IMAGE_FROM_TAR}" == "true" ]] && [[ -r ${REVERSE_PROXY_CONTAINER_IMAGE_TAR} ]]; then
		log_info "Loading proxy image from file ${REVERSE_PROXY_CONTAINER_IMAGE_TAR}"
		podman load --quiet --input ${REVERSE_PROXY_CONTAINER_IMAGE_TAR}
	fi
	log_info "Starting proxy from image ${CONTAINER_IMAGE} with run args as: ${PODMAN_RUN_ARGS}"
	podman run ${PODMAN_RUN_ARGS} ${CONTAINER_IMAGE}
	if [ $? -ne 0 ]; then
    log_error "Container ${CONTAINER_NAME} failed to start. Exiting."
    exitall 210
  fi
}

function start() {
	log_info "Starting reverse proxy"
	configure_openresty_configs
	create_ssl_cert
	prepare_podman_run_args
	override_podman_run_args
	run_container
}

function stop() {
	log_info "Stopping reverse proxy"
	podman stop "${CONTAINER_NAME}"
	# Clear the podman container, when auto restart flag is enabled, container would have started with "restart-unless-stopped" option.
	# Hence manual removal of the container required on the stop command.
	if should_auto_restart_container; then
		log_info "Removing the systemctl service to disable the auto start."
		run_command_or_exit sudo systemctl disable container-${CONTAINER_NAME}.service
		run_command_or_exit sudo systemctl stop container-${CONTAINER_NAME}.service
		run_command_or_exit rm -rf "/etc/systemd/system/container-${CONTAINER_NAME:?}.service"
	fi

  remove_iptables_rules
  remove_log_rotate
}

function remove_iptables_rules(){
  if ! is_iptables_running; then
    log_warn "Iptables not running."
    return
  fi

  for ENV in ${DEPLOYMENT_ENV_DIR}/*.env; do
    if [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/core.env" ] || [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/dirs.env" ]; then
      continue
    else
      remove_template_ports_from_iptables ${ENV}
    fi
  done


  if should_auto_restart_container; then
      save_iptables_rules
  fi
}

#
# Function to remove iptables rules for template ports in env file.
#
function remove_template_ports_from_iptables(){
  local ENV
  ENV=${1}

  local portsToRemoveList
  portsToRemoveList=""

  local template_type
  template_type=$(get_prop_val '^TEMPLATE_TYPE' ${ENV})
  local result_template_type
  result_template_type=$(get_component $template_type)
  local portkey
  portkey=$(echo "NGX_PRXY_${result_template_type}_PORT")
  local portValue
  portValue=$(get_prop_val "^${portkey}" ${ENV})
  portsToRemoveList+=${portValue}" "

  if [ "${template_type}" == "cuic_1261" ]; then
    local portkey
    portkey=$(echo "NGX_PRXY_${result_template_type}_DOC_PORT")
    local portValue
    portValue=$(get_prop_val "^${portkey}" ${ENV})
    portsToRemoveList+=${portValue}" "
  fi

  if [ "${template_type}" == "livedata_1261" ]; then
    local portkey
    portkey=$(echo "NGX_PRXY_${result_template_type}_SCKT_IO_PORT")
    local portValue
    portValue=$(get_prop_val "^${portkey}" ${ENV})
    portsToRemoveList+=${portValue}" "
  fi

  if [ "${template_type}" == "chat" ]; then
    local portList
    portList=$(grep 'NGX_CHAT_HOST._PROXY' ${ENV} | grep -v "NGX_CHAT_HOST1_PROXY" | cut -d ':' -f2)
    for port in ${portList[@]}
    do
      portsToRemoveList+=${port}" "
    done
  fi

  for port in ${portsToRemoveList}
  do
    EXTERNAL_IP=$(get_external_ip)
    local check_port_in_iptables_result
    check_port_in_iptables_result=$(iptables -L ALLOWED_PORTS -nv -t mangle | awk -v ip="$EXTERNAL_IP" '$9 == ip' | grep " dpt:${port}\>")
    if [ ! -z  "$check_port_in_iptables_result" ]; then
      log_info "Removing iptables rule for port ${port}"
      log_and_evaluate "iptables-save | grep -v \"d "${EXTERNAL_IP}"/32.*dport "${port}" \" | iptables-restore"
    fi
  done
}

function reload {
	log_info "Reloading reverse proxy"
	podman exec -it "${CONTAINER_NAME}" /usr/local/openresty/nginx/sbin/openresty_launcher.sh reload
}

function hot_reload {

	local real_container_name
	local test_container_name
	real_container_name="${1}"
	test_container_name="TMP_${real_container_name}"

	is_valid_container_to_reload "${real_container_name}" "${CONTAINER_IMAGE}"

	check_test_container_not_running "${test_container_name}"

	local real_working_dir
	local test_working_dir
	real_working_dir=${HOST_WORKING_DIR}
	test_working_dir="/tmp/${test_container_name}"

	
	CURRENT_CONFIG_FILE=$(get_new_temp_file)
	set_trap_with_preserve "cleanup ${CURRENT_CONFIG_FILE}" EXIT
	store_container_configs_in_file "${real_container_name}" "${CURRENT_CONFIG_FILE}"
	
	start_test_container "${test_container_name}" "${test_working_dir}" "${real_working_dir}"
	
	wait_for_container_to_comeup "${test_container_name}" 30 "test"
	# Now we are sure that test container is running.
	set_trap_with_preserve "cleanup_podman_container ${test_container_name} ${test_working_dir}" EXIT


	NEW_CONFIG_FILE=$(get_new_temp_file)
	set_trap_with_preserve "cleanup ${NEW_CONFIG_FILE}" EXIT
	store_container_configs_in_file "${test_container_name}" "${NEW_CONFIG_FILE}"
	
	log_info "Replacing port to original values on the openresty config file."
	# Relace the configs of test ports with real port values.
	replace_test_port_with_real_port "$NEW_CONFIG_FILE"


	display_config_diff_to_user "$CURRENT_CONFIG_FILE" "$NEW_CONFIG_FILE" 
	local prompt
	prompt="[WARN] Please confirm if you want to deploy the changed config? [Y|N]"
	if get_user_confirmation "${prompt}" "${AUTO_PROMT_CONFIMATION}"; then
		reload_new_openresty_configs "${real_container_name}" "${real_working_dir}"
	else
		log_error "Config reload rejected by user, Exiting."
		exitall 101
	fi

}


function clear_cache {
	log_info "Clearing reverse proxy cache"
	podman exec -it "${CONTAINER_NAME}" /usr/local/openresty/nginx/sbin/openresty_launcher.sh clear_cache
}


function prestart_hook() {
	# Prestart hook to do stuff before start
	return 0
}

function poststart_hook() {
	# Prestart hook to do stuff after start
	return 0
}

# Get the installer env value first
function popupate_installer_env() {
	for installer_env in "${INSTALLER_ENV_FILES[@]}"; 
	do
		check_file_exist_or_exit "${installer_env}"
		if [ -z "${INSTALLER_ENV}" ]; then
			# It is possible we can have multiple installer env files given in the commandline params
			# But the first file is the one actual installer.env all others are files that can be used to override the params in the env file.
			INSTALLER_ENV=${installer_env}
		fi
	done
}

function prepare_env() {
	set -o allexport
	for installer_env in "${INSTALLER_ENV_FILES[@]}"; 
	do
		check_file_exist_or_exit "${installer_env}"
		source "${installer_env}"
	done
	set +o allexport
}

function build_components_list() {
	#Iterates over the list of components env found and builds string variable that looks like "cloudconnect|cuic|finesse".

	if [ "${DEPLOYMENT_ENV_DIR}" == "" ]; then
		# If this is empty below code will consider the location of DEPLOYMENT_ENV_DIR as /
		# thats wrong, hence exit here after printing usage.
		log_error "Invalid env files location provided. Exiting."
		usage
		exitall 100
	fi
	local configured_components_list=""
	if [ "$(ls -A "${DEPLOYMENT_ENV_DIR}")" ]; then
		for ENV in "${DEPLOYMENT_ENV_DIR}"/*.env; do
			if [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/core.env" ] || [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/dirs.env" ]; then
				continue
			else
				local component_type
				component_type=$(get_prop_val 'TEMPLATE_TYPE' "${ENV}")
				local component_name
				component_name=$(get_component "${component_type}")
				COMPONENTS_COUNT["$component_name"]=$((COMPONENTS_COUNT["$component_name"] + 1))
				if [[ ! "$configured_components_list" =~ $component_type ]]; then
					configured_components_list+=$component_type"|"
				fi	
			fi
		done
	else
    		log_warn "No component env configs provided in ${DEPLOYMENT_ENV_DIR}."
	fi
	
	# After iterating over the ENV dir provided, we should able to get the list of configured components
	# If no env files provided on the given ENV dir, exit.
	if [ "${configured_components_list}" != "" ]; then
		configured_components_list=${configured_components_list::-1}
		add_configured_component_list_var \""${configured_components_list}"\"
	else
		log_error "No component configured. Exiting."
		exitall 100
	fi

}

function add_configured_component_list_var() {
	#Add configured component list in installer.env, also delete any existing entry of same var.
	local configured_components_list
	configured_components_list=${1}
	sed -i '/NGX_CONFIGURED_COMPONENTS_LIST/d' "${INSTALLER_ENV}"
	echo NGX_CONFIGURED_COMPONENTS_LIST="$configured_components_list" >>"${INSTALLER_ENV}"
}

function check_errors_if_any() {
	wait_for_container_to_comeup ${CONTAINER_NAME} 20
	log_info "Successfully started the container with name ${CONTAINER_NAME}"
}

function record_start_time() {
	START_TIME=$(get_last_timestamp "${HOST_LOGS_VOL}/error.log")
	log_info "Last time stamp on the error file. $START_TIME"
}

function parseArgs() {
	while getopts ":i:e:r:y" opt; do
		case "${opt}" in
			i )
				INSTALLER_ENV_FILES+=("${OPTARG}")
				;;
			e )
				DEPLOYMENT_ENV_DIR="${OPTARG}"
				;;
			r )
				container_name="${OPTARG}"
				;;
			y )
				AUTO_PROMT_CONFIMATION=true
				;;
			\? )
				log_info "Invalid option: -${OPTARG}"
				usage
				exitall 100
				;;
			: )
				log_info "Invalid value: -${OPTARG} requires a value."
				usage
				exitall 100
				;;
		esac
	done
	shift $((OPTIND -1))
}

function main() {
	if is_podman_installed && is_bc_installed; then
		parseArgs "$@"
		popupate_installer_env
		build_components_list
		prepare_env
		case "${@: -1}" in
			start)
				log_info "installer env is $INSTALLER_ENV, deployment env dir is $DEPLOYMENT_ENV_DIR"
				define_nginx_user
				check_cron_service_or_exit
				do_initial_validations_before_start_or_hot_reload "$INSTALLER_ENV" "$DEPLOYMENT_ENV_DIR" "$AUTO_PROMT_CONFIMATION"
				create_directories
				upgrade_configs_if_required "$INSTALLER_ENV" "$DEPLOYMENT_ENV_DIR"
				check_or_create_nginxuser "$INSTALLER_ENV" "${DEPLOYMENT_ENV_DIR}/core.env"
				check_valid_user_agent_validation "${DEPLOYMENT_ENV_DIR}/core.env"
				compare_customerenvs_masterenvs "$INSTALLER_ENV"
				validate_envs_with_main "$DEPLOYMENT_ENV_DIR"
				check_log_level_value $(get_prop_val "NGX_ERR_LOG_LEVEL" "${DEPLOYMENT_ENV_DIR}"/core.env)
				record_start_time
				prestart_hook
				suggest_podman_resource_params
				start
				poststart_hook
				check_errors_if_any
				handle_auto_start
				configure_iptables_rules
				configure_log_rotate
				;;

			stop)
				stop
				;;

			reload)
				reload
				;;

			clear_cache)
				clear_cache
				;;

			hot_reload)
				if [ -z "${container_name}" ]; then
					log_error "No container name specified to reload. Exiting."
					exitall 208
				fi
				log_info "Container name to reload $container_name"
				define_nginx_user
				do_initial_validations_before_start_or_hot_reload "$INSTALLER_ENV" "$DEPLOYMENT_ENV_DIR" "$AUTO_PROMT_CONFIMATION"
				record_start_time
				hot_reload "$container_name"
				check_errors_if_any
				;;

			*)
				usage
				;;
		esac
	else
		log_error "Please ensure podman is installed and running on this host."
		exitall 1
	fi
}

