--[[
   IDSPublicKey
   The purpose of IdSPublicKey is to hold the public key value of ids
   Also has functinality to fetch backend api and parse it. This is invoked by Nginx timer at every fixed interval
]]
local utils = require('utils')
local constants = require('constants')
local accept = "text/plain"
local publickeyconst = "PUBLICKEY"
local idspublickey = ngx.shared.idspublickey


local delay = tonumber(ngx.var.ids_publickey_poll_interval)
if delay == nil then delay = constants.DEFAULT_IDS_PUBLICKEY_POLLER_INTERVAL end
local fast_poller_interval = constants.FAST_POLLER_INTERVAL

--[[
   Function to populate IdS Public key
   
   Input:
      - url - url to get the public key of IdS
]]
local fetchPublicKey = function(url)
   ngx.update_time()
   local startTime = ngx.now()
   local http = require "resty.http"
   local httpc = http.new()

   local body = nil

   -- Fetch the response
   local res, err = httpc:request_uri(url, { method = "GET", headers = { ["Accept"] = accept, ["User-Agent"] = constants.USER_AGENT_HEADER  }, ssl_verify = false })
   ngx.log(ngx.NOTICE, "fetchPublicKey - fetching IdS public key from URL ", url)
   if res and res.status == ngx.HTTP_OK then
      body = res.body
      ngx.log(ngx.NOTICE, "fetchPublicKey - IdS Public key fetch success, public key is " , body)
      idspublickey:set(publickeyconst,body)
      ngx.log(ngx.INFO, "Updated the public key from IdS", body)
      -- update last update of the thirdpartyurl list
      utils.set_last_poll_time(constants.IDS_PUBLICKEY_POLLER_NAME)
   elseif res and res.status == ngx.HTTP_NOT_FOUND then
      ngx.log(ngx.NOTICE, "IdS version is < 12.6.2, there is no public key available.")
      idspublickey:set(publickeyconst, "NA")
      -- update last update of the thirdpartyurl list
      utils.set_last_poll_time(constants.IDS_PUBLICKEY_POLLER_NAME)
   else
      ngx.log(ngx.ERR, "API [", url, "] Failed with error: [", err, "] and response [", res and res.status ,"] on worker process: [", ngx.worker.id(), "]")
   end
   ngx.update_time()
   ngx.log(ngx.NOTICE, "For worker process: [", ngx.worker.id(), "] total parse time is: ", (ngx.now() - startTime))
   collectgarbage()
end


--[[
   Poll handler function function. This is used to by the scheudler function.
   
   Input:
      - prematureTimerExpiry - argument takes a boolean value indicating whether it is a premature timer expiration or not
         Premature timer expiration happens when the Nginx worker process is trying to shut down
      - url - url to get the public key of IdS
]]
local pollHandler = function (prematureTimerExpiry, url)
   if prematureTimerExpiry then return end
   if utils.has_value_populated(constants.IDS_PUBLICKEY_POLLER_NAME) and not utils.is_valid_poll(constants.IDS_PUBLICKEY_POLLER_NAME, delay) then
      return
   end
   fetchPublicKey(url)
end



--[[
   Function to schedule the Nginx timer at every fixed internal
   Input:
      - url - URL to get the IdS public key
]]
local poll = function (serverName, url)
   ngx.log(ngx.NOTICE, "Starting Nginx timer to fetch IdS Public key ", fast_poller_interval, " seconds for proxy server: ", serverName, " on worker process:", ngx.worker.id())
   local ok, err = ngx.timer.every(fast_poller_interval, pollHandler, url)
   if not ok then
      ngx.log(ngx.ERR, "Failure - while invoking Thread scheduler for public key: ", err, " on worker process: [", ngx.worker.id(), " ] and url: [", url, "]")
   else
      ngx.log(ngx.INFO, "Success - Thread scheduler for fetching public key on worker process: [", ngx.worker.id(), " ] and url: [", url, "]")
   end
end

local IdSPublicKeyPoller = {}


--[[
   Function to get the users list object for a proxy server name with backend upstream url which to be used for retrieving list of allowed users
   If Users List Object is not present, then initialise users list object, schedule nginx timer with handler at every fixed interval to refresh the users list.
   Input:
      - serverName - proxy server name
      - url - which to be used for retrieving list of allowed users using nginx timer scheduler
]]
function IdSPublicKeyPoller.startPoll(serverName, url)

	   local sharedinstance = ngx.shared.timerthreadsstore
 		local resty_lock = require "resty.lock"
        local value  =  sharedinstance:get("getidspublickeythreadstarted")
        if  value then       -- cache hit
            if utils.is_poller_running(constants.IDS_PUBLICKEY_POLLER_NAME) then
               ngx.log(ngx.DEBUG, "Poller for ", constants.IDS_PUBLICKEY_POLLER_NAME, " is running. [",ngx.worker.id(), " ]")
               return
            else
               ngx.log(ngx.ERR, "Poller for ", constants.IDS_PUBLICKEY_POLLER_NAME, " is stopped. worker id[",ngx.worker.id(), " ]")
            end 
        end    
		local lock = resty_lock:new("timerthreadsstore")
        local elapsed = lock:lock("idspublickey")
        if not elapsed then
              return
        end
	    -- lock successfully acquired!
	    -- someone might have already put the value into the cache
	    -- so we check it here again
	    -- Double check

	    local threadstarted, err = sharedinstance:get("getidspublickeythreadstarted")
	    if threadstarted == "started" then
         if utils.is_poller_running(constants.IDS_PUBLICKEY_POLLER_NAME) then
	        local ok, err = lock:unlock()
	        if not ok then
	            ngx.log(ngx.ERR, "Failed to unlock timer store in ids public key manager [", ngx.worker.id(), " ] ")
	        end
	        return
         else
            ngx.log(ngx.ERR, "Poller for ", constants.IDS_PUBLICKEY_POLLER_NAME, " is stopped. worker id[",ngx.worker.id(), " ]")
         end 
	    end
	    
		-- start the timer
      poll(serverName, url)
      fetchPublicKey(url)

      -- Item never expires, so not setting expiry.
      local success, err =  sharedinstance:set("getidspublickeythreadstarted", "started")
	    if not success then
            local success, err = lock:unlock()
            if not success then
            	ngx.log(ngx.ERR, "Failed to unlock timer store in userslistmanager [", ngx.worker.id(), " ] ")
                return 
            end
            return 
	    end

		-- unlock
        local success, err = lock:unlock()
        if not success then
        	ngx.log(ngx.ERR, "Failed to unlock timer store in userslistmanager [", ngx.worker.id(), " ] ")
            return
        end  
end

return IdSPublicKeyPoller

