--[[
	Lua file responsible for performing sso related operations
	Author: Sahil Narwal<sanarwal@cisco.com>
]]
local jwt = require("resty.jwt")
local validators = require("resty.jwt-validators")
local cjson = require("cjson")
local _strutils = require("strutils")

local tokenCache = ngx.shared.tokencache

-- Stores the list of IP addresses through which users who successfully authenticated, connected. Used to allow websocket connections.
local ipstore = ngx.shared.ipstore

--[[
Method to decode base64 encoded JWT payload issued by IdS. This is required as default behaviour
of lua-resty-jwt module is to parse it as json which results in error. This is hooked into lua-resty-jwt module.
IdS issued token structure is as follows:

Un-encrypted token: JWT token contains 3 parts delimited by dot
		1st part: Encoded header - Looks like {"alg": "HS256"}
		2nd part: Encoded token payload
			Sample:
				{
					"sub": "{\"scope\":[\"ccc_onprem_apps\"],\"rt\":\"7ddac088dc87018882132860f099eb93eeb5a398\",\"user_id\":\"1001001\",\"realm\":\"finesse.com\",\"upn\":\"1001001@finesse.com\",\"ids_id\":\"ids.autobot.cvp\",\"client_id\":\"bedbc55ee864b99c8f540886b23ae37c5189407e\",\"token\":\"a49348c18dabdd87bd6a1c680cb158d247c89552\",\"expiry\":1632866510802,\"usage\":\"access\",\"ver\":\"1.0\"}",
					"iss": "ids.autobot.cvp",
					"exp": 1632866510,
					"iat": 1632859310,
					"jti": "a49348c18dabdd87bd6a1c680cb158d247c89552"
				}
		3rd part: Encoded auth tag or signature - Token signature
 Sample token:
 eyJhbGciOiJIUzI1NiJ9.eyJzdWIiOiJ7XCJzY29wZVwiOltcImNjY19vbnByZW1fYXBwc1wiXSxcInJ0XCI6XCI3ZGRhYzA4OGRjODcwMTg4ODIxMzI4NjBmMDk5ZWI5M2VlYjVhMzk4XCIsXCJ1c2VyX2lkXCI6XCIxMDAxMDAxXCIsXCJyZWFsbVwiOlwiZmluZXNzZS5jb21cIixcInVwblwiOlwiMTAwMTAwMUBmaW5lc3NlLmNvbVwiLFwiaWRzX2lkXCI6XCJpZHMuYXV0b2JvdC5jdnBcIixcImNsaWVudF9pZFwiOlwiYmVkYmM1NWVlODY0Yjk5YzhmNTQwODg2YjIzYWUzN2M1MTg5NDA3ZVwiLFwidG9rZW5cIjpcImE0OTM0OGMxOGRhYmRkODdiZDZhMWM2ODBjYjE1OGQyNDdjODk1NTJcIixcImV4cGlyeVwiOjE2MzI4NjY1MTA4MDIsXCJ1c2FnZVwiOlwiYWNjZXNzXCIsXCJ2ZXJcIjpcIjEuMFwifSIsImlzcyI6Imlkcy5hdXRvYm90LmN2cCIsImV4cCI6MTYzMjg2NjUxMCwiaWF0IjoxNjMyODU5MzEwLCJqdGkiOiJhNDkzNDhjMThkYWJkZDg3YmQ2YTFjNjgwY2IxNThkMjQ3Yzg5NTUyIn0.FwRUMUG9uypMbHOCz01NG-MAwC31RjTOJpFOXT_53AY

 Encrypted token: JWT token contains 5 parts delimited by dot. 
		1st part: Encoded header - Looks like {"cty": "JWT", "enc": "A128CBC-HS256", "alg": "dir"}
		2nd part: Encoded encrypted key - not set by IdS
		3rd part: Encoded iv - Initialization vector - Initial random seed used for encryption
		4th part: Encoded cipher text - Encrypted Un-encrypted token(shared above) payload
		5th part: Encoded auth tag or signature - Token signature
 Sample token:
 eyJjdHkiOiJKV1QiLCJlbmMiOiJBMTI4Q0JDLUhTMjU2IiwiYWxnIjoiZGlyIn0..o_TPfBmNK0AClsLUCQFkeg.2J9F5P58Lv5vvWxeI5WKToMZczWNZPo8EcctSX6v1Sj9T3po9T5eCC0vtx2UixqCau3BoW4eAWqj21JeociAYqVaMJuQCPQcq7k5MttkVj0eyaU1gB3UiDXm-ty6siUmWK426Q2bVq7bIGu0gqzucN8_xK0ZboEZjMuJC0sLUa3Re4MlAneqkt97jVbhA2nM-oBLkfvkd4H4T1RdjArTRZS3rDnj8wdvaAddcPQ-k5xyGDdJLQyxX_-pWvsq7u-NIZBMk5yz3pj8fMM_YYWvVTQfaKs0JaB6lYiuz7koMzrrOcPxGsgIMm9TeNgN43ICsp68BspikiicqMVp-FqcBKvJOJUyTHOYJZFzPICYRwSPurPZ8B6xZyP8otHjN89iJWXMBrD5XnN-dnJWR_8GLJaMrlq1qSjanvtaR7cgDDpHUvYH8EgaRJuIM5GqjzM_cS7k8fuS0jYR8Js2TEVa7qqGThNpLhoGxPzyUQCDvweoPPiJYju__l6gvrYbS5IAp8WZVBq7CI3HBolJzbQ0TSC1H5PjSS1lQ5jqld_ns0xHYj0_xkoi0PuvKzN_D5YJ5H82eHFuAoGEvxYqy-hJl48cPbIe81RUdnmEWK-qhLCxpN_RQC4S7i9oO4LBfhu5HDj_wVTCJL1z8p5u7XZ1g08w98JkuGOKfv1rNcd1s_qbYJCR0NL6az_DgVwDUOjQLDABpk-HA1pgE95zQACEZ9PoSQBHrBlPmxpl8alryLvM0JIEQYLXIUArILg62iVt1sN6GBB9ehcVo5gNL10Ins-svcfOkUxWaS7FGWeoDp0vV_ThLHUk-pckmw3UWRdgkQuCFDm18W3AfGqeexCn7I_HLk83KdJPA2rkszMdpPLhSV9bkMpcSG2juX_8UYoJQuu78zgQiYTQptaOfOOrhfPa5y1fgmk4RUIEiOHtnw2f3Bzp-DKfFfXmIcdJ1qq2lqAg9lWUfkSfhk5PvZh0JQ.UktxBU4NvUEGIbfILBlMqw
]]
local payload_decoder = function(payload)
	local jwt_obj = jwt:load_jwt(payload)
	if not jwt_obj.valid then
		ngx.log(ngx.ERR, "[", ngx.worker.id() , "][", ngx.var.cookie_cc_username, "] Error decoding JWT payload. Error: ", jwt_obj.reason)
		error({reason="failed to decode JWT payload: " .. jwt_obj.reason})
	end
	return jwt_obj.payload
end

jwt:set_payload_decoder(payload_decoder)

local ssoutils = {}



--[[
	Method to cache a token in provided cache with predefined expiry.
	Token expiry is configured in maps.conf
]]
function ssoutils.cacheToken(token)
	local username = ngx.var.cookie_cc_username
	if (token == nil)
	then
		ngx.log(ngx.ERR, "[", ngx.worker.id() , "][", username, "] Ignoring nil token to be cached.")
	else
		local jwtExpiry = tonumber(ngx.var.jwt_expiry)
		tokenCache:add(token, "true", jwtExpiry)
		-- Allow websocket connectivity from this client IP address.
		local clientip = ngx.var.remote_addr
		ipstore:set(clientip, 1, jwtExpiry * 2)
		ngx.log(ngx.NOTICE, "[", ngx.worker.id() , "][", username, "] Token ", token, " cached with expiry ", jwtExpiry, " seconds.")
	end
end

--[[
	Method to validate a token in provided cache.
	Return true if token exist in cache and is not expired, false otherwise.
]]
function ssoutils.validateTokenFromCache(token)
   local value = tokenCache:get(token)
   return value ~= nil
end

--[[
	Method to decrypt and validate a token.
	Return true if token is valid i.e it is not expired and issued by valid issuer, false otherwise.
	Token decryption secret and valid issuers are configured in maps.conf
]]
function ssoutils.decryptAndValidateToken(token)
	local jwtPublicKeyOrSecretKey = nil
	-- when the system is < 12.6.2 then IdS token verification can be done with only private key, not the public key
	local publickey = ngx.shared.idspublickey:get("PUBLICKEY")
	if (publickey == nil) then
		ngx.log(ngx.ERR, "[ Problem in getting public key, it is nil. ]")
	elseif (publickey == "NA") then
		ngx.log(ngx.INFO, "[", ngx.worker.id() , "][", ngx.var.cookie_cc_username, "] IdS version is < 12.6.2 hence using only secret key for token validation")
		jwtPublicKeyOrSecretKey = ngx.decode_base64(ngx.var.jwt_secret)
	else
		ngx.log(ngx.INFO, "[", ngx.worker.id() , "][", ngx.var.cookie_cc_username, "] IdS version is >= 12.6.2 using cached public key.")
		jwtPublicKeyOrSecretKey = publickey
	end
   
   local jwt_obj = jwt:verify(jwtPublicKeyOrSecretKey, token, {
      exp = validators.opt_is_not_expired()
   })
   if (jwt_obj ~= nil and jwt_obj.verified == false) then
	   ngx.log(ngx.ERR, "[", ngx.worker.id() , "][", ngx.var.cookie_cc_username, "] Token verification failed with error: ", jwt_obj.reason)
   end
   return jwt_obj ~= nil and jwt_obj.verified == true
end

--[[
	@function _getTokenFromCookie
	@param set_cookie set cookie header table
	@return token cookie value matching token cookie name regex
]]
local _getTokenFromCookie = function(set_cookie)
	local token = nil
	for key, cookie_value in pairs(set_cookie) do
		if string.match(cookie_value, "^token-.*") then
			local cookie_parts = _strutils.split(cookie_value, ";")
			local cookie_name_value = _strutils.split(cookie_parts[1], "=")
			token=cookie_name_value[2]
			break
		end
	end
	return token
end

--[[
	Method to cache access token from response cookies.
	This is called from sso_valve configuration file when /sso/authcode
	endpoint is accessed and served with 302 response.
]]
function ssoutils.cacheTokenFromResponseCookies()
	local username = ngx.var.cookie_cc_username
    local headers, err = ngx.resp.get_headers()
    if err then
        ngx.log(ngx.ERR, "[", ngx.worker.id() , "][", username, "] Error accessing response headers while caching token from response cookies. Error: ", err)
        return ngx.exit(ngx.HTTP_INTERNAL_SERVER_ERROR)
    end
    local token = _getTokenFromCookie(headers["set-cookie"])
	if (token == nil)
	then
		ngx.log(ngx.ERR, "[", ngx.worker.id() , "][", username, "] No access token found in response cookies.")
	else
		ngx.log(ngx.NOTICE, "[", ngx.worker.id() , "][", username, "] Caching token from response cookies. Authcode from request: ", ngx.var.arg_code)
		ssoutils.cacheToken(token)
	end
end

--[[
	Method to cache access token from response body.
	This is called from sso_valve configuration file when /sso/authcode
	endpoint is accessed and served with 200 response or /sso/token endpoint
	is accessed to refresh an issued access token
]]
function ssoutils.cacheTokenFromResponseBody()
    if ngx.arg[2] then
        local resp_body = ngx.ctx.buffered
        local json_resp = cjson.decode(resp_body)
		if (json_resp.token == nil)
		then
			ngx.log(ngx.ERR, "[", ngx.worker.id() , "][", ngx.var.cookie_cc_username, "] No access token found in response body.")
		else
			ngx.log(ngx.NOTICE, "[", ngx.worker.id() , "][", ngx.var.cookie_cc_username, "] Caching token from response body.")
			ssoutils.cacheToken(json_resp.token)
		end
    else
        local chunk = ngx.arg[1]
        ngx.ctx.buffered = (ngx.ctx.buffered or "") .. chunk
    end
end

return ssoutils

