
--[[
   Unauthenticated Resources Manager 
   The purpose of Unauthenticated Resources Manager is to maintain the 3rd party gadget Resources List Object for each proxy server name.
   NOTE: Unauthenticated Resources Manager is defined per worker process and data is not shared among them
   Functionality:
   - Get the 3rd party Resources List, Desktop Url list for the each proxy server name and url from which allowed resources list needs to be determined.
   - If resources List Object is not present, then initialise resources list object, schedule nginx timer with handler at every fixed interval to refresh the users list.
   - Manually invoke the handler to populated for this first time instead of waiting for scheduler to populate.
]]

local ResourcesList = require('3rdpartygadgetresources_list')
local utils = require('utils')
local constants = require('constants')
local delay = constants.THIRD_PARTY_POLL_INTERVAL
local fast_poller_interval = constants.FAST_POLLER_INTERVAL

--[[
   Handler function to populate the resources list object with allowed url paths.
   
   Input:
      - premature - argument takes a boolean value indicating whether it is a premature timer expiration or not
         Premature timer expiration happens when the Nginx worker process is trying to shut down
      - thirdpartyGadgetPathsInstance - instance of Thirdparty Resources List Object
]]
local handler = function (premature, thirdpartyGadgetPathsInstance)
   if premature then return end
   ngx.update_time()
   local startTime = ngx.now()

   if utils.has_value_populated(constants.THIRD_PARTY_POLLER_NAME) and not utils.is_valid_poll(constants.THIRD_PARTY_POLLER_NAME, delay) then
      return
   end

   local ResourcesListXML = thirdpartyGadgetPathsInstance:fetchResourcesList()
   if ResourcesListXML ~= nil then
      thirdpartyGadgetPathsInstance:parseResources(ResourcesListXML)
   end
   ngx.update_time()
   ngx.log(ngx.NOTICE, "Handler for worker process: [", ngx.worker.id() , "] and url :[" , thirdpartyGadgetPathsInstance.url, "] takes total processing time as: [", (ngx.now() - startTime), "]")
   collectgarbage()
end

--[[
   Function to schedule the Nginx timer at every fixed internal
   Input:
      - thirdpartyGadgetPathsInstance - instance of resources List Object
]]
local startNginxTimer = function (thirdpartyGadgetPathsInstance)
   ngx.log(ngx.NOTICE, "Starting Nginx timer to fetch thirdarty Resource list every ", fast_poller_interval, " seconds for proxy server: ",thirdpartyGadgetPathsInstance.serverName, " on worker process:", ngx.worker.id())
   local ok, err = ngx.timer.every(fast_poller_interval, handler, thirdpartyGadgetPathsInstance)
   if not ok then
      ngx.log(ngx.NOTICE, "Failure - while invoking Thread scheduler for fetching thirdparty list: ", err, " on worker process: [", ngx.worker.id(), " ] and url: [", thirdpartyGadgetPathsInstance.url, "]")
   else
      ngx.log(ngx.NOTICE, "Success - Thread scheduler for fetching thirdparty resource list on worker process: [", ngx.worker.id(), " ] and url: [", thirdpartyGadgetPathsInstance.url, "]")
   end
   ngx.timer.at(0, handler, thirdpartyGadgetPathsInstance)
end

local UnauthenticatedResourcesManager = {}

--[[
   Function to get the thirdparty gadget urls which dont require authentication for a proxy server name with backend upstream url
   If thirdparty url List Object is not present, then initialise the resource list object, schedule nginx timer with handler at every fixed interval to refresh the  list from api.
   Input:
      - serverName - proxy server name
      - url - which to be used for retrieving list of allowed thirdparty urls  without authentication  using nginx timer scheduler
   
   Retuns:
      - Thirdparty reosurces Object responsible for getting the data from api.
]]
function UnauthenticatedResourcesManager.getThirdPartyGadgetResources (serverName, url)

		local sharedinstance = ngx.shared.timerthreadsstore
 		local resty_lock = require "resty.lock"
        local value  =  sharedinstance:get("thirdpartygadgeturlthreadstarted")
        if  value then       -- cache hit
         if utils.is_poller_running(constants.THIRD_PARTY_POLLER_NAME) then
            ngx.log(ngx.DEBUG, "Poller is running for ", constants.THIRD_PARTY_POLLER_NAME, " worker ID[", ngx.worker.id(), "], starting it.")
            return
         else
            ngx.log(ngx.ERR, "Poller stoppned for ", constants.THIRD_PARTY_POLLER_NAME, " worker ID[", ngx.worker.id(), "], starting it.")
         end
        end    
		local lock = resty_lock:new("timerthreadsstore")
        local elapsed = lock:lock("thirdpartyurlkey")
        if not elapsed then
              return
        end
	    -- lock successfully acquired!
	    -- someone might have already put the value into the cache
	    -- so we check it here again
	    -- Double check

	    local threadstarted, err = sharedinstance:get("thirdpartygadgeturlthreadstarted")
	    if threadstarted == "started" then
            if utils.is_poller_running(constants.THIRD_PARTY_POLLER_NAME) then
               local ok, err = lock:unlock()
               if not ok then
                     ngx.log(ngx.ERR, "Failed to unlock  in thirdparty resourcemanager[", ngx.worker.id(), " ] ")
               end
               return
            else
               ngx.log(ngx.ERR, "Poller stoppned for ", constants.THIRD_PARTY_POLLER_NAME, " worker ID[", ngx.worker.id(), "], starting it.")
            end
	    end
	    
	  -- Start the timer.
      local thirdpartyGadgetPathsInstance = ResourcesList:new(serverName, url)

      -- make the first call is done inline - to make sure the values are populated first before starting thread.
      local ResourcesListXML = thirdpartyGadgetPathsInstance:fetchResourcesList()
      if ResourcesListXML ~= nil then
         thirdpartyGadgetPathsInstance:parseResources(ResourcesListXML)
      end

      startNginxTimer(thirdpartyGadgetPathsInstance)

	  -- Item never expires, so not setting expiry.
      local success, err =  sharedinstance:set("thirdpartygadgeturlthreadstarted", "started")
	    if not success then
            local success, err = lock:unlock()
            if not success then
            	ngx.log(ngx.ERR, "Failed to unlock   in thirdparty resourcemanager[", ngx.worker.id(), " ] ")
                return 
            end
            return 
	    end

		-- unlock
        local success, err = lock:unlock()
        if not success then
        	ngx.log(ngx.ERR, "Failed to unlock   in thirdparty resourcemanager[", ngx.worker.id(), " ] ")
            return
        end   
end


return UnauthenticatedResourcesManager 
 
