local utils = {}

local _syslogwriter = require("syslogwriter")
local _strutils = require("strutils")
local constants = require("constants")


local upload_limit_cache_key = "upload_limit"
local upload_limit_cache_expiry = constants.UPLOAD_LIMIT_POLL_INTERVAL -- 15 minutes in seconds
local default_upload_limit = 1 * 1024 * 1024  -- Default to 1 MB if fetching fails
local fast_poller_interval = constants.FAST_POLLER_INTERVAL
local upload_limit_fetch_url = "https://${NGX_FIN_HOSTNAME}:8445/finesse/api/AppConfigurations/com.cisco.ccbu.finesse.api.clientlog.compressedfile.size?bypassServerCache=true"

-- check if inputed value contains 'Bearer'
function utils.isBearer(s)
   return s ~= nil and s:find("Bearer ") == 1
end

-- check if inputed value contains 'Basic'
function utils.isBasic(s)
   return s ~= nil and s:find("Basic ") == 1
end

-- returns the username by first element before ':'
function utils.getCredentialsFromBasic(str)
   local username, password= str:match("([^:]+):([^:]+)")
   return {username, password}
end

function utils.isComponentConfigured(component_list_str, component_str)
   return component_list_str ~= nil and component_list_str:find(component_str) ~= nil
end

-- function to split a string by a delimiter
function utils.splitByDelimiter(inputstr, sep)
    if sep == nil then
        sep = "%s"
    end
    local t = {}
    for str in string.gmatch(inputstr, "([^" .. sep .. "]+)") do
        table.insert(t, str)
    end
    return t
end

-- function to get the last item from a string list separated by a delimiter
function utils.getLastItem(inputstr, delimiter)
    local items = utils.splitByDelimiter(inputstr, delimiter)
    return items[#items]
end

-- decode function
function utils.urldecode(s)
    s = s:gsub('+', ' ')
    s = s:gsub('%%(%x%x)', function(h)
        return string.char(tonumber(h, 16))
    end)
    return s
end

-- function to extract the token from the Sec-WebSocket-Protocol header
function utils.extractAuthFromHeader(headerValue)


    local lastItem = utils.getLastItem(headerValue, ",")
    lastItem = lastItem:match("^%s*(.-)%s*$")  -- Trim any leading/trailing whitespace

    -- ngx.log(ngx.ERR, "Last item before decoding: ", lastItem)
    -- Decode the last item (URL decoding)
    lastItem = utils.urldecode(lastItem)


    -- check if the last item starts with "Authorization:" and extract the token
    if lastItem:find("^Authorization:") then
        local auth = lastItem:match("^Authorization:%s*(.+)$")
        if auth then
            return auth
        end
    end

    -- Debug log: No valid authorization token found
    ngx.log(ngx.ERR, "No valid authorization token found")
    return nil
end

function utils.generateHash(password)
   local hash = 7
   for c in password:gmatch "." do
       hash = hash * 31 + string.byte(c)
   end
   return tonumber(hash)
end

function utils.isWSAuthenticationEnabled()
   local isWSAuthEnabled = "${NGX_AUTHENTICATE_WEBSOCKET}"
   -- to avoid the problem, when finesse component itself not present in the deployment.
   if isWSAuthEnabled == "true" then
        return true
   else
        return false
   end
end


--[[
    check if the Ip is blocked, if blocked then return
 ]]
function utils.isIPBlocked(clientIp)
    -- Stores blocking user list
    local blockingresources = ngx.shared.blockingresources
   if blockingresources:get(clientIp) ~= nil then
      ngx.log(ngx.ERR, clientIp, " :: IP is already blocked...")
      ngx.header.content_type = "text/html"
      ngx.status = 418
      ngx.say(
         "<html><head><title>418 I'm a teapot</title></head><body><center><h1>418 I'm a teapot</h1></center></body></html>"
      )
      ngx.exit(ngx.status)
   end
end

-- unauthorized error will be sent
-- use the output of checkForBruteforceAttack method
function utils.redirectAndSendError(blockStatus)
    if blockStatus == 1 then
        local logMessage =
            ngx.var.remote_addr ..
            " will be blocked for " ..
                tonumber(ngx.var.ip_blocking_duration) / 60 .. " minutes for exceeding retry limit."
        ngx.log(ngx.EMERG, logMessage)
        _syslogwriter.logMessage(logMessage)
        ngx.header.content_type = "text/html"
        ngx.status = 418
        ngx.say(
            "<html><head><title>418 I'm a teapot</title></head><body><center><h1>418 I'm a teapot</h1></center></body></html>"
        )
        return ngx.exit(ngx.status)
    else
        return ngx.exit(ngx.HTTP_UNAUTHORIZED)
    end
end


-- At a time only one thread will be allowed to check system is  brute forced or not,
-- will be checked for all error scenarioes

function utils.checkForBruteforceAttack(clientIp)
    ngx.log(ngx.DEBUG, "Authentication failed for agent with client address [", clientIp, "]  ")
    local resty_lock = require "resty.lock"
    local lock = resty_lock:new("clientstorage")
    local elapsed = lock:lock("client" .. clientIp)
    if not elapsed then
        ngx.log(ngx.ERR, "Failed to acquire lock from client store [", ngx.worker.id(), " ] ")
        return 2
    end
    local blockStatus = utils.handleBruteforceAttack(clientIp)
    local ok, err = lock:unlock()
    if not ok then
        ngx.log(ngx.ERR, "Failed to unlock from client store [", ngx.worker.id(), " ] ")
        return 2
    end

    return blockStatus
end

-- handle Brute force attack
--If in 30 seconds no of failures are  more than 5
-- ip will be block for 30 minutes
function utils.handleBruteforceAttack(clientIp)
    -- This is to skip the bruteforce check for the ratelimit_disable_ip_list IPs configured.
    -- Usually this will include LB IPs and localhost IPs.
    -- limitkey will have the value '' if its whitelisted.
    -- For all other requests, it will have non '' value.
    if ngx.var.limitkey == "" then
        return 0
    end
    -- If error occurs, store the api failure count as 1  in a share dictionary for 1 minute , provided entry is not present yet.
    -- After 1 minute entry will be deleted.

    -- Stores remote user id for whom request is failed
    local clientstorage = ngx.shared.clientstorage

    if clientstorage:get(clientIp) == nil then
        local auth_failure_counting_window_secs = tonumber(ngx.var.auth_failure_counting_window_secs)
        clientstorage:set(clientIp, 1, auth_failure_counting_window_secs)
    else
        -- If entry is already there ( i.e. api failure happens within 1 minute) update same dictionary with the incremented failure counter and remaining expiry time
        clientstorage:set(clientIp, clientstorage:get(clientIp) + 1, clientstorage:ttl(clientIp))
        --  Counter value is exceeded than thresold value, store the user in a share dictionary for 10 minutes.
        --  and auth request coming for that user will not be handled for 10 minutes .
        -- ## These two variables indicate five auth failures from a client can be allowed in thirty seconds. if the threshold is crossed,client ip will be blocked.
        local auth_failure_threshold_for_lock = tonumber(ngx.var.auth_failure_threshold_for_lock)
        if (clientstorage:get(clientIp) > auth_failure_threshold_for_lock) then
            -- Stores blocking user list
            local blockingresources = ngx.shared.blockingresources
            -- ## This variable indicates duration of blocking a client to avoid brute force attack
            local ip_blocking_duration = tonumber(ngx.var.ip_blocking_duration)
            blockingresources:set(clientIp, clientIp, ip_blocking_duration)
            return 1
        end
    end
    return 0
end


function utils.checkPassword(credentials_stored, password)
    -- local password_stored= credentials_stored
    -- password_stored=password_stored
    -- If user is locked returning with 401 error
    local passwordhash = utils.generateHash(password)

    local clientIp = ngx.var.remote_addr
    if passwordhash ~= tonumber(credentials_stored) then
        -- check for Brute force attack
        local blockStatus = utils.checkForBruteforceAttack(clientIp)
        utils.redirectAndSendError(blockStatus)
    end

    local ipstore = ngx.shared.ipstore
    -- check if this ip is stored in ipstore
    
    if ipstore:get(clientIp) == nil then
        --password is matching but client is different, looks like agent is using another machine. Adding this machine's ip address to ipstore to enable ws authentication.
        ipstore:set(clientIp, 1, 900 * 2)
        ngx.log(ngx.NOTICE, "Added the client ip address to ipstore")
    end
    ngx.log(
        ngx.NOTICE,
        "Skipping the user authentication as password is cached & provided is correct, request running on worker : [",
        ngx.worker.id(),
        "]"
    )
end


function utils.makeAuthAPICall(authUrl, user, password, authorization)
    local http = require "resty.http"
    local httpc = http.new()
    local blockStatus = 0
    local clientIp = ngx.var.remote_addr
    -- Fetch the response
    local res, err =
        httpc:request_uri(
        authUrl,
        {method = "GET", headers = {["Accept"] = accept, ["Authorization"] = authorization, ["User-Agent"] = constants.USER_AGENT_HEADER }, ssl_verify = false}
    )

    if res and res.status == ngx.HTTP_OK then
        ngx.log(ngx.DEBUG, "Auth call success from upstream, response: [ ", res.status, " ] client IP: ", clientIp)
        local credentialsstore = ngx.shared.credentialsstore
        credentialsstore:set(user, utils.generateHash(password), 900)
        local ipstore = ngx.shared.ipstore
        ipstore:set(clientIp, 1, 900 * 2)
        ngx.log(ngx.DEBUG, "credentialsstore updated for user: [ ", user, " ] client IP: ", clientIp)
        return ngx.HTTP_OK, blockStatus
    else
        if res then
            if res.status == ngx.HTTP_UNAUTHORIZED then
                -- Check for Brute force attack
                ngx.log(ngx.DEBUG, "Auth call failed from upstream, response: [ ", res.status, " ] client IP: ", clientIp)
                blockStatus = utils.checkForBruteforceAttack(clientIp)
                return ngx.HTTP_UNAUTHORIZED, blockStatus
            else
                ngx.log(ngx.DEBUG, "Auth call failed from upstream, ignoring response code: [ ", res.status, " ] ")
                return ngx.HTTP_UNAUTHORIZED, blockStatus
            end
        end
    end
end


-- Check for any configured auth server names.
function utils.validateServerNames()
    -- Check for any configured auth server names. 
    local serverName = ngx.var.auth_server_name
    local validOrigins = ngx.var.valid_auth_origin
    if serverName ~= nil and serverName ~= "" then
        local serverName = ngx.var.auth_server_name
        local validOrigins = ngx.var.valid_auth_origin
        if validOrigins == nil then
            ngx.log(ngx.ERR, "NGX_VALID_REFERRERS needs to be configured in core.env.")
            return ngx.exit(ngx.HTTP_UNAUTHORIZED)
        end
        validOrigins = string.gsub(validOrigins, "%-", "_")
        local serverNameLocal = string.lower(string.gsub(serverName, "%-", "_"))
        if (string.match("|" .. validOrigins .. "|", "|" .. serverNameLocal .. "|") == nil) then
            ngx.log(ngx.ERR, "Origin header " .. serverName .. " is not present in allowed origins " .. validOrigins)
            return ngx.exit(ngx.HTTP_UNAUTHORIZED)
        end
    end
    return serverName
end


-- handle ws auth
-- If this request is /ws request - then return the authHeader 
--      Throw exception and break if the auth header is not present.
-- Else Check /ws auth is not enabled
--      Check the clientIP is already authenticated (older way of /ws validation)
--      If the clientIP is valid return 0 (flow can break here)
--      Else throw unauthorized exception
-- 
function utils.handleWSAuth(uri, clientIp)
    ngx.log(ngx.INFO, "It is the /ws request, checking for authentication.")
    if utils.isWSAuthenticationEnabled() then
        local wsAuthHeader = ngx.req.get_headers()["Sec-WebSocket-Protocol"]
        ngx.log(ngx.INFO, "It is the /ws request, checking for auth values.")
        if wsAuthHeader ~= nil then
            ngx.log(ngx.INFO, "Auth values are not null, validating...")
            -- ws_auth = utils.extractAuthFromHeader(wsAuthHeader)
            return utils.extractAuthFromHeader(wsAuthHeader)
        else
            ngx.log(
                ngx.ERR,
                "Authentication failed: (Request header missing auth) ",
                ngx.var.request,
                " from client ",
                clientIp
            )
            -- Check for brute force attack
            local blockStatus = utils.checkForBruteforceAttack(clientIp)
            utils.redirectAndSendError(blockStatus)     -- send error
        end
    else
        ngx.log(ngx.INFO, "It is the /ws request, but wsAuth is not configured, hence just checking clientip...")
        -- flag in reverse proxy set to not to check the auth headers
        -- need to fallback on the ip based validation
        -- The shared dictionary that stores the IP addresses of successfully authenticated users.
        local ipstore = ngx.shared.ipstore
        -- Fetching the client IP address to check if it is authorized to make a websocket connection.
        local clientIp = ngx.var.remote_addr

        if ipstore:get(clientIp) == nil then
            ngx.log(ngx.ERR, "The ip [", clientIp, "] is unauthorized to create a websocket connections")
            return ngx.exit(ngx.HTTP_UNAUTHORIZED)
        end
        return 0
    end
end


function utils.handleSSOAuth(authorization, username)
    local _authHeader = require("auth_header")
    local jwt_token = _authHeader.getBearerToken(authorization)
    local clientIp = ngx.var.remote_addr
    if _strutils.isEmpty(jwt_token) then
        ngx.log(
            ngx.ERR,
            "[",
            ngx.worker.id(),
            "][",
            username,
            "] No JWT provided in bearer auth header. Rejecting request ",
            ngx.var.request,
            " from client ",
            clientIp
        )
        -- Check for brute force attack
        local blockStatus = utils.checkForBruteforceAttack(clientIp)
        utils.redirectAndSendError(blockStatus)
    else
        utils.validateJWT(username, jwt_token)
    end
end


function utils.validateJWT(username, jwt_token)
    local blockStatus = 0
    local clientIp = ngx.var.remote_addr
    local _ssoutils = require("ssoutils")
    if _ssoutils.validateTokenFromCache(jwt_token) then
        -- Allow access if provided token exists in cache
    elseif _ssoutils.decryptAndValidateToken(jwt_token) then
        -- Allow access if provided token is valid IdS issued token
        -- Cache the newly encountered token locally to avoid decryption on next access
        ngx.log(ngx.NOTICE, "[", ngx.worker.id(), "][", username, "] Caching token post decryption and validation.")
        _ssoutils.cacheToken(jwt_token)
    else
        -- Block the user from accessing the Finesse API's
        ngx.log(
            ngx.ERR,
            "[",
            ngx.worker.id(),
            "][",
            username,
            "] Neither token ",
            jwt_token,
            " was found in token cache nor token was found to be valid after decryption. Rejecting request ",
            ngx.var.request,
            " from client ",
            clientIp
        )
        -- Check for brute force attack
        blockStatus = utils.checkForBruteforceAttack(clientIp)
        utils.redirectAndSendError(blockingStatus)
    end
end


function utils.handleBasicAuth(authUrl, authorization, serverName)
    --[[
        get the user name, password from authorization header.
        NOTE: IN CASE OF ANYTHING MISSING OR INVALID AUTHORIZATION HEADER, RETURN 401 UNATHORISED
    ]]
    local _authHeader = require("auth_header")
    local credentials = _authHeader.getUserFromAuthHeader(authorization)
    local user = credentials[1]
    local password = credentials[2]

    -- Stores the count of users from the response of api.(Need to put in shared mem or else we need to calculate whether the users data is populated
    -- for every request which is costly, even looping through get keys in dictionary is costly.
    local userscount = ngx.shared.userscount

    --[[
    Check if users list is present. There are chances that user list couldn't be populated because request to backend finesse server never got success
    In such case, ignore the user existence against users list and continue with intended finesse api
    Note: when the User poller stopped and caches are expired, in that case, the poller will be started and when the cache is yet to be populated by the poller
    We will have a race condidtion, hence adding last 2 if conditions to avoid the RC
    ]]
    if userscount:get("size") == nil or userscount:get("size") == 0 or utils.is_poller_running(constants.USERS_POLLER_NAME) == false or utils.has_cache_expired(constants.USERS_POLLER_NAME, constants.USERS_POLL_INTERVAL) then
        ngx.log(ngx.INFO, "userscount is nil or empty or poller ", constants.USERS_POLLER_NAME, " is not running, hence using make call directly.")
        local httpstatus, blockStatus = utils.makeAuthAPICall(authUrl, user, password, authorization)
        if httpstatus ~= ngx.HTTP_OK then
            utils.redirectAndSendError(blockStatus)
        end
        return
    else
        ngx.log(ngx.DEBUG, "userscount is not nil or not empty on poller ", constants.USERS_POLLER_NAME, " utils.is_poller_running(constants.USERS_POLLER_NAME) ", utils.is_poller_running(constants.USERS_POLLER_NAME), "utils.has_cache_expired(constants.USERS_POLLER_NAME, constants.USERS_POLL_INTERVAL) ", utils.has_cache_expired(constants.USERS_POLLER_NAME, constants.USERS_POLL_INTERVAL))
    end

    -- Stores the credentials
    local credentialsstore = ngx.shared.credentialsstore
    -- Stores the list of users returned from the API
    local shareduserlist = ngx.shared.userlist
    local sharedalternatehosts = ngx.shared.alternatehosts
    local userExists = shareduserlist:get(serverName .. user)
    --[[
    In case Desktop is loaded in RPB, CUIC Gadget calls will reach to RPA with Origin header as RPB. 
    In this case the serverName will be RPB and it will not find an entry with RPB<Agent_ID> in user list. 
    Get the alternate host for RPB, which will be RPA and checkin the userlist.
    ]]
    if not userExists then
        local alternateHost = sharedalternatehosts:get(serverName)
        if alternateHost ~= nil then
            userExists = shareduserlist:get(alternateHost .. user)
        end
    end
    --[[
    1 . Check if user exists in the users list.
    If present, return which continue with intended finesse api
    If not present, return with 401 UNATHORISED
    2. If user is present , check if the authentication is done with the service.
        i. If not done, make a rest call to the suthentication service.
        ii. Store the passwords hash in the cache.(valid for 15min ~ 900seconds)
    3. if authentication is already done, check for the password entered hash with the successful password hash saved in step 2.ii
        i. if same, proceed, else reject with 401.

    ]]
    if userExists then
        local credentials_stored = credentialsstore:get(user)
        local passwordstoredisEmpty = _strutils.isEmpty(credentials_stored)
        if not passwordstoredisEmpty then
            utils.checkPassword(credentials_stored, password)
        else
            local resty_lock = require "resty.lock"
            local lock = resty_lock:new("credentialsstore")
            local elapsed = lock:lock("user" .. user)
            if not elapsed then
                return
            end
            -- lock successfully acquired!
            -- someone might have already put the value into the cache
            -- so we check it here again
            -- Double check

            local credentialsStored, err = credentialsstore:get(user)
            if credentialsStored then
                local ok, err = lock:unlock()
                if not ok then
                    ngx.log(ngx.ERR, "Failed to unlock from credentialsstore  [", ngx.worker.id(), " ] ")
                end
                utils.checkPassword(credentialsStored, password)
                return
            end
            local httpstatus, blockStatus = utils.makeAuthAPICall(authUrl, user, password, authorization)
            -- unlock
            local ok, err = lock:unlock()
            if not ok then
                ngx.log(ngx.ERR, "Failed to unlock  [", ngx.worker.id(), " ] ")
                return
            end

            if httpstatus ~= ngx.HTTP_OK then
                utils.redirectAndSendError(blockStatus)
            end
        end
    else
        -- Check for brute force attack and block the user from accessing the Finesse API's
        local clientIp = ngx.var.remote_addr
        local blockStatus = utils.checkForBruteforceAttack(clientIp)
        utils.redirectAndSendError(blockStatus)
    end
end

function utils.has_valid_url_to_fetch_upload_limit()
    if upload_limit_fetch_url:find("{NGX_FIN_HOSTNAME}") then
        return false
    end
    return true
end

function utils.start_ids_poller(serverName)
    local idsPublicKeyUrl = "https://${NGX_PRXY_IDS_HOSTNAME}:${NGX_PRXY_IDS_PORT}/ids/v1/keys/token/public"
    if idsPublicKeyUrl:find("{NGX_PRXY_IDS_HOSTNAME}") then
        ngx.log(ngx.ERR, "Ids is not configured on the system, returning without starting the ids poller")
        return
    end
    ngx.log(ngx.DEBUG, "Ids is configured on the system, starting the ids poller...")
    local IdSPublicKeyPoller = require("ids_publickey_poller")
    IdSPublicKeyPoller.startPoll(serverName, idsPublicKeyUrl)
end


-- Function to fetch the upload limit from the upstream server
-- Fetch only when the cache is expired else return the cached value to avoid unneccessary upstream call
function utils.fetch_upload_limit()
    
    local http = require "resty.http"
    local httpc = http.new()
    local res, err = httpc:request_uri(upload_limit_fetch_url, {
        method = "GET",
        headers = { ["Accept"] =  "application/xml", ["User-Agent"] = constants.USER_AGENT_HEADER  }, 
        ssl_verify = false
    })
    if not res then
        ngx.log(ngx.ERR, "Failed to request: ", err)
        return nil
    end

    if res.status ~= 200 then
        ngx.log(ngx.ERR, "Request failed with status: ", res.status)
        return nil
    end
    local response_body = res.body

    -- Find the value inside <value> tags
    local value = response_body:match("<value>(%d+)</value>")
    if value then
        ngx.log(ngx.ERR, "fetch_upload_limit() new value from finesse ", value)
        return tonumber(value)
    else
        ngx.log(ngx.ERR, "Failed to find <value> in response")
        return nil
    end
end

function utils.set_upload_limit_cache(upload_limit, upload_limit_cache_expiry)
    if upload_limit and upload_limit_cache_expiry then
        ngx.log(ngx.ERR, "Updating upload limit cache value. New value is:", upload_limit, " bytes. Caching for next ", upload_limit_cache_expiry, " seconds")
        local map = ngx.shared.client_body_size
        map:set(upload_limit_cache_key, upload_limit, upload_limit_cache_expiry)
    end
end

function utils.get_and_update_upload_limit_cache()
    -- if the URL is not proper, there is no finesse component installed, we can safly skip fetching the URL
    if not utils.has_valid_url_to_fetch_upload_limit then
        return -1
    end
    local map = ngx.shared.client_body_size
    local upload_limit = map:get(upload_limit_cache_key)
    if not upload_limit then
        upload_limit = utils.fetch_upload_limit()
        if upload_limit then
            utils.set_upload_limit_cache(upload_limit, upload_limit_cache_expiry)
            utils.set_last_poll_time(constants.UPLOAD_LIMIT_POLLER_NAME)
        else
            upload_limit = default_upload_limit
            utils.set_upload_limit_cache(upload_limit, 60)   -- cache this for next 1 minute and retry
        end
    else 
        ngx.log(ngx.ERR, "Using upload limit value from cache ", upload_limit)
    end

    return upload_limit
end


function utils.checkClientLogSize()
    -- Get the cached upload limit or fetch from the upstream server
    local upload_limit = utils.get_and_update_upload_limit_cache()
    local content_length = tonumber(ngx.req.get_headers()["content-length"])
    if content_length and content_length > upload_limit then
        ngx.log(ngx.ERR, "413 Request Entity Too Large, allowed size:", upload_limit, ", actual size: ", content_length)
        return ngx.exit(413)
    end
end

function utils.is_poller_running(poller)

    local last_poll_info = ngx.shared.last_poll_info
    local last_poll = last_poll_info:get(poller .. constants.CHECKED_AT)

    if last_poll then
        ngx.log(ngx.DEBUG, "Poller for ", poller, ". Last checked at: ", last_poll)
        local elapsed_time = ngx.time() - last_poll
        if elapsed_time < (3 + fast_poller_interval)  then -- giving 3 seconds leeway, in last 8 seconds if there is no check on this poller assume timer is not running
            ngx.log(ngx.DEBUG, "Poll is happening for this poller: ", poller)
            return true
        end
    else
        ngx.log(ngx.DEBUG, "Last poll is not defined for: ", poller)
    end
    return false
end

function utils.set_last_poll_time(poller)
    local last_poll_info = ngx.shared.last_poll_info
    local poll_time = ngx.time()
    local last_poll = last_poll_info:get(poller)
    if last_poll then
        ngx.log(ngx.INFO, "Poller for ", poller, ". Last poll at: ", last_poll, " replacing it with ", poll_time)
    else
        ngx.log(ngx.INFO, "Last poll is not defined for: ", poller, " Setting poll time: ", poll_time)
    end
    last_poll_info:set(poller, poll_time)
end

function utils.has_cache_expired(poller, cache_validity)
    ngx.log(ngx.DEBUG, "Check cache expirty for poller ", poller, " cache validity ", cache_validity)
    local last_poll_info = ngx.shared.last_poll_info
    local poll_time = ngx.time()
    local last_poll = last_poll_info:get(poller)
    if last_poll then
        local elapsed_time = ngx.time() - last_poll
        if elapsed_time < cache_validity then
            ngx.log(ngx.DEBUG, "Cache updated before ", elapsed_time, " for poller. Cache is valid ", poller, " cache validity ", cache_validity)
            return false
        else
            ngx.log(ngx.DEBUG, "Cache updated before ", elapsed_time, " for poller. Cache is valid ", poller, " cache validity ", cache_validity)
        end
    end
    return true
end

-- function to check atleast one valid poll for this poller is done
function utils.has_value_populated(poller) 
    local last_poll_info = ngx.shared.last_poll_info
    local last_poll = last_poll_info:get(poller)
    if last_poll then
        return true
    end
    return false
end

function utils.is_valid_poll(poller, delay)
    delay = tonumber(delay)
    -- update the last checked for this poller
    -- helps us to check timer is running for this poller
    local last_poll_info = ngx.shared.last_poll_info
    local poll_time = ngx.time()
    last_poll_info:set(poller .. constants.CHECKED_AT, poll_time)
    if delay <= fast_poller_interval then
        return true
    end
    local last_poll_info = ngx.shared.last_poll_info
    local last_poll = last_poll_info:get(poller)
    if last_poll then
        ngx.log(ngx.DEBUG, "Poller for ", poller, ". Last poll at: ", last_poll)
        local elapsed_time = ngx.time() - last_poll
        if elapsed_time < (delay - fast_poller_interval)  then -- validate this atleast happening with valid dealy
            ngx.log(ngx.DEBUG, "Poll ignored: Less than ", delay ," seconds since last poll.")
            return false
        end
    else
        ngx.log(ngx.DEBUG, "Last poll is not defined for: ", poller)
    end
    return true
end

return utils

