#!/bin/bash

SCRIPTPATH=$(dirname ${0})

source ${SCRIPTPATH}/setenv.sh

TMP_FILE=$(mktemp -q /tmp/env.XXXXXX) || { echo "[ERROR] Failed to create temporary file. Exiting."; exitall 351; }

env_var_list=""

function prepareenv() {
	set -o allexport
	source ${1}
	env_var_list=$(env | grep '^NGX_' | column -t -s '=' | awk '{ print "${"$1"}" }' | tr "\n" ":")
	logInfoOnlyToFile "env_var_list to be substituted: ${env_var_list}"
	set +o allexport
}

function generate() {
	local TEMPLATE=${1}
	# Retain the values of environment vairable values passed for this script
	source $TMP_FILE >/dev/null 2>&1
	envsubst ${env_var_list} < ${TEMPLATE} > ${TEMPLATE}.tmp && mv -f ${TEMPLATE}.tmp ${TEMPLATE}
	# Configure SSL DHPARAM config if file exists
	if [[ ! -z ${NGX_SSL_DHPARAM} ]]; then
		local search_exp="[[:blank:]]*#[[:blank:]]*ssl_dhparam"
		sed -i'' -e "/^${search_exp}/s/${search_exp}/ssl_dhparam/g" ${TEMPLATE}
	fi
}

function main() {
	local ENV=${1}
	local TEMPLATE=${2}
	env | grep '^NGX' > "${TMP_FILE}"
	if [[ -f ${TEMPLATE} ]]; then
		prepareenv ${ENV}
		logInfoOnlyToFile "Generating config from template ${TEMPLATE} as per env ${ENV}"
		generate ${TEMPLATE}
	elif [[ -d ${TEMPLATE} ]]; then
		prepareenv ${ENV}
		for FILE in $(find ${TEMPLATE} -type f); do
			logInfoOnlyToFile "Generating config from template ${FILE} as per env ${ENV}"
			generate ${FILE}
		done
	else
		logInfoOnlyToFile "Unable to ascertain whether template ${TEMPLATE} is a file or directory."
	fi
	# :? is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
	rm -rf ${TMP_FILE:?}
}

function usage() {
	logInfoOnlyToFile "USAGE: sh -c 'generate_config.sh <ENV-FILE> <TEMPLATE-FILE/TEMPLATE-DIR>'"
	logInfoOnlyToFile "All params are mandatory"
	# :? is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
	rm -rf ${TMP_FILE:?}
}

[ "$#" -ne 2 ] && ( usage && exit 1 ) || main "$@"
