#!/bin/bash

#
# install_os_settings.sh: Installs os specific hardning configurations.
# Run this as root.
#
# Copyright: Cisco Systems Copyright (c) 2021 by Cisco
#
SCRIPTPATH=$(dirname ${0})

source "${SCRIPTPATH}/../common_utils.sh"
source "${SCRIPTPATH}/../iptables_utilities.sh"

LOG_FILE="${SCRIPTPATH}/install_os_settings.log"
ext_ip_config='true'
ALLOW_PRIVATE_SUBNETS='false'
ESCAPE_DDOS_RULES='false'
CONFIGURE_KERNEL='false'
CONFIGURE_IPTABLES='false'
DISABLE_IPTABLES='false'
NETWORK_FILE_PATH="/etc/reverse-proxy"
NETWORK_FILE="network.env"
OVERRIDE_EXTERNAL_INTERFACE_CONFIRMATION='false'
readonly IPTABLES_CONF='/etc/sysconfig/iptables'
readonly SYSCTL='/etc/sysctl.conf'

declare -a ALLOWED_HOSTS_FOR_PING
declare -a SSH_INTERFACES
declare -a RATELIMIT_BYPASS_LIST
declare -a EXTERNAL_INTERFACE
ALLOW_ICMP_PING_RULE="-A PREROUTING -p ICMP --icmp-type 8 -s SOURCE_IP -j ACCEPT -m comment --comment \" Added by Cisco Reverse Proxy Installer \""
ALLOW_ICMP_PING_RULE_MARKER="ALLOW_ICMP_PING_RULE_MARKER_START"

function logInfo()
{
   echo `date "+%m/%d/%Y %H:%M:%S :"` "$@"
   echo `date "+%m/%d/%Y %H:%M:%S :"` "$@" >> ${LOG_FILE}
}

function logError()
{
    local message="$(date "+%m/%d/%Y %H:%M:%S :") $@"
    log_error "$message"
    echo "$message" >> "${LOG_FILE}"
}

function logWarn()
{
     local message="$(date "+%m/%d/%Y %H:%M:%S :") $@"
     log_warn "$message"
     echo "$message" >> "${LOG_FILE}"
}

function usage() {
	echo "USAGE: ${0} [OPTIONAL_ARGS]"
	echo "OPTIONAL_ARGS: -k -e -a -i <external-interface> -y -p <source-ip1> -p <source-ip2> ..."
	echo "-k: configure kernel hardening"
	echo "-e: escape ddos iptables configuration. This option is ignored if -i or iptables configuration option is not given"
	echo "-a: allow private subnets in iptables configuration. This option is ignored if -i or iptables configuration option is not given"
	echo "-i: configure iptables hardening with given external interface."
	echo "-d: disable iptables hardening. This option should not be used in conjunction with -i, -e and -a options."
	echo "-y: escape external interface confirmation. This option is ignored if -i or iptables configuration option is not given"
	echo "-p: allowed source ip for ICMP ping messages. By default ICMP ping is blocked for all hosts. This option is ignored if -i or iptables configuration option is not given"
	echo ""
	echo "Example usage: ${0} -k -e -a -i eth1 -y -p allowed.host.for.ping.1 -p allowed.host.for.ping.2 "
	exit 1
}

function parse_args() {
	while getopts ":hkeadi:yp:" opt; do
		case ${opt} in
			h )
				usage
				;;
			k )
				CONFIGURE_KERNEL='true'
				;;
		  a )
		    if [ "${DISABLE_IPTABLES}" == "true" ]; then
          echo "Invalid option: option -d cannot be used with -i, -e or -a arguments."
          exit 1
        fi
		    CONFIGURE_IPTABLES='true'
        ALLOW_PRIVATE_SUBNETS='true'
        ;;
      e )
        if [ "${DISABLE_IPTABLES}" == "true" ]; then
          echo "Invalid option: option -d cannot be used with -i, -e or -a arguments."
          exit 1
        fi
        CONFIGURE_IPTABLES='true'
        ESCAPE_DDOS_RULES='true'
        ;;
			i )
			  if [ "${DISABLE_IPTABLES}" == "true" ]; then
          echo "Invalid option: option -d cannot be used with -i, -e or -a arguments."
          exit 1
        fi
				CONFIGURE_IPTABLES='true'
				EXTERNAL_INTERFACE+=("${OPTARG}")
				;;
		  d )
		    if [ "${CONFIGURE_IPTABLES}" == "true" ]; then
          echo "Invalid option: option -d cannot be used with -i, -e or -a arguments."
          exit 1
        fi
        DISABLE_IPTABLES='true'
        ;;
			p )
				ALLOWED_HOSTS_FOR_PING+=("${OPTARG}")
				;;
		  y )
		    OVERRIDE_EXTERNAL_INTERFACE_CONFIRMATION='true'
		    ;;
			\? )
				logInfo "Invalid option: -${OPTARG}" >&2
				usage
				;;
      ':')
        logInfo "MISSING ARGUMENT for option -- ${OPTARG}" >&2
				usage
        ;;
      *)
        logInfo "UNIMPLEMENTED OPTION -- ${opt}" >&2
				usage
        ;;
		esac
	done
	shift $((OPTIND -1))
}

function backup_file() {
	local FILE_TO_BACKUP=${1}
	local FORCE_BACKUP=${2}
	local BACKUP_FILE="${FILE_TO_BACKUP}.bak"
	logInfo "Taking backup of file ${FILE_TO_BACKUP} -> ${BACKUP_FILE}"
	if [ -f ${FILE_TO_BACKUP} ]; then
		if [ ! -f ${BACKUP_FILE} ] || [ "${FORCE_BACKUP}" == "true" ]; then
			\cp -fp ${FILE_TO_BACKUP} ${BACKUP_FILE}
			logInfo "File ${FILE_TO_BACKUP} backed up at ${BACKUP_FILE}"
		else
			logWarn "File ${FILE_TO_BACKUP} not backed up: Backup already exists and force backup was disabled"
		fi
	else
		logWarn "File ${FILE_TO_BACKUP} not backed up: File do not exist"
	fi
}

function is_almalinux() {
  if ! [ -f /etc/os-release ]; then
    return 1
  fi
  os_name=$(awk -F= '/^ID=/ {print $2}' /etc/os-release | tr -d '"')
  if [ "$os_name" = "almalinux" ]; then
    return 0
  fi
  return 1
}

function is_rhel94() {
  if ! [ -f /etc/os-release ]; then
    return 1
  fi
  os_name=$(awk -F= '/^ID=/ {print $2}' /etc/os-release | tr -d '"')
  if [ "$os_name" = "rhel" ]; then
	rhel_version=$(cat /etc/os-release | grep "VERSION_ID=\"9.4\"")
	if  [[ -z "$rhel_version" ]]; then
    	return 1
  	fi
    return 0
  fi
  return 1
}

function is_supported_kernel() {
  if ! [ -f /etc/os-release ]; then
    return 1
  fi
  rhel_version=$(cat /etc/os-release | grep "VERSION_ID=\"$1\"")
  if  [[ -z "$rhel_version" ]]; then
    return 1
  fi
  return 0
}

function is_iptables_installed_and_enabled() {
    if systemctl is-active --quiet iptables; then
        if systemctl is-enabled --quiet iptables; then
            return 0
        else
            logError "Iptables service is installed but not enabled."
            logError "Enable iptables service to configure iptable rules"
            exit 1
        fi
    else
        logError "Iptables service is not running."
        logError "Install iptables service to configure iptable rules"
        exit 1
    fi
}


function append_rules_after() {
	local marker="${1}"
	local line_to_add="${2}"
	local file="${3}"
	sed -i -e "/${marker}/a ${line_to_add}" "${file}"
}

function configure_iptables() {
	if is_iptables_installed_and_enabled; then
		local temp_file=$(mktemp)
		cat ${SCRIPTPATH}/iptables.conf >> ${temp_file}
		add_iptables_rules_for_ping ${temp_file}
		add_ddos_iptables_rules ${temp_file}
		backup_file ${IPTABLES_CONF} "true"
		logInfo "Updating iptables configuration"
		cat ${temp_file} > ${IPTABLES_CONF}
		# :? needed for all rm -rf as a preventive measure to not to delete the entire filesystem if the value is empty.
		rm -rf ${temp_file:?}
		iptables-restore < ${IPTABLES_CONF}
		if [ $? -ne 0 ]; then
		  logError "ERROR : Configuring Iptables failed."
		  exit 101
		fi
		logInfo "Iptables configurations loaded"
	fi
}

function add_ddos_iptables_rules(){

  if [[ "${ALLOW_PRIVATE_SUBNETS}" == "true" ]]; then
  		sed -i '/BLOCK_PRIVATE_SUBNETS_START/,/BLOCK_PRIVATE_SUBNETS_END/{/BLOCK_PRIVATE_SUBNETS_START/!{/BLOCK_PRIVATE_SUBNETS_END/!d}}' "${1}"
  fi
  if [[ "${ESCAPE_DDOS_RULES}" == "true" ]]; then
      sed -i '/DDOS_CONFIGURATION_START/,/DDOS_CONFIGURATION_END/{/DDOS_CONFIGURATION_START/!{/DDOS_CONFIGURATION_END/!d}}' "${1}"
  fi

  for interface in "${EXTERNAL_INTERFACE[@]}";
  do
    sed -i "s/EXTERNAL_INTERFACE/${interface}/g" "${1}"
  done
}

function add_iptables_rules_for_ping(){
  for source in "${ALLOWED_HOSTS_FOR_PING[@]}";
  do
    local rule=$(echo ${ALLOW_ICMP_PING_RULE} | sed "s~SOURCE_IP~${source}~g")
    logInfo "Adding iptables rule to allow ping: ${rule}"
    append_rules_after "${ALLOW_ICMP_PING_RULE_MARKER}" "${rule}" "${1}"
  done
}

# XDMCP provides unencrypted remote access through the Gnome Display Manager (GDM) which does not provide for the confidentiality 
# and integrity of user passwords or the remote session. If a privileged user were to login using XDMCP,
# the privileged user password could be compromised due to typed XEvents and keystrokes will traversing over
# the network in clear text.
function update_gdm_custom_conf() {
    local config_file="/etc/gdm/custom.conf"
    local section="[xdmcp]"
    local setting="Enable=false"

    if [[ ! -f "$config_file" ]]; then
        logError "Error: $config_file not found."
        return 1
    fi
    #Check if the setting already exists under [xdmcp]
    if grep -q -P "(?<=\[xdmcp\]\s*).*${setting}" "$config_file"; then
        logInfo "Setting already exists under [xdmcp]."
        return 0
    fi
    #Insert the setting after [xdmcp] section
    awk -v section="$section" -v setting="$setting" '
    $0 == section {
        print $0
        print setting
        next
    }
    { print $0 }
    ' "$config_file" > /tmp/custom.conf.tmp && mv /tmp/custom.conf.tmp "$config_file"
    logInfo "Updated /etc/gdm/custom.conf successfully."
}

# If an account has an empty password, anyone could log in and run commands with the privileges of that account. 
# Accounts with empty passwords should never be used in operational environments.
function remove_nullok() {
    local files=("/etc/pam.d/system-auth" "/etc/pam.d/password-auth")
    for file in "${files[@]}"; do
        if [[ ! -f "$file" ]]; then
            logError "Error: $file not found."
            continue
        fi
        sed -i 's/nullok//g' "$file"
        logInfo "Removed 'nullok' from $file."
    done
}

# Configuring this setting for the SSH daemon provides additional assurance that remote login via SSH will require a password,
# even in the event of misconfiguration elsewhere.
function update_permit_empty_passwords() {
    local config_file="/etc/ssh/sshd_config.d/00-complianceascode-hardening.conf"
    local setting="PermitEmptyPasswords no"

	if [[ -f "$config_file" ]]; then
		if grep -q "^PermitEmptyPasswords" "$config_file"; then
        	sed -i 's/^PermitEmptyPasswords.*/PermitEmptyPasswords no/' "$config_file"
        	logInfo "Updated 'PermitEmptyPasswords' to 'no' in $config_file."
			return 0
		fi
    fi
    echo "$setting" >> "$config_file"
    logInfo "Added 'PermitEmptyPasswords no' to $config_file."
}

function other_rhel9_specific_hardening() {
	# Kernel core dumps may contain the full contents of system memory at the time of the crash. 
	# Kernel core dumps consume a considerable amount of disk space and may result in denial of service by
	# exhausting the available space on the target file system partition. Unless the system is used for kernel 
	# development or testing, there is little need to run the kdump service.
	systemctl mask --now kdump.service

	
	# Centralized cryptographic policies simplify applying secure ciphers across an operating system and the 
	# applications that run on that operating system. 
	# Use of weak or untested encryption algorithms undermines the purposes of utilizing encryption
	# to protect data.
	update-crypto-policies --set DEFAULT:NO-SHA1

	update_gdm_custom_conf

	remove_nullok

	update_permit_empty_passwords
}

function configure_kernel() {
  backup_file ${SYSCTL}
  if is_almalinux; then
    cat ${SCRIPTPATH}/sysctl_almalinux.conf > ${SYSCTL}
    logInfo "Added kernel configurations to sysctl for AlmaLinux"
  elif is_rhel94; then
    cat ${SCRIPTPATH}/sysctl_rhel9.4.conf > ${SYSCTL}
    logInfo "Added kernel configurations to sysctl for RHEL"
	other_rhel9_specific_hardening
  elif is_supported_kernel "7"; then
    cat ${SCRIPTPATH}/sysctl_centos7.conf > ${SYSCTL}
    logInfo "Added kernel configurations to sysctl for centos7"
  elif is_supported_kernel "8"; then
    cat ${SCRIPTPATH}/sysctl_centos8.conf > ${SYSCTL}
    logInfo "Added kernel configurations to sysctl for centos8"
  else
    logInfo "Not a compatible OS"
    return 1
  fi
  sysctl -p -q
  logInfo "Reloaded kernel sysctl"
  return 0
}

function check_user_have_permissions_to_run_iptables_commands(){
  local user=$(whoami)
  local check_iptables_command=$(iptables --version | echo $?)
  local check_ip_command=$(ip -V | echo $?)
  if [ $check_iptables_command -ne 0 ] || [ $check_ip_command -ne 0 ]; then
    logError "ERROR : User ${user} does not have permissions to run iptables or ip command."
    exit 101
  fi
}

function exit_if_current_user_is_connected_from_external_interface() {

  local source_ip_of_current_ssh_session=$(echo $SSH_CLIENT | awk '{print $1}')
  local ips_of_external_interface=$(get_ips_of_interface ${EXTERNAL_INTERFACE})
  local pattern=$(echo ${ips_of_external_interface} | tr ',' '|' | sed 's/^/\\b/; s/$/\\b/')
  local count=$(ss -tnp | grep 'ssh' | grep -E "$pattern" | awk '{print $5}' | grep -c ${source_ip_of_current_ssh_session})
  if [ "$count" -gt 0 ]; then
  	logError "ERROR : Current shell session is connected through External/Internet facing Network Interface ${EXTERNAL_INTERFACE}. Please connect through Internal/Non-Internet facing Network Interface."
    exit 1
  fi
}

#
# Function to get number of IP addresses of ssh clients connected through external interface
#
function get_number_of_ssh_sessions_connected_through_external_interface() {
  local NUMBER_OF_SHELL_CLIENT_IPs_EXTERNAL_INTERFACE
  local ips_of_external_interface=$(get_ips_of_interface ${EXTERNAL_INTERFACE})
  local pattern=$(echo ${ips_of_external_interface} | tr ',' '|' | sed 's/^/\\b/; s/$/\\b/')
  NUMBER_OF_SHELL_CLIENT_IPs_EXTERNAL_INTERFACE=$(ss -tnp | grep 'ssh' | awk '{print $4}' | grep -E "$pattern" | wc -l)
  echo "$NUMBER_OF_SHELL_CLIENT_IPs_EXTERNAL_INTERFACE"
}

function external_interface_confirmation_or_exit() {
    local NUMBER_OF_SHELL_CLIENT_IPs_EXTERNAL_INTERFACE=$(get_number_of_ssh_sessions_connected_through_external_interface)
    logInfo "External/Internet facing Network Interface configured is ${EXTERNAL_INTERFACE}"
    logInfo "WARNING : Shell connections originating from this interface will be blocked and logins prevented, for security."
    logInfo "WARNING : Number of connected shell client IP addresses for given interface : "${NUMBER_OF_SHELL_CLIENT_IPs_EXTERNAL_INTERFACE}""
    logInfo "WARNING : Once this change is applied even ongoing connections will cease to work"
    logInfo "WARNING : Please ensure that your shell interface is through alternate interfaces before continuing"
    logInfo "Please confirm shell access from ${EXTERNAL_INTERFACE} can be blocked  (y/n)"
    read confirmation
    if [ "${confirmation}" != "y" ]; then
      logInfo "Entered ${confirmation}. Exiting"
      exit 100;
    fi
}

function check_external_interface_in_networkfile() {
	key="EXTERNAL_INTERFACE"
  	value=$EXTERNAL_INTERFACE
	
	if [ -e "$NETWORK_FILE_PATH/$NETWORK_FILE" ]; then
		if grep -q "$value" "$NETWORK_FILE_PATH/$NETWORK_FILE"; then
			logInfo "WARNING : $value already configured as the external interface"
			read -p "Do you want to re-run the IPTable configurations to secure $value? [y/n]" answer
			if echo "$answer" | grep -Eq '^[nN]$'; then
				logInfo "Not re-running Iptables to secure external interface $value"
				ext_ip_config='false'
			fi
		elif [ -s "$NETWORK_FILE_PATH/$NETWORK_FILE" ]; then
			read -p "External interface value already exists but is different. Do you want to change it? [y/n] " answer
			if echo "$answer" | grep -Eq '^[nN]$'; then
				logInfo "Not updating the external interface. It will continue to be $value"
				logInfo "Not re-running external interface security configurations"
				ext_ip_config='false'
			fi
		fi
	fi
}

function create_update_external_interface_in_networkfile() {
	if [ -e "$NETWORK_FILE_PATH/$NETWORK_FILE" ]; then
		if ! grep -q "$value" "$NETWORK_FILE_PATH/$NETWORK_FILE"; then	
			echo -n "" > "$NETWORK_FILE_PATH/$NETWORK_FILE"
			echo "$key=$value" >> "$NETWORK_FILE_PATH/$NETWORK_FILE"
			logInfo "External Interface value updated successfully"
		fi
	else
		mkdir -p $NETWORK_FILE_PATH
		touch $NETWORK_FILE_PATH/$NETWORK_FILE
		echo "$key=$value" >> "$NETWORK_FILE_PATH/$NETWORK_FILE"
		logInfo "External interface added in $NETWORK_FILE_PATH/$NETWORK_FILE"
	fi
}

function check_if_any_container_is_running(){
  local container_count=$(podman ps -q | wc -l)
  if [ $container_count -gt 0 ]; then
    logError "ERROR : There are ${container_count} container(s) running. Please stop all container(s) before applying iptables security hardening."
    exit 1
  fi
}

function disable_iptables_hardening(){
  iptables -t mangle -F
  rc=$?
  if [ $rc -ne 0 ]; then
    logError "ERROR : iptables hardening disable failed"
    exit 1
  fi
  backup_file ${IPTABLES_CONF} "true"
  echo -n > ${IPTABLES_CONF}
  logInfo "INFO : iptables hardening disabled successfully"
}

function main() {
	parse_args "$@"

	if [[ "${CONFIGURE_IPTABLES}" == "true" ]]; then
	  check_if_any_container_is_running
    check_interface_or_exit ${EXTERNAL_INTERFACE}
    EXTERNAL_INTERFACE=$(echo $EXTERNAL_INTERFACE | tr -d ' ')
		check_external_interface_in_networkfile
		if [[ "${ext_ip_config}" == "true" ]]; then
			is_ip_installed
			is_route_installed
			check_user_have_permissions_to_run_iptables_commands
			if [[ "${OVERRIDE_EXTERNAL_INTERFACE_CONFIRMATION}" == "false" ]]; then
				external_interface_confirmation_or_exit
			fi
			exit_if_current_user_is_connected_from_external_interface
			configure_iptables
      create_update_external_interface_in_networkfile
		fi
	fi
	if [[ "${CONFIGURE_KERNEL}" == "true" ]]; then
		configure_kernel
	fi
	if [[ "${DISABLE_IPTABLES}" == "true" ]]; then
	  check_user_have_permissions_to_run_iptables_commands
    disable_iptables_hardening
  fi
	logInfo "Configuration script has run successfully"	
}

[ "$#" -lt 1 ] && usage || main "$@"
