#!/bin/bash

SCRIPTPATH=$(dirname ${0})
source ${SCRIPTPATH}/common_utils.sh


# Compare the given 2 version, check which on is greater
# Returns -1 -> if v1 < v2
# Returns 0 -> if v1 == v2
# Returns 1 -> if v1 > v2
function compare_versions() {
    local version1
    local version2
    version1="$1"
    version2="$2"

    result=$(echo "${version1} > ${version2}" | bc -l)
    if [ $result -eq 1 ]; then
        echo "1"
        return
    fi
    result=$(echo "${version1} < ${version2}" | bc -l)
    if [ $result -eq 1 ]; then
        echo "-1"
        return
    fi 
    echo "0"
}

# Function to findout the list of upgrade functions to be called.
# Arg 1: version path as comma seperate string
# Arg 2: current version of the config file
# Arg 3: corresponding upgrade script 
# Find the upgrade route, if the arg1 is  "0.0,1.0,2.0,3.0,4.0" and arg2 is 2.0
# function returns the valid functions to be called to complete the upgrade ie: "upgrade_2dot0_to_3dot0 upgrade_3dot0_to_4dot0"
function get_upgrade_functions() {
    local version_path
    local current_version
    local upgrade_script

    version_path="$1"       # EG: "0.0,1.0,2.0,3.0,4.0"
    current_version="$2"    # Eg: "2.0"
    upgrade_script="$3"     # To valid the upgrade function is present

    IFS=',' read -ra versions <<< "${version_path}"

    local upgrade_functions=""
    local found_current_version=false


    for version in "${versions[@]}"; do
        if $found_current_version && [ "$(compare_versions "$current_version" "$version")" -lt 0 ]; then
            local function_name
            function_name="upgrade_${current_version//./dot}_to_${version//./dot}"
            if is_function_present_in_file "${function_name}" "${upgrade_script}"; then
                upgrade_functions+="${function_name} "
                # move to the next version, update current version value
                current_version="${version}"
            else
                log_error "The required function named ${function_name} is not present in upgrade script ${upgrade_script}. Exiting upgrade."
                exitall 238
            fi
        fi

        if [ "$(compare_versions "$version" "$current_version")" -eq 0 ]; then
            found_current_version=true
        fi
    done

    echo "$upgrade_functions"
}

# Checks the funcion name is present on the file or not
# Exit otherwise
function is_function_present_in_file() {
    local function_name
    local file_path

    function_name="${1}"
    file_path="${2}"

    if grep -q "^[[:space:]]*function[[:space:]]\+$function_name[[:space:]]*(" "$file_path"; then
        return 0
    fi
    return 1
}


# Function to update the existing property value
function update_property() {
    local env_file
    local property_name
    local new_value

    env_file="$1"
    property_name="$2"
    new_value="$3"

    # If given property is not present add it and return
    if ! is_property_in_file "$property_name" "$env_file"; then
        add_multiline_comment_and_property "$env_file" "" "$property_name" "$new_value"
        return 0
    fi
    
    if grep -q "^[[:space:]]*$property_name[[:space:]]*=" "$env_file"; then
        sed -i "s/^.*$property_name.*=.*/$property_name=\"$new_value\"/" "$env_file"
        return 0
    else
        return 1
    fi
}

# Function checks the given property is present on the property file
# Arg 1: property name to check
# Arg 2: property file name
# returns 0 if the property name is present on the file
function is_property_in_file() {
    local property_name
    local property_file
    property_name=${1}
    property_file=${2}
    if grep -q "^[[:space:]]*$property_name[[:space:]]*=" "$property_file"; then
        return 0
    fi
    return 1
}

# Function to add property with comment line inserting before that
# Arg 1: Property filename
# Arg 2: Multiline comments if any
# Arg 3: Property name
# Arg 4: Property value to be added
function add_multiline_comment_and_property() {
    local property_file
    local comment
    local property_name
    local property_value
    local noquotes

    property_file="$1"
    comment="$2"
    property_name="$3"
    property_value="$4"
    noquotes="$5"

    if is_property_in_file "$property_name" "$property_file"; then
        return 1
    else
        if [ -n "$comment" ]; then
            echo -e "$comment" >> "$property_file"
        fi
        if [ -n "$noquotes" ]; then
            echo "$property_name=\"$property_value\"" >> "$property_file"
        else
            echo "$property_name=$property_value" >> "$property_file"
        fi
        return 0
    fi
}

# Function to remove a line and add a property at the first line of a file
# Arg 1: Property filename
# Arg 2: Existing line to remove if already present 
# Arg 3: Property name
# Arg 4: Property value to be added
function add_or_update_config_version() {
    local file
    local line_to_remove
    local property_name
    local property_value

    file="$1"
    line_to_remove="$2"
    property_name="$3"
    property_value="$4"

    sed -i "/^[[:space:]]*$line_to_remove[[:space:]]*=/d" "$file"
    sed -i "1i$property_name=$property_value" "$file"
}

# Function to handle adding the IP address for the cloudconnect allowed list
# add the IP address only if the value already don't contain it
# Arg 1: Property filename
# Arg 2: Property name
# Arg 3: Property value to be added
function append_value_if_not_present() {
    local file
    local property
    local value
    file="$1"
    property="$2"
    value="$3"

    if is_property_in_file "$property" "$file"; then
        # Find the existing value of the property
        existing_value=$(grep "^[[:space:]]*$property[[:space:]]*=" "$file" | sed -E 's/^.*'"$property"'.*="(.*)"$/\1/')
        
        #If the given value is not present then update the property
        if [[ ! "$existing_value" == *"$value"* ]]; then
            # Append the new value to the existing value within the existing quotes
            updated_value="$existing_value|$value"
            
            # Update back the file with the new value (append)
            sed -i 's@^.*'"$property"'.*=.*$@'"$property=\"$updated_value\""'@' "$file"
        fi
    else
        # Property not exists, add it with the value within double quotes
        echo "$property=\"$value\"" >> "$file"
    fi
}

# Function to remove a property from file
# Arg 1: Property filename
# Arg 2: Property key to be removed from the file
function remove_property() {
    local file
    local property
    file="$1" 
    property="$2" 

    # Use sed to remove the line containing the property
    sed -i "/^.*$property.*=/d" "$file"
}


# Function to do the upgrade of the config files
function upgrade_configs_if_required() {
	local current_config_version_in_env_file
	local config_upgrade_file
	local template_type
	local version
	local DEPLOYMENT_ENV_DIR
	local INSTALLER_ENV
	INSTALLER_ENV="$1"
	DEPLOYMENT_ENV_DIR="$2"
	
	log_info "Checking for upgrades ..."

    # Mapping to temp file to original env file mapping
    # after all the upgrades are compete, do in one go.
    declare -A TMP_FILE_TO_ORG_FILE_MAPPING

	config_upgrade_file=$(get_upgrade_file "installer")
	log_info "Config verifying file $INSTALLER_ENV by $config_upgrade_file"
	check_and_upgrade_config "${INSTALLER_ENV}" "$config_upgrade_file"

	for env_file in ${DEPLOYMENT_ENV_DIR}/*.env; do		
		template_type=$(get_template_type "${env_file}" "${DEPLOYMENT_ENV_DIR}")
		config_upgrade_file=$(get_upgrade_file "${template_type}")
		log_info "Config verifying file $env_file by $config_upgrade_file"
		check_and_upgrade_config "${env_file}" "$config_upgrade_file"
	done
    if [ ${#TMP_FILE_TO_ORG_FILE_MAPPING[@]} -gt 0 ]; then
        log_info "Replacing existing env files with the upgraded envs ..."
    fi
	
    for key in "${!TMP_FILE_TO_ORG_FILE_MAPPING[@]}"; do
        log_info "Copying file from ${key} to ${TMP_FILE_TO_ORG_FILE_MAPPING[$key]}"
        yes | cp -Rfp "${key}" "${TMP_FILE_TO_ORG_FILE_MAPPING[$key]}"
    done

    log_info "Finished checking for upgrades ..."
}

# Function to do check and upgrade the config file
# Arg 1: Property filename
# Arg 2: Corresponding upgrade script
# Checks both the files are present before executing
function check_and_upgrade_config() {
	local env
	local config_upgrade_file
	local version
	local target_env_ver
	local upgrade_path
	local upgrade_functions
    local version_path
    local current_env_ver
	env=${1}
	config_upgrade_file=${2}

	current_env_ver=$(get_config_version "${env}")
	target_env_ver=$(get_max_version ${config_upgrade_file})

	if [ "$(compare_versions "${current_env_ver}" "${target_env_ver}")" -lt 0 ]; then
        log_info "Target config version from upgrade script {${config_upgrade_file}} is ${target_env_ver}"
		log_info "Found older config version. Performing automated env upgrade for file ${env}"
        local version_path=$(get_version_path "${config_upgrade_file}")
		upgrade_functions=$(get_upgrade_functions "${version_path}" "${current_env_ver}" "${config_upgrade_file}")
		log_info "Upgrade path found for env ${env}: ${upgrade_functions}"
		upgrade_to_latest_config "${config_upgrade_file}" "${env}" "${upgrade_functions}"
	else
        log_info "Env file – ${env} is on latest version ${current_env_ver} - OK."
	fi

}

# Get latest version for the component
# Arg 1: Property filename
# Returns config_version value from the config file
function get_config_version() {
	local file_path
	file_path=$1
	local version
	version=$(get_prop_val '^CONFIG_VERSION' "${file_path}")
	if [ -z "$version" ]; then
		version="0"
	fi
	echo ${version}
}

# Function to return the version path defined in the file
# Arg 1: Property/shell script filename
# Returns the VERSION_PATH value present on that file
function get_version_path() {
    local file_name
    file_name=$1
    local version_path
    version_path=$(get_prop_val_space_stripped 'VERSION_PATH' ${file_name})
    if [ -z "$version_path" ]; then
		version_path="0"
	fi
	echo ${version_path}
}

# Function to return the max version defined in version path
# Arg 1: Property/shell script filename
# Returns the max version value from the list of versions available in VERSION_PATH
function get_max_version() {
    local file_name
    local max_version
    file_name=$1
    local version_paths=$(get_version_path "$1")

    # Check if the input string is empty or contains only commas
    if [ -z "$version_paths" ] || [[ "$version_paths" == ,* ]]; then
        echo "0"
        return
    fi

    # Use awk to print the last field separated by commas
    echo "$version_paths" | awk -F',' '{print $NF}'
}

# Function to get the valid upgrade.sh file for the given component
# Arg 1: component name
# Returns the corresponding upgrade script for the component from migration directory
function get_upgrade_file() {
	local component_name
	component_name=${1}
	local file_path
	file_path="${REVERSE_PROXY_CONFIG_UPGRADE_SCRIPTS_DIR}/${component_name}_upgrade.sh"
    if [ ! -e "$file_path" ]; then
        log_error "Upgrade file not present for the given component: {${component_name}}, upgrade script path: {$file_path}. Exiting."
        exitall 256
    fi
	echo ${file_path}
}

# Function to do the upgrade to latest config
# Arg 1: Upgrade script file
# Arg 2: Env file to upgrade
# Arg 3: List of functions to be called on the upgrade script
function upgrade_to_latest_config() {
	local upgrade_file
	local list_of_functions
	local tmp_conf_file
	local env_file_to_upgrade


	upgrade_file=${1}
	env_file_to_upgrade=${2}
	list_of_function=${3}

	tmp_conf_file=$(get_new_temp_file)
	set_trap_with_preserve "cleanup ${tmp_conf_file}" EXIT

	yes | cp -Rfp "${env_file_to_upgrade}" "${tmp_conf_file}"
    TMP_FILE_TO_ORG_FILE_MAPPING["$tmp_conf_file"]="${env_file_to_upgrade}"
	(
        log_info "Sourcing file: ${upgrade_file}"
		source ${upgrade_file}
		for function in $list_of_function; do
			log_info "Executing function $function"
			eval "$function \"$tmp_conf_file\""
		done
        log_info "Upgrade successful for the env ${env_file_to_upgrade}"
	)
	
}

# Function to check if the given comma seperated input string is in ascending order
# Arg 1: Comma seperated string
# Functions checks the given string is in ascending order or not.
function is_sorted() {
    local input
    local len
    input=${1}
    
    IFS=',' read -ra arr <<< "$input"
    len=${#arr[@]}

    for ((i = 1; i < len; i++)); do
        if [ "$(compare_versions "${arr[i-1]}" "${arr[i]}")" -gt 0 ]; then
            return 1
        fi
    done
    return 0
}
