#!/bin/bash
##
# Copyright (c) Cisco Systems, Inc.
# All rights reserved.
#
# This script troubleshoot reverse proxy in a podman container
##

declare -A UPSTREAM_ENDPOINTS
UPSTREAM_ENDPOINTS['finesse']='/finesse/api/SystemInfo'
UPSTREAM_ENDPOINTS['chat']=''
UPSTREAM_ENDPOINTS['ids']='/ids/v1/status'
UPSTREAM_ENDPOINTS['cuic']='/cuic/rest/status'
#UPSTREAM_ENDPOINTS['livedata']=''
UPSTREAM_ENDPOINTS['cuic_1261']='/cuic/rest/status'
#UPSTREAM_ENDPOINTS['livedata_1261']=''
UPSTREAM_ENDPOINTS['cloudconnect']='/drapi/v1/ping'
UPSTREAM_ENDPOINTS['idp-adfs3']='/adfs/services/trust/mex'

declare -A REVERSE_PROXY_ENDPOINTS
REVERSE_PROXY_ENDPOINTS['finesse']='/desktop/api/DesktopConfig'
REVERSE_PROXY_ENDPOINTS['chat']=''
REVERSE_PROXY_ENDPOINTS['ids']='/ids/v1/status'
REVERSE_PROXY_ENDPOINTS['cuic']='/cuic/rest/status'
#REVERSE_PROXY_ENDPOINTS['livedata']=''
REVERSE_PROXY_ENDPOINTS['cuic_1261']='/cuic/rest/status'
#REVERSE_PROXY_ENDPOINTS['livedata_1261']=''
# For external CC, there is no publically available API to check health, using task api, which will return 403 response, that is fine to see the status of CC.
REVERSE_PROXY_ENDPOINTS['cloudconnect']='/drapi/v1/tasks'
REVERSE_PROXY_ENDPOINTS['idp-adfs3']='/adfs/services/trust/mex'

declare -A HOSTPORT_TEMPLATE_MAP
PROXY_CONTAINER_NAME=""
PROTOCOL="https://"
TROUBLESHOOT_UPSTREAM="check_upstream_status.sh"
COMMON_UTILS="common_utils.sh"
CONTAINER_HOME=/home

SCRIPTPATH=$(dirname ${0})
source ${SCRIPTPATH}/launcher_utilities.sh
source ${SCRIPTPATH}/common_utils.sh

# Function checks whether the proxy container name provided & is running
# show usage and exit if name is empty or null
function validate_container_name() {
	if test -z "${1}"
    	then
       		log_error "Container name not provided. Please provide the container name to complete troubleshooting."
        	usage
    	else
    		local message="Checking current run status of container ${1}"
        	# check if podman container is provided and is running
        	if  is_container_running ${1} > /dev/null ; then
        		log_info "${message} - LIVE"
        	else
        		log_error "${message} - FAILED"
				exitall 1
			fi
    fi
}

# Function makes curl request to upstream endpoint with write-out parameter and returns then response 
function invoke_api_request() {
	if [[ "${1}" == "finesse" ]]; then
		curl -k -s -m 20 -X GET -H "User-Agent: Mozilla/5.0 Troubleshoot Edg/1" "${2}"
	else
		curl -s -k -m 20 -I -X GET -H "User-Agent: Mozilla/5.0 Troubleshoot Edg/1" -w "%{http_code},%{time_total}" -o /dev/null "${2}"
	fi
}


function check_localip_in_allowed_cc_client_ips() {
	local dep_dir=$(get_deployment_env)

	# Find the value of NGX_CLOUDCONNECT_CLIENT_IPS from any file in the directory
	value=$(grep -rhoP '^NGX_CLOUDCONNECT_CLIENT_IPS=.*' "$dep_dir" 2>/dev/null)

	# Check if a value was found
	if [[ -z "$value" ]]; then
		log_warn "If cloudconnect is configured, please add the current system IP to the NGX_CLOUDCONNECT_CLIENT_IPS before starting the container for the cloudconnect troubleshooting to work properly." 
		return 1
	fi

	# Get the current local IP of the system dynamically
	current_ip=$(ip -4 route get 1.1.1.1 | awk '{print $7}' | head -n 1)

	# Ensure we got an IP
	if [[ -z "$current_ip" ]]; then
		log_warn "If cloudconnect is configured, please add the current system IP to the NGX_CLOUDCONNECT_CLIENT_IPS before starting the container for the cloudconnect troubleshooting to work properly."
		return 1
	fi

	# Convert value into an array using '|' as a delimiter
	IFS='|' read -ra ip_list <<< "$value"

	# Check if current_ip exists in the array
	for ip in "${ip_list[@]}"; do
		if [[ "$ip" == "$current_ip" ]]; then
			return 0
		fi
	done
	log_warn "If cloudconnect is configured, please add the current system IP to the NGX_CLOUDCONNECT_CLIENT_IPS before starting the container for the cloudconnect troubleshooting to work properly."
	return 1

}
# Function reads the endpoint env file and invokes curl on each endpoint available,
# the response is then used to get total time & status code, these data are then printed
function validate_api_response() {
	local response_time
	local http_status
	local response
	for line in $(cat "${1}" | grep "proxy")
	do
	# TODO: Skipping components due to endpoint unavailability, to be removed later
	if [[ "$line" == *"chat"* || "$line" == *"livedata"*  ]]; then
		continue
	fi
    	IFS='=' read -ra ARR <<< "${line}"
    	# remove prefix 'proxy_' from component name
    	local component="$( echo ${ARR[0]} | awk -F_ '{print $2}' )"
    	local hostname="$( echo "${ARR[1]}" | awk -F/ '{print $3}' )"
    	if [ $component == "finesse" ]; then
    		evaluate_finesse_status "$component" "$hostname" "${ARR[1]}"
    	else
			local response=$( invoke_api_request "" "${ARR[1]}" )
			response_time=$( get_response_time "$response" )
        	http_status=$( get_http_status "${response}" )
			if [[ $http_status == "403" && $component == "cloudconnect" ]]; then
				# We get 403 reponse for cc when the current IP is not in allowed list
				check_localip_in_allowed_cc_client_ips
			fi
        	print_response "External" "${http_status}" "${component}" "${hostname}" "${response_time}"
        fi
    done
}

# Function that evaluates finesse api response time, http status and configuration
#arg1: component name
#arg2: proxy hostname
#arg3: finesse endpoint
function evaluate_finesse_status() {
	local status_desc
	local response=$( invoke_api_request "" "${3}" )
	local response_time=$( get_response_time "$response" )
	local finesse_status=$( get_http_status "$response" )
	if [[ "$finesse_status" == "200" ]]; then
		status_desc=$( validate_finesse_configuration "${1}" "${2}" "${ARR[1]}" )
		# if finesse upstream config is not OK then ignore finesse http status so that it can show custom message.
		if [[ "$status_desc" != "OK" ]]; then
			finesse_status=""
		fi
	fi
    print_response "External" "${finesse_status}" "${1}" "${2}" "${response_time}" "${status_desc}"
}

#Function validates finesse upstream configuration 
#arg1: component name
#arg2: proxy hostname
#arg3: finesse endpoint
function validate_finesse_configuration() {
	local status_desc
	local response=$( invoke_api_request "${1}" "${3}" )
	local config_host=$( echo $response | sed -En 's/.*"host":"([^"]*).*/\1/p' )
	# {...otherprops...,"isProxyRequest":true} - this should match
	local isProxyRequest=$( echo $response | sed -En 's/.*"isProxyRequest":([^,}]*).*/\1/p' )
	if [[ $isProxyRequest == "true" && ${config_host,,} == "${2,,}" ]]; then
		status_desc="OK"
	else
		status_desc="Invalid finesse upstream proxy configuration."
	fi
	echo $status_desc
}

# Function gets the configured working directory for provided proxy container
function get_deployment_env() {
	local container_working_dir
	hostconfig_arr=$( podman container inspect -f '{{range .HostConfig.Binds}}{{printf "%s\n" .}}{{end}}' "${PROXY_CONTAINER_NAME}" )
	for dir in $hostconfig_arr
	do
		if [[ "$dir" == *"/etc/localtime"* ]]; then
			continue
		else
			IFS=":" read -a arr <<< $dir
			container_working_dir=$( echo "${arr[0]}" | sed 's/\(.*\)\/.*/\1/' )	# Note: we need remove last dir alone
			break
		fi
	done
	echo $container_working_dir
}

# Function creates temporary endpoint env file to be used for validating component status
function create_endpoint_file() {
	local tmp_endpoint_file=$(get_new_temp_file "/tmp/troubleshoot.XXXXXX")
	set_trap_with_preserve "cleanup ${tmp_endpoint_file}" EXIT
	echo $tmp_endpoint_file
}

# Function reads component files in envClone directory and builds endpoints for each component available
# Exits if the envClone directory is empty.
function process_component_env_files() {
	ENV_DIR=${1}/envClone
	# Check if working directory has some value
	check_dir_exist_or_exit "${ENV_DIR}"
	# As core.env and dirs.env is already processed so we are checking to be greater than 2
	if [ "$(ls -A ${ENV_DIR} | wc -l )" -gt 2 ]; then
		for ENV in ${ENV_DIR}/*.env; do
			if [ "${ENV}" == "${ENV_DIR}/core.env" ] || [ "${ENV}" == "${ENV_DIR}/dirs.env" ]; then
				continue
			else
				process_env_host_port "${ENV}" "$2"
			fi
		done
	else
    	log_error "No component env configs provided in ${ENV_DIR}, Exiting."
    	exitall 1
	fi
}

# Check if duplicate host:port exists, exit if duplicate entry found.
function validate_hostport(){
	local hostport=$1
	local template_type=$2
	local existing_val=${HOSTPORT_TEMPLATE_MAP[$hostport]}
	if [ -v ${HOSTPORT_TEMPLATE_MAP[$hostport]} ]; then
		HOSTPORT_TEMPLATE_MAP[$hostport]="${template_type}"
	else
		log_error "Error while processing ${template_type} env, ${hostport} is already configured in component '${existing_val}'."
		exitall 1
	fi
}

# Function reads the host and port for both upstream and proxy for provided component
# if port not found in component env will get default port from VALID_UPSTREAM_PORTS map
# exit if duplicate host:port entry found
# builds the endpoint which will be written into file in troubleshoot working directory
function process_env_host_port() {
	local ENV=${1}
	local template_type=$(get_prop_val '^TEMPLATE_TYPE' ${ENV})
	local result_template_type=$(get_component $template_type)
	local hostnamekey=$(echo "NGX_${result_template_type}_HOSTNAME")
	local proxy_hostname_key=$(echo "NGX_PRXY_${result_template_type}_HOSTNAME")
	local portkey=$(echo "NGX_${result_template_type}_PORT")
	local proxy_port_key=$(echo "NGX_PRXY_${result_template_type}_PORT")
	local hostname=$(get_prop_val "^${hostnamekey}" ${ENV})
	local proxy_hostname=$(get_prop_val "^${proxy_hostname_key}" ${ENV})
	local port=$(get_prop_val "^${portkey}" ${ENV})
	local proxy_port=$(get_prop_val "^${proxy_port_key}" ${ENV})
	# check if port is null then set from map
	if [ -z "$port" ]; then
		port=$(echo "${VALID_UPSTREAM_PORTS[$template_type]}" | cut -d',' -f1)
	elif [ -z "$proxy_port" ]; then
		proxy_port=$(echo "${VALID_UPSTREAM_PORTS[$template_type]}" | cut -d',' -f1)
	fi
	local hostport="$hostname:$port"
	local proxy_hostport="$proxy_hostname:$proxy_port"
	# Check if duplicate host:port exists
	validate_hostport "$hostport" "$template_type"
	validate_hostport "$proxy_hostport" "$template_type"
	echo "upstream_${template_type}=${PROTOCOL}${hostport}${UPSTREAM_ENDPOINTS[$template_type]}" >> $2
	echo "proxy_${template_type}=${PROTOCOL}${proxy_hostport}${REVERSE_PROXY_ENDPOINTS[$template_type]}" >> $2
}

# Function copies the necessary env and script files into proxy container and
# executes the script to validate upstreams
function validate_upstream_components() {
	# copy the env & script files to container
	podman cp ${1} ${PROXY_CONTAINER_NAME}:${CONTAINER_HOME}
	podman cp ${TROUBLESHOOT_UPSTREAM} ${PROXY_CONTAINER_NAME}:${CONTAINER_HOME}
	podman cp ${COMMON_UTILS} ${PROXY_CONTAINER_NAME}:${CONTAINER_HOME}
	local endpoint_file="$(echo ${1} | awk -F/ '{print $3}' )"
	# execute troubleshoot_upstream.sh with upstream env
	podman exec -it ${PROXY_CONTAINER_NAME} bash -c "sh ${CONTAINER_HOME}/${TROUBLESHOOT_UPSTREAM} ${endpoint_file}"
}

# Function validates the reverse proxy components with provided endpoint env file
function validate_reverse_proxy_components() {
	echo "************************* Checking External Configuration *************************"
	print_header
	validate_api_response "${1}"
	echo "***********************************************************************************"
}

function usage() {
	echo "USAGE: ${0} [options...] "
	echo "Options: -r <PROXY-CONTAINER-NAME>"
	echo "PROXY-CONTAINER-NAME is a mandatory parameters to begin troubleshooting"
	echo "Example usage: ${0} -r reverse-proxy.cisco.com"
	exit 1
}

function parseArgs() {
	if [ "$#" -eq 0 ]; then
  	usage
	fi
	while getopts ":r:" opt; do
		case "${opt}" in
			r )
				validate_container_name "${OPTARG}"
				PROXY_CONTAINER_NAME="${OPTARG}"
				;;
			\? )
				log_info "Invalid option: ${OPTARG}"
				usage
				exitall 100
				;;
			: )
				log_info "Invalid value: -${OPTARG} requires a value."
				usage
				exitall 100
				;;
		esac
	done
}

main() {
	parseArgs "$@"
	deployment_env_dir=$(get_deployment_env)
	# Check if working directory exist
    check_dir_exist_or_exit "${deployment_env_dir}"
	# Create troubleshoot directory with required permission
	tmp_endpoint_file=$(create_endpoint_file)
	# read env files for provided components and create component-host mapping file
	process_component_env_files "${deployment_env_dir}" ${tmp_endpoint_file}
	# invoke endpoints from within the container
    validate_upstream_components ${tmp_endpoint_file}
    # invoke endpoints with proxy host
    validate_reverse_proxy_components ${tmp_endpoint_file} 
}
main "$@"
